#
# Copyright (C) 2010 Alexander Taler <dissent@0--0.org>
#

# This file is part of hsh.

# hsh is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# hsh is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with hsh.  If not, see <http://www.gnu.org/licenses/>.

######################################################################

# The format of the user configuration file is to be determined.

import logging

import hsh.jobs

from hsh.exceptions import *
from view import View
from job_view import JobView

class ListView(View, hsh.jobs.JobManagerListener, hsh.jobs.JobListener):
    """Displays a list of job information.  Instances need to be configured by
    specifying a format for display and a function for filtering those to
    display.

    It contains a list of job display objects, and tracks and highlights a
    focussed one.
    """

    # self.jobs is a list of jobs.Job objects.  curjob is an index into
    # self.jobs, which specifies the currently highlighted job, or None if
    # there are no jobs in the list.

    # Drawing of the job output is controlled by two variables: tail and
    # disp_pos.  disp_pos indicates which line of the window should anchor the
    # drawing of the current job, and tail indicates if it should be anchoring
    # the head or tail of the job.  disp_pos can be negative if the anchored
    # bit is in fact not currently visible.

    def __init__(self, display, name, job_filter, display_format,
                 header="", header_face="list.header", show_job_output=False,
                 sync_main_job=False,
                 highlight_face="list.highlight",
                 jobheader_face="list.jobheader"):
        if name:
            self.name = name
        super(ListView, self).__init__(display)
        self.jobs = []
        self.curjob = None
        self.disp_pos = 0
        self.tail = False
        self.job_filter = job_filter
        self.header_fmt = header
        self.header_face = self.display.faces[header_face]
        self.highlight_face = self.display.faces[highlight_face]
        self.jobheader_face = self.display.faces[jobheader_face]
        self.display_format = display_format
        self.show_job_output = show_job_output
        self.sync_main_job = sync_main_job
        self.last_curjob = None
        # job_draw_lines tracks where jobs were drawn during the last draw.
        # keys are indices in the jobs array, values are pairs: (the screen
        # line where the job started, the number of lines in the job's output).
        # Jobs which weren't drawn in the last pass won't be included.
        self.job_draw_lines = {}
        # If show_tail is True, then disp_pos will be adjusted to ensure the
        # tail of the current job is visible.
        self.show_tail = False

        # Line to indicate some output has not be drawn.
        self.trunc_line = hsh.content.TextLine()
        rg = self.trunc_line.append_region(' ...\n')
        rg.face = display.faces["wrapmark"]

        # Updated with each draw to reflect the size of the window.
        self.cur_width = 1
        self.cur_height = 1

        hsh.jobs.manager.register_listener(self)

    def set_visible_job(self, job):
        # Called when the main visible job has changed, in case the view wants
        # to update internal structures to acknowledge this.
        if self.sync_main_job:
            for i in range(len(self.jobs)):
                if self.jobs[i] == job:
                    self._set_curjob(newjob=i)
                    return
            self._set_curjob(newjob=None)

    ######################################################################
    # JobManagerListener interface

    def on_add_job(self, job):
        """Add a single job to the list."""
        if not self.job_filter(job):
            return

        job.register_listener(self)

        # Insert the new job into the sorted list.
        i = 0
        while i < len(self.jobs):
            if job.jid < self.jobs[i].jid:
                self.jobs[i:i] = [job]
                break
            i += 1
        if i == len(self.jobs):
            self.jobs.append(job)

        if job.is_new:
            # Update disp_pos for the change to curjob, so that everything
            # stays in the same place on the screen.
            if (i - 1) in self.job_draw_lines:
                dp = self.job_draw_lines[i - 1]
                self.disp_pos = dp[0] + dp[1]
                if self.disp_pos >= self.cur_height:
                    self.disp_pos = self.cur_height - 2
            else:
                self.disp_pos = 0
            self.tail = False
            self.show_tail = True

        # Update cursor and display.
        if job.is_new or self.curjob is None:
            self._set_curjob(newjob=i)
        elif self.curjob >= i:
            self._set_curjob(next=True, setdisp=True)

    def on_remove_job(self, job):
        """A job is removed from the manager, so remove it from the list."""
        job.unregister_listener(self)
        for i in xrange(len(self.jobs)):
            if job.jid == self.jobs[i].jid:
                del self.jobs[i]
                if len(self.jobs) == 0:
                    self._set_curjob(newjob=None)
                # Current job is an index into the array, so needs to be
                # updated if it's on or after the deleted job.
                elif self.curjob is not None and self.curjob >= i:
                    self._set_curjob(prev=True, setdisp=True)
                self.set_dirty()
                return

    ######################################################################
    # jobs.JobListener interface

    def on_job_output(self, job):
        if job in self.jobs and not self.job_filter(job):
            self.on_remove_job(job)
        self.set_dirty()

    def on_job_terminate(self, job):
        if job in self.jobs and not self.job_filter(job):
            self.on_remove_job(job)
        self.set_dirty()

    def on_job_raw_output(self, job, text, channel):
        if job in self.jobs and not self.job_filter(job):
            self.on_remove_job(job)
        self.set_dirty()

    ######################################################################
    # Job List specific functions

    def _load_job(self):
        """Ask the job manager to load a new job on disk, which will be
        inserted at the front of the list.  Return the loaded job or None."""
        # The job manager fetch calls the JobListener interface to add the job
        # to self.jobs.  However, the job may not be filtered out, so keep
        # trying if it didn't appear.  But don't waste too long searching for
        # an older job, give up if a match doesn't come quickly.
        if len(self.jobs) == 0:
            fj = hsh.jobs.manager.get_last_job()
        else:
            fj = hsh.jobs.manager.get_prev_job(self.jobs[0].jid)

        tc = 0
        while fj and (len(self.jobs) == 0 or self.jobs[0] != fj):
            fj = hsh.jobs.manager.get_prev_job(fj.jid)
            tc += 1
            if tc == 100:
                return None

        return fj

    def _set_curjob(self, newjob=None, next=False, prev=False, setdisp=False):
        """Change self.curjob, and update global state, but don't update
        display.  newjob is the new value for curjob, which can be None if the
        job list is empty. If next or prev is True then newjob is ignored and
        curjob will be moved forward or back as appropriate.  If setdisp is
        True the disp_pos will be updated for the change."""
        oldcurjob = self.curjob
        if len(self.jobs) == 0:
            self.curjob = None
        elif next:
            if self.curjob is None:
                self.curjob = 0
            elif self.curjob + 1 < len(self.jobs):
                self.curjob += 1
        elif prev:
            if self.curjob is None:
                self.curjob = 0
            elif self.curjob > 0:
                self.curjob -= 1
        elif newjob >= 0 and newjob < len(self.jobs):
            self.curjob = newjob
        if setdisp:
            if self.curjob in self.job_draw_lines:
                self.disp_pos = self.job_draw_lines[self.curjob][0]
                if self.disp_pos < 0:
                    self.disp_pos = 0
                if self.disp_pos > self.cur_height - 2:
                    self.disp_pos = self.cur_height - 2
            else:
                self.disp_pos = 0
            self.tail = False
            self.show_tail = False
        # Make sure the job is shown in the job view.
        if self.sync_main_job and oldcurjob != self.curjob:
            self.display.display_job(self.current_job())

    def current_job(self):
        """Return the current job as a hsh.jobs.Job object."""
        if self.curjob is None:
            return None
        else:
            return self.jobs[self.curjob]

    def get_predecessor(self, job):
        """Given a job, fetch one from the list which falls before it based on
        jobid.  Return None if no predecessor could be found.  Load jobs from
        disk if needed."""
        # Load jobs backwards to an appropriate point.
        while len(self.jobs) == 0 or job.jid <= self.jobs[0].jid:
            rj = self._load_job()
            if rj is None:
                return

        # Search the list for its immediate predecessor.
        last_job = None
        for i_job in self.jobs:
            if i_job.jid < job.jid:
                last_job = i_job
        return last_job

    def get_successor(self, job):
        """Given a job, fetch one from the list which falls after it based on
        jobid.  Return None if no successor could be found.  Load jobs from
        disk if needed."""
        # If job is before the beginning of the list, load some earlier jobs.
        while len(self.jobs) == 0 or job.jid < self.jobs[0].jid:
            rj = self._load_job()
            if rj is None:
                break

        for i_job in self.jobs:
            if i_job.jid > job.jid:
                return i_job

    def _format_job_header(self, drawjob, height, width):
        """Internal function to format a job's header for drawing."""
        if drawjob == self.curjob:
            jhface = self.highlight_face
        else:
            jhface = self.jobheader_face
        jd = self.display.get_job_view(self.jobs[drawjob])
        rawh = (self.display_format % jd.header_info()).split('\n')
        return View.format_lines(self.display, rawh, height, width,
                                 face=jhface, wrap=False)

    def set_face(self, region):
        """Given a region, set its face."""
        super(ListView, self).set_face(region)
        if "source" in region.__dict__:
            region.face = self.display.faces["job." + region.source]

    def draw(self, win, force_redraw):

        if (not force_redraw and not self.is_dirty() and 
            self.curjob == self.last_curjob):
            return False

        self.set_dirty(None)
        self.last_curjob = self.curjob
        self.job_draw_lines = {}

        self.draw_header(win)

        cwin = self._content_win(win)
        cwin.leaveok(1)
        cwin.clear()

        (height, width) = cwin.getmaxyx()
        self.cur_width = width
        self.cur_height = height

        if self.curjob is None:
            cwin.refresh()
            cwin.leaveok(0)
            return True

        ##########
        # Assemble the formatted lines of the job list view into an array.

        # This code is placed in a loop, separate from the actual drawing
        # process, so that if things need to be changed, the loop can be
        # restarted without affecting the screen.

        # The drawing is anchored around a line, anchor, and the formatted
        # output separated into two lists, prelines for output before anchor,
        # and postlines for output on and after it.  anchor is always the first
        # line of output for the anchorjob.

        # anchor and anchorjob are derived from curjob, disp_pos, and tail, but
        # always refer to the top line of a job, even when tail is True.

        # The process for assembling formatted output is:

        #   * Loop through jobs before anchor job, assembling enough lines to
        #     fill space before anchor.
        #   * Make sure there's no blank lines at the top
        #   * If the current job isn't visible, choose a new visible one and 
        #     start again.
        #   * Loop through jobs after anchor job, assembling enough lines to
        #     fill space after anchor.
        #   * If the current job isn't visible, choose a new visible one and 
        #     start again.

        while True:
            # Assembled formatted output.
            prelines = hsh.content.TextContent()
            postlines = hsh.content.TextContent()

            if self.tail:
                if self.show_tail and self.disp_pos >= height:
                    self.disp_pos = height - 1
                if self.show_tail and self.disp_pos < 0:
                    self.disp_pos = 0
            # Determine the job anchored at the top, and its top line.
            if not self.tail:
                anchor = self.disp_pos
                anchorjob = self.curjob
            else:
                anchor = self.disp_pos + 1
                anchorjob = self.curjob + 1

            #####
            # Preceding jobs
            drawjob = anchorjob - 1
            drawline = anchor - 1  # number of lines left to draw
            while drawjob >= 0 and drawline >= 0:
                # Calculate the job header
                flines = self._format_job_header(drawjob, drawline + 1, width)

                # Include job output if necessary
                if self.show_job_output:
                    # Ugh: wasteful recalculation of formatted content.
                    jd = self.display.get_job_view(self.jobs[drawjob])
                    jolines = View.format_lines(self.display, jd.content,
                                                drawline + 1, width,
                                                tail=(jd.show_output == -1))

                    if 0 <= jd.show_output and jd.show_output < len(jolines):
                        jolines[jd.show_output:] = [self.trunc_line]

                    flines += jolines

                self.job_draw_lines[drawjob] = \
                    (anchor - len(prelines) - len(flines), len(flines))
                prelines[0:0] = flines
                drawjob -= 1
                drawline -= len(flines)

            # Make sure there's no blank space at the top
            if len(prelines) < anchor:
                self.disp_pos -= (anchor - len(prelines))
                anchor -= (anchor - len(prelines))

            #####
            # Post jobs output
            drawjob = anchorjob
            drawline = anchor
            while drawjob < len(self.jobs) and drawline <= height:
                # Start with the header:
                flns = self._format_job_header(drawjob,height-drawline+1,width)

                # Include job output if necessary
                if self.show_job_output:
                    # Ugh: wasteful recalculation of formatted content.
                    jd = self.display.get_job_view(self.jobs[drawjob])
                    hl = height - drawline - len(flns) + 1
                    jolns = View.format_lines(self.display,jd.content,hl,width)

                    if 0 <= jd.show_output and jd.show_output < len(jolns):
                        jolns[jd.show_output:] = [self.trunc_line]

                    flns += jolns

                self.job_draw_lines[drawjob] = \
                         (anchor + len(postlines), len(flns))
                postlines.extend(flns)
                drawline += len(flns)
                drawjob += 1

            # If nothing's visible, don't let them move the display any further.
            if anchor + len(postlines) <= 0:
                logging.debug("nothing visible")
                self.disp_pos += 1 - (anchor + len(postlines))
                # Continue the main loop to calculate again.
                continue

            # Check for visibility of current job.
            firstvisijob = lastvisijob = None

            jobids = self.job_draw_lines.keys()
            jobids.sort()
            for job in jobids:
                jbounds = self.job_draw_lines[job]
                if jbounds[0] < height and jbounds[0] + jbounds[1] >= 0:
                    # The job is visible
                    if firstvisijob is None:
                        firstvisijob = job
                    lastvisijob = job
            
            # If curjob's not visible, change it and continue the main loop.
            if self.curjob < firstvisijob and firstvisijob is not None:
                logging.debug("Switching curjob: %s -> %s" %
                              (self.curjob, firstvisijob))
                self._set_curjob(newjob=firstvisijob, setdisp=True)
                continue
            if self.curjob > lastvisijob and lastvisijob is not None:
                logging.debug("Switching curjob: %s -> %s" %
                              (self.curjob, lastvisijob))
                self._set_curjob(newjob=lastvisijob)
                if self.curjob in self.job_draw_lines:
                    cjdl = self.job_draw_lines[self.curjob]
                    self.disp_pos = cjdl[0] + cjdl[1] - 1
                else:
                    self.disp_pos = height - 1
                self.tail = True
                continue

            if self.show_tail:
                # Ensure that the tail of the current job is visible.
                cjdl = self.job_draw_lines[self.curjob]
                if cjdl[0] + cjdl[1] > height:
                    self.tail = True
                    self.disp_pos = height - 1
                    logging.debug("Repositioning for tail to be visible.")
                    continue

            # Finished, so break the loop
            break

        #####
        # Actual drawing
        for j in range(height):
            offset = j - anchor
            if offset >= len(postlines):
                break
            if offset < 0:
                self.draw_line(prelines[offset], j, cwin)
            else:
                self.draw_line(postlines[offset], j, cwin)

        cwin.refresh()
        cwin.leaveok(0)
        return True

    ######################################################################
    # Functions bound to keystrokes

    # Functions bound to keystrokes which override parent's functions.
    def move_up(self, ki):
        if not self.show_job_output:
            self.prev_job(ki)
        else:
            self.disp_pos += ki.num
        self.show_tail = False

    def move_down(self, ki):
        if not self.show_job_output:
            self.next_job(ki)
        else:
            self.disp_pos -= ki.num

    def move_job_top(self, ki):
        logging.debug("move_job_top")
        # Move to the top of the current job.
        if self.curjob is None or self.curjob not in self.job_draw_lines:
            return
        if self.job_draw_lines[self.curjob][0] >= 0:
            return
        self.disp_pos = 0
        self.show_tail = False
        self.tail = False

    def move_job_bottom(self, ki):
        # Move to the bottom of the current job.
        if self.curjob is None or self.curjob not in self.job_draw_lines:
            return
        if self.job_draw_lines[self.curjob] <= self.cur_height:
            return
        self.disp_pos = self.cur_height - 1
        self.show_tail = True
        self.tail = True

    def page_up(self, ki):
        if not self.show_job_output:
            ki.num *= self.cur_height / len(self.display_format.split('\n')) 
            self.prev_job(ki)
        else:
            ki.num *= self.cur_height
            self.move_up(ki)
        self.show_tail = False

    def page_down(self, ki):
        if not self.show_job_output:
            ki.num *= self.cur_height / len(self.display_format.split('\n')) 
            self.next_job(ki)
        else:
            ki.num *= self.cur_height
            self.move_down(ki)

    def hide_output(self, ki):
        if self.curjob is not None:
            cj = self.display.get_job_view(self.jobs[self.curjob])
            if cj.show_output == -1:
                cj.show_output = 3
            elif cj.show_output == 3:
                cj.show_output = 0
            elif cj.show_output == 0:
                cj.show_output = -1
            self.tail = False
            self.show_tail = False

    def next_job(self, ki):
        for i in range(ki.num):
            if len(self.jobs) == 0:
                return
            self._set_curjob(next=True, setdisp=True)

    def prev_job(self, ki):
        for i in range(ki.num):
            if (not self.curjob):
                lj = self._load_job()
                if lj is None:
                    return # Couldn't find an earlier job
            self._set_curjob(prev=True, setdisp=True)

    def restart_job(self, ki):
        self.display.get_job_view(self.current_job()).restart()

    def select_job(self, ki):
        self.display.display_job(self.current_job(), show=True)

    def input_command(self, ki):
        # Copy the current job's command line to input
        self.display.views["InputView"].insert_text(self.current_job().cmdline)
        self.display.give_focus(self.display.views["InputView"])
        self.display.views["InputView"].set_dirty()

    def delete_job(self, ki):
        # Request to delete the current job.  This is escalated to the job
        # manager, and the on_remove_job() callback handles the work.
        dj = self.current_job()
        if dj is None:
            return
        if dj.get_state() == "Running":
            self.display.show_alert("Can't delete an active job")
            return
        hsh.jobs.manager.forget_job(dj)
