#
# Copyright (C) 2010 Alexander Taler <dissent@0--0.org>
#

# This file is part of hsh.

# hsh is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# hsh is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with hsh.  If not, see <http://www.gnu.org/licenses/>.

######################################################################

import logging
import os

import hsh.jobs
import hsh.command
import hsh.config
import hsh.system

from hsh.content.text import TextList

from hsh.exceptions import *
from content_view import ContentView

class InputView(ContentView):
    def __init__(self, display):
        super(InputView, self).__init__(display, TextList([""]))
        self.textf.set_format(wrap = False)
        self.editable = True
        self.header_fmt = hsh.config.header_input
        # Initialize header_size with something sensible
        self.header_size = len(self.header_fmt.split('\n'))

        # Parse tree object for the user's typed command.
        self.command_text = hsh.command.CommandToken()

        # _history_past_end is True when forward_history() (the down arrow) has
        # been used to go past the last entry on the history list, and the user
        # sees a blank line again.
        self._history_past_end = False

    def header_info(self):
        hi = {'username' : hsh.system.get_username(),
              'hostname' : hsh.system.get_hostname(),
              'cwd'      : hsh.system.get_cwd(),
              'expanded_cmd' : hsh.command.format_cmd(self.command_text.expand()),
              'expanded_arg' : "N/A",
              }
        cs = self.command_text.completions(self.textf.cursor_pos[1])
        if cs:
            cs = map(lambda x: cs[0] + x, cs[1:])
        else:
            cs = []
        hi['completions'] = " ".join(cs)

        return hi

    def insert_text(self, text):
        self.command_text.insert(self.textf.cursor_pos[1], text)
        super(InputView, self).insert_text(text)

    # Functions bound to keys

    # First batch are for editing, and call super.  Note that commandtoken
    # doesn't represent multiple lines (currently) so they don't bother with
    # multiline conversions.
    def delete_left(self, ki):
        if self.textf.cursor_pos[1] > 0:
            self.command_text.delete(self.textf.cursor_pos[1] - 1)
        super(InputView, self).delete_left(ki)

    def delete_right(self, ki):
        if len(self.command_text) > self.textf.cursor_pos[1]:
            self.command_text.delete(self.textf.cursor_pos[1])
        super(InputView, self).delete_right(ki)

    def delete_line(self, ki):
        self.command_text.delete(self.textf.cursor_pos[1],
                                 len(self.command_text) - self.textf.cursor_pos[1])
        super(InputView, self).delete_line(ki)

    # This batch of bound functions do not call super.
    def do_command(self, ki):
        # Create a new job - usually Enter key.
        self._history_past_end = True
        self._do_command(start=True)

    def cancel(self, ki):
        # Cancel editing, but store the command line in history.
        self._history_past_end = True
        self._do_command(start=False)

    def _do_command(self, start=True):
        try:
            j = hsh.jobs.manager.create_job(self.command_text, start)
        except JobCreationException, jce:
            if not (not start and str(jce) == "No program to execute"):
                self.display.show_alert(str(jce))
            return

        if start and not "dontchangeview" in j.get_directives():
            self.display.display_sessionlist()

        # Switch focus away from a suspended exclusive job so control isn't
        # just given right back to.
        if j.get_state() == "Suspended":
            self.display.next_window()

        self._clear_text()

    def _clear_text(self):
        self.command_text = hsh.command.CommandToken()
        self.textf.set_format(trim = 0)
        self.textf.move_top()
        self.text[:] = [""]
        self.set_dirty()

    def _move_history(self, ki, forward):
        hist = self.display.get_view("CommandHistory")

        # If what's currently typed on the command line isn't blank and doesn't
        # match the current history item, add it to the history before loading
        # the next command line.
        histjob = hist.current_job()
        histlastjob = hist.last_job()
        curtxt = str(self.command_text)
        if (curtxt != "" and (histjob is None or histjob.cmdline != curtxt)):
            self._do_command(start=False)
            # Restore original history position
            hist.set_current_job(histjob)
        else:
            self._clear_text()

        for i in range(ki.num):
            ki.num = 1
            if forward:
                if histjob == histlastjob:
                    # At end of history, so go to a blank line
                    self._history_past_end = True
                else:
                    hist.next_job(ki)
            else:
                if (not self._history_past_end or histjob != histlastjob):
                    hist.prev_job(ki)
                self._history_past_end = False
            histjob = hist.current_job()
            histlastjob = hist.last_job()
        hist.set_dirty()

        if hist.current_job() is not None and not self._history_past_end:
            self.insert_text(hist.current_job().cmdline)
        

    def back_history(self, ki):
        self._move_history(ki, False)

    def forward_history(self, ki):
        self._move_history(ki, True)

    def main_handle(self, ki):
        self.display.get_main_view().putch(ki.key, ki.meta)

    def tab_complete(self, ki):
        cl = self.command_text.completions(self.textf.cursor_pos[1])
        if cl is None or len(cl) <= 1: return
        cl = cl[1:]
        # Since completions are sorted, the substring shared by the first and
        # last is shared by all.
        app = ""
        for i in range(len(cl[0])):
            if cl[0][i] != cl[-1][i]:
                break
            app += cl[0][i]
        if len(app) > 0:
            self.insert_text(app)

    def dump_parse_tree(self, ki):
        logging.info("\n" + "".join(self.command_text.dump()))
