#
# Copyright (C) 2010 Alexander Taler <dissent@0--0.org>
#

# This file is part of hsh.

# hsh is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# hsh is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with hsh.  If not, see <http://www.gnu.org/licenses/>.

######################################################################

# The format of the user configuration file is to be determined.

import curses

import logging

import hsh.jobs
import hsh.content.text
import hsh.content.text_formatter

from hsh.exceptions import *
from hsh.lib import *
from view import View
from job_view import JobView

class ListView(View, hsh.jobs.JobManagerListener, hsh.jobs.JobListener):
    """Displays a list of job information.  Instances need to be configured by
    specifying a format for display and a function for filtering those to
    display.

    It contains a list of job display objects, and tracks and highlights a
    focussed one.
    """

    # self.jobs is a list of jobs.Job objects.  curjob is an index into
    # self.jobs, which specifies the currently highlighted job, or None if
    # there are no jobs in the list.

    # Drawing of the job output is controlled by two variables: tail and
    # disp_pos.  disp_pos indicates which line of the window should anchor the
    # drawing of the current job, and tail indicates if it should be anchoring
    # the head or tail of the job.  disp_pos can be negative if the anchored
    # bit is in fact not currently visible.

    def __init__(self, display, name, job_filter, job_header_fmt,
                 header="", show_job_output=False, sync_main_job=False):
        if name:
            self.name = name
        super(ListView, self).__init__(display)
        self.jobs = []
        self.curjob = None
        self.disp_pos = 0
        self.tail = False
        self.job_filter = job_filter
        self.header_fmt = header
        self.job_header_fmt = job_header_fmt
        self.show_job_output = show_job_output
        self.sync_main_job = sync_main_job
        self.last_curjob = None
        # If show_tail is True, then disp_pos will be adjusted to ensure the
        # tail of the current job is visible.
        self.show_tail = False

        # job_options contains drawing options for each job specific to this
        # list view object.  Keys are job objects, values are dicts containing
        # these keys:
        #   formatter_header -> TextFormatter object for the header info
        #   formatter_output -> TextFormatter object for the job output
        #   job_drawn_lines  -> Where job was last drawn
        #                      (first screen line used; line count;
        #                       offset from top of job output to value 0 of
        #                       this tuple either positive from top or negative
        #                       from bottom)
        self.job_options = {}

        # List of jobs included in the last draw.
        self.jobs_drawn_last = []

        # Updated with each draw to reflect the size of the window.
        (self.cur_height, self.cur_width) = self.display.get_size()

        hsh.jobs.manager.register_listener(self)

    def set_visible_job(self, job):
        """Implementation of abstract method from View."""
        if self.sync_main_job:
            self.set_current_job(job)

    ######################################################################
    # JobManagerListener interface

    def on_add_job(self, job):
        """Add a single job to the list."""
        if not self.job_filter(job):
            return

        job.register_listener(self)
        self.job_options[job] = {}
        self._set_job_option("job_drawn_lines", (0,2,0), job)

        # Formatters for the job headers and job output go in options
        jhf = hsh.content.text_formatter.TextFormatter([])
        jhf.set_format(wrap=False, face_view_name=self.get_name(),
                       width=self.cur_width)
        self._set_job_option("formatter_header", jhf, job)

        jtext = self.display.get_job_view(job).text
        jof = hsh.content.text_formatter.TextFormatter(jtext)
        jof.set_format(face_view_name=self.get_name(), width=self.cur_width)
        self._set_job_option("formatter_output", jof, job)

        # Insert the new job into the sorted list.
        i = 0
        while i < len(self.jobs):
            if job.jid < self.jobs[i].jid:
                self.jobs[i:i] = [job]
                break
            i += 1
        if i == len(self.jobs):
            self.jobs.append(job)

        if job.is_new or self.curjob is None:
            self._set_curjob(newjob=i, dispmode="head")
            self.show_tail = True
        elif self.curjob >= i:
            self._set_curjob(next=True, dispmode="still")

    def on_remove_job(self, job):
        """A job is removed from the manager, so remove it from the list."""
        job.unregister_listener(self)
        for i in xrange(len(self.jobs)):
            if job.jid == self.jobs[i].jid:
                if self.curjob < i:
                    # Deletion doesn't affect the current job
                    del self.jobs[i]
                else:
                    # Deletion affects the current job, so unset it first.
                    cj = self.curjob
                    self._set_curjob(newjob=None)
                    del self.jobs[i]
                    if cj > 0:
                        self._set_curjob(newjob=cj-1)
                    elif len(self.jobs) > 0:
                        self._set_curjob(newjob=0, dispmode="head")
                self.set_dirty()
                return

    ######################################################################
    # jobs.JobListener interface

    def on_job_output(self, job):
        if job in self.jobs and not self.job_filter(job):
            self.on_remove_job(job)
        self.set_dirty()

    def on_job_terminate(self, job):
        if job in self.jobs and not self.job_filter(job):
            self.on_remove_job(job)
        self.set_dirty()

    def on_job_raw_output(self, job, text, channel):
        if job in self.jobs and not self.job_filter(job):
            self.on_remove_job(job)
        self.set_dirty()

    ######################################################################
    # List View specific functions

    def _get_job_header_size(self):
        "Return the number of lines used by each job header."
        return len(self.job_header_fmt.split('\n'))

    def _load_job(self):
        """Ask the job manager to load a new job on disk, which will be
        inserted at the front of the list.  Return the loaded job or None."""
        # The job manager fetch calls the JobListener interface to add the job
        # to self.jobs.  However, the job may be filtered out, so keep
        # trying if it didn't appear.  But don't waste too long searching for
        # an older job, give up if a match doesn't come quickly.
        if len(self.jobs) == 0:
            fj = hsh.jobs.manager.get_last_job()
        else:
            fj = hsh.jobs.manager.get_prev_job(self.jobs[0].jid)

        tc = 0
        while fj and (len(self.jobs) == 0 or self.jobs[0] != fj):
            fj = hsh.jobs.manager.get_prev_job(fj.jid)
            tc += 1
            if tc == 100:
                return None

        return fj

    def _get_job_option(self, option, job=None):
        """Fetch an option from self.job_options."""
        if job is None:       job = self.curjob
        if job is None:       return
        if type(job) == int:  job = self.jobs[job]
        return (self.job_options[job])[option]

    def _set_job_option(self, option, value, job=None):
        """Set an option in self.job_options."""
        if job is None:       job = self.curjob
        if job is None:       return
        if type(job) == int:  job = self.jobs[job]
        (self.job_options[job])[option] = value

    def _set_curjob(self, newjob=None, next=False, prev=False, dispmode=""):
        """Update the current job and related state.

        dispmode takes one of the following values to define how the display
        position should be updated:

         - "" - Don't update the display position.
         - "tail" - Show the tail of the new current job.
         - "head" - Show the head of the new current job.
         - "still" - Update display position to keep the screen as still as
           possible.

        @param newjob: The new current job as in index into self.jobs or None.
        @type newjob: integer
        @param next: True to use the job after current job instead of newjob
        @type next: boolean
        @param prev: True to use the job before current job instead of newjob
        @type prev: boolean
        @param dispmode: How to update display position.
        @type dispmode: string
        """
        oldcurjob = self.curjob
        if len(self.jobs) == 0:
            self.curjob = None
        elif next:
            if self.curjob is None:
                self.curjob = 0
            elif self.curjob + 1 < len(self.jobs):
                self.curjob += 1
        elif prev:
            if self.curjob is None:
                self.curjob = 0
            elif self.curjob > 0:
                self.curjob -= 1
        elif newjob >= 0 and newjob < len(self.jobs):
            self.curjob = newjob
        else:
            self.curjob = None

        if (oldcurjob == self.curjob) or (self.curjob is None):
            return

        if dispmode:
            max_dp = self.cur_height - self._get_job_header_size()
            jdl = list(self._get_job_option("job_drawn_lines"))
            if self.current_job() not in self.jobs_drawn_last:
                # jdl is out of date, so try to fix it
                if self.jobs[self.curjob - 1] in self.jobs_drawn_last:
                    pl = self._get_job_option("job_drawn_lines", self.curjob-1)
                    jdl[0] = pl[0] + pl[1]
                else:
                    jdl[0] = 0

            if dispmode == "tail":
                self.tail = True
                self.disp_pos = clamp(1, jdl[0] + jdl[1], self.cur_height - 1)
            elif dispmode == "head":
                self.tail = False
                newdp = jdl[0]
                if jdl[0] + jdl[1] > self.cur_height:
                    newdp = self.cur_height - jdl[1]
                self.disp_pos = clamp(0, newdp, max_dp)
            elif dispmode == "still":
                if jdl[0] >= 0:
                    self.disp_pos = min(jdl[0], max_dp)
                    self.tail = False
                elif jdl[0] + jdl[1] <= self.cur_height:
                    self.disp_pos = max(1, jdl[0] + jdl[1] - 1)
                    self.tail = True
                else:
                    self.disp_pos = jdl[0]
                    self.tail = False

        # Make sure the job is shown in the job view.
        if self.sync_main_job:
            self.display.display_job(self.current_job())

        if oldcurjob is not None:
            jf = self._get_job_option('formatter_header', oldcurjob)
            jf.set_format(face_default = "listjobheader")
        jf = self._get_job_option('formatter_header')
        jf.set_format(face_default = "highlight")

    def current_job(self):
        """Return the current job as a hsh.jobs.Job object."""
        if self.curjob is None:
            return None
        else:
            return self.jobs[self.curjob]

    def last_job(self):
        """Return the last job as a hsh.jobs.Job object or None."""
        if len(self.jobs) > 0:
            return self.jobs[-1]

    def set_current_job(self, job):
        """Update the list's current job to the given hsh.jobs.Job object, and
        change display position so it's visible.  If the job object isn't in
        the list, set the current job to None."""
        for i in range(len(self.jobs)):
            if self.jobs[i] == job:
                self._set_curjob(i, dispmode="still")
                return
        self._set_curjob(newjob=None)

    def get_predecessor(self, job):
        """Given a job, fetch one from the list which falls before it based on
        jobid.  Return None if no predecessor could be found.  Load jobs from
        disk if needed."""
        # Load jobs backwards to an appropriate point.
        while len(self.jobs) == 0 or job.jid <= self.jobs[0].jid:
            rj = self._load_job()
            if rj is None:
                return

        # Search the list for its immediate predecessor.
        last_job = None
        for i_job in self.jobs:
            if i_job.jid < job.jid:
                last_job = i_job
        return last_job

    def get_successor(self, job):
        """Given a job, fetch one from the list which falls after it based on
        jobid.  Return None if no successor could be found.  Load jobs from
        disk if needed."""
        # If job is before the beginning of the list, load some earlier jobs.
        while len(self.jobs) == 0 or job.jid < self.jobs[0].jid:
            rj = self._load_job()
            if rj is None:
                break

        for i_job in self.jobs:
            if i_job.jid > job.jid:
                return i_job

    ######################################################################
    # Drawing and formatting code

    def draw_cursor(self, win):
        """Draw cursor at the current job."""
        cy, cx = (0, 0)
        if self.current_job() in self.jobs_drawn_last:
            cy = self._get_job_option("job_drawn_lines")[0]
        cy = clamp(0, cy, self.cur_height - 1)

        # I want to use win.move(cy, cx), but it doesn't work for me.
        (py, px) = win.getparyx()
        # Account for the header size
        self.display.scr.move(py + cy + self.header_size, px + cx)

        curses.curs_set(2)  # Make the cursor visible again.
        self.display.scr.refresh()
        return

    def draw(self, win, force_redraw, search=None):

        if (not force_redraw and not self.is_dirty() and 
            self.curjob == self.last_curjob):
            return False

        self.set_dirty(None)
        self.last_curjob = self.curjob

        self.draw_header(win)

        cwin = self._content_win(win)
        cwin.leaveok(1)
        cwin.clear()

        (height, width) = cwin.getmaxyx()
        self.cur_height = height
        if self.cur_width != width:
            self.cur_width = width
            for jo in self.job_options.values():
                jo['formatter_header'].set_format(width=width)
                jo['formatter_output'].set_format(width=width)

        if self.curjob is None:
            cwin.refresh()
            cwin.leaveok(0)
            return True

        ##########
        # Assemble the formatted lines of the job list view into an object.

        # This code is placed in a loop, separate from the actual drawing
        # process, so that if things need to be changed, the loop can be
        # restarted without affecting the screen.

        # The drawing is anchored around a line, anchor and a job whose first
        # line of output is on that line, anchorjob.  These values are derived
        # from curjob, disp_pos, and tail.

        # The process for assembling formatted output is:

        #   * Loop through jobs before anchor job, assembling enough lines to
        #     fill space before anchor.
        #   * Make sure there's no blank lines at the top
        #   * Loop through jobs after anchor job, assembling enough lines to
        #     fill space after anchor.
        #   * If the current job isn't visible, choose a new visible one and 
        #     start again.

        while True:
            self.jobs_drawn_last = []

            if self.tail and self.show_tail:
                self.disp_pos = clamp(0, self.disp_pos, height - 1)

            # Determine the job anchored at the top, and its top line.
            if not self.tail:
                anchor = self.disp_pos
                anchorjob = self.curjob
            else:
                anchor = self.disp_pos + 1
                anchorjob = self.curjob + 1

            # Assembled formatted output.
            fmto = ListView.JobFormatAssembler(self, height, anchor)

            #####
            # Preceding jobs
            drawjob = anchorjob - 1
            while drawjob >= 0 and fmto.needs_more_above():
                jdl = fmto.add_job_above(drawjob, search)
                self._set_job_option("job_drawn_lines", jdl, drawjob)
                self.jobs_drawn_last.insert(0, self.jobs[drawjob])

                drawjob -= 1

            # Make sure there's no blank space at the top
            if fmto.needs_more_above():
                self.disp_pos -= 1
                anchor -= 1
                continue # recalculate job_drawn_lines

            #####
            # Post jobs output
            drawjob = anchorjob
            while drawjob < len(self.jobs) and fmto.needs_more_below():
                jdl = fmto.add_job_below(drawjob, search)
                self._set_job_option("job_drawn_lines", jdl, drawjob)
                self.jobs_drawn_last.append(self.jobs[drawjob])

                drawjob += 1

            # If nothing's visible, readjust the display.
            if fmto.distance_to_visible() > 0:
                logging.debug("Nothing visible, adjusting position")
                self.disp_pos += fmto.distance_to_visible()
                continue

            # If curjob's not visible, change it and continue the main loop.
            nj = fmto.check_visible(self.curjob)
            if nj != self.curjob:
                logging.debug("Current job not visible, switching")
                self.show_tail = nj < self.curjob
                self._set_curjob(newjob=nj, dispmode="still")
                continue

            if self.show_tail and not fmto.tail_visible(self.curjob):
                self.tail = True
                self.disp_pos = height - 1
                logging.debug("Repositioning for tail to be visible.")
                continue

            # Finished, so break the loop
            break

        #####
        # Actual drawing
        j = 0
        for line in fmto.get_lines():
            self.draw_line(line, j, cwin)
            j += 1

        cwin.refresh()
        cwin.leaveok(0)
        return True

    class JobFormatFetcher(object):
        """Class to simplify accessing a job's formatted header and output.

        It's a very restrictive class, and can only be used by making the
        following three calls in order:

         1. B{Construction}: Provide a job and formatting direction.

         2. B{Line Skipping}: call skip_lines_head() or skip_lines_tail() to
            give the number of lines which don't need to be formatted.

         3. B{Output Accessing}: call one of get_lines_head(), get_lines_tail(),
            search_lines_head() or search_lines_tail(), which start from where
            the previous skip call ended.  The get functions return the
            requested number of formatted lines and the search functions return
            a line number where the search matched.
        """
        # Internal position is recorded by self.dp as coordinates in the body,
        # and self.hp as line in the formatted header
        def __init__(self, view, job, tail, search):
            """@param view: The ListView this is formatting for.
            @param job: Index of job to format.
            @param tail: Boolean, true means format from the tail.
            @param search: Compiled regexp for search highlighting.
            """
            self.view = view
            self.job = job
            self.h_fmter = self.view._get_job_option("formatter_header", job)
            self.o_fmter = self.view._get_job_option("formatter_output", job)
            self.o_fmter.set_format(reverse=tail)
            self.j_view = self.view.display.get_job_view(self.view.jobs[job])
            self.search = search

            rawhl = (self.view.job_header_fmt %
                     self.j_view.header_info()).split('\n')
            self.header = self.h_fmter.get_lines(len(rawhl) * 2,
                                                 search=self.search,
                                                 text=rawhl)

            if tail:
                self.dp = self.o_fmter.get_text().end()
                self.dp[0] += 1
                self.dp[1] = 0
                self.hp = len(self.header)
            else:
                self.dp = [0,0]
                self.hp = 0

        def skip_lines_tail(self, count):
            if count <= 0: return 0
            skc = 0
            if self.view.show_job_output:
                skc = self.o_fmter.coord_up_fline(self.dp, clamp=True,
                                                  count=count)
            if skc < count:
                self.hp = min(count - skc, len(self.header))
                skc += self.hp
            return skc

        def skip_lines_head(self, count):
            if count <= 0: return 0
            self.hp = min(count, len(self.header))
            skc = 0
            if self.hp < count and self.view.show_job_output:
                skc = self.o_fmter.coord_down_fline(self.dp, clamp=True,
                                                    count=count-self.hp,
                                                    beyond=1)
            return skc + self.hp

        def get_lines_tail(self, count):
            """@return: The formatted text
            @rtype: MutableText"""
            if count <= 0: return []
            if self.view.show_job_output:
                olns = self.o_fmter.get_lines(count, display_pos=self.dp,
                                              search=self.search)
            else:
                olns = []
            if len(olns) < count:
                hct = min(count - len(olns), len(self.header), self.hp)
                olns[0:0] = self.header[-hct:]
            return olns

        def get_lines_head(self, count):
            """@return: The formatted text and a boolean which is True if more
            was available.
            @rtype: (MutableText, boolean)"""
            if count <= 0: return [], True
            hct = min(self.hp + count, len(self.header))
            olns = self.header[self.hp:hct]
            glmore = [True]
            if len(olns) < count and self.view.show_job_output:
                olns.extend(self.o_fmter.get_lines(count - len(olns),
                                                   display_pos=self.dp,
                                                   search=self.search,
                                                   more=glmore))
            return (olns, glmore[0])

        def search_lines_tail(self, pattern):
            "@return: Line from top of job where pattern matches or None"
            if self.view.show_job_output:
                m = self.o_fmter.search(pattern, self.dp, forward=False)
                if m is not None:
                    return (len(self.header) +
                            self.o_fmter.coord_diff(m[0], [0,0])[0])
            for i in range(self.hp - 1, -1, -1):
                match = pattern.search(str(self.header[i]))
                if match:
                    return i

        def search_lines_head(self, pattern):
            "@return: Line from top of job where pattern matches or None"
            for i in range(self.hp, len(self.header)):
                match = pattern.search(str(self.header[i]))
                if match:
                    return i
            if self.view.show_job_output:
                m = self.o_fmter.search(pattern, self.dp, forward=True)
                if m is not None:
                    return (len(self.header) +
                            self.o_fmter.coord_diff(m[0], [0,0])[0])

    class JobFormatAssembler(object):
        """Small class to assemble the formatted job output.

        It is created with a reference to a view and parameters for assembling
        output.  C{height} is the number of lines which need to be filled, and
        C{anchor} is a line relative to the top of the height around which job
        output is placed.

        Once all job output is added, the formatted lines can be fetched in one
        call.
        """
        def __init__(self, view, height, anchor):
            """@param view: The ListView being formatted for.
            @param height: The number of lines to format.
            @param anchor: The line around which job output is placed.
            """
            self.view = view
            self.height = height
            self.anchor = anchor

            self.formatted_lines = hsh.content.text.MutableText()
            # The lines between done_up and done_down are accounted for.
            # anchor line belongs to the jobs below.
            self.done_up = anchor - 1
            self.done_down = anchor

            self.drawnjobs = []
            self.more = None

        def needs_more_above(self):
            return self.done_up >= 0

        def needs_more_below(self):
            return self.done_down < self.height

        def add_job_above(self, job, search):
            """
            Put a job's output at the top of the formatted output, above the
            anchor and any other jobs already added with this function.

            @return: job_drawn_lines data
            """
            if self.done_up < 0: return (self.done_up, 0)
            jfa = ListView.JobFormatFetcher(self.view, job, True, search)

            skl = jfa.skip_lines_tail(self.done_up + 1 - self.height)
            self.done_up -= skl
            if self.done_up + 1 > self.height: return (self.done_up + 1, skl)

            lns = jfa.get_lines_tail(self.done_up + 1)
            if len(lns) == 0: return (self.done_up + 1, skl)
            self.done_up -= len(lns)
            self.formatted_lines[0:0] = lns
            self.drawnjobs.insert(0, (job, self.done_up + 1,
                                      self.done_up + 1 + skl + len(lns) - 1))

            if self.more is None and self.done_up + 1 < self.height:
                self.more = self.done_up + 1 + skl + len(lns) > self.height

            return (self.done_up + 1, skl + len(lns), - (skl + len(lns)))

        def add_job_below(self, job, search):
            """
            Put a job's output at the bottom of the formatted output, below the
            anchor and any other jobs already added with this function.

            @return: job_drawn_lines data
            """
            if self.done_down >= self.height: return (self.done_down, 0)
            jfa = ListView.JobFormatFetcher(self.view, job, False, search)

            skl = jfa.skip_lines_head(-self.done_down)
            self.done_down += skl
            if self.done_down < 0: return (self.done_down - skl, skl)

            (lns, self.more) = jfa.get_lines_head(self.height - self.done_down)

            self.done_down += len(lns)
            self.formatted_lines.extend(lns)
            self.drawnjobs.append((job, self.done_down - len(lns) - skl,
                                   self.done_down - 1))

            return (self.done_down - len(lns) - skl, len(lns) + skl, 0)

        def get_lines(self):
            """Return the formatted lines
            @rtype: MutableText
            """
            return self.formatted_lines

        def distance_to_visible(self):
            """The number of lines from the last formatted line to the top of
            the visible area.

            This value is positive and relevant when anchor is negative and not
            enough output has been added below to reach the visible area.

            @rtype: int
            """
            return -self.done_down + 1

        def check_visible(self, job):
            """Check if the provided job is visible.  If it is, return it,
            otherwise return a newjob which is.

            @return: a job index into the View's .jobs array.
            @rrtype int
            """
            if len(self.drawnjobs) == 0:
                return job
            return clamp(self.drawnjobs[0][0], job, self.drawnjobs[-1][0])

        def tail_visible(self, job):
            """Return True if the tail of the given job is visible."""
            if self.check_visible(job) != job: return False
            for dj in self.drawnjobs:
                if dj[0] == job:
                    if dj[2] < self.height - 1:
                        return True
                    elif dj[2] == self.height - 1:
                        return not self.more
                    else:
                        return False
            return False

    ######################################################################

    def move_search(self, pattern, forward=True, from_end=False):
        """Update the display position so the next match of the given pattern
        is visible."""
        if pattern is None: return
        self.set_dirty()

        # Determine the first visible job
        jobid = self.curjob
        while jobid >= 0:
            if self.jobs[jobid] not in self.jobs_drawn_last: return
            jdl = self._get_job_option("job_drawn_lines", jobid)
            if jdl[0] <= 0 and jdl[0] + jdl[1] >= 1:
                break
            jobid -= 1

        if jobid < 0: return False

        # Create and skip the JFF for the visible job, handling the search
        # starting in the middle of output.
        if jdl[2] >= 0:
            # Output position is measured as lines from the head.
            jff = ListView.JobFormatFetcher(self, jobid, False, None)
            skip = - jdl[0] + jdl[2]
            if forward: skip += 1
            jff.skip_lines_head(skip)
        else:
            # Output position is measured as lines from the tail.
            jff = ListView.JobFormatFetcher(self, jobid, True, None)
            skip = - jdl[2]
            if forward: skip += 1
            jff.skip_lines_tail(skip)

        if forward:
            while jobid < len(self.jobs):
                ml = jff.search_lines_head(pattern)
                if ml is not None:
                    self._set_curjob(jobid)
                    self.disp_pos = -ml
                    self.tail = False
                    self.show_tail = False
                    return True

                jobid += 1
                if jobid < len(self.jobs):
                    jff = ListView.JobFormatFetcher(self, jobid, False, None)
                
        else:
            while True:
                ml = jff.search_lines_tail(pattern)
                if ml is not None:
                    self._set_curjob(jobid)
                    self.disp_pos = -ml
                    self.tail = False
                    self.show_tail = False
                    return True

                firstjob = False
                jobid -= 1
                if jobid < 0:
                    # Try and load an older job if one's available.
                    lj = self._load_job()
                    if lj is None:
                        return False
                    jobid = 0
                jff = ListView.JobFormatFetcher(self, jobid, True, None)

        return False

    ######################################################################
    # Functions bound to keystrokes

    # Functions bound to keystrokes which override parent's functions.
    def move_up(self, ki):
        if not self.show_job_output:
            self.prev_job(ki)
        else:
            self.disp_pos += ki.num
        self.show_tail = False

    def move_down(self, ki):
        if not self.show_job_output:
            self.next_job(ki)
        else:
            self.disp_pos -= ki.num

    def move_job_top(self, ki):
        # Move the top of the current job to the top of the screen.
        self.disp_pos = 0
        self.show_tail = False
        self.tail = False

    def move_job_bottom(self, ki):
        # Move the bottom of the current job to the bottom of the screen.
        self.disp_pos = self.cur_height - 1
        self.show_tail = True
        self.tail = True

    def page_up(self, ki):
        if not self.show_job_output:
            ki.num *= self.cur_height / self._get_job_header_size() 
            self.prev_job(ki)
        else:
            ki.num *= self.cur_height
            self.move_up(ki)
        self.show_tail = False

    def page_down(self, ki):
        if not self.show_job_output:
            ki.num *= self.cur_height / self._get_job_header_size() 
            self.next_job(ki)
        else:
            ki.num *= self.cur_height
            self.move_down(ki)

    def hide_output(self, ki):
        if self.curjob is not None:
            jf = self._get_job_option('formatter_output')
            cur_truncate = jf.get_format('truncate')
            if cur_truncate == None: jf.set_format(truncate=5)
            if cur_truncate == 5:    jf.set_format(truncate=0)
            if cur_truncate == 0:    jf.set_format(truncate=None)

            # If the job was occupying the top of the screen, make sure it's
            # properly visible after.
            if self.current_job() in self.jobs_drawn_last:
                if self._get_job_option("job_drawn_lines")[0] <= 0:
                    self.disp_pos = 0
            self.tail = False
            self.show_tail = False

    def next_job(self, ki):
        for i in range(ki.num):
            if len(self.jobs) == 0:
                return
            self._set_curjob(next=True, dispmode="head")
            self.show_tail = False

    def prev_job(self, ki):
        for i in range(ki.num):
            if (not self.curjob):
                lj = self._load_job()
                if lj is None:
                    return # Couldn't find an earlier job
            self._set_curjob(prev=True, dispmode="head")
            self.show_tail = False

    def restart_job(self, ki):
        self.display.get_job_view(self.current_job()).restart()

    def edit_job(self, ki):
        self.display.get_job_view(self.current_job()).edit_job(ki)

    def select_job(self, ki):
        self.display.display_job(self.current_job(), show=True)

    def input_command(self, ki, overwrite):
        """Insert the current job's command line to input view, and if
        overwrite is True, remove any existing input."""
        newtext = self.current_job().cmdline
        if overwrite:
            self.display.views["InputView"].cancel(ki)
        self.display.views["InputView"].insert_text(newtext)
        self.display.give_focus(self.display.views["InputView"])
        self.display.views["InputView"].set_dirty()

    def delete_job(self, ki):
        # Request to delete the current job.  This is escalated to the job
        # manager, and the on_remove_job() callback handles the work.
        dj = self.current_job()
        if dj is None:
            return
        if dj.get_state() == "Running":
            self.display.show_alert("Can't delete an active job")
            return
        hsh.jobs.manager.forget_job(dj)

    def toggle_wrap(self, ki):
        cur_wrap = self._get_job_option('formatter_header').get_format('wrap')
        for jo in self.job_options.values():
            jo['formatter_header'].set_format(wrap=not cur_wrap)
            jo['formatter_output'].set_format(wrap=not cur_wrap)
        self.set_dirty()
