#
# Copyright (C) 2010 Alexander Taler <dissent@0--0.org>
#

# This file is part of hsh.

# hsh is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# hsh is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with hsh.  If not, see <http://www.gnu.org/licenses/>.

######################################################################

# A one-line view used for entering search patterns.

import logging
import re

import hsh.content.text
from hsh.content.text import TextList

from view import View
from content_view import ContentView

class SearchView(ContentView):
    def __init__(self, display):
        super(SearchView, self).__init__(display, TextList([""]))
        self.textf.set_format(wrap = False, prefix = "i/")
        self.editable = True
        self.header_size = 0
        self.case_match = True

    def get_pattern(self):
        """Return a compiled regular expression of the current search string."""
        if len(self.text[0]) > 0:
            try:
                if self.case_match:
                    return re.compile(self.text[0])
                else:
                    return re.compile(self.text[0], re.IGNORECASE)
            except:
                # probably an unclosed parenthesis or something
                pass

    def highlight_regions(self, line, search_face):
        """Given a TextLine object, return a new one with the parts that match
        the search highlighted."""

        patt = self.get_pattern()
        if patt is None:
            return hsh.content.text.TextLine(line)

        line_ret = hsh.content.text.TextLine()
        for region in line.regions:
            match = patt.search(str(region))
            while match is not None:
                if match.start() > 0:
                    # Append the bit before the match as a region
                    line_ret.append_region(region[0:match.start()], region)
                if match.end() > match.start():
                    # Append the matched text
                    line_ret.append_region(region[match.start():match.end()],
                                           attrs = {"face":search_face,
                                                    "source":"search"})
                if match.end() == 0:
                    # Don't get caught in a loop when hitting a zero length
                    # match at the beginning of the string.
                    line_ret.append_region(region[0:1], region)
                region[0:match.end() or 1] = []
                if len(region) > 0:
                    match = patt.search(str(region))
                else:
                    match = None
            # No match so copy over the rest
            if len(region) > 0:
                rg = line_ret.append_region(region)

        return line_ret

    ######################################################################
    # Commands bound to key strokes.

    # History browsing is certainly on the cards
    def back_history(self, ki):
        pass

    def forward_history(self, ki):
        pass

    def main_handle(self, ki):
        self.display.get_main_view().putch(ki.key, ki.meta)

    def close_search(self, ki):
        self.display.display_search(show=False)

    def search_forward(self, ki):
        self.display.get_main_view().move_search(self.get_pattern(),
                                                 forward=True)

    def search_backward(self, ki):
        self.display.get_main_view().move_search(self.get_pattern(),
                                                 forward=False)

    def toggle_case_match(self, ki):
        self.case_match = not self.case_match
        self.textf.set_format(prefix = self.case_match and "i/" or "I/")
