;;; html-writer-mode.el --- Major mode for composing html files.

;; Author: Shawn K. Quinn <skquinn@speakeasy.net>
;; Maintainer: Shawn K. Quinn <skquinn@speakeasy.net>
;; Created: 01 Feb 1994
;; $Id: html-writer-mode.el,v 1.2 2005/05/23 17:03:16 skquinn Exp $
;; Keywords: HTML major-mode

;; Copyright (C) 1994 Nelson Minar
;; Copyright (C) 1995 Nelson Minar and Ulrik Dickow

;; Copyright (C) 1998,1999,2001,2004,2005 Nelson Minar, Ulrik Dickow,
;; and Shawn K. Quinn

;; This is a fork of html-helper-mode intended to be up-to-date with W3C
;; HTML 4.01 Strict and HTML authoring practices in the
;; "post-browser-wars" era (2001-later).

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, 51 Franklin Street, Fifth Floor,
;; Boston, MA 02110-1301, USA.

;;; Commentary:
;;{{{

;; Installation:
;;   Add this line in your .emacs:
;;     (autoload 'html-writer-mode "html-writer-mode" "Yay HTML" t)
;;   To invoke html-writer-mode automatically on .html files, do this:
;;     (setq auto-mode-alist (cons '("\\.html$" . html-writer-mode) auto-mode-alist))
;;
;;   This mode requires another lisp file, tempo.el. This can be
;;     retrieved from ftp://ftp.lysator.liu.se/pub/emacs/tempo.el
;;   Xemacs users need to have auc-menu installed.
;;   Emacs 18 users need to have auc-menu and add-hook installed.
;;   If your OS has broken 14 character filenames
;;      this mode will also work with the name "html-mode.el".

;; Configuration:
;;   see the "user variables" section, or the documentation on
;;   configuration in http://www.santafe.edu/~nelson/tools/ (for the
;;   original html-helper-mode). There are variables you want to
;;   configure, particularly html-writer-address-string and
;;   html-writer-use-expert-menu

;; Description:
;;   html-writer-mode makes it easier to write HTML documents. This mode
;;   handles inserting HTML codes in a variety of ways (keybindings,
;;   menus, completion in the buffer). It also supports indentation,
;;   timestamps, skeletons for new documents, hilit19 patterns, and a
;;   variety of other things. For the full skinny, see the HTML
;;   documentation that comes with the package or the original
;;   html-helper-mode documentation at
;;   http://www.santafe.edu/~nelson/tools/

;; Thank yous:
;;   Nelson Minar (current e-mail address unknown) for the original
;;     html-helper-mode.
;;   David Kgedal <davidk@lysator.liu.se> for the tempo code which
;;     forms the core of the HTML insertion, as well as the HTML+ tags.
;;   Marc Hedlund <march@europa.com> for general encouragement and
;;     many helpful suggestions, especially with HTML/2.0 compliance
;;     and form design.
;;   Ulrik Dickow <dickow@nbi.dk> for the font-lock code
;;   Denis Howe <dbh@doc.ic.ac.uk> for writing browse-url.
;;   Magnus Homann <d0asta@dtek.chalmers.se> and Jamshid Afshar
;;     <jamshid@ses.com> for timestamp suggestions.
;;   Everyone who sent Nelson a version of menus (16 in all!)
;;   Marc Andreessen <marca@mcom.com> for writing the original html-mode

;; Current versions of html-writer-mode may be downloaded from:
;; <URL:http://savannah.nongnu.org/projects/html-writer-mode/>

;; The newest version of the original html-helper-mode, should you need
;; it, should always be available from:
;; <URL:http://www.santafe.edu/~nelson/tools/>

;; If you are upgrading from Nelson Minar's html-helper-mode:
;; 1. Open your .emacs in Emacs
;; 2. Change all occurences of html-helper to html-writer
;;    (M-x query-replace RET html-helper RET html-writer RET !)
;; 3. If using tables.el, replace it with the html-writer-mode
;;    compatible version, and replace hhm-table with hwm-table as above
;;    (M-x query-replace RET hhm-table RET hwm-table RET !). If you've
;;    customized your tables.el, you may need to merge the changes
;;    manually.
;; 4. Save your .emacs
;; 5. Quit and restart Emacs to make sure your changes work as intended.
;; 6. If using the timestamp functionality, change "hhmts" to "hwmts" in
;;    your current documents and new document templates, as this has
;;    changed to reflect the new name.

;; There is limited backward compatibility built in (html-writer-mode
;; still provides "html-helper-mode") , but in the interest
;; of consistency and to allow for future changes that may break
;; backward compatibility all symbol name changes that referenced
;; the original html-helper-mode were changed to begin with
;; "html-writer".

;; Nelson originally wrote this code using folding.el, a wonderful
;; folding editor minor mode for emacs. That's what the strange {{{
;; comments are for, even though I (Shawn) don't use folding.el but I
;; left them in anyway.

;;}}}

;;; Code:

(defconst html-writer-mode-version (substring "$Revision: 1.2 $" 11 15))
;;{{{ user variables

;; Set this to be whatever signature you want on the bottom of your pages.
(defvar html-writer-address-string ""
  "*The default author string of each file.")

;; Features; these are all good to have on. (see also tempo.el)

(defvar html-writer-use-expert-menu nil
  "*If not nil, then use the full HTML menu.")

(defvar html-writer-do-write-file-hooks t
  "*If not nil, then modify `local-write-file-hooks' to do timestamps.")

(defvar html-writer-build-new-buffer t
  "*If not nil, then insert `html-writer-new-buffer-strings' for new buffers.")

;; variables to configure (these defaults are reasonable.)

;; skquinn - updated to be HTML 4.01 Strict

(defvar html-writer-htmldtd-version "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01//EN\">\n"
  "*Version of HTML DTD you're using.")

(defvar html-writer-user-menu nil
  "*Extra items to put in the HTML expert menu.
The value of this symbol is appended to the beginning of the expert
menu that is handed off to easymenu for definition. It should be a
list of vectors or lists which themselves are vectors (for submenus).")

(defvar html-writer-basic-offset 2
  "*Basic indentation size used for list indentation")

(defvar html-writer-item-continue-indent 4
  "*Indentation of lines that follow a <li> item.
Default is 4, the length of things like \"<li>\" and \"<dd>\".")

(defvar html-writer-never-indent t
  "*If not nil, the indentation code for html-writer is turned off.")

;; hooks (see also tempo.el)

(defvar html-writer-mode-hook nil
  "*Hook run when html-writer-mode is started.")

(defvar html-writer-load-hook nil
  "*Hook run when html-writer-mode is loaded.")

(defvar html-writer-timestamp-hook 'html-writer-default-insert-timestamp
  "*Hook called for timestamp insertion.
Override this for your own timestamp styles.")

;; strings you might want to change

;; skquinn - changed for HTML 4.01 Strict compliance.

(defvar html-writer-new-buffer-template
  '(html-writer-htmldtd-version
    "<html> <head>\n"
    "<title>" p "</title>\n</head>\n\n"
    "<body>\n"
    "<h1>" p "</h1>\n\n"
    p
    "\n\n<hr>\n"
    "<address>" html-writer-address-string "</address>\n"
    "<p>\n" 
    html-writer-timestamp-start
    html-writer-timestamp-end
    "\n</p>"
    "\n</body> </html>\n")
  "*Template for new buffers.
Inserted by `html-writer-insert-new-buffer-strings' if
`html-writer-build-new-buffer' is set to t")

(defvar html-writer-timestamp-start "<!-- hwmts start -->\n"
  "*Start delimiter for timestamps.
Everything between `html-writer-timestamp-start' and
`html-writer-timestamp-end' will be deleted and replaced with the output
of the functions `html-writer-timestamp-hook' if
`html-writer-do-write-file-hooks' is t")

(defvar html-writer-timestamp-end "<!-- hwmts end -->"
  "*End delimiter for timestamps.
Everything between `html-writer-timestamp-start' and
`html-writer-timestamp-end' will be deleted and replaced with the output
of the function `html-writer-insert-timestamp' if
`html-writer-do-write-file-hooks' is t")

;; control over what types of tags to load. By default, we load all the
;; ones we know of.

(defvar html-writer-types-to-install
  '(anchor list header logical phys textel entity image head form contain object revise misc)
  "*List of tag types to install when html-writer-mode is first loaded.
If you want to not install some type of tag, override this variable.
Order is significant: menus go in this order.")

;; emacs18 detection.

(defvar html-writer-emacs18
  (and (boundp 'emacs-version)
       (or (and (boundp 'epoch::version) epoch::version)
           (string-lessp emacs-version "19")))
  "I'll do minimal emacs18 support, grumble.")

;;}}} end of user variables

(require 'tempo)			;essential part of html-writer-mode
(condition-case nil			;menu support, standard in emacs19
    (require 'auc-menu)			;add-on for XEmacs. *why* does this
  (error (require 'easymenu)))		;package have to have two names?

;;{{{ html-writer-mode-syntax-table and html-writer-mode-abbrev-table

;; emacs doesn't seem to be able to really handle SGML like syntax. In
;; particular, comments are a loss.
;; We do try this, though: give < and > matching semantics

(defvar html-writer-mode-syntax-table nil
  "Syntax table for html-writer.")

(if html-writer-mode-syntax-table
    ()
  (setq html-writer-mode-syntax-table (make-syntax-table text-mode-syntax-table))
  (modify-syntax-entry ?<  "(>  " html-writer-mode-syntax-table)
  (modify-syntax-entry ?>  ")<  " html-writer-mode-syntax-table)
  (modify-syntax-entry ?\" ".   " html-writer-mode-syntax-table)
  (modify-syntax-entry ?\\ ".   " html-writer-mode-syntax-table)
  (modify-syntax-entry ?'  "w   " html-writer-mode-syntax-table))

(defvar html-writer-mode-abbrev-table nil
  "Abbrev table used while in html-writer-mode.")
(define-abbrev-table 'html-writer-mode-abbrev-table ())

;;}}}
;;{{{ type based keymap and menu variable and function setup

;; Our basic keymap.
(defvar html-writer-mode-map (make-sparse-keymap)
  "Keymap for html-writer")
(defvar html-writer-mode-menu nil
  "Menu for html-writer. Clobbered and rebuilt by `html-writer-install-menu'")

;; html-writer-mode has a concept of "type" of tags. Each type is a
;; list of tags that all go together in one keymap and one menu.
;; Types can be added to the system after html-writer has been loaded,
;; briefly by doing html-writer-add-type-to-alist, then
;; html-writer-install-type, then html-writer-add-tag (for each tag)
;; then html-writer-rebuild-menu. See the mode documentation for more detail.

(defconst html-writer-type-alist nil
  "Alist: type of tag -> keymap, keybinding, menu, menu string.
Add to this with `html-writer-add-type-to-alist'.")

;;{{{ accessor functions for html-writer-type-alist

(defun html-writer-keymap-for (type)
  "Accessor function for alist: for type, return keymap or nil"
  (nth 0 (cdr-safe (assq type html-writer-type-alist))))

(defun html-writer-key-for (type)
  "Accessor function for alist: for type, return keybinding or nil"
  (nth 1 (cdr-safe (assq type html-writer-type-alist))))

(defun html-writer-menu-for (type)
  "Accessor function for alist: for type, return menu or nil"
  (nth 2 (cdr-safe (assq type html-writer-type-alist))))

(defun html-writer-menu-string-for (type)
  "Accessor function for alist: for type, return menustring or nil"
  (nth 3 (cdr-safe (assq type html-writer-type-alist))))

(defun html-writer-normalized-menu-for (type)
  "Helper function for building menus from submenus: add on string to menu."
  (cons (html-writer-menu-string-for type)
	(eval (html-writer-menu-for type))))

;;}}}

(defun html-writer-add-type-to-alist (type)
  "Add a type specification to the alist.
The spec goes (type . (keymap-symbol keyprefix menu-symbol menu-string)).
See code for an example."
  (setq html-writer-type-alist (cons type html-writer-type-alist)))

;; Here are the types provided by html-writer-mode.
(mapcar 'html-writer-add-type-to-alist
  '((entity  . (nil nil html-writer-entity-menu "Insert Character Entities"))
    (textel  . (nil nil html-writer-textel-menu "Insert Text Elements"))
    (head    . (html-writer-head-map "\C-c\C-b" html-writer-head-menu "Insert Structural Elements"))
    (header  . (html-writer-header-map "\C-c\C-t" html-writer-header-menu "Insert Headers"))
    (anchor  . (html-writer-anchor-map "\C-c\C-a" html-writer-anchor-menu "Insert Hyperlinks"))
    (logical . (html-writer-logical-map "\C-c\C-s" html-writer-logical-menu "Insert Logical Styles"))
    (phys    . (html-writer-phys-map "\C-c\C-p" html-writer-phys-menu "Insert Physical Styles"))
    (list    . (html-writer-list-map "\C-c\C-l" html-writer-list-menu "Insert List Elements"))
    (form    . (html-writer-form-map "\C-c\C-f" html-writer-form-menu "Insert Form Elements"))
    (contain . (html-writer-contain-map "\C-c\C-c" html-writer-contain-menu "Insert Containers"))
    (revise  . (html-writer-revise-map "\C-c\C-r" html-writer-revise-menu "Insert Revision Marks"))
    (object  . (html-writer-object-map "\C-c\C-o" html-writer-object-menu "Insert Objects"))
    (misc    . (html-writer-misc-map "\C-c+" html-writer-misc-menu "Add Miscellaneous Attributes"))
    (image   . (html-writer-image-map "\C-c\C-i" html-writer-image-menu "Insert Inlined Images"))
    ))

;; Once html-writer-mode is aware of a type, it can then install the
;; type: arrange for keybindings, menus, etc.

(defconst html-writer-installed-types nil
  "The types that have been installed (used when building menus).
There is no support for removing a type once it has been installed.")

(defun html-writer-install-type (type)
  "Install a new tag type: add it to the keymap, menu structures, etc.
For this to work, the type must first have been added to the list of types
with html-writer-add-type-to-alist."
  (setq html-writer-installed-types (cons type html-writer-installed-types))
  (let ((keymap (html-writer-keymap-for type))
	(key (html-writer-key-for type))
	(menu (html-writer-menu-for type))
	(menu-string (html-writer-menu-string-for type)))
    (and key
	 (progn
	   (set keymap nil)
	   (define-prefix-command keymap)
	   (if html-writer-emacs18
	       (progn
		 (set keymap (make-sparse-keymap))
		 (define-key html-writer-mode-map key (eval keymap)))
	     (define-key html-writer-mode-map key keymap))))
    (and menu
	 (progn
	   (set menu nil)))))

;; install the default types.
(mapcar 'html-writer-install-type html-writer-types-to-install)

;; special mode keys
(mapcar
 (function (lambda (l) (define-key html-writer-mode-map (car l) (nth 1 l))))
 '(("\C-x " tempo-complete-tag)            ;skquinn - changed for Win32 use
   ("\M-\C-f" tempo-forward-mark)
   ("\M-\C-b" tempo-backward-mark)))       

;; Extra commands that HTML writer supports that aren't insertions
(defvar html-writer-mode-functions-map nil
  "Keymap for extra HTML mode functions")
(define-prefix-command 'html-writer-mode-functions-map)
(define-key html-writer-mode-map "\C-c\C-z"
  'html-writer-mode-functions-map)
(define-key html-writer-mode-functions-map "t"
  'html-writer-insert-timestamp-delimiter-at-point)

;; indentation keys - only rebind these if the user wants indentation
(if html-writer-never-indent
    ()
  (define-key html-writer-mode-map "\t" 'html-writer-indent-command)
  (define-key html-writer-mode-map "\C-m" 'newline-and-indent))

;; browse url stuff
(if (fboundp 'browse-url-of-file)
    (define-key html-writer-mode-functions-map "v" 'browse-url-of-file))
(if (and (boundp 'browse-url-browser-function) (fboundp browse-url-browser-function))
    (define-key html-writer-mode-functions-map "u" browse-url-browser-function))

;;}}}

;;{{{ html-writer-add-tag function for building basic tags

(defvar html-writer-tempo-tags nil
  "List of tags used in completion.")

;; this while loop is awfully Cish
;; isn't there an emacs lisp function to do this?
(defun html-writer-string-to-symbol (input-string)
  "Given a string, downcase it and replace spaces with -.
We use this to turn menu entries into good symbols for functions.
It's not entirely successful, but fortunately emacs lisp is forgiving."
  (let* ((s (copy-sequence input-string))
	 (l (1- (length s))))
    (while (> l 0)
      (if (char-equal (aref s l) ?\ )
	  (aset s l ?\-))
      (setq l (1- l)))
    (concat "html-" (downcase s))))


(defun html-writer-add-tag (l)
  "Add a new tag to html-writer-mode.
Builds a tempo-template for the tag and puts it into the
appropriate keymap if a key is requested. Format:
`(html-writer-add-tag '(type keybinding completion-tag menu-name template doc)'"
  (let* ((type (car l))
	 (keymap (html-writer-keymap-for type))
	 (menu (html-writer-menu-for type))
	 (key (nth 1 l))
	 (completer (nth 2 l))
	 (name (nth 3 l))
	 (tag (nth 4 l))
	 (doc (nth 5 l))
	 (command (tempo-define-template (html-writer-string-to-symbol name)
					 tag completer doc
					 'html-writer-tempo-tags)))

    (if (null (memq type html-writer-installed-types))    ;type loaded?
	t                                                 ;no, do nothing.
      (if (stringp key)			                  ;bind key somewhere?
	  (if keymap			                  ;special keymap?
	      (define-key (eval keymap) key command)      ;t:   bind to prefix
	    (define-key html-writer-mode-map key command));nil: bind to global
	t)
      (if menu				                  ;is there a menu?
	  (set menu			                  ;good, cons it in
	       (cons (vector name command t) (eval menu))))
      )))

;; for backwards compatability
(fset 'html-writer-add-cookie 'html-writer-add-tag)

;;}}}

;;{{{ most of the HTML tags

;; These tags are an attempt to be HTML/2.0 compliant, with the exception
;; of container <p>, <li>, <dd>, <dt> (we adopt 3.0 behaviour).
;; For reference see <URL:http://www.w3.org/hypertext/WWW/MarkUp/MarkUp.html>

;; order here is significant: within a tag type, menus and mode help
;; go in the reverse order of what you see here. Sorry about that, it's
;; not easy to fix.

;; skquinn - modifications:

;; * A lot of these are now changed for HTML 4.01 compliance. Most
;; notably: The IMG tag has had the variations that did not embed the
;; ALT attribute removed (ALT is required in 4.01 as it should have been
;; from the beginning). STRIKE and U are gone; if you want them you'll
;; have to add them back yourself.

;; * Addition for DIV and SPAN generic containers.

;; * STYLE is added, but do consider a seperate CSS stylesheet for
;; ease of maintenance.

;; * An "entity" for splitting paragraphs easily. I added this for ease
;; of use for people (like myself, of course) who write long paragraphs
;; directly in HTML, instead of writing in plain text first, and adding
;; the paragraph marks later.

;; * Changed keystrokes for the registered trademark and copyright
;; entities to something more easily remembered. (Let's face it, 'C-c $'
;; and 'C-c @' are hard to remember.)

;; * A whole bunch of OBJECT has been added.

;; skquinn to be done:

;; * _Maybe_ add the plethora of attributes that have come along since
;; HTML 2.0 and 3.2, including STYLE, CLASS, ID, and the many ON*
;; scripting/Javascript attributes (onmouseover, onmouseout, etc) or
;; provide another shortcut that can do these. (CLASS and ID already
;; exist for SPAN and DIV; they are of very limited utility without
;; them.)

;; * _Maybe_ (again) add most of the other new HTML 4.0 tags. (Some of
;; them are already added, such as INS, DEL, ABBR, ACRONYM, OBJECT,
;; SPAN, DIV, STYLE, SCRIPT.)

;; * Add bindings to automatically run programs like Tidy, Demoronizer,
;; WWWis, etc on the current buffer. (Of limited utility if you're using
;; an HTML preprocessor like Orb, or editing PHP or ASP source, and I
;; think in a sense you can already do this with Emacs.)

;; * Consider a new version for XHTML 1.0/1.1, probably to be called
;; xhtml1-writer-mode. (In retrospect, this should probably have been
;; html4-writer-mode while I was changing the name. Eh, oh well.)

(mapcar
 'html-writer-add-tag
 '(
   ;;entities (and stuff like "split paragraph" that doesn't have
   ;;another place to go)
   (entity  "\C-c#"   "&#"              "Ascii or Unicode Character"      ("&#" (r "Unicode: ") ";"))
   (entity  "\C-c\""  "&quot;"          "Quotation mark"  ("&quot;"))
   (entity  "\C-cr"   "&reg;"           "Registered"      ("&reg;"))
   (entity  "\C-cc"   "&copy;"          "Copyright"       ("&copy;"))
   (entity  "\C-c-"   "&shy;"           "Soft Hyphen"     ("&shy;"))
   (entity  "\C-c "   "&nbsp;"		"Nonbreaking Space"  ("&nbsp;"))
   (entity  "\C-c&"   "&amp;"		"Ampersand"	  ("&amp;"))
   (entity  "\C-c>"   "&gt;"	  	"Greater Than"       ("&gt;"))
   (entity  "\C-c<"   "&lt;"		"Less Than"	  ("&lt;"))
   (entity  "\C-cp"   "</p>"		"Split Paragraph"	  ("</p>\n<p>"))
   ;; logical styles
   (logical "q"       "<q>"		"Quote inline"         	  ("<q>" (r "Quote: ") "</q>"))
   (logical "b"       "<blockquote>"		"Blockquote"           	  ("<blockquote>" (r "Quote: ") "</blockquote>"))
   (logical "c"       "<code>"		"Code"           	  ("<code>" (r "Code: ") "</code>"))
   (logical "x"       "<samp>"		"Sample"         	  ("<samp>" (r "Sample code") "</samp>"))
   (logical "r"       "<cite>"		"Citation"       	  ("<cite>" (r "Citation: ") "</cite>"))
   (logical "k"       "<kbd>"		"Keyboard Input"       	  ("<kbd>" (r "Keyboard: ") "</kbd>"))
   (logical "v"       "<var>"		"Variable"       	  ("<var>" (r "Variable: ") "</var>"))
   (logical "d"       "<dfn>"		"Definition"     	  ("<dfn>" (r "Definition: ") "</dfn>"))
   (logical "a"	      "<address>"	"Address"		  ("<address>" r "</address>"))
   (logical "v"	      "<ab"             "Abbreviation"		  ("<abbr title=\"" (p "Title: ") "\">" r "</abbr>"))
   (logical "m"	      "<acr"            "Acronym"		  ("<acronym title=\"" (p "Title: ") "\">" r "</acroynm>"))
   (logical "e"       "<em>"		"Emphasized"     	  ("<em>" (r "Text: ") "</em>"))
   (logical "s"       "<strong>"	"Strong"         	  ("<strong>" (r "Text: ") "</strong>"))
   (logical "p"       "<pre>"		"Preformatted"   	  ("<pre>" (r "Text: ") "</pre>"))

   ;;physical styles
   (phys    "i"       "<i>"		"Italic"             ("<i>" (r "Text: ") "</i>"))
   (phys    "b"	      "<b>"    		"Bold"               ("<b>" (r "Text: ") "</b>"))
   (phys    "t"       "<tt>"		"Teletype [fixed]"   ("<tt>" (r "Text: ") "</tt>"))

   ;;headers
   (header  "6"       "<h6>"		"Header 6"       	  ("<h6>" (r "Header: ") "</h6>"))
   (header  "5"       "<h5>"		"Header 5"       	  ("<h5>" (r "Header: ") "</h5>"))
   (header  "4"       "<h4>"		"Header 4"       	  ("<h4>" (r "Header: ") "</h4>"))
   (header  "3"       "<h3>"		"Header 3"       	  ("<h3>" (r "Header: ") "</h3>"))
   (header  "2"       "<h2>"		"Header 2"       	  ("<h2>" (r "Header: ") "</h2>"))
   (header  "1"	      "<h1>"     	"Header 1"       	  ("<h1>" (r "Header: ") "</h1>"))

   ;; forms
   (form    "o"       "<option>"        "Option"                 (& "<option>" > ))
   (form    "v"       "<option value"   "Option with Value"      (& "<option value=\"" (r "Value: ") "\">" >))
   (form    "s"       "<select"		"Selections"	          ("<select name=\"" (p "Name: ") "\">\n<option>" > "\n</select>")"<select")
   (form    "z"	      "<input"		"Reset Form"    	  ("<input type=\"RESET\" value=\"" (p "Reset button text: ") "\">"))
   (form    "b"	      "<input"		"Submit Form"   	  ("<input type=\"SUBMIT\" value=\"" (p "Submit button text: ") "\">"))
   (form    "i"	      "<input"		"Image Field"   	  ("<input type=\"IMAGE\" name=\"" (p "Name: ") "\" src=\"" (p "Image URL: ") "\">"))
   (form    "h"       "<input"          "Hidden Field"            ("<input type=\"HIDDEN\" name=\"" (p "Name: ") "\" value=\"" (p "Value: ") "\">"))
   (form    "p"	      "<textarea"	"Text Area"	  ("<textarea name=\"" (p "Name: ") "\" rows=\"" (p "Rows: ") "\" cols=\"" (p "Columns: ") "\">" r "</textarea>"))
   (form    "c"	      "<input"		"Checkbox"   	          ("<input type=\"CHECKBOX\" name=\"" (p "Name: ") "\">"))
   (form    "r"	      "<input"		"Radiobutton"   	  ("<input type=\"RADIO\" name=\"" (p "Name: ") "\">"))
   (form    "t"	      "<input"		"Text Field"	          ("<input type=\"TEXT\" name=\"" (p "Name: ") "\" size=\"" (p "Size: ") "\">"))
   (form    "f"	      "<form"           "Form"		          ("<form action=\"" (p "Action: ") "\" method=\"" (p "Method: ") "\">\n</form>\n"))

   ;;lists
   (list    "t"       "<dt>"            "Definition Item"         (& "<dt>" > (p "Term: ") "</dt>\n<dd>" > (r "Definition: ") "</dd>"))
   (list    "l"       "<li>"            "List Item"               (& "<li>" r "</li>" > (r "Item: ")))
   (list    "o"	      "<ol>"		"Ordered List"   	  (& "<ol>" > "\n<li>" r "</li>" > (r "Item: ") "\n</ol>" >))
   (list    "d"	      "<dl>"		"Definition List" 	  (& "<dl>" > "\n<dt>" > (p "Term: ") "</dt>\n<dd>" > (r "Definition: ") "</dd>\n</dl>" >))
   (list    "u"	      "<ul>"		"Unordered List" 	  (& "<ul>" > "\n<li>" > (r "Item: ") "</li>\n</ul>" >))

   ;;anchors
   (anchor  "n"	      "<a name="	"Link Target"	  ("<a name=\"" (p "Anchor name: ") "\">" (r "Anchor text: ") "</a>"))
   (anchor  "l"	      "<a href="        "Hyperlink"          	  ("<a href=\"" (p "URL: ") "\">" (r "Anchor text: ") "</a>"))

   ;;graphics
   (image   "i"       "<img alt="	"Image"	  ("<img alt=\"" (p "Alternate text: ") "\" src=\"" (r "Image URL: ") "\">"))
   (image   "u"       "<img usemap="	"Image with client-side imagemap" ("<img usemap=\"" (r "URL of usemap: ") "\" alt=\"" (p "Alternate text: ") "\" src=\"" (r "Image URL: ") "\">"))
   (image   "d"       nil	"Image with dimensions"	  ("<img width=\"" (p "Width: ") "\" height=\"" (p "Height: ") "\" alt=\"" (p "Alternate text: ") "\" src=\"" (r "Image URL: ") "\">"))
   (image   "l"       "<img longdesc="	"Image with longdesc"	  ("<img longdesc=\"" (p "URL of long description: ") "\" alt=\"" (p "Alternate text: ")  " src=\"" (r "Image URL: ") "\">"))
   (image   "f"       nil	"Image full attributes"	  ("<img src=\"" (r "Image URL: ") "\" alt=\"" (p "Alternate text: ") "\" width=\"" (p "Width: ") "\" height=\"" (p "Height: ")  "\" longdesc=\"" (p "URL of long description: ") "\">"))

   ;;misc containers
   (contain "sn"       "<span"    "Span (normal)"      ("<span>" r "</span>"))
   (contain "sc"       nil    "Span with class"      ("<span class=\"" (p "Class: ") "\">" r "</span>"))
   (contain "si"       nil    "Span with id"      ("<span id=\"" (p "ID: ") "\">" r "</span>"))
   (contain "sb"       nil    "Span with class and id"      ("<span class=\"" (p "Class: ") "\" id=\"" (p "ID: ") "\">" r "</span>"))
   (contain "dn"       "<div"    "Division (normal)"  ("<div>" r "</div>"))
   (contain "dc"       nil    "Division with class"      ("<div class=\"" (p "Class: ") "\">" r "</div>"))
   (contain "di"       nil    "Division with id"      ("<div id=\"" (p "ID: ") "\">" r "</div>"))
   (contain "db"       nil    "Division with class and id"      ("<div class=\"" (p "Class: ") "\" id=\"" (p "ID: ") "\">" r "</div>"))


   ;;revision marks
   (revise  "in"       "<ins"     "Insertion (normal)" ("<ins>" r "</ins>"))
   (revise  "it"       nil     "Insertion with time" ("<ins time=\"" (p "Time: ") "\">" r "</ins>"))
   (revise  "dn"       "<del"     "Deletion (normal)" ("<del>" r "</del>"))
   (revise  "dt"       nil     "Deletion with time" ("<del time=\"" (p "Time: ") "\">" r "</del>"))

   ;;text elements
   (textel  "\C-c="    nil		"Horizontal Line"	  (& "<hr>\n"))
   (textel  "\C-c\C-m" nil		"Line Break"		  ("<br>\n"))
   (textel  "\e\C-m"  nil		"Paragraph"	  ("\n<p>" r "</p>"))

   ;;head elements
   (head    "H"       "<head>"          "Head"            ("<head>\n" p "</head>\n"))
   (head    "B"       "<body>"          "Body"            ("<body>\n" p "</body>\n"))
   (head    "i"	      "<isindex>"	"Isindex"         ("<isindex>\n"))
   (head    "n"	      "<nextid>"	"Nextid"          ("<nextid>\n"))
   (head    "s"	      "<style>"	        "Style"           ("<style>\n" p "</style>\n"))
   (head    "S"	      "<script"	        "Script"          ("<script>\n" p "</script>\n"))
   (head    "h"       "<meta http-equiv=" "HTTP Equivalent" ("<meta http-equiv=\"" (p "Equivalent: ") "\" content=\"" (r "Content: ") "\">\n"))
   (head    "m"       "<meta name="     "Meta Name"       ("<meta name=\"" (p "Name: ") "\" content=\"" (r "Content: ") "\">\n"))
   (head    "l"	      "<link"		"Link"            ("<link href=\"" p "\">"))
   (head    "b"       "<base"		"Base"            ("<base href=\"" r "\">"))
   (head    "t"	      "<title>"		"Title"           ("<title>" (r "Document title: ") "</title>"))

   ;;objects
   (object     "o"       "<object"         "Object"          ("<object classid=\"" (p "Class ID:") "\">" r "</object>"))
   (object     "d"       nil               "Object with data, type, codebase, and codetype" ("<object classid=\"" (p "Class ID:") "\" data=\"" p "\" type=\"" p "\" codebase=\"" p "\" >" r "</object>"))
   (object     "h"       nil               "Object with height and width" ("<object classid=\"" (p "Class ID:") "\" height=" p " width=" p ">" r "</object>"))
   (object     "p"       "<param"          "Parameter"       ("<param name=\"" (p "Name:") "\" value=\"" (p "Value:") "\">"))
   (object     "t"       nil     "Parameter with value type" ("<param name=\"" (p "Name:") "\" value=\"" (p "Value:") "\" valuetype=\"" (p "Value Type (DATA, REF, or OBJECT):") "\">"))

   ;;miscellaneous (attributes)
   (misc       "i"       nil     "Id attribute" (" id=\"" (p "Id:") "\""))
   (misc       "c"       nil     "Class attribute" (" class=\"" (p "Class:") "\""))
   (misc       "s"       nil     "Style attribute" (" style=\"" (p "Style:") "\""))

 ))


;;}}}
;;{{{ html-writer-smart-insert-item

;; there are two different kinds of items in HTML - those in regular
;; lists <li> and those in dictionaries <dt>..<dd>
;; This command will insert the appropriate one depending on context.

(defun html-writer-smart-insert-item (&optional arg)
  "Insert a new item, either in a regular list or a dictionary."
  (interactive "*P")
  (let ((case-fold-search t))
    (if
        (save-excursion
          (re-search-backward "<li>\\|<dt>\\|<ul>\\|<ol>\\|<dd>\\|<menu>\\|<dir>\\|<dl>" nil t)
          (looking-at "<dt>\\|<dl>\\|<dd>"))
        (tempo-template-html-definition-item arg)
      (tempo-template-html-list-item arg))))

;; special keybindings in the prefix maps (not in the list of tags)
(and (boundp 'html-writer-list-map)
     (define-key html-writer-list-map "i" 'html-writer-smart-insert-item))

;; and, special menu bindings
(and (boundp 'html-writer-list-menu)
     (setq html-writer-list-menu
	   (cons '["List Item" html-writer-smart-insert-item t] html-writer-list-menu)))

;;}}}
;;{{{ menu support

;; menus are built for easymenu. html-writer-add-tag builds
;; submenus based on tag type, the expert menu code lumps them
;; together into one list and calls easy-menu-define

(defvar html-writer-novice-menu
  '("HTML"
    ["Insert Paragraph" tempo-template-html-paragraph t]
    ["Insert Hyperlink" tempo-template-html-hyperlink t]
    ["Insert Big Header" tempo-template-html-header-2 t]
    ["Insert Unordered List" tempo-template-html-unordered-list t]
    ["Insert List Item" html-writer-smart-insert-item t]
    ["Insert Inlined Image" tempo-template-html-image-with-alternate-text t]
    ["Turn on Expert Menu" html-writer-toggle-expert-menu t])
  "Menu for novices, only installed if `html-writer-use-expert-menu' is nil")

(defun html-writer-menu nil
  "Return the proper menu. Looks at `html-writer-use-expert-menu'"
  (if html-writer-use-expert-menu
      (html-writer-expert-menu)
    html-writer-novice-menu))

(defun html-writer-rebuild-menu nil
  "Rebuild and install the HTML menu (using `easy-menu-define').
If `html-writer-use-expert-menu' is nil, then just use a novice menu."
  (let ((menu (html-writer-menu)))
    (easy-menu-remove menu)
    (easy-menu-define html-writer-mode-menu-symbol
		      html-writer-mode-map "HTML menus" menu)
    (easy-menu-add menu html-writer-mode-map)))

(defun html-writer-toggle-expert-menu (&optional arg)
  "Toggle full HTML menus. Optional arg acts like minor-mode args."
  (interactive "P")
  (setq html-writer-use-expert-menu
	(if (null arg) (not html-writer-use-expert-menu)
	  (> (prefix-numeric-value arg) 0)))
  (html-writer-rebuild-menu))

;; If browse-url loaded, add this in the novice menu.
(if (fboundp 'browse-url-of-file)
    (setq html-writer-novice-menu
	  (append html-writer-novice-menu
		  (list ["Load This Buffer in Browser" browse-url-of-file t]))))

;; Expert menus: consed up out of html-writer-installed-types
(defun html-writer-expert-menu ()
  "This menu is based on the current value of `html-writer-installed-types'.
This function can be called again, it redoes the entire menu."
  ;; first, reset this so we can call this again and again.
  (setq html-writer-mode-menu nil)

  ;; Cons in the toggle of the menu
  (setq html-writer-mode-menu
	(cons '["Turn on Novice Menu"
		html-writer-toggle-expert-menu t]
	      html-writer-mode-menu))

  ;; Now add in user-provided menu stuff
  (setq html-writer-mode-menu
	(append html-writer-user-menu html-writer-mode-menu))

  ;; Now cons in the browse-url functions
  (if (fboundp 'browse-url-of-file)
    (setq html-writer-mode-menu
	  (cons '["Load this Buffer in Browser" browse-url-of-file t]
		html-writer-mode-menu)))
  (if (and (boundp 'browse-url-browser-function) (fboundp browse-url-browser-function))
    (setq html-writer-mode-menu
	  (cons (vector "Browse URL at point" browse-url-browser-function t)
		html-writer-mode-menu)))

  ;; cons in the timestamp delimiters
  (setq html-writer-mode-menu
	(cons '["Insert Timestamp Delimiter"
		html-writer-insert-timestamp-delimiter-at-point t]
	      html-writer-mode-menu))

  ;; now cons up the main menu out of the submenus
  (mapcar
   (function (lambda (type)
	       (setq html-writer-mode-menu
		     (cons (html-writer-normalized-menu-for type)
			   html-writer-mode-menu))))
	  html-writer-installed-types)

  ;; now tack on our name
  (setq html-writer-mode-menu (cons "HTML" html-writer-mode-menu))
  html-writer-mode-menu)

(html-writer-rebuild-menu)

;;}}}

;;{{{ context guessing

;; guess where we are in indented lists based on the last list token.
;; it would be much better to try to match </ul> to <ul>, and </ol> to <ol>
;; etc, but that is pretty unwieldy and slow.
;; Note, we make select/option look like a list structure too, so indentation
;; works. This is a bit weird, but it's ok.

(defvar html-writer-any-list-item-start "<li>\\|<dt>\\|<dd>\\|<option\\|<th>\\|<td>")
(defvar html-writer-any-list-item-end "</li>\\|</dt>\\|</dd>\\|</th>\\|</td>")
(defvar html-writer-any-list-start "<dl>\\|<ul>\\|<ol>\\|<menu>\\|<dir>\\|<select\\|<table\\|<tr>")
(defvar html-writer-any-list-end "</dl>\\|</ul>\\|</ol>\\|</menu>\\|</dir>\\|</select>\\|</table>\\|</tr>")
(defvar html-writer-any-list
  (format "\\(%s\\)\\|\\(%s\\)\\|\\(%s\\)\\|\\(%s\\)"
	  html-writer-any-list-start
	  html-writer-any-list-end
	  html-writer-any-list-item-start
	  html-writer-any-list-item-end))

(defvar html-writer-indentation-list
  (format "\\(%s\\)\\|\\(%s\\)\\|\\(%s\\)"
	  html-writer-any-list-start
	  html-writer-any-list-end
	  html-writer-any-list-item-start))
(defvar html-writer-search-limit 2000 "limit on how far back we search")

(defun html-writer-context-symbol ()
  "Return the symbol the last match (against `html-writer-any-list') found."
  (cond ((match-beginning 1) 'list-start)
	((match-beginning 2) 'list-end)
	((match-beginning 3) 'item-start)
	((match-beginning 4) 'item-end)
	(t 'error)))

(defun html-writer-guess-prev-context ()
  "Figure out the last list-type tag before point relevant to indentation.
Returns 'item-start if the last list tag is a list item start
        'start      if the last list tag is the start of a list
        'end        if the last list tag is the end of a list.
Ignores list item ends, because those aren't reliable for indentation."
  (save-excursion
    (let* ((lim (max (point-min) (- (point) html-writer-search-limit)))
	   (context (if (re-search-backward html-writer-indentation-list lim t)
			(html-writer-context-symbol)
		      nil)))
      (cons context (current-indentation)))))

(defun html-writer-print-prev-context ()
  (interactive)
  (message "%s" (html-writer-guess-prev-context)))

;;}}}
;;{{{ indentation

(defvar html-writer-print-indent-info nil
  "If t, indent will print out information as a message.")

(defun html-writer-indent-command ()
  "Command for indenting HTML to the appropriate column.
Calls `html-writer-indent' which tries to examine how many levels down
in nested lists we are and does the appropriate indentation.'
See also `html-writer-basic-offset', `html-writer-item-continue-indent',
and `html-writer-never-indent'."
  (interactive)
  (html-writer-indent))

;; some ideas borrowed from cc-mode.el.
;; Basic logic:
;;   if this line is some sort of list token, indent according to prev context:
;;     if previous context was a list-end or item-start, use its indentation
;;     if previous context was a list start, indent forward basic-offset
;;     ignore previous list-ends, their indentation is unreliable.
;;     then if this is some sort of list-item, do special case fixups:
;;       if this is a item start or end and previous was a list end, go back
;;           item-continue-indent (the </ul> is probably indented for an <li>
;;       if this is a list end and previous was a list end, go back
;;           item-continue-indent (the </ul> is probably indented for an <li>
;;       if this is a list end and prev *not* a list end, go back basic-offset
;;   else if this line is not a list item, and previous line is a item-start
;;     indent continue-indent, because this is part of the item


(defun html-writer-indent ()
  "Indentation workhorse function."
  (if html-writer-never-indent
      ()
    (let ((m (point-marker))
	  (bol (progn (beginning-of-line) (point))))

      ;; unindent the line
      (delete-region (point) (progn (back-to-indentation) (point)))

      (let* ((where (html-writer-guess-prev-context))
	     (prev-context (car where))
	     (this-context nil)
	     (previ (cdr where))
	     (newi (cond
		    ((eq prev-context 'list-end) previ)
		    ((eq prev-context 'item-start) previ)
		    ((eq prev-context 'list-start) (+ previ html-writer-basic-offset))
		    (t previ))))

	;; newi is set to the basic indentation, now adjust indentation
	;; based on what the current line is.
	(if (looking-at html-writer-any-list)
	    (progn
	      (setq this-context (html-writer-context-symbol))
	      (cond
	       ;; item start or end and last line was a list-end: go backwards
	       ((and
		 (or (eq this-context 'item-start) (eq this-context 'item-end))
		 (eq prev-context 'list-end))
		(setq newi (- newi html-writer-item-continue-indent)))

	       ;; end of list and last line was an end: go backwards twice
	       ((and (eq this-context 'list-end) (eq prev-context 'list-end))
		(setq newi (- newi html-writer-item-continue-indent html-writer-basic-offset)))

	       ;; Any other end of list? Indent negative
	       ((and (eq this-context 'list-end))
		(setq newi (- newi html-writer-basic-offset)))

	       ;; start of list and last line beginning of item, go forwards
	       ((and (eq this-context 'list-start) (eq prev-context 'item-start))
		(setq newi (+ newi html-writer-item-continue-indent)))))

	  ;; default: no special case, indent forward for text
	  (cond
	   ;; last line an item? Beginning of continued item - go forward
	   ((eq prev-context 'item-start)
	    (setq newi (+ newi html-writer-item-continue-indent)))))

	(if html-writer-print-indent-info
	    (message "Last Context: %s, This Context: %s, Previous: %s New: %s" prev-context this-context previ newi))

	;; just in case
	(if (< newi 0)
	    (setq newi 0))
	(indent-to newi newi)

	;; adjust point to where it was before, or at start of indentation
	(goto-char (marker-position m))
	(if (< (current-column) (current-indentation))
	    (back-to-indentation))))))

;;}}}
;;{{{ completion finder for tempo

(defvar html-writer-completion-finder
  (if html-writer-emacs18
      'html-writer-emacs18-completion-finder
    "\\(\\(<\\|&\\).*\\)\\=")
  "Passed to tempo-use-tag-list, used to find tags to complete.")

;; The regexp finds everything between the last < or & and point,
;; which is good enough to match the tags HTML might complete.
;; emacs18 doesn't have the \= for regexps, though, so we do something
;; more hackish.

(defun html-writer-emacs18-completion-finder ()
  "Unfortunately emacs18 doesn't support \\= in regexps, so we do this hack.
If you have problems with it, maybe you should upgrade to emacs19 :-)"
  (let* ((where nil)
         (s (buffer-substring
             (point)
             (setq where (save-excursion
                           (re-search-backward "<\\|&" (min (point-min) 100) t)
                           (point))))))
    (cons s where)))

;;}}}

;;{{{ timestamps

(defun html-writer-update-timestamp ()
  "Basic function for updating timestamps.
It finds the timestamp in the buffer by looking for
`html-writer-timestamp-start', deletes all text up to
`html-writer-timestamp-end', and runs `html-writer-timestamp-hook' which
will should insert an appropriate timestamp in the buffer."
  (save-excursion
    (goto-char (point-max))
    (if (not (search-backward html-writer-timestamp-start nil t))
	(message "timestamp delimiter start was not found")
      (let ((ts-start (+ (point) (length html-writer-timestamp-start)))
	    (ts-end (if (search-forward html-writer-timestamp-end nil t)
			(- (point) (length html-writer-timestamp-end))
		      nil)))
	(if (not ts-end)
	    (message "timestamp delimiter end was not found. Type C-c C-t to insert one.")
	  (delete-region ts-start ts-end)
	  (goto-char ts-start)
	  (run-hooks 'html-writer-timestamp-hook)))))
  nil)

(defun html-writer-default-insert-timestamp ()
  "Default timestamp insertion function."
  (let ((time (current-time-string)))
    (insert "Last modified: "
	    (substring time 0 20)
	    (nth 1 (current-time-zone))
	    " "
	    (substring time -4)
	    "\n")))

(defun html-writer-insert-timestamp-delimiter-at-point ()
  "Simple function that inserts timestamp delimiters at point.
Useful for adding timestamps to existing buffers."
  (interactive)
  (insert html-writer-timestamp-start)
  (insert html-writer-timestamp-end))

;;}}}
;;{{{ html-writer-insert-new-buffer-strings

(tempo-define-template "html-skeleton" html-writer-new-buffer-template
		       nil
		       "Insert a skeleton for a HTML document")

(defun html-writer-insert-new-buffer-strings ()
  "Insert `html-writer-new-buffer-strings'."
  (tempo-template-html-skeleton))

;;}}}

;;{{{ html-writer-mode

(defun html-writer-mode ()
  "Mode for editing HTML documents.
For more documentation and the newest version,
see http://www.santafe.edu/~nelson/tools/

The main function html-writer-mode provides is a menu and keybindings
for the HTML tags one inserts when writing HTML documents. Selecting
the menu item or typing the key sequence for a command inserts the
corresponding tag and places point in the right place. If a prefix
argument is supplied, the tags is instead wrapped around the region.
Alternately, one can type in part of the tag and complete it with M-TAB.

There is also code for indentation, timestamps, skeletons for new
documents, and lots of other neat features.

\\{html-writer-mode-map}
Written by nelson@santafe.edu, http://www.santafe.edu/~nelson/
"
  (interactive)
  (kill-all-local-variables)

  (use-local-map html-writer-mode-map)
  (setq local-abbrev-table html-writer-mode-abbrev-table)
  (set-syntax-table html-writer-mode-syntax-table)

  (setq mode-name "HTML writer")
  (setq major-mode 'html-writer-mode)

  (make-local-variable 'comment-start)
  (make-local-variable 'comment-end)
  (make-local-variable 'comment-column)
  (make-local-variable 'comment-start-skip)
  (make-local-variable 'indent-line-function)

  ;; font-lock setup for various emacsen: XEmacs, Emacs 19.29+, Emacs <19.29.
  ;; By Ulrik Dickow <dickow@nbi.dk>.  (Last update: 05-Sep-1995).
  (cond	((string-match "XEmacs\\|Lucid" (emacs-version)) ; XEmacs/Lucid
	 (put major-mode 'font-lock-keywords-case-fold-search t))
	;; XEmacs (19.13, at least) guesses the rest correctly.
	;; If any older XEmacsen don't, then tell me.
	;;
	((string-lessp "19.28.89" emacs-version) ; Emacs 19.29 and later
	 (make-local-variable 'font-lock-defaults)
	 (setq font-lock-defaults '(html-writer-font-lock-keywords t t)))
	;;
	(t ; Emacs 19.28 and older
	 (make-local-variable 'font-lock-keywords-case-fold-search)
	 (make-local-variable 'font-lock-keywords)
	 (make-local-variable 'font-lock-no-comments)
	 (setq font-lock-keywords-case-fold-search t)
	 (setq font-lock-keywords html-writer-font-lock-keywords)
	 (setq font-lock-no-comments t)))

  (setq comment-start "<!-- "
	comment-end " -->"
	comment-start-skip "<!--[ \t]*"
	comment-column 0
	indent-line-function 'html-writer-indent)

  (tempo-use-tag-list 'html-writer-tempo-tags html-writer-completion-finder)

  (if html-writer-do-write-file-hooks
      (add-hook 'local-write-file-hooks 'html-writer-update-timestamp))

  (if (and html-writer-build-new-buffer (zerop (buffer-size)))
      (html-writer-insert-new-buffer-strings))

  (easy-menu-add (html-writer-menu) html-writer-mode-map)

  (run-hooks 'text-mode-hook)
  (run-hooks 'html-mode-hook)
  (run-hooks 'html-writer-mode-hook))

;;}}}

;;{{{ patterns for font-lock

;; By Ulrik Dickow <dickow@nbi.dk>.
;;
;; Originally aimed at Emacs 19.29.  Later on disabled syntactic fontification
;; and reordered regexps completely, to be compatible with XEmacs (it doesn't
;; understand OVERRIDE=`keep').
;;
;; We make an effort on handling nested tags intelligently.

;; font-lock compatibility with XEmacs/Lucid and older Emacsen (<19.29).
;;
(if (string-match "XEmacs\\|Lucid" (emacs-version))
    ;; XEmacs/Lucid
    ;; Make needed faces if the user hasn't already done so.
    ;; Respect X resources (`make-face' uses them when they exist).
    (let ((change-it
	   (function (lambda (face)
		       (or (if (fboundp 'facep)
			       (facep face)
			     (memq face (face-list)))
			   (make-face face))
		       (not (face-differs-from-default-p face))))))
      (if (funcall change-it 'html-writer-bold-face)
	  (copy-face 'bold 'html-writer-bold-face))
      (if (funcall change-it 'html-writer-italic-face)
	  (copy-face 'italic 'html-writer-italic-face))
      (if (funcall change-it 'html-writer-underline-face)
	  (set-face-underline-p 'html-writer-underline-face t))
      (if (funcall change-it 'font-lock-variable-name-face)
	  (set-face-foreground 'font-lock-variable-name-face "salmon"))
      (if (funcall change-it 'font-lock-reference-face)
	  (set-face-foreground 'font-lock-reference-face "violet")))
  ;; Emacs (any version)
  ;;
  ;; Note that Emacs evaluates the face entries in `font-lock-keywords',
  ;; while XEmacs doesn't.  So XEmacs doesn't use the following *variables*,
  ;; but instead the faces with the same names as the variables.
  (defvar html-writer-bold-face 'bold
    "Face used as bold.  Typically `bold'.")
  (defvar html-writer-italic-face 'italic
    "Face used as italic.  Typically `italic'.")
  (defvar html-writer-underline-face 'underline
    "Face used as underline.  Typically `underline'.")
  ;;
  (if (string-lessp "19.28.89" emacs-version)
      () ; Emacs 19.29 and later
    ;; Emacs 19.28 and older
    ;; Define face variables that don't exist until Emacs 19.29.
    (defvar font-lock-variable-name-face 'font-lock-doc-string-face
      "Face to use for variable names -- and some HTML keywords.")
    (defvar font-lock-reference-face 'underline ; Ugly at line breaks
      "Face to use for references -- including HTML hyperlink texts.")))

(defvar html-writer-font-lock-keywords
  (let (;; Titles and H1's, like function defs.
	;;   We allow for HTML 3.0 attributes, like `<h1 align=center>'.
	(tword "\\(h1\\|title\\)\\([ \t\n]+[^>]+\\)?")
	;; Names of tags to boldify.
	(bword "\\(b\\|h[2-4]\\|strong\\)\\([ \t\n]+[^>]+\\)?")
	;; Names of tags to italify.
	(iword "\\(address\\|cite\\|em\\|i\\|var\\)\\([ \t\n]+[^>]+\\)?")
	;; Regexp to match shortest sequence that surely isn't a bold end.
	;; We simplify a bit by extending "</strong>" to "</str.*".
	;; Do similarly for non-italic and non-title ends.
	(not-bend (concat "\\([^<]\\|<\\([^/]\\|/\\([^bhs]\\|"
			  "b[^>]\\|"
			  "h\\([^2-4]\\|[2-4][^>]\\)\\|"
			  "s\\([^t]\\|t[^r]\\)\\)\\)\\)"))
	(not-iend (concat "\\([^<]\\|<\\([^/]\\|/\\([^aceiv]\\|"
			  "a\\([^d]\\|d[^d]\\)\\|"
			  "c\\([^i]\\|i[^t]\\)\\|"
			  "e\\([^m]\\|m[^>]\\)\\|"
			  "i[^>]\\|"
			  "v\\([^a]\\|a[^r]\\)\\)\\)\\)"))
	(not-tend (concat "\\([^<]\\|<\\([^/]\\|/\\([^ht]\\|"
			  "h[^1]\\|t\\([^i]\\|i[^t]\\)\\)\\)\\)")))
    (list ; Avoid use of `keep', since XEmacs will treat it the same as `t'.
     ;; First fontify the text of a HREF anchor.  It may be overridden later.
     ;; Anchors in headings will be made bold, for instance.
     '("<a\\s-+href[^>]*>\\([^>]+\\)</a>"
       1 font-lock-reference-face t)
     ;; Tag pairs like <b>...</b> etc.
     ;; Cunning repeated fontification to handle common cases of overlap.
     ;; Bold complex --- possibly with arbitrary other non-bold stuff inside.
     (list (concat "<" bword ">\\(" not-bend "*\\)</\\1>")
	   3 'html-writer-bold-face t)
     ;; Italic complex --- possibly with arbitrary non-italic kept inside.
     (list (concat "<" iword ">\\(" not-iend "*\\)</\\1>")
	   3 'html-writer-italic-face t)
     ;; Bold simple --- first fontify bold regions with no tags inside.
     (list (concat "<" bword ">\\("  "[^<]"  "*\\)</\\1>")
	   3 'html-writer-bold-face t)
     ;; Any tag, general rule, just after bold/italic stuff.
     '("\\(<[^>]*>\\)" 1 font-lock-type-face t)
     ;; Titles and level 1 headings (anchors do sometimes appear in h1's)
     (list (concat "<" tword ">\\(" not-tend "*\\)</\\1>")
	   3 'font-lock-function-name-face t)
     ;; Underline is rarely used. Only handle it when no tags inside.
     '("<u>\\([^<]*\\)</u>" 1 html-writer-underline-face t)
     ;; Forms, anchors & images (also fontify strings inside)
     '("\\(<\\(form\\|i\\(mg\\|nput\\)\\)\\>[^>]*>\\)"
       1 font-lock-variable-name-face t)
     '("</a>" 0 font-lock-keyword-face t)
     '("\\(<a\\b[^>]*>\\)" 1 font-lock-keyword-face t)
     '("=[ \t\n]*\\(\"[^\"]+\"\\)" 1 font-lock-string-face t)
     ;; Large-scale structure keywords (like "program" in Fortran).
     ;;   "<html>" "</html>" "<body>" "</body>" "<head>" "</head>" "</form>"
     '("</?\\(body\\|form\\|h\\(ead\\|tml\\)\\)>"
       0 font-lock-variable-name-face t)
     ;; HTML special characters
     '("&[^;\n]*;" 0 font-lock-string-face t)
     ;; SGML things like <!DOCTYPE ...> with possible <!ENTITY...> inside.
     '("\\(<![a-z]+\\>[^<>]*\\(<[^>]*>[^<>]*\\)*>\\)"
       1 font-lock-comment-face t)
     ;; Comments: <!-- ... -->. They traditionally override anything else.
     ;; It's complicated 'cause we won't allow "-->" inside a comment, and
     ;; font-lock colours the *longest* possible match of the regexp.
     '("\\(<!--\\([^-]\\|-[^-]\\|--[^>]\\)*-->\\)"
       1 font-lock-comment-face t)))
    "Additional expressions to highlight in HTML writer mode.")

;;}}}
;;{{{ patterns for hilit19

;; Define some useful highlighting patterns for the hilit19 package.
;; These will activate only if hilit19 has already been loaded.
;; Thanks to <dickow@nbi.dk> for some pattern suggestions

;; SKQ - added script and style, for what good it does. Others to be
;; added.

(if (featurep 'hilit19)
    (hilit-set-mode-patterns
     'html-writer-mode
     '(("<!--" "-->" comment)
       ("<![a-z]+\\>[^<>]*\\(<[^>]*>[^<>]*\\)*>" nil comment) ;<!DOCTYPE ...>
       ("<title>" "</title>" defun)
       ("<h[1-6]>" "</h[1-6]>" bold) ;only colour inside tag
       ("<a\\b" ">" define)
       ("</a>" nil define)
       ("<img\\b" ">" include)
       ("<option\\|</?select\\|<input\\|</?form\\|</?textarea\\|</?script\\|</?style" ">" include)
       ;; First <i> highlighting just handles unnested tags, then do nesting
       ("<i>[^<]*</i>" nil italic)
       ("<b>" "</b>" bold)
       ("<i>" "</i>" italic)
       ("<u>" "</u>" underline)
       ("&[^;\n]*;" nil string)
       ("<" ">" keyword))
     nil 'case-insensitive)
  nil)

;;}}}

(provide 'html-writer-mode)
(provide 'html-helper-mode)             ;backward compatibility
(provide 'html-mode)			;for 14 character filename
(run-hooks 'html-load-hook)
(run-hooks 'html-helper-load-hook)      ;backward compatibility
(run-hooks 'html-writer-load-hook)

;;; html-writer-mode.el ends here
