#!/usr/bin/perl -w

# IE2G  - converts Microsoft Internet Explorer favorites to Galeon bookmarks
# 
# Version:   0.24  (2002-03-17)
#
# Copyright  (C) 2002 Francesco Poli, <e_frx@inwind.it>
#                     Via Amici 19 50131 Firenze, Italy
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
# Note: This script should be usable under any GNU/Linux system with a Perl
#       interpreter. At least, I hope so... ;-)
# 
#

# Parse command line arguments.
my $semicolon_sep_arg = join ";", ";", @ARGV, ";";
if ($semicolon_sep_arg =~ /;--version;/)
  {
    print
    "IE2G 0.24\nCopyright (C) 2002 Francesco Poli\n",
    "IE2G comes with ABSOLUTELY NO WARRANTY.\n",
    "This is free software, and you are welcome to redistribute it\n",
    "under the terms of the GNU General Public License.\n",
    "For more information about these matters, see the file named COPYING.\n";
    exit;
  }
elsif ($semicolon_sep_arg =~ /;--help;/)
  {
    print
    "Usage: $0 SOURCE DEST VERSION\n\n",
    "Find all Microsoft Internet Explorer favorites in the subtree with root",
    " in\nSOURCE and convert them into a Galeon bookmark file.\n",
    "This file is named `bookmarks.ie2g' and written in directory DEST.\n",
    "(bookmarks.ie2g should be manually copied to ~/.galeon/bookmarks.xml",
    " in\norder to get the converted bookmarks as permanent Galeon",
    " bookmarks)\n(Warning:",
    " file to be overwritten should have been manually backed up!)\n",
    "Argument VERSION specifies target Galeon version: it must be 0.8.2 or",
    " 0.12\n\n",
    "With the option --version, output version information and exit.\n",
    "With the option --help, display this help and exit.\n\n",
    "NOTE: IE2G is designed to convert Internet Explorer favorites that are\n",
    "stored in a filesystem subtree.\n",
    "For those versions of Internet Explorer that save favorites in a single\n",
    "file, you will need a different tool... sorry! :-(\n\n",
    "Report bugs to <e_frx\@inwind.it>.\n";
    exit;
  }
elsif (@ARGV != 3) 
  {
    die "$0: wrong number of arguments\n",
    "Try `$0 --help' for more information.\n";
  }
else
  {
    ($source, $dest, $version) = @ARGV;
  }


SWITCH:
  {
    if ($version eq "0.8.2") { $target = 1; $indent = ""  ; last SWITCH; }
    if ($version eq "0.12")  { $target = 2; $indent = "  "; last SWITCH; }
    die "$0: third argument must be 0.8.2 or 0.12\n",
    "Try `$0 --help' for more information.\n";
  }

# Discard final slash, if present.
$source =~ s/\/$//;
$dest   =~ s/\/$//;

# Open output file.
open XML, ">$dest/bookmarks.ie2g"
    or die "$0: cannot write $dest/bookmarks.ie2g\n";

# Initialize element counter.
$id = 0;

# Initialize warning arrays.
@w_book_not_valid    = ();
@w_book_not_readable = ();
@w_dir_not_exec      = ();

# Start recursion.
&create_category($source, 1, $indent, $target, "");

# Close output file.
close XML;

# Write warnings to stderr.
if (@w_book_not_valid)
  {
    warn "$0: WARNING: the following files don't seem to be",
         " valid IE favorite files:\n";

    foreach my $not_valid(@w_book_not_valid)
      {
        warn "$not_valid\n";
      }

    warn
    "Since their URLs were unavailable, fictitious URLs have been written\n\n";
  }

if (@w_book_not_readable)
  {
    warn "$0: WARNING: the following files don't seem to be readable:\n";

    foreach my $not_readable(@w_book_not_readable)
      {
        warn "$not_readable\n";
      }

    warn
    "Since their URLs were unreadable, fictitious URLs have been written\n\n";
  }

if (@w_dir_not_exec)
  {
    warn "$0: WARNING: the following directories don't seem to be",
         " executable:\n";

    foreach my $not_exec(@w_dir_not_exec)
      {
        warn "$not_exec\n";
      }

    warn
    "Since their content was unreachable, they have been assumed empty\n\n";
  }

exit;


   ##########################################################################


# Write a category or folder tag corresponding to directory $dir.
# If $header is 1 create header and root tag, otherwise create normal tag.
# $indent is an indentation string.  $target identifies target Galeon version
# (1 --> 0.8.2, 2 --> 0.12).  $wherewasi represents current directory path
# before changing to $dir.

sub 
create_category
{

    # Get parameters.
    my ($dir, $header, $indent, $target, $wherewasi) = @_;

    # Change working directory, if possible.
    unless (chdir $dir)
      {
        if ($header == 1)
          {
            # Explain what's wrong.
            warn "$0: WARNING: cannot cd to $dir\n";

            # Open and close root tag.
            print XML "<?xml version=\"1.0\"?>\n<bookmarks>\n";

            if ($target == 2)
              {
                print XML "  <folder id=\"0\" name=\"Bookmarks\"",
                          " expanded=\"TRUE\"",
                          " default_bookmarks_root=\"TRUE\"/>\n";
              }

            print XML "</bookmarks>\n";

            return;
          }
        else
          {
            # Directory is not executable.
            push @w_dir_not_exec, $wherewasi."/".$dir;

            # Report activity on stdout.
            print $indent."opening category \"".$dir."\"\n";

            # Filter category or folder name (to enforce XML syntax).
            (my $name = $dir) =~ s/&/%26/g;
            $name =~ s/</%3C/g;
            $name =~ s/>/%3E/g;
            $name =~ s/"/%22/g;

            # Open category or folder tag.
            if ($target == 1)
              {
                print XML $indent."<category name=\"".$name."\"",
                          " create_toolbar=\"FALSE\" expanded=\"TRUE\"",
                          " notes=\"\"/>\n";
                $id++;
              }
            else
              {
                print XML $indent."<folder id=\"".++$id."\"",
                          " name=\"".$name."\" expanded=\"TRUE\"/>\n";
              }

            # Report activity on stdout.
            print $indent."closing category \"".$dir."\"\n";

            return;
          }
      }

    # This will store current directory.
    my $whereami;

    # A flag to remember we must finish to write a tag.
    my $stmissing = 0;

    if ($header == 1)
      {
        # Where am I now?
        $whereami = $dir;

        # Report activity on stdout.
        print "Starting bookmark conversion...\n\n";

        # Open root tag.
        print XML "<?xml version=\"1.0\"?>\n<bookmarks>\n";

        if ($target == 2)
          {
            print XML "  <folder id=\"0\" name=\"Bookmarks\"",
                      " expanded=\"TRUE\"",
                      " default_bookmarks_root=\"TRUE\"";
            $stmissing = 1;
          }
      }
    else
      {
        # Where am I now?
        $whereami = $wherewasi."/".$dir;

        # Report activity on stdout.
        print $indent."opening category \"".$dir."\"\n";

        # Filter category or folder name (to enforce XML syntax).
        (my $name = $dir) =~ s/&/%26/g;
        $name =~ s/</%3C/g;
        $name =~ s/>/%3E/g;
        $name =~ s/"/%22/g;

        # Open category or folder tag.
        if ($target == 1)
          {
            print XML $indent."<category name=\"".$name."\"",
                      " create_toolbar=\"FALSE\" expanded=\"TRUE\"",
                      " notes=\"\"";
            $stmissing = 1;
            $id++;
          }
        else
          {
            print XML $indent."<folder id=\"".++$id."\"",
                      " name=\"".$name."\" expanded=\"TRUE\"";
            $stmissing = 1;
          }
      }

    # List and sort (case-insensitively) current category content.
    my @list = sort {uc($a) cmp uc($b)} `ls -F`;

    # Initialize subcategories array.
    my @subcategories = ();

    # Initialize sites array.
    my @sites = ();

    # Remember sites and subcategories.
    foreach my $element(@list)
      {
        # ls -F appends * to files which have executable flag set, so
        # remove *'s, if present.
        $element =~ s/\*$//;

        if ($element =~ /\.url$/)
          {
	    push @sites, $element;
          }
        elsif ($element =~ /\/$/)
          {
            push @subcategories, $element;
          }
      }

    # Is our category or folder tag empty (i.e. no sites or subcategories
    # within it) or not?
    my $tag_not_empty = (@sites or @subcategories);

    # If we have neither subcategories or sites then
    # our category or folder tag has to be closed with a slash.
    if ($stmissing)
      {
        if ($tag_not_empty)
          { 
            print XML ">\n";
          }
        else
          {
            print XML "/>\n";
          }
      }

    # Create sites.
    foreach my $site(@sites)
      {
        &create_site($site, "  $indent", $target, $whereami);
      }

    # Create subcategories.
    foreach my $subdir(@subcategories)
      {
        # Discard newline and slash.
        chomp $subdir;
        $subdir =~ s/\/$//;

        &create_category($subdir, 0, "  $indent", $target, $whereami);
      }

    if ($header == 1)
      {
        # Close root tag if it hasn't been closed already.
        if ($target == 2 and $tag_not_empty)
          {
            print XML "  </folder>\n";
          }

        print XML "</bookmarks>\n";

        # Report activity on stdout.
        print
        "\nBookmark conversion completed!\n$id elements processed!\n\n",
        "Remember to manually backup ~/.galeon/bookmarks.xml\n",
        "and then manually copy bookmarks.ie2g to it.\n\n";
      }
    else
      {
        # Close category tag if it hasn't been closed already.
        if ($tag_not_empty)
          {
            if ($target == 1)
              {
                print XML $indent."</category>\n";
              }
            else
              {
                print XML $indent."</folder>\n";
              }
          }

        # Report activity on stdout.
        print $indent."closing category \"".$dir."\"\n";
      }

    # Change back working directory.
    chdir "..";

}

   ##########################################################################


# Write a site tag corresponding to file $filename.
# $indent is an indentation string.  $target identifies target Galeon version
# (1 --> 0.8.2, 2 --> 0.12).  $whereami represents current directory path.

sub 
create_site
{

    # Get parameters.
    my ($filename, $indent, $target, $whereami) = @_;

    # Discard newline.
    chomp $filename;

    # Open file to be analysed, if possible.
    my $is_unreadable = 0;
    unless (open BOOK, "<$filename")
      {
        # File doesn't seem to be readable.
        $is_unreadable = 1;
        push @w_book_not_readable, $whereami."/".$filename;
      }

    # Extract the URL, if possible.
    my $url;
    if ($is_unreadable)
      {
        # Setup a fictitious URL.
        $url = "this is not a URL";
      }
    else
      {
        my $is_OK = 0;
        CHECK: while (my $line = <BOOK>)
          {
            if ($line =~ /^\[InternetShortcut\]/)
              {
                while ($line = <BOOK>)
                  {
                    if ($line =~ /^URL=(.*)$/)
                      {
                        $is_OK = 1;
                        $url = $1;
                        last CHECK;
                      }
                  }
              }
          }

        if ($is_OK)
          {
            # Analysed file seems OK.
            $url =~ s/\r$//;
          }
        else
          {
            # Analysed file doesn't seem to be a valid IE favorite file.
            push @w_book_not_valid, $whereami."/".$filename;

            # Setup a fictitious URL.
            $url = "this is not a URL";
          }
      }

    # Discard ".url".
    $filename =~ s/\.url$//;

    # Filter site name (to enforce XML syntax).
    (my $name = $filename) =~ s/&/%26/g;
    $name =~ s/</%3C/g;
    $name =~ s/>/%3E/g;
    $name =~ s/"/%22/g;

    # Filter URL (to enforce XML syntax).
    $url =~ s/&/&amp;/g;
    $url =~ s/</&lt;/g;
    $url =~ s/>/&gt;/g;
    $url =~ s/"/&quot;/g;

    # Insert site tag.
    if ($target == 1)
      {
        print XML $indent."<site name=\"".$name."\"",
                  " url=\"".$url."\" nick=\"\" notes=\"\"/>\n";
        $id++;
      }
    else
      {
        print XML $indent."<site id=\"".++$id."\" name=\"".$name."\"",
                  " url=\"".$url."\"/>\n";
      }

    # Report activity on stdout.
    print $indent."creating bookmark \"".$filename."\"\n";

    # Close the analysed file, if open.
    unless ($is_unreadable) { close BOOK; }

}


