/* iMBR_editor.c, v. 1.7, 2019-01-21 */
/* iMBR_editor.c, v. 1.6, 2018-01-28 */
/* iMBR_editor.c, v. 1.5, 2018-01-10 */
/* iMBR_editor.c, v. 1.4, 2017-12-26 */
/* iMBR_editor.c, v. 1.3, 1997-08-05 */

/*
 * Copyright (c) 1997-2019 Alexei G. Malinin <Alexei.Malinin@mail.ru>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */


#include <stdint.h>
#include <inttypes.h>
#include <ctype.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <math.h>
#include <assert.h>

#include "OS.h"

#pragma pack(1)


int to_ASCII(int ch) {
        return (ch & 0177);
}

#if !defined strlcpy
/*	$OpenBSD: strlcpy.c,v 1.15 2016/10/16 17:37:39 dtucker Exp $	*/

/*
 * Copyright (c) 1998, 2015 Todd C. Miller <Todd.Miller@courtesan.com>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/types.h>
#include <string.h>

/*
 * Copy string src to buffer dst of size dsize.  At most dsize-1
 * chars will be copied.  Always NUL terminates (unless dsize == 0).
 * Returns strlen(src); if retval >= dsize, truncation occurred.
 */
size_t
strlcpy(char *dst, const char *src, size_t dsize)
{
	const char *osrc = src;
	size_t nleft = dsize;

	/* Copy as many bytes as will fit. */
	if (nleft != 0) {
		while (--nleft != 0) {
			if ((*dst++ = *src++) == '\0')
				break;
		}
	}

	/* Not enough room in dst, add NUL and traverse rest of src. */
	if (nleft == 0) {
		if (dsize != 0)
			*dst = '\0';		/* NUL-terminate dst */
		while (*src++)
			;
	}

	return(src - osrc - 1);	/* count does not include NUL */
}
#endif

#if !defined strtonum
/*	$OpenBSD: strtonum.c,v 1.8 2015/09/13 08:31:48 guenther Exp $	*/

/*
 * Copyright (c) 2004 Ted Unangst and Todd Miller
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <errno.h>
#include <limits.h>
#include <stdlib.h>

#define	INVALID		1
#define	TOOSMALL	2
#define	TOOLARGE	3

long long
strtonum(const char *numstr, long long minval, long long maxval,
    const char **errstrp)
{
	long long ll = 0;
	int error = 0;
	char *ep;
	struct errval {
		const char *errstr;
		int err;
	} ev[4] = {
		{ NULL,		0 },
		{ "invalid",	EINVAL },
		{ "too small",	ERANGE },
		{ "too large",	ERANGE },
	};

	ev[0].err = errno;
	errno = 0;
	if (minval > maxval) {
		error = INVALID;
	} else {
		ll = strtoll(numstr, &ep, 10);
		if (numstr == ep || *ep != '\0')
			error = INVALID;
		else if ((ll == LLONG_MIN && errno == ERANGE) || ll < minval)
			error = TOOSMALL;
		else if ((ll == LLONG_MAX && errno == ERANGE) || ll > maxval)
			error = TOOLARGE;
	}
	if (errstrp != NULL)
		*errstrp = ev[error].errstr;
	errno = ev[error].err;
	if (error)
		ll = 0;

	return (ll);
}
#endif


#define NUL			'\0'
#define BEL			'\a'
#define BS			'\b'
#define DEL			127
#define LF			'\n'
#define CR			'\r'
#define SPACE			' '
#define ASTERISK		'*'
#define ESC			'\033'

/*
 * partition table (PT) entry (E) format:
 * - 0x00	BYTE		boot indicator (0x80: active, 0x00: inactive)
 * - 0x01	BYTE		start head
 * - 0x02	WORD		start cylinder, sector
 * - 0x04	BYTE		system type
 * - 0x05 	BYTE		end head
 * - 0x06	WORD		end cylinder, sector
 * - 0x08	LONG		start LBA sector
 * - 0x0C	LONG		number of sectors in a partition
 */
typedef struct _PTE {
	uint8_t			boot_flag;
	uint8_t			SH;
	uint16_t		STS;
	uint8_t			ID;
	uint8_t			EH;
	uint16_t		ETS;
	uint32_t		S_LBA_S;
	uint32_t		LBA_Ss_number;
} PT_ENTRY;

#define PTE_SIZE		16
#define PT_ENTRIES		4
#define ACTIVE_PTE		0x80

#define active_PTE(pte)		(  (uint8_t)(ACTIVE_PTE)  & (pte).boot_flag)
#define set_boot_flag(pte)	(  (uint8_t)(ACTIVE_PTE)  | (pte).boot_flag)
#define clear_boot_flag(pte)	(~((uint8_t)(ACTIVE_PTE)) & (pte).boot_flag)

#define MBR_SIZE		512					// 0x0200
#define iMBR_CODE_SIZE		440					// 0x01B8
#define OS_NAME_SIZE		8
#define iMBR_PROMT_SIZE		5
#define ONE_TICK		0.05492					// in seconds

struct _iMBR {
	uint8_t			code[
					iMBR_CODE_SIZE
					-sizeof(uint8_t)		// CHECK_TIMEOUT
					-sizeof(int16_t)		// TIMEOUT
					-PT_ENTRIES*OS_NAME_SIZE	// OS_NAMEs
					-OS_NAME_SIZE			// ANSWER
					-iMBR_PROMT_SIZE		// PROMPT
				];
	char			PROMPT[iMBR_PROMT_SIZE];
	char			ANSWER[OS_NAME_SIZE];
	char			OS_NAMEs[PT_ENTRIES][OS_NAME_SIZE];
	int16_t			TIMEOUT;				// in ticks
	uint8_t			CHECK_TIMEOUT;
};

typedef struct _MBR {
	struct _iMBR		iMBR;
	uint32_t		NT_SIGNATURE;
	uint16_t		UNUSED;
	PT_ENTRY		PTE[PT_ENTRIES];
	uint16_t		SIGNATURE;
} MBR;


const char * const err_msg1=	"the device/file writing error... ";
const char * const err_msg2=	"the device/file reading error... ";
const char * const err_msg3=	"load the iMBR bootcode... ";
const char * const err_msg4=	"0<= partition number <=3... ";
const char * const err_msg7=	"this partition is empty... ";
const char * const msg1=	"enter partition number: ";
const char * const msg2=	"the MBR has been written... ";
const char * const msg3=	"enter the OS name: ";
const char * const msg4=	"enter the timeout (0..1799 seconds): ";
const char * const msg5=	"press any key... ";
const char * const msg0=	"press the key: ";

#define MAX_msg_len		44

void message(const char * const msg) {
	printf("%c", CR);
	for (int i=0; i<MAX_msg_len; ++i)
		printf("%c", SPACE);
	printf("%c", CR);
	printf("%s", msg);
}

int get_answer(const char * const msg) {
	message(msg);
	return (get_ch());
}

void print_partition_geometry(int p, const PT_ENTRY * const PTE) {
	message("");
	printf("partition:			%i\r\n", p);
	printf("start head:			%" PRIu8 "\r\n", PTE->SH);
	printf("start cylinder/sector, hex:	%" PRIX16 "\r\n", PTE->STS);
	printf("end head, hex:			%" PRIu8 "\r\n", PTE->EH);
	printf("end cylinder/sector, hex:	%" PRIX16 "\r\n", PTE->ETS);
	printf("start LBA sector:		%" PRIu32 "\r\n", PTE->S_LBA_S);
	printf("number of LBA sectors:		%" PRIu32 "\r\n", PTE->LBA_Ss_number);
	printf("\r\n");
	get_answer(msg5);
}

void get_OS_name(char *name) {
	int i, editing;

	message(msg3);
	i=strlen(name);
	printf("%s", name);
	editing=1;
	while (editing) {
		int ch;

		switch (ch=get_ch()) {
			case EOF:
				clearerr(stdin);
				break;
			case ESC:
			case CR:
				name[i]=(char)NUL;
				editing=0;
				break;
			case BS:
			case DEL:
				if (i>0) {
					if ((ch=putchar(BS))!=EOF)
						if ((ch=putchar(SPACE))!=EOF)
							if ((ch=putchar(BS))!=EOF)
								--i;
					if (ch==EOF)
						clearerr(stdout);
				}
				break;
			default:
				if (isprint(ch=to_ASCII(ch)))
					if (i<OS_NAME_SIZE-1) {
						if (putchar(ch)!=EOF)
							name[i++]=(char)ch;
						else
							clearerr(stdout);
					}
				break;
		}
	}
}

void get_timeout(int16_t *timeout) {
	int i, editing;
	char t[5];

	*timeout=(int16_t)lround((double)(*timeout)*ONE_TICK);
	printf(msg4);
	if (snprintf(t, sizeof(t), "%" PRIi16, *timeout)==-1) {
		printf("snprintf() error: %s (line %i)\n", __FILE__, __LINE__-1);
		abort();
	}
	i=strlen(t);
	printf("%s", t);
	editing=1;
	while (editing) {
		int ch;

		switch (ch=get_ch()) {
			case EOF:
				clearerr(stdin);
				break;
			case ESC:
				editing=0;
				break;
			case CR:
				{
					const char *err_msg;
					int16_t tt;

					t[i]=(char)NUL;
					tt=(int16_t)strtonum(t, 0, 1799, &err_msg);
					if (!err_msg) {
						*timeout=tt;
						editing=0;
					}
				}
				break;
			case BS:
			case DEL:
				if (i>0) {
					if ((ch=putchar(BS))!=EOF)
						if ((ch=putchar(SPACE))!=EOF)
							if ((ch=putchar(BS))!=EOF)
								--i;
					if (ch==EOF)
						clearerr(stdout);
				}
				break;
			case '0':
			case '1':
			case '2':
			case '3':
			case '4':
			case '5':
			case '6':
			case '7':
			case '8':
			case '9':
				if ((unsigned)i<sizeof(t)-1) {
					if (putchar(ch)!=EOF)
						t[i++]=(char)ch;
					else
						clearerr(stdout);
				}
				break;
		}
	}
	*timeout=(int16_t)lround((double)(*timeout)/ONE_TICK);
}

void correct_MBR(int *iMBR_loaded, MBR *mbr) {
	if ((*iMBR_loaded=(!strncmp(mbr->iMBR.PROMPT, "OS: ", iMBR_PROMT_SIZE)))) {
		// partitions must not be empty
		for (int p=0; p<PT_ENTRIES; ++p)
			if (!mbr->PTE[p].ID) {
				mbr->PTE[p].boot_flag=clear_boot_flag(mbr->PTE[p]);
				*mbr->iMBR.OS_NAMEs[p]=(char)NUL;
			}
		// OS names must be unique or empty
		for (int i=0; i<PT_ENTRIES; ++i)
			for (int j=i+1; j<PT_ENTRIES; ++j)
				if (!strncmp(mbr->iMBR.OS_NAMEs[i], mbr->iMBR.OS_NAMEs[j], OS_NAME_SIZE))
					*mbr->iMBR.OS_NAMEs[i]=(char)NUL;
		// if TIMEOUT > 0 then CHECK_TIMEOUT must be > 0
		mbr->iMBR.CHECK_TIMEOUT=mbr->iMBR.TIMEOUT?1:0;
	}
}

int main(int argc, char *argv[]) {
	MBR mbr;
	int editing;

	assert((sizeof(PT_ENTRY)==PTE_SIZE));
	assert((sizeof(struct _iMBR)==iMBR_CODE_SIZE));
	assert((sizeof(MBR)==MBR_SIZE));

	if (argc!=4) {
		printf("Usage: %s <an iMBR bootcode file> <a device/file with an MBR to edit> <a device/file with the MBR to write>\r\n", argv[0]);
		exit(EXIT_FAILURE);
	}

	get_tty_mode();
	set_tty_raw_mode();

	memset(&mbr, 0, sizeof(mbr));

	editing=1;
	while (editing) {
		int iMBR_loaded;

		correct_MBR(&iMBR_loaded, &mbr);

		clear_screen();

		printf("The iMBR bootcode file:\t\t\t%s\r\n", argv[1]);
		printf("The device/file with an MBR to edit:\t%s\r\n", argv[2]);
		printf("The device/file with the MBR to write:\t%s\r\n", argv[3]);

		printf(		"+-----------+----------+-----------+---------+\r\n");
		printf(		"| partition | ID,  hex | boot flag | OS name |\r\n");
		printf(		"+-----------+----------+-----------+---------+\r\n");
		for (int p=0; p<PT_ENTRIES; ++p) {
			printf(	"|     %i     |    %02" PRIX8 "    |     %c     |", p, mbr.PTE[p].ID, active_PTE(mbr.PTE[p])?ASTERISK:SPACE);
			if (iMBR_loaded)
				printf(" %*s ", OS_NAME_SIZE-1, mbr.iMBR.OS_NAMEs[p]);
			else
				for (int i=0; i<(OS_NAME_SIZE-1)+2; ++i)
					printf("%c", SPACE);
			printf("|\r\n");
		}
		printf(		"+-----------+----------+-----------+---------+\r\n");

		printf(		"+------------+------------+------------------+\r\n");
		printf(		"|    iMBR    | default OS | timeout, seconds |\r\n");
		printf(		"+------------+------------+------------------+\r\n");

		if (iMBR_loaded)
			printf(	"|############|    %*s |             %4" PRIi16 " |\r\n", OS_NAME_SIZE-1, mbr.iMBR.ANSWER, (int16_t)lround((double)mbr.iMBR.TIMEOUT*ONE_TICK));
		else
			printf(	"|            |            |                  |\r\n");
		printf(		"+------------+------------+------------------+\r\n");

		printf("  <1> read the MBR\r\n");
		printf("  <2> load the iMBR bootcode\r\n");
		printf("  <3> set an OS name\r\n");
		printf("  <4> switch a partition boot flag\r\n");
		printf("  <5> set the default OS\r\n");
		printf("  <6> set the timeout\r\n");
		printf("  <7> write the MBR\r\n");
		printf("  <8> print a partition geometry\r\n");
		printf("<Esc> exit\r\n\r\n");

		switch (get_answer(msg0)) {
			case ESC:
				clear_screen();
				editing=0;
				continue;
			case '7':
				{
					uint8_t b[MBR_SIZE], *p;
					size_t nbytes=sizeof(b);
					int fd, err_c;

					memmove(b, &mbr, sizeof(b));			// b=mbr;

					err_c=1;
					while ((fd=open(argv[3], W_OPEN_FLAGS, S_IRUSR|S_IWUSR))==-1)
						if (errno!=EINTR)
							break;

#ifdef WINDOWS
					if (fd==-1)
						while ((fd=open(argv[3], WE_OPEN_FLAGS))==-1)
							if (errno!=EINTR)
								break;
#endif

					if (fd!=-1) {
						for (p=b; nbytes>0; ) {
							ssize_t w;

							w=write(fd, p, nbytes);
							if (w>0) {
								p+=w;
								nbytes-=w;
							}
							else {
								if (w==-1)
									if (errno!=EINTR)
										break;
								if (w==0)
									break;		// ERROR?
								else
									break;		// unreachable
							}
						}
						while ((err_c=close(fd))==-1)
							if (errno!=EINTR)
								break;
					}
					if ((nbytes > 0) || (err_c != 0))
						get_answer(err_msg1);
					else
						get_answer(msg2);
				}
				break;

			case '1':
				{
					uint8_t b[MBR_SIZE], *p;
					size_t nbytes=sizeof(b);
					int fd;

					while ((fd=open(argv[2], R_OPEN_FLAGS))==-1)
						if (errno!=EINTR)
							break;
					if (fd!=-1) {
						for (p=b; nbytes>0; ) {
							ssize_t r;

							r=read(fd, p, nbytes);
							if (r>0) {
								p+=r;
								nbytes-=r;
							}
							else {
								if (r==-1)
									if (errno!=EINTR)
										break;
								if (r==0)
									break;		// EOF
								else
									break;		// unreachable
							}
						}
						while (close(fd)==-1)
							if (errno!=EINTR)
								break;
					}
					if (nbytes==0)
						memmove(&mbr, b, sizeof(b));		// mbr=b;
					else
						get_answer(err_msg2);
				}
				break;

			case '2':
				{
					uint8_t b[iMBR_CODE_SIZE], *p;
					size_t nbytes=sizeof(b);
					int fd;

					while ((fd=open(argv[1], R_OPEN_FLAGS))==-1)
						if (errno!=EINTR)
							break;
					if (fd!=-1) {
						for (p=b; nbytes>0; ) {
							ssize_t r;

							r=read(fd, p, nbytes);
							if (r>0) {
								p+=r;
								nbytes-=r;
							}
							else {
								if (r==-1)
									if (errno!=EINTR)
										break;
								if (r==0)
									break;		// EOF
								else
									break;		// unreachable
							}
						}
						while (close(fd)==-1)
							if (errno!=EINTR)
								break;
					}
					if (nbytes==0)
						memmove(&mbr.iMBR, b, sizeof(b));	// mbr.iMBR=b;
					else
						get_answer(err_msg2);
				}
				break;

			case '3':
				if (iMBR_loaded) {
					int p;

					p=get_answer(msg1)-'0';
					if ((p>=0) && (p<PT_ENTRIES))
						if (mbr.PTE[p].ID)
							get_OS_name(mbr.iMBR.OS_NAMEs[p]);
						else
							get_answer(err_msg7);
					else
						get_answer(err_msg4);
				}
				else
					get_answer(err_msg3);
				break;
			case '4':
				if (iMBR_loaded) {
					int p;

					p=get_answer(msg1)-'0';
					if ((p>=0) && (p<PT_ENTRIES))
						if (mbr.PTE[p].ID)
							mbr.PTE[p].boot_flag=active_PTE(mbr.PTE[p])?clear_boot_flag(mbr.PTE[p]):set_boot_flag(mbr.PTE[p]);
						else
							get_answer(err_msg7);
					else
						get_answer(err_msg4);
				}
				else
					get_answer(err_msg3);
				break;
			case '5':
				if (iMBR_loaded) {
					// get_OS_name(mbr.iMBR.ANSWER);
					int p;

					p=get_answer(msg1)-'0';
					if ((p>=0) && (p<PT_ENTRIES))
						strlcpy(mbr.iMBR.ANSWER, mbr.iMBR.OS_NAMEs[p], sizeof(mbr.iMBR.ANSWER));
					else
						get_answer(err_msg4);
				}
				else
					get_answer(err_msg3);
				break;
			case '6':
				if (iMBR_loaded)
					get_timeout(&mbr.iMBR.TIMEOUT);
				else
					get_answer(err_msg3);
				break;
			case '8':
				{
					int p;

					p=get_answer(msg1)-'0';
					if ((p>=0) && (p<PT_ENTRIES))
						print_partition_geometry(p, &mbr.PTE[p]);
					else
						get_answer(err_msg4);
				}
				break;
		}
	}

	restore_tty_mode();
	return (EXIT_SUCCESS);
}


