/* MBR.S, v. 1.7, 2019-01-21 */
/* MBR.S, v. 1.6, 2018-01-28 */
/* MBR.S, v. 1.5, 2018-01-11 */
/* MBR.S, v. 1.4, 2017-12-17 */
/* MBR.S, v. 1.3, 1997-08-05 */

/*
 * Copyright (c) 1997-2019 Alexei G. Malinin <Alexei.Malinin@mail.ru>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */



/*
 * BIOS loads us at physical address 0x7C00; we set our stack to start
 * at 0000:0x7BFC and we relocate to 0000:0x0600
 *
 * environment:
 * - CPU			Intel 8086/8088
 * - mode			real
 * - registers:
 *   - %dl			boot drive (usually 0x80)
 *   - %dh			should be preserved
 *   - %es:%di			should be preserved, points to "$PnP"
 *     				installation check structure
 *     				(ideally, %es:%di is passed on to the PBR for
 *     				later use by the loaded operating system,
 *     				but PnP-enabled operating systems typically
 *     				also have fallback methods to retrieve the
 *     				PnP BIOS entry point later on so that most
 *     				operating systems do not rely on this)
 *
 * algorithm:
 * - disable interrupts
 * - set up our stack
 * - preserve the registers (see above)
 * - relocate to 0000:0x0600
 * - enable interrupts
 * - choose the active (0x80) primary partition
 * - load the corresponding PBR to 0000:0x7C00
 * - check the PBR signature:
 *   - the byte at 0000:0x7C00+0x01FE must be equal to 0x55
 *   - the byte at 0000:0x7C00+0x01FF must be equal to 0xAA
 *   = so the word at 0000:0x7C00+0x01FE must be equal to 0xAA55
 * - set up the environment for the PBR:
 *   - %ds:%si [& %ds:%bp]	point to the 16-byte MBR partition table entry
 *     				(in the relocated MBR) corresponding with
 *     				the chosen PBR
 *   - restore preserved registers
 * - run the PBR
 *
 * diagnostics:
 * - '?'			the bootable OS was not found
 * - '!'			the PBR read error
 *
 * memory layout:
 * - 0x7C00-0x7DFF		BIOS loads us here and we load a PBR here
 * - 0x0600-0x07FF		we relocate to here
 * - 0x7BFC-[0x0802]		our stack (grows down), 28 Kbytes
 *
 * partition table (PT) entry format:
 * - 0x00	BYTE		boot indicator (0x80: active, 0x00: inactive)
 * - 0x01	BYTE		start head
 * - 0x02	WORD		start cylinder, sector
 * - 0x04	BYTE		system type
 * - 0x05 	BYTE		end head
 * - 0x06	WORD		end cylinder, sector
 * - 0x08	LONG		start LBA sector
 * - 0x0C	LONG		number of sectors in a partition
 *
 * in the case of a partition that extends beyond the 8 GB boundary,
 * the LBA values will be correct, the CHS values will have their
 * maximums typically (C,H,S): (1023,255,63)
 *
 */

#define MBR_SIZE		0x0100		/* in words */
#define BR_SIGNATURE_OFFSET	0x01FE
#define BR_SIGNATURE_VALID	0xAA55
#define BR_ADDR			0x7C00
#define R_MBR_ADDR		0x0600
#define STACK_ADDR		0x7BFC
#define PT_OFFSET		0x01BE
#define PT_ENTRIES		4
#define PT_ENTRY_SIZE		16
#define ACTIVE_PARTITION_FLAG	0x80
#define OS_NAME_SIZE		8
#define NT_SIGNATURE_OFFSET	0x01B8		/* NT disk signature offset */


/*
 * BIOS interrupt 0x13 (Extensions - INSTALLATION CHECK):
 * - call with:
 *   - %ah	0x41
 *   - %bx	0x55AA
 *   - %dl	drive (0x80 for 1st HDD, 0x81 for 2nd,.. 0xFF for the last)
 * - returns:
 *   - CF:
 *     - set	failure (extensions are not supported)
 *       - %ah	error code (0x01 - invalid function)
 *     - clear	success
 *       - %ah	major version of extensions
 *         - 0x01	1.x
 *         - 0x20	2.0 / EDD-1.0
 *         - 0x21	2.1 / EDD-1.1
 *         - 0x30	EDD-3.0
 *       - %al	internal use
 *       - %bx	0xAA55 (must be verified!)
 *       - %dx	extension version
 *       - %cx	API subset support bitmap
 *         - 0x0001	extended disk access functions are supported
 *           		(%ah: 0x42-0x44, 0x47, 0x48)
 *         - 0x0002	removable drive controller functions are supported
 *           		(%ah: 0x45, 0x46, 0x48, 0x49;
 *           		int 0x15, %ah: 0x52)
 *         - 0x0004	enhanced disk drive (EDD) functions are supported
 *           		(%ah: 0x48, 0x4E),
 *           		extended drive parameter table is valid
 *         - 0xFFF8	reserved (0)
 *
 * BIOS interrupt 0x13 (Extensions - EXTENDED READ):
 * - call with:
 *   - %ah	0x42
 *   - %dl	drive (0x80 for 1st HDD, 0x81 for 2nd,.. 0xFF for the last)
 *   - %ds:%si	command packet segment:offset
 * - returns:
 *   - command packet's sector count field set to the number of
 *     successfully transferred sectors
 *   - CF:
 *     - set	failure (extensions are not supported)
 *       - %ah	error code (0x01 - invalid function)
 *     - clear	success
 *       - %ah	0x00
 *
 * - command packet:
 *   - 0x0000	BYTE	packet size (0x10 or 0x18)
 *   - 0x0001	BYTE	reserved (should be 0)
 *   - 0x0002	WORD	sectors to transfer (max 127)
 *   - 0x0004	LONG	transfer buffer segment:offset
 *   - 0x0008	QUAD	starting sector number (for non-LBA devices, compute as
 *     			(Cylinder*NumHeads+SelectedHead)*SectorPerTrack+\
 *     				SelectedSector-1
 *   - 0x0010	QUAD	(EDD-3.0, optional) 64-bit flat address of a transfer
 *   			buffer; used if DWORD at 0x04 is 0xFFFF:0xFFFF
 *

 * BIOS interrupt 0x13 (READ SECTORS FROM DISK INTO MEMORY):
 * - call with:
 *   - %ah	0x02
 *   - %al	number of sectors to read (must be nonzero)
 *   - %ch	low eight bits of cylinder number
 *   - %cl	sector number 1-63 (bits 0-5), high two bits of cylinder
 *     		(bits 6-7, hard disk only)
 *   - %dh	head number
 *   - %dl	drive number (bit 7 set for hard disk)
 *   - %es:%bx	-> data buffer
 * - returns:
 *   - CF:
 *     - set	error
 *       - %ah	error code
 *       - %al	number of transferred sectors
 *     - clear	success
 *       - %al	0x00 OR number of transferred sectors (depends on BIOS)
 * - notes:	errors on a floppy may be due to the motor failing to spin up
 *   		quickly enough; the read should be retried at least three times,
 *   		resetting the disk with %ah=00 between attempts
 * - bugs:	when reading from floppies, some AMI BIOSes (around 1990-1991)
 *   		trash the byte following the data buffer, if it is not arranged
 *   		to an even memory boundary; a workaround is to either make the
 *   		buffer word aligned (which may also help to speed up things),
 *   		or to add a dummy byte after the buffer; apparently some BIOSes
 *   		have bugs that may destroy %dx on return or not properly set the
 *   		Carry Flag; on the original IBM AT BIOS (1984/01/10) this
 *   		function does not disable interrupts for hard disks
 *   		(%dl >= 0x80)
 *
 * BIOS interrupt 0x13 (RESET DISK SYSTEM):
 * - call with:
 *   - %ah	0x00
 *   - %dl	drive number
 * - returns:
 *   - CF:
 *     - set	error (%ah: status)
 *     - clear	success (%ah: 0)
 * - notes:	forces controller to recalibrate drive heads (seek to track 0)
 *
 */

/*
 * parameters:	%bp -> PT entry, %dl: drive
 * returns:	CF
 * modifies:	%ax, %bx, %cx, %dx, %si, CF
 */
#if (!defined(LBA) && !defined(CHS))
#define READ_SECTOR							;\
	read_sector:							;\
		/* check for LBA support */				;\
		push	%dx						;\
		movw	$0x55AA, %bx					;\
		movb	$0x41, %ah					;\
		int	$0x13						;\
		pop	%dx						;\
		jc	CHS_read_sector					;\
		/* check that %bl and %bh have been exchanged */	;\
		cmpw	$0xAA55, %bx					;\
		jne	CHS_read_sector					;\
		/* do we have "read" available? */			;\
		testb	$0x01, %cl					;\
		jz	CHS_read_sector					;\
	LBA_read_sector:						;\
		/* load LBA sector number from the PT entry */		;\
		movw	$LBA_CMD, %si					;\
		movw	8(%bp), %cx					;\
		movw	%cx, LBA_SECTOR					;\
		movw	10(%bp), %cx					;\
		movw	%cx, LBA_SECTOR+2				;\
		movb	$0x42, %ah					;\
		int	$0x13						;\
		jnc	read_sector__end				;\
		/* LBA reading failed, let us try CHS reading */	;\
	CHS_read_sector:						;\
		movw	$BR_ADDR, %bx					;\
		movw	2(%bp), %cx					;\
		movb	1(%bp), %dh					;\
		/* cli */						;\
		/* read one sector */					;\
		movw	$0x0201, %ax					;\
		int	$0x13						;\
		/* sti */						;\
		jnc	read_sector__end				;\
		movb	$('!'), %al					;\
		call	print_ch					;\
		/* unnecessary: CF is already set */			;\
		/* stc */						;\
	read_sector__end:						;\

#endif

#if (defined(LBA) && !defined(CHS))
#define READ_SECTOR							;\
		.ascii		"1234567890123"				;\
	read_sector:							;\
		/* check for LBA support */				;\
		push	%dx						;\
		movw	$0x55AA, %bx					;\
		movb	$0x41, %ah					;\
		int	$0x13						;\
		pop	%dx						;\
		jc	CHS_read_sector					;\
		/* check that %bl and %bh have been exchanged */	;\
		cmpw	$0xAA55, %bx					;\
		jne	CHS_read_sector					;\
		/* do we have "read" available? */			;\
		testb	$0x01, %cl					;\
		jz	CHS_read_sector					;\
	LBA_read_sector:						;\
		/* load LBA sector number from the PT entry */		;\
		movw	$LBA_CMD, %si					;\
		movw	8(%bp), %cx					;\
		movw	%cx, LBA_SECTOR					;\
		movw	10(%bp), %cx					;\
		movw	%cx, LBA_SECTOR+2				;\
		movb	$0x42, %ah					;\
		int	$0x13						;\
		jnc	read_sector__end				;\
		/* LBA reading failed, let us try CHS reading */	;\
	CHS_read_sector:						;\
		stc							;\
		jnc	read_sector__end				;\
		movb	$('!'), %al					;\
		call	print_ch					;\
		/* unnecessary: CF is already set */			;\
		/* stc */						;\
	read_sector__end:						;\

#endif

#if (!defined(LBA) && defined(CHS))
#define READ_SECTOR							;\
		.ascii		"123456789012345678901234567890"	;\
		.ascii		"123456789012345"			;\
	read_sector:							;\
	CHS_read_sector:						;\
		movw	$BR_ADDR, %bx					;\
		movw	2(%bp), %cx					;\
		movb	1(%bp), %dh					;\
		/* cli */						;\
		/* read one sector */					;\
		movw	$0x0201, %ax					;\
		int	$0x13						;\
		/* sti */						;\
		jnc	read_sector__end				;\
		movb	$('!'), %al					;\
		call	print_ch					;\
		/* unnecessary: CF is already set */			;\
		/* stc */						;\
	read_sector__end:						;\

#endif



/*
 * BIOS interrupt 0x16 (CHECK FOR A KEYSTROKE):
 * - call with:
 *   - %ah	0x01
 * - returns:
 *   - ZF:
 *     - set	no keystroke is available
 *     - clear	a keystroke is available
 *   - %al	ASCII character
 *   - %ah	BIOS scan code
 * - notes:	if a keystroke is present, it is not removed from the keyboard
 *   		buffer; however, any extended keystrokes which are not
 *   		compatible with 83/84-key keyboards are removed by IBM and most
 *   		fully-compatible BIOSes in the process of checking whether
 *   		a non-extended keystroke is available; some (older) clone BIOSes
 *   		do not discard extended keystrokes and manage function %ah=0x00
 *   		and %ah=0x10 the same; the K3PLUS v6.00+ int 0x16 BIOS
 *   		replacement doesn't discard extended keystrokes (same as with
 *   		functions 0x10 and 0x20), but will always translate prefix 0xE0
 *   		to 0x00; this allows old programs to use extended keystrokes and
 *   		should not cause compatibility problems
 *
 */

#define CHK_KEY								;\
		movb	$1, %ah						;\
		int	$0x16						;\


/*
 * BIOS interrupt 0x16 (GET A KEYSTROKE):
 * - call with:
 *   - %ah	0x00
 * - returns:
 *   - %al	ASCII character
 *   - %ah	BIOS scan code
 * - notes:	on extended keyboards, this function discards any extended
 *   		keystrokes, returning only when a non-extended keystroke is
 *   		available; the BIOS scan code is usually, but not always,
 *   		the same as the hardware scan code processed by int 0x09;
 *   		it is the same for ASCII keystrokes and most unshifted
 *   		special keys (F-keys, arrow keys, etc), but differs for
 *   		shifted special keys; some (older) clone BIOSes do not
 *   		discard extended keystrokes and manage function %ah=0x00
 *   		and %ah=0x10 the same; the K3PLUS v6.00+ int 0x16 BIOS
 *   		replacement doesn't discard extended keystrokes (same as
 *   		with functions 0x10 and 0x20), but will always translate
 *   		prefix 0xE0 to 0x00; this allows old programs to use
 *   		extended keystrokes and should not cause compatibility
 *   		problems
 *
 */

#define GET_CH								;\
		xor	%ah, %ah					;\
		int	$0x16						;\



	.file		"MBR.S"
	.text
	.code16
	.globl		START
	.space		R_MBR_ADDR, 0
	START:
		cli
		/* set up our stack */
		xor	%ax, %ax
		mov	%ax, %ss
		movw	$STACK_ADDR, %sp
		/* preserve %di and %es */
		push	%di
		push	%es
		/* relocate us */
		mov	%ax, %ds
		mov	%ax, %es
		movw	$BR_ADDR, %si
		movw	$START, %di
		movw	$MBR_SIZE, %cx
		cld
		rep movsw
		jmp	$0, $1f
	1:
		sti
		/* preserve %dx */
		movw	%dx, SAVED_DX
		/* set our timer tick int */
		movw	$timer_tick_int, %dx
		/* %ax is already zeroed
		xor	%ax, %ax
		 */
		call	set_timer_tick_int
		/* save original timer tick int vector */
		push	%ax
		push	%dx
	choose_OS:
		call	clear_screen
		movw	$PROMPT, %si
		call	print_msg
		movw	$ANSWER, %bx		/* %bx -> ANSWER */
		/*
		cmpw	$0, (%bx)
		je	get_OS_NAME
		 */
		mov	%bx, %si
		call	print_msg
	chk_keystroke:
		CHK_KEY
		jnz	get_OS_NAME
	chk_timeout:
		testb	$0xFF, CHECK_TIMEOUT
		jz	get_OS_NAME
		cmpw	$0, TIMEOUT
		jg	chk_keystroke
		movb	$0, CHECK_TIMEOUT
		jmp	scan_PT
	get_OS_NAME:
		movb	$0, CHECK_TIMEOUT
		movw	$-1, %di
	calc_ANSWER_len:			/* %di: ANSWER length,        */
		inc	%di			/* range: 0..(OS_NAME_SIZE-1) */
		cmpb	$0, (%bx, %di)		/* (%bx, %di) == 0            */
		jne	calc_ANSWER_len

	get_ch:
		GET_CH
	chk_ch:
		cmpb	$0, %al			/* NUL */
		je	get_ch
		cmpb	$0x07, %al		/* BEL */
		je	get_ch
		cmpb	$0x0A, %al		/* LF */
		je	get_ch
	chk_DEL:
		cmpb	$127, %al		/* DEL */
		je	chk_min_len
	chk_BS:
		cmpb	$0x08, %al		/* BS */
		jne	chk_CR
	chk_min_len:
		/* check MIN ANSWER length */
		cmpw	$0, %di
		je	get_ch
		/* delete the last ch from ANSWER */
		dec	%di
		movb	$0, (%bx, %di)
		jmp	choose_OS
	chk_CR:
		cmpb	$0x0D, %al		/* CR */
		je	scan_PT
		/* check MAX ANSWER length */
		cmpw	$OS_NAME_SIZE-1, %di
		jge	get_ch
		/* append ch to ANSWER */
		movb	%al, (%bx, %di)
		inc	%di
		call	print_ch
		jmp	get_ch
	scan_PT:
		/* find the partition: from the last PT entry to the 1st one */
		movw	$(R_MBR_ADDR+PT_OFFSET+(PT_ENTRIES-1)*PT_ENTRY_SIZE), \
			%bp
		/* check OS names: from the last to the first */
		movw	$(OS_NAMES+(PT_ENTRIES-1)*OS_NAME_SIZE), %si
		movw	$ANSWER, %di
		movw	$PT_ENTRIES, %cx
	chk_PT_entry:
		/* is this partition active? */
		testb	$ACTIVE_PARTITION_FLAG, (%bp)
		jnz	chk_PT_entry_name
	go_to_next_PT_entry:
		subw	$PT_ENTRY_SIZE, %bp
		subw	$OS_NAME_SIZE, %si
		loop	chk_PT_entry
	no_bootable_OS:
		/* the bootable OS was not found */
		movb	$('?'), %al
		call	print_ch
	wait_for_a_keystroke:
		GET_CH
		jmp	choose_OS
	chk_PT_entry_name:			/* OS names are unique or "" */
		push	%cx
		push	%si
		push	%di
		movw	$OS_NAME_SIZE, %cx
		/* cld */
		repe cmpsb
		pop	%di
		pop	%si
		pop	%cx
		jne	go_to_next_PT_entry

	load_PBR:				/* %bp -> PT entry */
		/*
		 * clear the signature in the PBR buffer,
		 * such clearing seems unnecessary but some BIOSes have bugs!
		movw	$0, (BR_ADDR+BR_SIGNATURE_OFFSET)
		 */
		movw	SAVED_DX, %dx
		READ_SECTOR
		jc	wait_for_a_keystroke
		/* check the signature */
		cmpw	$BR_SIGNATURE_VALID, (BR_ADDR+BR_SIGNATURE_OFFSET)
		jne	no_bootable_OS
		call	clear_screen
		/* restore saved timer tick int vector */
		pop	%dx
		pop	%ax
		call	set_timer_tick_int
		/* set the environment */
		mov	%bp, %si
		/* %ax is already zeroed
		xor	%ax, %ax
		 */
		/* %bx and %cx zeroing is unnecessary */
		xor	%bx, %bx
		xor	%cx, %cx
		/* restore preserved registers */
		movw	SAVED_DX, %dx
		pop	%es
		pop	%di
		/* run the PBR */
		/* this does not work everywhere
		jmp	BR_ADDR
		 */
		/* but this works */
		jmp	PBR
		/*
		jmp	$0, $BR_ADDR
		 */



/*
 * BIOS interrupt 0x10 (GET CURRENT VIDEO MODE):
 * - call with:
 *   - %ah	0x0F
 * - returns:
 *   - %ah	number of character columns
 *   - %al	video mode
 *   - %bh	video page
 *
 * BIOS interrupt 0x10 (SET VIDEO MODE):
 * - call with:
 *   - %ah	0x00
 *   - %al	desired video mode
 * - returns:
 *   - %al	video mode flag
 */

/*
 * parameters:	
 * returns:	
 * modifies:	%al, %ah
 */
	clear_screen:
		push	%bx
		movb	$0x0F, %ah
		int	$0x10
		movb	%bh, VIDEO_PAGE
		xor	%ah, %ah
		int	$0x10
		pop	%bx
		ret



/*
 * BIOS interrupt 0x10 (TELETYPE VIDEO OUTPUT):
 * - call with:
 *   - %ah	0x0E
 *   - %al	character to output
 *   - %bh	video page
 *   - %bl	foreground color (for graphics modes only)
 * - returns:
 * - notes:	displays the character on the screen, advancing the
 *   		cursor and scrolling the screen as necessary;
 *   		characters 0x07 (BEL), 0x08 (BS), 0x0A (LF), 0x0D (CR)
 *   		are interpreted and do the expected things;
 *   		IBM PC ROMs dated 1981/4/24 and 1981/10/19 require
 *   		that %bh be the same as the current active video page
 * - bugs:	if the output causes the screen to scroll, %bp is
 *   		destroyed by BIOSes for which %ah=0x06 destroys %bp
 */

/*
 * parameters:	%si
 * returns:	
 * modifies:	%al, %ah, %si
 */
	print_msg:
		/* cld */
		lodsb
		test	%al, %al
		jz	print__end
		call	print_ch
		jmp	print_msg

/*
 * parameters:	%al
 * returns:	
 * modifies:	%ah
 */
	print_ch:
		push	%bx
		push	%bp
		movb	$0x0E, %ah
		movb	VIDEO_PAGE, %bh
		movb	$1, %bl			/* as in OpenBSD mbr.S */
		int	$0x10
		pop	%bp
		pop	%bx
	print__end:
		ret



/*
 * BIOS interrupt 0x1C (SYSTEM TIMER TICK):
 * - notes:	this interrupt is automatically called on each clock tick
 *   		by the int 0x08 handler; not available on NEC 9800 series PCs
 */

/*
 * parameters:	%ax, %dx
 * returns:	%ax, %dx
 * modifies:	%ax, %dx
 */
	set_timer_tick_int:
		cli
		xchgw	%dx, (4*0x1C)
		xchgw	%ax, (4*0x1C+2)
		sti
		ret

/*
 * parameters:	
 * returns:	
 * modifies:	
 */
	timer_tick_int:
		/* without "%cs:" modifier TIMEOUT does not work! */
		cmpw	$0, %cs:TIMEOUT
		jle	timer_tick_int__end
		decw	%cs:TIMEOUT
	timer_tick_int__end:
		iret



/* LBA sectors reading command packet */
	LBA_CMD:
		.byte		0x10		/* command packet size    */
		.byte		0x00		/* reserved               */
		.word		0x0001		/* sectors to transfer: 1 */
		.word		BR_ADDR		/* target buffer offset   */
		.word		0		/* target buffer segment  */
	LBA_SECTOR:
		.quad		0		/* sector number          */

	SAVED_DX:
		.word		0

	VIDEO_PAGE:
		.byte		0


	PROMPT:
		.string		"OS: "

	ANSWER:					/* also default OS name */
		.space		OS_NAME_SIZE, 0	/* "C" string */

/* OS names are unique or empty ("") */
	OS_NAMES:				/* "C" strings */
		.space		OS_NAME_SIZE*4, 0

	TIMEOUT:				/* signed, in ticks: 0-32767, */
		.word		27		/* or 0-1799 seconds,         */
						/* 1 tick equals 54.92 ms     */

/* if TIMEOUT > 0 then CHECK_TIMEOUT must be > 0 */
	CHECK_TIMEOUT:				/* for checking TIMEOUT only */
		.byte		1		/* before (and NOT after!)   */
						/* any pressed key           */


	. = R_MBR_ADDR+NT_SIGNATURE_OFFSET
	NT_SIGNATURE:
		.long		0xAAAAAAAA

	. = R_MBR_ADDR+PT_OFFSET
	PT:

	. = R_MBR_ADDR+BR_SIGNATURE_OFFSET
	BR_SIGNATURE:
		.word		0xAAAA

	. = R_MBR_ADDR+0x200

	. = BR_ADDR
	PBR:


