/* make_test_disk.c, v. 1.7, 2019-01-21 */
/* make_test_disk.c, v. 1.6, 2018-01-28 */
/* make_test_disk.c, v. 1.5, 2018-01-10 */
/* make_test_disk.c, v. 1.4, 2017-11-17 */

/*
 * Copyright (c) 1997-2019 Alexei G. Malinin <Alexei.Malinin@mail.ru>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */


#include <stdint.h>
#include <inttypes.h>
#include <ctype.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <assert.h>

#pragma pack(1)

/*
 * partition table (PT) entry (E) format:
 * - 0x00	BYTE		boot indicator (0x80: active, 0x00: inactive)
 * - 0x01	BYTE		start head
 * - 0x02	WORD		start cylinder, sector
 * - 0x04	BYTE		system type
 * - 0x05 	BYTE		end head
 * - 0x06	WORD		end cylinder, sector
 * - 0x08	LONG		start LBA sector
 * - 0x0C	LONG		number of sectors in a partition
 */
typedef struct _PTE {
	uint8_t			boot_flag;
	uint8_t			SH;
	uint16_t		STS;
	uint8_t			ID;
	uint8_t			EH;
	uint16_t		ETS;
	uint32_t		S_LBA_S;
	uint32_t		LBA_Ss_number;
} PT_ENTRY;

#define PT_ENTRIES		4
#define PTE_SIZE		16
#define BR_SIZE			512
#define MBR_CODE_SIZE		446

typedef struct _MBR {
	uint8_t			code[MBR_CODE_SIZE];
	PT_ENTRY		PTE[PT_ENTRIES];
	uint16_t		BR_SIGNATURE;
} MBR;

typedef struct _PBR {
	uint8_t			code[BR_SIZE-sizeof(uint16_t)];
	uint16_t		BR_SIGNATURE;
} PBR;

#define R_OPEN_FLAGS		(O_RDONLY)
#define W_OPEN_FLAGS		(O_CREAT|O_WRONLY|O_SYNC)

// returns:
// - 1: ERROR
// - 0: OK
int read_BR(const char * const BR_file_name, void *br) {
	uint8_t b[BR_SIZE], *p;
	size_t nbytes=sizeof(b);
	int fd;
	int R;

	R=0;
	while ((fd=open(BR_file_name, R_OPEN_FLAGS))==-1)
		if (errno!=EINTR)
			break;
	if (fd!=-1) {
		for (p=b; nbytes>0; ) {
			ssize_t r;

			r=read(fd, p, nbytes);
			if (r>0) {
				p+=r;
				nbytes-=r;
			}
			else {
				if (r==-1)
					if (errno!=EINTR)
						break;
				if (r==0)
					break;		// EOF
				else
					break;		// unreachable
			}
		}
		while (close(fd)==-1)
			if (errno!=EINTR)
				break;
	}
	if (nbytes==0)
		memmove(br, b, sizeof(b));		// br=b;
	else
		R=1;
	return (R);
}

// returns:
// - 1: ERROR
// - 0: OK
int write_BR(const char * const BR_file_name, off_t BR_file_offset, const void * const br) {
	uint8_t b[BR_SIZE], *p;
	size_t nbytes=sizeof(b);
	int fd, err_c;
	int R;

	R=0;
	memmove(b, br, sizeof(b));						// b=br;

	err_c=1;
	while ((fd=open(BR_file_name, W_OPEN_FLAGS, S_IRUSR|S_IWUSR))==-1)
		if (errno!=EINTR)
			break;
	if (fd!=-1) {
		if (lseek(fd, BR_file_offset, SEEK_SET)==BR_file_offset) {
			for (p=b; nbytes>0; ) {
				ssize_t w;

				w=write(fd, p, nbytes);
				if (w>0) {
					p+=w;
					nbytes-=w;
				}
				else {
					if (w==-1)
						if (errno!=EINTR)
							break;
					if (w==0)
						break;				// ERROR?
					else
						break;				// unreachable
				}
			}
			while ((err_c=close(fd))==-1)
				if (errno!=EINTR)
					break;
		}
	}
	if ((nbytes > 0) || (err_c != 0))
		R=1;
	return (R);
}

#define READ_BR(file_name, BR, BR_name, BR_index) {					\
	if (read_BR((file_name), (BR))) {						\
		printf("%s: %s%s reading error\n", (file_name), (BR_name), (BR_index));	\
		return (EXIT_FAILURE);							\
	}										\
}

#define WRITE_BR(file_name, file_offset, BR, BR_name, BR_index) {			\
	if (write_BR((file_name), (file_offset), (BR))) {				\
		printf("%s: %s%s writing error\n", (file_name), (BR_name), (BR_index));	\
		return (EXIT_FAILURE);							\
	}										\
}

int main(int argc, char *argv[]) {
	MBR mbr;
	PBR pbr[PT_ENTRIES];

	assert((sizeof(PT_ENTRY)==PTE_SIZE));
	assert((sizeof(MBR)==BR_SIZE));

	if ((argc<4)||(argc>7)) {
		printf("Usage:		%s <HDD> <MBR> <PBR0> [<PBR1> [<PBR2> [<PBR3>]]]\r\n", argv[0]);
		printf("Example 1:	%s /dev/rsd3c ./mbr \"\" ./pbr1\r\n", argv[0]);
		printf("Example 2:	%s /dev/rsd3c /dev/rsd3c ./pbr0 \"\" \"\" /tmp/pbr3\r\n", argv[0]);
		exit(EXIT_FAILURE);
	}

	READ_BR(argv[2], &mbr, "MBR", "");
	WRITE_BR(argv[1], 0, &mbr, "MBR", "");
	printf("MBR (from %s) start LBA sector: %" PRIu32 " (offset in bytes: %llu)\n", argv[2], 0, 0llu);

	for (int p=0; p<PT_ENTRIES; ++p)
		if (mbr.PTE[p].S_LBA_S!=0) {
			off_t file_offset;
			int i;
			char *file_name;
			char ps[2];

			// This is not calculated correctly for an unknown reason!
			// file_offset=(mbr.PTE[p].S_LBA_S)*(BR_SIZE);
			// printf("%llu\n", (unsigned long long)file_offset);

			// This is calculated correctly.
			file_offset=mbr.PTE[p].S_LBA_S;
			file_offset*=BR_SIZE;
			// printf("%llu\n", (unsigned long long)file_offset);

			file_name=((i=3+p)<argc)?argv[i]:NULL;
			snprintf(ps, sizeof(ps), "%i", p);
			READ_BR(file_name, &(pbr[p]), "PBR", ps);
			WRITE_BR(argv[1], file_offset, &(pbr[p]), "PBR", ps);
			printf("PBR%i (from %s) start LBA sector: %" PRIu32 " (offset in bytes: %llu)\n",
				p, file_name, mbr.PTE[p].S_LBA_S, (unsigned long long)file_offset);
		}

	return (EXIT_SUCCESS);
}


