/*
   Copyright (C) 2010 Stephane Pion
   This file is part of Intifada.

    Intifada is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Intifada is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Intifada.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef STANDALONE_PCAP_ASTERIX_SOURCE_HXX_
# define STANDALONE_PCAP_ASTERIX_SOURCE_HXX_

# include <intifada/intifada-config.h>
# include <Source_Reader.hxx>
# include <string>
# include <istream>
# include <stdexcept>
# include <set>

class Operation;

class Standalone_PCAP_Asterix_Source : public Source_Reader
{
public:
  typedef Source_Reader inherited;
private:
  typedef struct internal_pcap_hdr_s {
          uint32_t magic_number;   /* magic number */
          uint16_t version_major;  /* major version number */
          uint16_t version_minor;  /* minor version number */
          int32_t  thiszone;       /* GMT to local correction */
          uint32_t sigfigs;        /* accuracy of timestamps */
          uint32_t snaplen;        /* max length of captured packets, in octets */
          uint32_t network;        /* data link type */
  } internal_pcap_hdr_t;

  typedef struct internal_pcaprec_hdr_s {
          uint32_t ts_sec;         /* timestamp seconds */
          uint32_t ts_usec;        /* timestamp microseconds */
          uint32_t incl_len;       /* number of octets of packet saved in file */
          uint32_t orig_len;       /* actual length of packet */
  } internal_pcaprec_hdr_t;

  typedef std::set<uint32_t> uint32_set_t;

public:
  Standalone_PCAP_Asterix_Source(std::istream& source,const std::string& family,Operation &op);

  virtual ~Standalone_PCAP_Asterix_Source();

  /// Open source
    /**
     *  After this call, source could be accessed
     *
     * @return -1 if something goes wrong (errno=ENOTSUP)
     * @return 0 if source is opened
     */
    virtual int open();

    /// Read part source
    /**
     * read coherent source part
     * @return -1 if part couldn't be read
     * @return >0 if part is acquired size is returned
     *
     */
    virtual int read();

    void operator()();

    void register_mac_address(const std::string& mac);

private:
    int read_global_header();

    int read_packet_header();

    int read_packet_data();


    /// Read bytes from stream according to byte order
    /**
     *
     * @param target where to store byte stream
     * @throw std::out_of_range if stream could'nt be readed
     */
    template<typename T> void read_bytes(T& target,bool network_order=false)throw(std::out_of_range);

private:
  std::istream& source_;

  internal_pcap_hdr_t global_header_;
  bool swap_order_;

  internal_pcaprec_hdr_t packet_header_;

  uint8_t* buf_;
  int buf_size_;
  uint32_set_t macs_list_;

  std::string family_;
  int idx_;
  Operation& op_;
};

template <typename T>
void Standalone_PCAP_Asterix_Source::read_bytes(T& target,bool network_order)throw(std::out_of_range)
{
  size_t target_size=sizeof(target);
  uint8_t* buf=new uint8_t[target_size];

  source_.read(reinterpret_cast<char *>(buf),target_size);
  if(source_)
    {
    target=0;
    size_t shift=target_size;
    bool swap_order=swap_order_;
    if(network_order==true)
      {
        swap_order=false;
      }
    if(swap_order!=false)
      {
      shift=0;
      }
    for(size_t idx=0;idx<target_size;++idx)
      {
      if(swap_order!=true)
        {
        --shift;
        }
      target|=buf[idx] << (shift*8);
      if(swap_order!=false)
        {
        ++shift;
        }
      }
    }
  else
    {
    throw std::out_of_range(std::string());
    }
  delete[]buf;
  return;
}

#endif // PCAP_ASTERIX_SOURCE_HXX_
