/* parse.c -- parses lines written by the user
   Copyright (C) 2004 Julio A. Becerra

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#define _POSIX_SOURCE
#define _ISOC99_SOURCE

#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <stdio.h>
#include <ctype.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "common.h"
#include "parse.h"
#include "user_iface.h"
#include "protocol.h"
#include "contact_list.h"
#include "uconfig.h"
#include "transport.h"
#include "misc.h"

#define ADD_C 1
#define DEL_C 2
#define HELP_C 3
#define NICK_C 4
#define PORT_C 5
#define QUIT_C 6
#define CLOSE_C 7
#define INFO_C 8

#define MAX_W 10

#define ERR_UNK (ERR_LOCAL -1)
#define ERR_AMB (ERR_LOCAL -2)


static int get_words (char **t, const char *str, int max);
static void free_words (char **t, int max);
static int command_id (char *str);
static void help (int command_id);
static void print_contact_info (contact_t *c);

void 
pa_parse_line (contact_t *win_contact, const char *line)
{
	char *words [MAX_W];
	int co;
	int co_h;
	int w;
	char *ptr;
	contact_t *ct;
	port_t port;
	
	if (line [0] != '/') {
		if (win_contact) {
			pr_send_text (win_contact, line);
		}
		return;
	}
	
	for (w = 0; w < MAX_W; w++)
		words [w] = NULL;
	
	//ui_output_err (line);
	
	w = get_words (words, line, MAX_W);
	co = command_id (words [0]+1);
	
	switch (co) {
		case ADD_C:
			if (w > 2) {
				ct = malloc (sizeof (contact_t));
				if (ct == NULL)
					dmx_stop ();
				strncpy (ct->nick, words [1], MAX_NICK-1);
				ct->nick [MAX_NICK-1]='\0';
				strncpy (ct->hname, words [2], MAX_HNAME-1);
				ct->hname [MAX_HNAME-1]='\0';
				if (w == 4)
					ct->port = (port_t) strtoul (words [3],
					                             NULL, 10);
				else
					ct->port = DEFAULT_PORT;
				ct->next = NULL;
				ct->state.ip = 0;
				ct->state.socket=-1;
				ct->state.hello_pending=TRUE;
				ct->state.in_buffer=NULL;
				ct->state.in_buf_size=0;
				ct->state.in_buf_rcvd=0;
				ct->state.out_buffer=NULL;
				ct->state.out_buf_size=0;
				ct->state.hello_timeout=(timer_id)-1;
				ct->state.ui_info=NULL;
				cl_add_contact (ct);
			}
			else
				ui_output_err ("Not enough parameters.");
			break;
		case DEL_C:
			if (w > 1)
				cl_remove_contact (words [1]);
			else
				ui_output_err ("Not enough parameters.");
			break;
		case HELP_C:
			if (w > 1) {
				co_h = command_id (words [1]);
				if (co_h == ERR_UNK)
					ui_output_err ("No help for %s.", 
							words[1]);
				else if (co_h == ERR_AMB)
					ui_output_err ("Ambiguous command: %s", 
							words[1]);
				else
					help (co_h);
			}
			else
				help (0);
			break;
		case NICK_C:
			if (w > 1) {
				strncpy (cfg.nick, words[1], MAX_NICK-1);
				cfg.nick [MAX_NICK-1] = '\0';
				cf_save ();
				ui_redraw_nick ();
				pr_send_hello (BROADCAST);
			}
			else
				ui_output_err ("Not enough parameters.");
			break;
		case PORT_C:
			if (w > 1) {
				port = (port_t) strtoul (words[1], &ptr, 10);
				if (*ptr != '\0')
					ui_output_err ("Invalid port number.");
				else if (cfg.listen_port == port) {
					ui_output_info ("That's the current "
					                "listen port.");
				}
				else {
					cfg.listen_port = port;
					cf_save ();
					pr_send_hello (BROADCAST);
					ui_output_info ("Listen port changed "
					                "to %u. You need to "
					                "restart ipchat.",
					                cfg.listen_port);
				}
			}
			else
				ui_output_info ("You are listening on port %u.",
				                cfg.listen_port);
			break;		
		case QUIT_C:
			dmx_stop ();
			break;
		case CLOSE_C:
			if (w > 1) {
				contact_t *c = cl_find_by_nick (words[1]);
				if (c == NULL) {
					ui_output_err ("Contact %s not found.",
					               words[1]);
				}
				else {
					ui_close_win (c);
				}
			}
			else
				ui_close_cur_win ();
			break;
		case INFO_C:
			if (w > 1) {
				contact_t *c = cl_find_by_nick (words[1]);
				if (c == NULL) {
					ui_output_err ("Contact %s not found.",
					               words[1]);
				}
				else {
					print_contact_info (c);
				}
			}
			else
				ui_output_err ("Not enough parameters.");
			break;
		case ERR_UNK:
			ui_output_err ("Unknown command: %s", words [0]+1);
			break;
		case ERR_AMB:
			ui_output_err ("Ambiguous command: %s", words [0]+1);
			break;
		case ERR:
			dmx_stop();
	}

	free_words (words, MAX_W);
}

static int 
get_words (char **t, const char *str, int max)
{
	int i = 0, j = 0, w = 0;
	
	while (w < max) {
		if (isspace (str [j]) || str[j] == '\0') {
			if (i != j) {
				t[w] = (char*) malloc ((j-i+1) * sizeof (char));
				strncpy (t[w], str+i, j-i);
				t[w][j-i]='\0';
				w++;
			}
			while (isspace (str[j]))
				j++;
			if (str[j] == '\0')
				break;
			i = j;			
		}
		else
			j++;
	}

	return w;
}

static void 
free_words (char **t, int max)
{
	int i = 0;

	while (i<max && t[i] != NULL) {
		free (t[i]);
		i++;
	}
}

static int 
command_id (char *str)
{
	int i = 0;
	int ret = ERR_UNK;
	char *c;

	c = malloc (strlen (str) * sizeof (char));
	if (c == NULL)
		return ERR;

	strcpy (c, str);

	while (c[i] != '\0') {
		c[i] = tolower ((int) c[i]);
		i++;
	}
	
	switch (*c) {
		case 'a':
			if (strncmp (c, "add", strlen (c)) == 0)
				ret = ADD_C;		
			break;
		case 'c':
			if (strncmp (c, "close", strlen (c)) == 0)
				ret = CLOSE_C;
			break;
		case 'd':
			if (strncmp (c, "delete", strlen (c)) == 0)
				ret = DEL_C;
			break;
		case 'h':
			if (strncmp (c, "help", strlen (c)) == 0)
				ret = HELP_C;
			break;
		case 'i':
			if (strncmp (c, "info", strlen (c)) == 0)
				ret = INFO_C;
			break;
		case 'n':
			if (strncmp (c, "nick", strlen (c)) == 0)
				ret = NICK_C;
			break;
		case 'p':
			if (strncmp (c, "port", strlen (c)) == 0)
				ret = PORT_C;
			break;
		case 'q':
			if (strncmp (c, "quit", strlen (c)) == 0)
				ret = QUIT_C;
			break;
	}

	free (c);
	return ret;	
}

static void
help (int command_id)
{
	switch (command_id) {
		case ADD_C:
			ui_output_info ("");
			ui_output_info ("ADD <nick> <host> [<port>]");
			ui_output_info ("");
			ui_output_info ("Creates a new contact. Host may be "
			                "given as an IP address or as a "
			                "hostname.");
			break;
		case CLOSE_C:
			ui_output_info ("");
			ui_output_info ("CLOSE [<nick>]");
			ui_output_info ("");
			ui_output_info ("Closes current or specified window.");
			break;
		case DEL_C:
			ui_output_info ("");
			ui_output_info ("DELETE <nick>");
			ui_output_info ("");
			ui_output_info ("Deletes a contact.");
			break;
		case HELP_C:
			ui_output_info ("");
			ui_output_info ("HELP [<command>]");
			ui_output_info ("");
			ui_output_info ("Shows help on commands.");
			break;
		case INFO_C:
			ui_output_info ("");
			ui_output_info ("INFO <nick>");
			ui_output_info ("");
			ui_output_info ("Shows information about a contact.");
			break;
		case NICK_C:
			ui_output_info ("");
			ui_output_info ("NICK <new nick>");
			ui_output_info ("");
			ui_output_info ("Changes your nick.");
			break;
		case PORT_C:
			ui_output_info ("");
			ui_output_info ("PORT [<port>]");
			ui_output_info ("");
			ui_output_info ("Shows or changes your listening "
			                "port.");
			break;
		case QUIT_C:
			ui_output_info ("");
			ui_output_info ("QUIT");
			ui_output_info ("");
			ui_output_info ("Ends your ipchat session.");
			break;
		default:
			ui_output_info ("");
			ui_output_info ("\\b\\3Available commands");
			ui_output_info ("\\b\\3------------------");
			ui_output_info ("\\bADD");
			ui_output_info ("\\bCLOSE");
			ui_output_info ("\\bDELETE");
			ui_output_info ("\\bHELP");
			ui_output_info ("\\bINFO");
			ui_output_info ("\\bNICK");
			ui_output_info ("\\bPORT");
			ui_output_info ("\\bQUIT");
			ui_output_info ("");
			ui_output_info ("For information on a command use "
			                "/help <command>.");
			ui_output_info ("Command name abbreviations are "
			                "allowed (eg: /p).");
			ui_output_info ("Text formatting: \\b\\\\b for bold, "
			                "\\u\\\\u for underline\\u, \\\\1 to "
			                "\\\\7 for colors, \\\\0 to reset.");
			ui_output_info ("");
	}
}

static void
print_contact_info (contact_t *c)
{
	ip_t old_ip = c->state.ip;

	/* user may use /info to force IP resolution */
	tr_resolv (c);
	if (old_ip == 0) {
		/* try to connect */
		tr_connect (c);
	}

	ui_output_info ("");
	ui_output_info ("\\b\\3Contact Info");
	ui_output_info ("\\b\\3------------");
	ui_output_info ("\\b\\2Nickname:      \\0%s", c->nick);
	ui_output_info ("\\b\\2Host:          \\0%s", c->hname);
	if (c->state.ip == 0) {
		ui_output_info ("\\b\\2IP address:    \\0Not found!");
	}
	else {
		ui_output_info ("\\b\\2IP address:    \\0%s",
		                ipv4_to_string (c->state.ip));
	}
	ui_output_info ("\\b\\2Listen port:   \\0%u", c->port);
	ui_output_info ("\\b\\2State:         \\0%s",
	                c->state.socket < 0 ? "Disconnected" :
	                c->state.hello_pending ? "Waiting HELLO..."
	                                       : "Connected");
	ui_output_info ("");
}
