/* ui_common.h -- common header for the ncurses user interface
   Copyright (C) 2004 Maximiliano Pin

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <ncurses.h>
#include "user_iface.h"
#include "contact_list.h"
#include "parse.h"

#define TBUF_SIZE 16384		/* size of text buffers (power of 2) */
#define IBUF_SIZE (256 - 1)	/* size of input buffers (1 for '\0') */
#define MAX_HISTORY 64		/* input history size (per window) */
#define MAX_CONTACT_WINS 12	/* one for each function key (TODO grow!) */

#define SEQCH '\\'		/* attr sequence start character */
#define SEQST "\\"		/* the same as string (can cpp do this?) */

#define BLD SEQST "b"		/* bold */
#define UND SEQST "u"		/* underline */
#define RST SEQST "0"		/* reset */
#define RED SEQST "1"		/* red */
#define GRE SEQST "2"		/* green */
#define YEL SEQST "3"		/* yellow */
#define BLU SEQST "4"		/* blue */
#define MAG SEQST "5"		/* magenta */
#define CYA SEQST "6"		/* cyan */
#define WHI SEQST "7"		/* white */
#define NL "\n"			/* new line */

#define INF_PREF NL BLD YEL "[--]" RST " "   /* prefix for info messages */
#define ERR_PREF NL BLD RED "[!!]" BLD " "   /* prefix for info messages */

#define NICK_PREF BLU BLD "<" WHI            /* prefix for people nicks */
#define NICK_SUFF BLU ">" RST " "            /* suffix for people nicks */
#define MY_NICK_PREF BLU BLD "<" YEL         /* prefix for user's nick */
#define MY_NICK_SUFF BLU ">" RST " "         /* suffix for user's nick */

/* TODO all here or all in ui_output... */
#define NICK_PREF_SUFF_LEN 3         /* length of nick prefix + suffix */

#define TSTAMP_PREF GRE "[" WHI      /* prefix for time stamps */
#define TSTAMP_SUFF GRE "]" RST      /* suffix for time stamps */
#define TSTAMP_PREF_SUFF_LEN 2       /* length of time stamp prefix + suffix */
#define TSTAMP_LEN 5                 /* hh:mm */

extern WINDOW *win_text;	/* window where messages appear */
extern WINDOW *win_input;	/* window for line edition */
extern WINDOW *win_contacts;	/* window for the list of contacts */
extern WINDOW *win_nick;	/* window where user's nick appears */

/* Position and size of ncurses windows. */
extern struct win_info {
	int     pos_y;
	int     pos_x;
	int     nlines;
	int     ncols;
} i_text, i_input, i_contacts, i_nick;

/* Every contact with an open window has a pointer to this in its state. */
typedef struct {
	char    ibuf[IBUF_SIZE+1]; /* line edition buffer (1 for '\0') */
	int     ibuf_len;          /* current length of the line */
	int     ishow;             /* index of first character shown */
	int     icursor;           /* index of the char where the cursor is */
	char   *hist[MAX_HISTORY]; /* history cyclic buffer */
	int     hist_tail;         /* index of oldest history line */
	int     hist_head;         /* next to use (if equal, hist is void) */
	int     hist_pos;          /* position where the user is editing */
	char    tbuf[TBUF_SIZE];   /* cyclic buffer of text */
	int     tbuf_tail;         /* oldest char (must be start of line) */
	int     tbuf_head;         /* next to use (if equal, tbuf is void) */
	int     align;             /* column where text is aligned */
	BOOL    alarm;             /* something happened in the window */
	int     pos;               /* position of this win in array cwin */
	contact_t *contact;        /* contact associated */
} contact_win;

/* Some notes on contact windows:
   - There is always at least one window. If no contact is connected, window 0
     is a default window with 'contact' set to NULL in the contact_win. The
     first contact to connect uses this window (and sets the 'contact' field).
   - If 'contact' is not NULL, contact->state.ui_info must point back to the
     contact_win structure.
   - 'cwin' is sorted by key (element 0 has key F1, element 1 has key F2...).
   - Yes, 'pos' is redundant, but it's faster this way.
   - This scheme will help understanding how the input buffer works.

                ________________________
                | This is the window.  |
                |                      |
                |      cur->icursor(21)|
                |                  |   |
                |cur->ishow(3)     |   |
 text not shown ||                \|/  |  text not shown (from ishow+ncols)
             \  +----------------------+    |
             012|345678901234567890c234|567890   <-- Text writen by the user
                +----------------------+
                \______________________/
                    i_input.ncols(22)
             \_______________________________/
                    cur->ibuf_len(31)
*/

extern contact_win *cwin[MAX_CONTACT_WINS]; /* contact windows */
extern int cwins;                           /* number of contact windows */
extern contact_win *cur;                    /* current contact window */
extern contact_win *vcur;                   /* visualized contact window */
extern short seq_err_pair;                  /* color pair to show seq error */
extern int show_text_idx;                   /* last char shown, for PgUp/PgDn */

/* Prototypes - ui_input */
extern void set_good_ishow ();
extern void cb_read_key (int fd, void *data);
extern void change_window (int idx);

/* Prototypes - ui_output */
extern void paint_all ();
extern void paint_text ();
extern void paint_text_bottom ();
extern int move_text_index_up (int *pidx, int lines);
extern int move_text_index_down (int *pidx, int lines);
extern void paint_input ();
extern void paint_nick ();
extern void paint_contacts ();
extern void update ();
extern void remake_windows ();
extern void remake_nick_windows ();
extern void recalc_windows ();

/* Prototypes - ui_format */
extern void build_line (const char *pref, const char *nick, const char *suff,
                        const char *text, int lpref, int lnick, int lsuff,
                        int ltext);
extern void add_text (const char *txt, int len);
extern int print_cur_text (int from, int to);
extern void proc_format (WINDOW *win, const char *str, int n);
extern int format_waddnstr (WINDOW *win, const char *str, int n, BOOL ps);
extern void calc_align ();
