/* ui_input.c -- user input handling for the ncurses user interface
   Copyright (C) 2004 Maximiliano Pin

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#define _POSIX_SOURCE

#include <ctype.h>		/* iscntrl, isdigit */
#include <stdio.h>		/* stdlib.h (on some systems) */
#include <stdlib.h>		/* malloc, realloc, free, atoi */
#include <string.h>		/* memcpy, memcmp */
#include "ui_common.h"
#include "uconfig.h"

/* Prototypes */
static void input_left ();
static void input_right ();
static void input_up ();
static void input_down ();
static void input_backspace ();
static void input_delete ();
static void input_enter ();
static void recover_hist ();
static inline BOOL repeated_input ();
static void update_text ();

/* Callback function, called when the user presses a key. */
void
cb_read_key (int fd, void *data)
{
	int     c = wgetch (win_input);
	int     i;

	if (c >= KEY_F(1) && c <= KEY_F(12)) {
		change_window (c - KEY_F(1));
		return;
	}

	switch (c) {
	case ERR:
	case KEY_RESIZE:
		return;

		/*
		   case CTL_D:
		   case TAB:
		   case KEY_F(1):
		   case KEY_IC: ???
		 */
	case KEY_LEFT:
		input_left ();
		break;
	case KEY_RIGHT:
		input_right ();
		break;
	case KEY_UP:
		input_up ();
		break;
	case KEY_DOWN:
		input_down ();
		break;
	/* TODO erasechar()? */
	case KEY_BACKSPACE:
	case '\b':
	case 0x7f:		/* ASCII DEL */
		input_backspace ();
		break;
	case KEY_DC:
		input_delete ();
		break;
	case KEY_ENTER:
	case '\n':
		input_enter ();
		break;
	case KEY_HOME:
		cur->icursor = cur->ishow = 0;
		break;
	case KEY_END:
		cur->icursor = cur->ibuf_len;
		if (cur->icursor >= cur->ishow + i_input.ncols)
			cur->ishow = cur->icursor - i_input.ncols + 1;
		break;
	case KEY_PPAGE:
		move_text_index_up (&show_text_idx, i_text.nlines / 2);
		update_text ();
		break;
	case KEY_NPAGE:
		move_text_index_down (&show_text_idx, i_text.nlines / 2);
		update_text ();
		break;
	case '\f':		/* CTRL+F */
		remake_windows ();
		return;
	default:
		if (cur->ibuf_len == IBUF_SIZE || c > 255 || iscntrl (c)) {
			beep ();
			return;
		}

		for (i = cur->ibuf_len - 1; i >= cur->icursor; i--)
			cur->ibuf[i + 1] = cur->ibuf[i];

		cur->ibuf_len++;
		cur->ibuf[cur->icursor++] = (char)c;

		if (cur->icursor - cur->ishow >= i_input.ncols) {
			cur->ishow++;
		}
		break;
	}

	paint_input ();
	wnoutrefresh (win_input);
	doupdate ();
}

/* Change show possition of input buffer if needed (eg. after term resize). */
void
set_good_ishow ()
{
	if (cur->icursor >= (cur->ishow + i_input.ncols))
		cur->ishow = cur->icursor - i_input.ncols + 1;
}

/* Change current contact window to window 'idx' (the first is 0). */
void
change_window (int idx)
{
	if (idx >= 0 && idx < cwins) {
		cur = vcur = cwin[idx];
		set_good_ishow ();  /* in case the window was resized */

		if (cur->alarm)
			cur->alarm = FALSE;  /* make "(*)" disappear */

		paint_all ();
	}
	else
		ui_output_err ("Window %d does not exist.", idx + 1);
}

/* Move cursor to the left. */
static void
input_left ()
{
	if (cur->icursor > 0)
		cur->icursor--;
	else
		beep ();

	if (cur->icursor < cur->ishow)
		cur->ishow--;
}

/* Move cursor to the right. */
static void
input_right ()
{
	if (cur->icursor < cur->ibuf_len)
		cur->icursor++;
	else
		beep ();

	if (cur->icursor >= cur->ishow + i_input.ncols)
		cur->ishow++;
}

/* Recover previous line of history. */
static void
input_up ()
{
	if (cur->hist_pos != cur->hist_tail) {
		if (cur->hist_pos == cur->hist_head) {
			cur->hist[cur->hist_pos] = realloc (
			           cur->hist[cur->hist_pos], cur->ibuf_len + 1);
			memcpy (cur->hist[cur->hist_pos], cur->ibuf,
			        cur->ibuf_len);
			cur->hist[cur->hist_pos][cur->ibuf_len] = '\0';
		}
		if (cur->hist_pos == 0)
			cur->hist_pos = MAX_HISTORY;
		cur->hist_pos--;
		recover_hist ();
	}
	else
		beep ();
}

/* Recover next line of history. */
static void
input_down ()
{
	if (cur->hist_pos != cur->hist_head) {
		cur->hist_pos = (cur->hist_pos + 1) % MAX_HISTORY;
		recover_hist ();
	}
	else
		beep ();
}

/* Backspace pressed. */
static void
input_backspace ()
{
	int i;

	if (cur->icursor > 0) {
		input_left ();
		if (cur->icursor == cur->ishow)
			cur->ishow = (cur->ishow > 8) ? cur->ishow - 8 : 0;
		cur->ibuf_len--;
		for (i = cur->icursor; i < cur->ibuf_len; i++)
			cur->ibuf[i] = cur->ibuf[i + 1];
	}
	else
		beep ();
}

/* Delete pressed. */
static void
input_delete ()
{
	int i;

	if (cur->icursor < cur->ibuf_len) {
		cur->ibuf_len--;
		for (i = cur->icursor; i < cur->ibuf_len; i++)
			cur->ibuf[i] = cur->ibuf[i + 1];
	}
	else
		beep ();
}

/* Enter pressed. Send input buffer to text buffer and to module 'parse'. */
static void
input_enter ()
{
	char *new_hist;
	int   new_head;
	BOOL  send_it = TRUE;
	int   change_win = 0;

	/* if the user is watching old text (PgUp), move to the bottom */
	if (show_text_idx != cur->tbuf_head) {
		paint_text_bottom ();
	}

	/* build and output the line (not for commands) */
	if (cur->ibuf_len == 0 || cur->ibuf[0] != '/') {
		if (!cur->contact || cur->contact->state.hello_pending) {
			ui_output_err ("Contact not connected.");
			send_it = FALSE;
		}
		else {
			build_line (MY_NICK_PREF, cfg.nick, MY_NICK_SUFF,
			            cur->ibuf, strlen (MY_NICK_PREF),
			            strlen (cfg.nick), strlen (MY_NICK_SUFF),
			            cur->ibuf_len);
		}
	}

	/* check if it's /<num> */
	if (cur->ibuf_len > 1 && cur->ibuf[0] == '/' && isdigit(cur->ibuf[1])) {
		cur->ibuf[cur->ibuf_len] = '\0';
		change_win = atoi (&cur->ibuf[1]);
		send_it = FALSE;
	}

	/* add to history (don't add repeated lines) */
	if (cur->ibuf_len > 0 && !repeated_input ()) {
		new_head = (cur->hist_head + 1) % MAX_HISTORY;
		if (new_head == cur->hist_tail) {
			/* rotate */
			free (cur->hist[new_head]);
			cur->hist_tail = (new_head + 1) % MAX_HISTORY;
		}
		cur->hist[new_head] = cur->hist[cur->hist_head];
		new_hist = malloc (cur->ibuf_len + 1);
		if (!new_hist) {
			dmx_stop ();
			return;
		}
		memcpy (new_hist, cur->ibuf, cur->ibuf_len);
		new_hist[cur->ibuf_len] = '\0';
		cur->hist[cur->hist_head] = new_hist;
		cur->hist_head = new_head;
	}
	cur->hist_pos = cur->hist_head;

	if (send_it) {
		/* TODO convert to UTF-8? (maybe in parse, when sending) */
		/* the buffer is not null-terminated, but parse requires
		   that; this is why ibuf is one byte longer */
		cur->ibuf[cur->ibuf_len] = '\0';
		/* send to parse */
		pa_parse_line (cur->contact, cur->ibuf);
	}

	/* reset edit environment */
	cur->ibuf_len = cur->icursor = cur->ishow = 0;

	/* change window if requested */
	if (change_win) {
		change_window (change_win - 1);
	}
}

/* Copy the line in current history position to input buffer. */
static void
recover_hist ()
{
	int len = strlen (cur->hist[cur->hist_pos]);
	memcpy (cur->ibuf, cur->hist[cur->hist_pos], len);
	cur->ibuf_len = len;
	cur->icursor = len;
	if (len < i_input.ncols)
		cur->ishow = 0;
	else
		cur->ishow = len - i_input.ncols + 1;
}

/* Check whether the line in the current input buffer is the same as the
   last line in the history. */
static inline BOOL
repeated_input ()
{
	int prev, len;

	prev = cur->hist_head;
	if (prev == cur->hist_tail)
		return FALSE;

	if (prev == 0)
		prev = MAX_HISTORY;
	prev--;

	len = strlen (cur->hist[prev]);
	return (len == cur->ibuf_len &&
	        memcmp (cur->ibuf, cur->hist[prev], len) == 0);
}

/* Repaint text, for PgUp/PgDn. */
static void
update_text ()
{
	werase (win_text);
	paint_text ();
	wnoutrefresh (win_text);
	doupdate ();
}
