/* ui_output.c -- painting in the ncurses user interface
   Copyright (C) 2004 Maximiliano Pin

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#define _POSIX_SOURCE
#define _ISOC99_SOURCE		/* snprintf */

#include <stdio.h>		/* snprintf */
#include <string.h>		/* strlen */
#include "ui_common.h"
#include "uconfig.h"

#define COLS_CONTACTS 24	/* width of contacts window */
#define PROMPT "> "		/* show this next to the nick */

int show_text_idx = 0;		/* last char shown, for PgUp/PgDn */

/* Prototypes */
static void mvresize (WINDOW *w, struct win_info *wi);

/* Paint all. */
void
paint_all ()
{
	paint_text_bottom ();
	paint_contacts ();
	paint_nick ();
	paint_input ();

	update ();
}

/* Paint visible text of text buffer in text window. */
void
paint_text ()
{
	WINDOW *w = win_text;
	int     y, x, i, n;

	werase (w);
	getmaxyx (w, y, x);

	/* begin in last line minus number of lines */
	i = show_text_idx;
	n = move_text_index_up (&i, y);
	if (i == show_text_idx) {
		move_text_index_down (&show_text_idx, 1);
		i = show_text_idx;
		n = move_text_index_up (&i, y);
	}

	wmove (w, y - (n ? n : 1), 0);

	/* print from 'i' to head */
	calc_align ();
	print_cur_text (i, show_text_idx);
	wstandend (w);  /* clear all attributes */
}

/* Paint last lines of text buffer (move to the bottom). */
void
paint_text_bottom ()
{
	show_text_idx = cur->tbuf_head;
	paint_text ();
}

/* Move index specified number of lines up. */
int
move_text_index_up (int *pidx, int lines)
{
	int n = 0;

	while (*pidx != cur->tbuf_tail) {
		if (*pidx > 0)
			(*pidx)--;
		else
			*pidx = TBUF_SIZE - 1;

		/* this is approximate, one line of text may occupy
		   more than one line in the window, but ncurses window
		   scrolling does the trick */
		/* TODO calculate how many real lines are used by each line,
		   in order to make PgUp/PgDn work better */
		if (cur->tbuf[*pidx] == '\n' && ++n == lines)
			break;
	}

	return n;
}

/* Move index specified number of lines down. */
int
move_text_index_down (int *pidx, int lines)
{
	int n = 0;

	while (*pidx != cur->tbuf_head) {
		*pidx = (*pidx + 1) % TBUF_SIZE;
		/* TODO calculate how many real lines are used by each line,
		   in order to make PgUp/PgDn work better */
		if (cur->tbuf[*pidx] == '\n' && ++n == lines)
			break;
	}

	return n;
}

/* Paint visible text of input buffer in input window. */
void
paint_input ()
{
	WINDOW *w = win_input;
	int     y, x, i, show;
#ifdef EDIT_NO_COLOR /* TODO document, parametrize or remove */
	int     precurs;
#endif

	werase (w);
	getmaxyx (w, y, x);

	/* Paint top line */
	wmove (w, 0, 0);
	for (i = 0; i < x; i++)
		waddch (w, ACS_HLINE);
	mvwaddch (w, 0, i_text.ncols - i_nick.ncols, ACS_BTEE);

	show = cur->ibuf_len - cur->ishow;
	if (show >= i_input.ncols) {
		show = i_input.ncols;
		if (cur->icursor == cur->ibuf_len) {
			/* leave place for the cursor */
			show--;
		}
	}
#ifdef EDIT_NO_COLOR /* TODO document or remove */
	precurs = cur->icursor - cur->ishow;
	mvwaddnstr (w, 1, precurs, &(cur->ibuf[cur->icursor]), show - precurs);
	mvwaddnstr (w, 1, 0, &(cur->ibuf[cur->ishow]), precurs);
#else
	wmove (w, 1, 0);
	proc_format (w, cur->ibuf, cur->ishow);
	format_waddnstr (w, &(cur->ibuf[cur->ishow]), show, TRUE);
	wstandend (w);
	wmove (w, 1, cur->icursor - cur->ishow);
#endif
}

/* Paint prompt (with local nick). */
void
paint_nick ()
{
	WINDOW *w = win_nick;
	int     y, x, i;

	werase (w);
	getmaxyx (w, y, x);

	/* Paint top line */
	wmove (w, 0, 0);
	for (i = 0; i < x; i++)
		waddch (w, ACS_HLINE);

	wcolor_set (w, COLOR_YELLOW, NULL);
	mvwaddstr (w, 1, 0, cfg.nick);
	wcolor_set (w, COLOR_BLUE, NULL);
	wattr_on (w, WA_BOLD, NULL);
	mvwaddstr (w, 1, strlen (cfg.nick), PROMPT);
	wstandend (w);
}

/* Paint list of contacts. */
void
paint_contacts ()
{
	WINDOW *w = win_contacts;
	int     y, x, i, ncon;
	contact_t *c;
	contact_win *cw;
	char    stmp[COLS_CONTACTS];

	werase (w);
	getmaxyx (w, y, x);

	/* Paint side line */
	for (i = 0; i < y; i++)
		mvwaddch (w, i, 0, ACS_VLINE);

	/* number of connected contacts is the same as number of windows,
	   except when only the default window exists */
	if (cwins == 1 && !cur->contact)
		ncon = 0;
	else
		ncon = cwins;

	for (c = contacts; c; c = c->next) {
		cw = (contact_win *)(c->state.ui_info);
		if (cw) {
			i = cw->pos;
			if (i < y) {
				wmove (w, i++, 1);
				if (cw == vcur)
					wattr_on (w, WA_BOLD, NULL);
				snprintf (stmp, x, "%s[%c%d] %s %s",
				          i > 9 ? " " : "  ",
				          i > 12 ? '/' : 'F', i, c->nick,
				          (cw == vcur && c->state.hello_pending)
					               ? "-<-" :
				          (cw == vcur) ? "<-" :
					  c->state.hello_pending ? "--" :
					  cw->alarm ? "(*)" : "");
				wcolor_set (w, COLOR_YELLOW, NULL);
				waddnstr (w, stmp, 6);
				wcolor_set (w, COLOR_WHITE, NULL);
				waddstr (w, stmp + 6);
			}
		}
		else if (ncon < y && c->nick[0]) {
			wmove (w, ncon++, 1);
			wcolor_set (w, COLOR_BLUE, NULL);
			snprintf (stmp, x, "       %s", c->nick);
			waddstr (w, stmp);
		}
		wstandend (w);
	}
}

/* Update the terminal (after painting). */
void
update ()
{
	wnoutrefresh (win_text);
	wnoutrefresh (win_contacts);
	wnoutrefresh (win_nick);
	wnoutrefresh (win_input);
	doupdate ();
}

/* Remake output (useful on Ctrl+L and terminal resize). */
void
remake_windows ()
{
	/* exit and return to curses mode, so we get the new size */
	endwin ();
	refresh ();

	recalc_windows ();
	mvresize (win_text, &i_text);
	mvresize (win_input, &i_input);
	mvresize (win_contacts, &i_contacts);
	mvresize (win_nick, &i_nick);

	set_good_ishow ();
	paint_all ();
}

/* Remake windows involved in the change of user's nick. */
void
remake_nick_windows ()
{
	remake_windows ();
	remake_windows ();
	/* TODO this is what should be done, but for some reason
	   it doesn't work: */
#if 0
	recalc_windows ();
	mvresize (win_input, &i_input);
	mvresize (win_nick, &i_nick);

	set_good_ishow ();
	paint_nick ();
	paint_input ();
	update ();
#endif
}

/* Recalculate the position and size of every curses window. */
void
recalc_windows ()
{
	int     y, x, w_nick;

	getmaxyx (stdscr, y, x);
	w_nick = strlen (cfg.nick) + strlen (PROMPT);

	i_text.pos_y = 0;
	i_text.pos_x = 0;
	i_text.nlines = y > 2 ? y - 2 : 1;
	i_text.ncols = x > COLS_CONTACTS ? x - COLS_CONTACTS : 1;

	i_input.pos_y = i_text.nlines;
	i_input.pos_x = w_nick;
	i_input.nlines = 2;
	i_input.ncols = x - w_nick;

	i_contacts.pos_y = 0;
	i_contacts.pos_x = i_text.ncols;
	i_contacts.nlines = i_text.nlines;
	i_contacts.ncols = x - i_text.ncols;

	i_nick.pos_y = i_text.nlines;
	i_nick.pos_x = 0;
	i_nick.nlines = 2;
	i_nick.ncols = w_nick;
}

/* Set position and size of a curses window. */
static void
mvresize (WINDOW * w, struct win_info *wi)
{
	mvwin (w, wi->pos_y, wi->pos_x);
	wresize (w, wi->nlines, wi->ncols);
}
