/* tcp.c -- wrapper to simplify the use of TCP sockets
   Copyright (C) 2004 Maximiliano Pin

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#define _POSIX_SOURCE

#include <sys/types.h>
#include <stdio.h>		/* stdlib.h (on some systems) */
#include <stdlib.h>		/* sys/socket.h (on some systems) */
#include <sys/socket.h>		/* socket, getsockopt, setsockopt, bind,
				   listen */
#include <unistd.h>		/* fcntl, close */
#include <fcntl.h>		/* fcntl */
#include <string.h>		/* memset */
#include <errno.h>		/* errno */
#include <netinet/in.h>		/* htonl, htons, ntohl */
#include "tcp.h"

#define LISTEN_BACKLOG 15	/* max length of input conn. queue */

extern int errno;

/* Create a non-blocking socket connecting to ip:port (both in host order).
   When select() reports write availability, call tcp_connect_result() to
   see whether the connection was successful. This functions returns the file
   descriptor of the socket if the connection process begins successfully,
   and returns ERR on any error. */
sock_t
tcp_connect (ip_t ip, port_t port)
{
	struct sockaddr_in peeraddr;
	sock_t s;
	int ret;

	s = socket (AF_INET, SOCK_STREAM, 0);
	CHECK (s >= 0);

	fcntl (s, F_SETFL, O_NONBLOCK);
	memset (&peeraddr, 0, sizeof (peeraddr));
	peeraddr.sin_family = AF_INET;
	peeraddr.sin_addr.s_addr = htonl (ip);
	peeraddr.sin_port = htons (port);

	ret = connect (s, (struct sockaddr *)&peeraddr, sizeof (peeraddr));
	if (ret == 0 || errno == EINPROGRESS) {
		return s;
	}

	close (s);
error:
	return ERR;
}

/* Returns OK if the connection process of 'socket' was successful,
   otherwise it returns ERR and sets errno to the error code. */
int
tcp_connect_result (sock_t socket)
{
	int err, ret;
	socklen_t len = sizeof (err);

	ret = getsockopt (socket, SOL_SOCKET, SO_ERROR, &err, &len);
	CHECK (ret == 0);
	errno = err;
	CHECK (err == 0);

	return OK;
error:
	return ERR;
}

/* Create a non-blocking socket listening at the specified TCP port, on all
   interfaces. Returns the file descriptor on success, ERR on error. */
sock_t
tcp_listen (port_t port)
{
	struct sockaddr_in servaddr;
	sock_t s;
	int ret;
	static const int on = 1;

	s = socket (AF_INET, SOCK_STREAM, 0);
	CHECK_J (s >= 0, error_ret);

	fcntl (s, F_SETFL, O_NONBLOCK);
	setsockopt (s, SOL_SOCKET, SO_REUSEADDR, &on, sizeof (on));
	memset (&servaddr, 0, sizeof (servaddr));
	servaddr.sin_family = AF_INET;
	servaddr.sin_addr.s_addr = htonl (INADDR_ANY);
	servaddr.sin_port = htons (port);

	ret = bind (s, (struct sockaddr *)&servaddr, sizeof (servaddr));
	CHECK (ret == 0);

	ret = listen (s, LISTEN_BACKLOG);
	CHECK (ret == 0);

	return s;

error:
	close (s);
error_ret:
	return ERR;
}

/* Accept the next pending input connection request in 'socket'. Returns
   the connected socket and the originator IP in *pip, in host order.
   It returns ERR on error. */
sock_t
tcp_accept (sock_t socket, ip_t *pip)
{
	sock_t s;
	struct sockaddr_in cliaddr;
	socklen_t clilen;

	clilen = sizeof (cliaddr);
	s = accept (socket, (struct sockaddr *)&cliaddr, &clilen);
	if (s < 0)
		s = ERR;
	else
		*pip = ntohl (cliaddr.sin_addr.s_addr);

	return s;
}
