/*
* opendb2.c

Copyright (C) 2008-2011 Alessandro Vesely

This file is part of Ipqbdb.

Ipqbdb is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Ipqbdb is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Ipqbdb.  If not, see <http://www.gnu.org/licenses/>.

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#define __USE_BSD 1
// for u_int in db.h
#include <sys/types.h>
#include <db.h>

#include <syslog.h>
#include <math.h>

#include "config_names.h"
#include "dbstruct.h"

#include "utf8_util.h"

#include <assert.h>

/*
* retrieve the last used index for description table.
* db is the handle for the primary (bynumber) table.
* return -1 on error.
*/
int last_descr_ndx(DB *db)
{
	DBC *curs;
	int rtc = db->cursor(db, NULL, &curs, 0);
	if (rtc)
	{
		db->err(db, rtc, "cannot create cursor");
		return -1;
	}

	DBT key, data;
	char buf[1];
	int ndx = 0;
	memset(&key, 0, sizeof key);
	memset(&data, 0, sizeof data);
	key.data = &ndx;
	key.ulen = key.size = sizeof ndx;
	key.flags = DB_DBT_USERMEM;
	data.data = buf;
	data.ulen = sizeof buf;
	data.flags = DB_DBT_USERMEM | DB_DBT_PARTIAL;

	rtc = curs->c_get(curs, &key, &data, DB_LAST);
	if (rtc == DB_NOTFOUND)
		ndx = 0; // descr table is empty, start from 1
	else if (rtc)
	{
		ndx = -1;
		db->err(db, rtc, "cannot read cursor");
	}

	curs->c_close(curs);
	return ndx;
}

/*
* get description number by descr, possibly inserting it
*/
int get_reason_id(int *reason_id, char *reason_string,
	app_private *ap, DB *db1, DB *db2, int verbose)
{
	DBT key1, key2, data;
	DBC *curs;
	union
	{
		bynumber_descr_t drec;
		char fill[sizeof(bynumber_descr_t) + IPQBDB_DESCR_MAX_LENGTH + 1];
	} u;
	int id = 0, rtc = 0;

	memset(&key1, 0, sizeof key1);
	memset(&key2, 0, sizeof key2);
	memset(&data, 0, sizeof data);
	memset(&u, 0, sizeof u);

	// write cursor: prevent writing the description table until c_close below
	rtc = db2->cursor(db2, NULL, &curs, DB_WRITECURSOR);
	if (rtc)
	{
		db2->err(db2, rtc, "cannot get write cursor");
		return rtc;
	}

	if ((key2.size = strlen(reason_string)) >= IPQBDB_DESCR_MAX_LENGTH)
	{
		char *end =
			begin_utf8(reason_string,
				reason_string + IPQBDB_DESCR_MAX_LENGTH - 1);
		*end = 0;
		key2.size = end - reason_string;
		report_error(ap, LOG_WARNING,
			"reason too long, max=%d: truncated to \"%s\"\n",
			IPQBDB_DESCR_MAX_LENGTH - 1, reason_string);
	}

	key2.data = reason_string;
	key2.size += 1; // always include terminating 0
	key1.data = &id;
	key1.size = key1.ulen = sizeof id;
	key1.flags = key2.flags = DB_DBT_USERMEM;
	
	strcpy(u.drec.descr, reason_string);
	size_t wsize = key2.size + offsetof(bynumber_descr_t, descr);

	data.data = &u.drec.chk;
	data.size = data.ulen = data.dlen = sizeof u.drec.chk;
	data.doff = offsetof(bynumber_descr_t, chk);
	data.flags = DB_DBT_USERMEM | DB_DBT_PARTIAL;
	rtc = curs->c_pget(curs, &key2, &key1, &data, DB_SET);
	if (rtc == 0 && u.drec.chk == IPQBDB_CHK_SIGNATURE)
	{
		if (verbose)
			printf("found reason id %d for \"%s\"\n", id, reason_string);
		if (reason_id)
			*reason_id = id;
	}

	else if (rtc != DB_NOTFOUND)
	{
		db2->err(db2, rtc, "lookup %s", reason_string);
	}

	else if (db1 && (id = last_descr_ndx(db1)) >= 0)
	{
		++id;
		data.data = &u.drec;
		data.size = data.ulen = wsize;
		data.doff = data.dlen = 0;
		data.flags = DB_DBT_USERMEM;

		u.drec.chk = IPQBDB_CHK_SIGNATURE;
		u.drec.created = time(0);
		rtc = db1->put(db1, NULL, &key1, &data, DB_NOOVERWRITE);
		if (rtc == 0)
		{
			if (verbose)
				printf("inserted reason id %d for \"%s\"\n", id, reason_string);
			if (reason_id)
				*reason_id = id;
		}
	}
	
	curs->c_close(curs);
	
	if (rtc && (db1 || rtc != DB_NOTFOUND))
		db1->err(db1, rtc,
			"unable to insert id %d reason \"%s\"",
			id, reason_string);

	return rtc;
}

/*
* update the relevant *_cnt fields in the description.
* ip_data pointer indicates delete operation.
* if tloc is not NULL, it is used, otherwise current time is queried.
* return non-zero on error (logged)
*/
int set_descr_cnt(DB *db1, int descr_id, int is_new,
	ip_data_t const *dead, time_t const *tloc)
{
	assert(is_new == 0 || dead == NULL);

	bynumber_descr_t drec;
	DBT key, data;
	DBC *curs;
	char const *what = "dunno";
	int rtc;

	memset(&key, 0, sizeof key);
	memset(&data, 0, sizeof data);
	key.data = &descr_id;
	key.size = sizeof descr_id;
	key.flags = DB_DBT_USERMEM;
	data.data = &drec;
	data.size = data.ulen = data.dlen = sizeof drec;
	data.flags = DB_DBT_USERMEM | DB_DBT_PARTIAL;

	// write cursor: prevent writing the description table until c_close below
	rtc = db1->cursor(db1, NULL, &curs, DB_WRITECURSOR);
	if (rtc)
	{
		db1->err(db1, rtc, "no write cursor");
		return rtc;
	}

	rtc = curs->c_get(curs, &key, &data, DB_SET);
	if (rtc == 0)
	{
		if (data.size == sizeof drec && drec.chk == IPQBDB_CHK_SIGNATURE)
		{
			time_t *tp = NULL;
			if (dead == NULL)
			{
				tp = &drec.last_added;
				drec.caught_cnt += 1;
				if (is_new)
					drec.rec_cnt += 1;
			}
			else
			{
				tp = &drec.last_removed;
				drec.block_cnt += dead->block_cnt;
				drec.decay_add_cnt += dead->decay_add_cnt;
			}

			if (tloc)
				*tp = *tloc;
			else
				time(tp);
			
			if ((rtc = db1->put(db1, NULL, &key, &data, 0)) != 0)
				what = "write";
		}
		else
		{
			rtc = 1;
			what = "record size/format";
		}
	}
	else what = "get";

	curs->c_close(curs);

	if (rtc)
		db1->err(db1, rtc, "set_descr_cnt on %s key=%d", what, descr_id);

	return rtc;
}

/*
* get description by number, with formatting options
*/
int get_descr(DB *db, int id, int opt, char *dest, size_t size)
{
	union
	{
		bynumber_descr_t drec;
		char fill[sizeof(bynumber_descr_t) + IPQBDB_DESCR_MAX_LENGTH + 1];
	} u;
	char *const descr = &u.drec.descr[0];
	char idbuf[32];
	size_t dlen = 0, ilen = 0, alloc = 0;
	int rtc;

	if (db)
	{
		DBT key, data;

		memset(&key, 0, sizeof key);
		memset(&data, 0, sizeof data);
		key.data = &id;
		key.size = sizeof id;
		key.flags = DB_DBT_USERMEM;
		data.data = &u;
		data.size = data.ulen = data.dlen = sizeof u;
		data.flags = DB_DBT_USERMEM | DB_DBT_PARTIAL;

		rtc = db->get(db, NULL, &key, &data, 0);
		if (rtc == 0)
		{
			if (u.drec.chk == IPQBDB_CHK_SIGNATURE)
			{
				// data.size should include trailing 0
				dlen = data.size - 1 - offsetof(bynumber_descr_t, descr);
				if (dlen >= IPQBDB_DESCR_MAX_LENGTH)
					dlen = IPQBDB_DESCR_MAX_LENGTH - 1;
#if 0
				else if (dlen < 0)
					dlen = 0;
#endif
				if (descr[dlen] != 0)
				{
					char *t = begin_utf8(&descr[0], &descr[dlen]);
					dlen = t - &descr[0];
					descr[dlen] = 0;
				}
			}
			else
			{
				rtc = -1;
				db->errx(db, "description %d record corrupted or old format", id);
			}
		}
		else if (rtc != DB_NOTFOUND)
			db->err(db, rtc, "get descr %d", id);
	}
	else
		rtc = -1;

	if (rtc || (opt & get_descr_add_id))
	{
		ilen = sprintf(idbuf, "%d", id);
		alloc = ilen + 1; // separating space
		opt |= get_descr_add_id;
	}

	if (alloc >= size)
		goto error_exit;

	if (rtc == 0 && (opt & get_descr_quoted) && dlen > 0)
	{
		alloc += 2; // quotes only around original
	}
	else if (rtc || dlen == 0)
	/*
	* meta-description, parethesized if descr not quoted
	*/
	{
		char *p = descr;
		if ((opt & get_descr_quoted) == 0)
			*p++ = '(';

		if (rtc == 0)
		{
			assert(dlen == 0);
			strcpy(p, "void");
		}
		else if (rtc == DB_NOTFOUND)
			strcpy(p, "undef");
		else // error or db == NULL
			strcpy(p, "unavail");
		if ((opt & get_descr_quoted) == 0)
			strcat(descr, ")");

		dlen = strlen(descr);
		opt &= ~get_descr_quoted;
	}

	/*
	* truncate description if too long.
	* shrink up to 5 chars of description as in "Hi..."
	* (size includes the trailing 0)
	*/
	if (alloc + dlen + 1 >= size)
	{
		if (dlen > 5 && alloc + 5 < size)
		{
			char *trunc = begin_utf8(descr, &descr[size - alloc - 4]);
			dlen = trunc - &descr[0];
			if (dlen < 2)
				goto error_exit;
			strcpy(trunc, "...");
			dlen += 3;
		}
		else
			goto error_exit;
	}

	alloc += dlen;
	alloc += 1; // terminating 0

	assert(alloc <= size);

	if (opt & get_descr_quoted)
		*dest++ = '"';
	if (dlen)
	{
		memcpy(dest, descr, dlen);
		dest += dlen;
	}
	if (opt & get_descr_quoted)
		*dest++ = '"';
	if (opt & get_descr_add_id)
	{
		*dest++ = ' ';
		memcpy(dest, idbuf, ilen);
		dest += ilen;
	}
	*dest = 0;
	return 0;

	error_exit:
	{
		if (size && dest)
			*dest = 0;
		return -1;
	}
}


/*
* callback for describing the secondary key
*/
static int bydescr(DB *db2, DBT const* key1, DBT const* data1, DBT *key2)
{
	(void)db2; (void)key1;
	memset(key2, 0, sizeof *key2);
	key2->data = (char*)data1->data + offsetof(bynumber_descr_t, descr);
	if ((key2->size = data1->size - offsetof(bynumber_descr_t, descr)) >
		 sizeof(bynumber_descr_t)) key2->size = 0;
	return 0;
}


/*
* open the descr table, with a secondary index based on descrs
* along with a primary based on int. The env is already opened.
*/
int open_descrdb(char *fname, DB_ENV *db_env, DB** db1, DB** db2)
{
	int rtc;
	char const *what = "dunno";

	*db1 = *db2 = NULL;

	rtc = db_create(db1, db_env, 0);
	if (rtc)
	{
		what = "db1_create";
		goto error_exit;
	}

	rtc = (*db1)->open(*db1, NULL, fname, "bynumber", DB_BTREE, DB_CREATE, 0);
	if (rtc)
	{
		what = "DB->open(primary)";
		goto error_exit;
	}

	rtc = db_create(db2, db_env, 0);
	if (rtc)
	{
		what = "db2_create";
		goto error_exit;
	}


	rtc = (*db2)->open(*db2, NULL, fname, "bydescr", DB_HASH, DB_CREATE, 0);
	if (rtc)
	{
		what = "DB->open(secondary)";
		goto error_exit;
	}

	rtc = (*db1)->associate(*db1, NULL, *db2, bydescr, 0);
	if (rtc == 0)
		return 0;

	what = "DB->associate";

	error_exit:
	{
		db_env->err(db_env, rtc, "cannot %s", what);
		if (*db2)
		{
			(*db2)->close(*db2, 0);
			*db2 = NULL;
		}
		if (*db1)
		{
			(*db1)->close(*db1, 0);
			*db1 = NULL;
		}
	}
	return 1;
}

int
check_whitelist(DB *db, uint32_t ip_addr, double *decay)
/*
* search the whitelist db for ip_addr and set decay if found.
* return values:
* 1: not found, the decay must be set by the caller to some initial default
* 0: found, if decay is 0.0 the ip should not be blocked at all.
* other: db error (logged)
*/
{
	int rtc;
	DBT key, data;
	ip_white_t whi;

	memset(&key, 0, sizeof key);
	memset(&data, 0, sizeof data);

	key.data = &ip_addr;
	key.size = key.ulen = sizeof ip_addr;
	key.flags = data.flags = DB_DBT_USERMEM;
	data.data = &whi;
	data.ulen = sizeof whi;

	rtc = db->get(db, NULL, &key, &data, 0);
	if (rtc == 0 && data.size == sizeof whi && whi.chk == IPQBDB_CHK_SIGNATURE)
	{
		if (!isnormal(whi.decay) || whi.decay < 0.0)
			whi.decay = 0.0;
		*decay = whi.decay;
	}
	else if (rtc == DB_NOTFOUND)
	{
		rtc = 1;
	}
	else if (rtc)
	{
		db->err(db, rtc, "get whitelist");
	}
	else
	{
		rtc = 1;
		db->errx(db, "bad whitelist record");
	}

	return rtc;
}

int write_block(DB *db, uint32_t ip_addr, double decay, int reason_id,
	int *probability, int force, write_block_cb verbose_fn, DB *db_d)
{
	DBC *curs;
	int rtc, rtc2 = 0, old = 0;
	ip_data_t ip_data, old_data;
	DBT key, data;
	time_t now;

	memset(&key, 0, sizeof key);
	memset(&data, 0, sizeof data);
	key.data = &ip_addr;
	key.ulen = key.size = sizeof ip_addr;
	key.flags = data.flags = DB_DBT_USERMEM;
	data.data = &ip_data;
	data.ulen = sizeof ip_data;
	memset(&ip_data, 0, sizeof ip_data);
	memset(&old_data, 0, sizeof old_data);

	time(&now);

	// a write cursor locks the database
	rtc = db->cursor(db, NULL, &curs, DB_WRITECURSOR);
	if (rtc)
	{
		db->err(db, rtc, "cannot create write cursor");
		return -1;
	}

	rtc = curs->c_get(curs, &key, &data, DB_SET);
	if (rtc == 0)
	{
		if (data.size != sizeof ip_data || ip_data.chk != IPQBDB_CHK_SIGNATURE)
			rtc = 1;
		else
		{
			memcpy(&old_data, &ip_data, sizeof old_data);
			old = 1;
			if (ip_data.decay < decay || (force & write_block_force_decay) != 0)
			{
				ip_data.decay = decay;
				ip_data.reason_id = reason_id;
			}
			ip_data.last_update = now;
			if (RAND_MAX > ip_data.probability &&
				RAND_MAX - ip_data.probability >= ip_data.probability)
					ip_data.probability *= 2;
			else
				ip_data.probability = RAND_MAX;
			
			const int initial_probability = *probability;
			if (ip_data.probability < initial_probability ||
				(force & write_block_force_probability) != 0)
			{
				ip_data.probability = initial_probability;
				ip_data.reason_id = reason_id;
			}
			if ((force & write_block_force_reason) != 0)
				ip_data.reason_id = reason_id;

			if (ip_data.probability > IPQBDB_PROBABILITY_THRESHOLD &&
				old_data.probability <= IPQBDB_PROBABILITY_THRESHOLD)
			{
				ip_data.decay += IPQBDB_DECAY_ADDEND_ON_THRESHOLD;
				ip_data.decay_add_cnt += 1;
			}

			ip_data.caught_cnt += 1;

			rtc2 = curs->c_put(curs, &key, &data, DB_CURRENT);
			if (rtc2 == 0)
				*probability = ip_data.probability;
		}
	}
	else if (rtc == DB_NOTFOUND)
	{
		rtc = 0;
		ip_data.decay = decay;
		ip_data.last_update = ip_data.created = now;
		ip_data.probability = *probability;
		ip_data.reason_id = reason_id;
		ip_data.caught_cnt = 1;
		ip_data.chk = IPQBDB_CHK_SIGNATURE;
		data.size = data.ulen;
		rtc2 = curs->c_put(curs, &key, &data, DB_KEYFIRST);
	}

	curs->c_close(curs);

	if (rtc == 1)
		db->errx(db, "bad block record");
	else if (rtc)
		db->err(db, rtc, "reading curs block");
	if (rtc2)
		db->err(db, rtc, "writing curs block");
	
	if (rtc == 0 && rtc2 == 0)
	{
		if (db_d)
			set_descr_cnt(db_d, reason_id, !old, NULL, &now);
			
		if (verbose_fn)
			verbose_fn(ip_addr, &ip_data, old? &old_data: NULL, db_d);
	}
	
	return rtc || rtc2;
}

void fputs_initial_count_help()
{
	fputs("\nThe initial-count determines the probability for newly inserted "
		"records,\nsuch that doubling the initial probability that many times "
		"results\nin a probability of 100%. Its max value is the number of "
		"bits in RAND_MAX.\n", stdout);
}
