/* FAKED libnfqnetlink.c: generic library for FAKE access to nf_queue
 *
 * (C) 2005 by Harald Welte <laforge@gnumonks.org>
 * Copyright (C) 2010-2011 Alessandro Vesely
 *
This file is part of Ipqbdb.

Ipqbdb is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Ipqbdb is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Ipqbdb.  If not, see <http://www.gnu.org/licenses/>.


NOTICE:
=======

FUBAR
This work has been derived from src/libnetfilter_queue.c and utils/nfqnl_test.c
of an old libnetfilter_queue distribution. The content of the original files
has been completely faked, so that the result cannot serve the purpose of the
original code. This only serves to link a test program that so that it can run
without root privileges and without disturbing the kernel.

HOW THIS WORKS:
===============

The main function runs in the client. It is assumed it calls nfq_open as a
first thing. Sanity checks are run there, it also retrieves an environment
variable that has the name of an in-file that with test details, and creates
the nfq_handle structure filled with the data for the test.

The in-file may contain packets, commands, and comments. Commands are defined
in cmd_strings.h; the "group" command groups packets. In-file is first read
on opening nfq_handle, and continues until there are enough packets to be sent.

The socketpair is used to pass parsed in-file data to the client. A call to
nfq_handle_packet is expected after sending. That function executes any command
and then calls all registered callbacks. The noop command is used to yield
control to the client. Other commands are useful to set test values, e.g. the
database, in order for the client to take certain actions. The client's
decision is displayed when it calls nfq_set_verdict or similar functions.

When the in-file terminates, the socketpair is closed, letting the client's
call to recv terminate with EBADF. nfq_fd will return -123 after that.

*/

#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>
#include <time.h>
#include <errno.h>
#include <limits.h>
#include <wait.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <libnetfilter_queue/libnetfilter_queue.h>

#include "ipv4_util.h"

#include <assert.h>

#define TEST_QUEUE_MAX 8
#define TEST_CMD_MAX 100
#define TEST_ARGS_MAX TEST_CMD_MAX
#define MY_PREFIX "TESTJUDGE"
static const char my_prefix[] = MY_PREFIX;

struct nfq_q_handle
{
	struct nfq_handle *h;
	nfq_callback *cb;
	void *data;
};

struct nfq_data
{
	time_t timestamp;
	union ipv4
	{
		struct fields
		{
			unsigned version:4;
			unsigned hlen:4;
			unsigned tos:8;
			unsigned len:16;
			unsigned id:16;
			unsigned flag:3;
			unsigned frag:13;
			unsigned ttl:8;
			unsigned proto:8;
			unsigned chk:16;
			unsigned source:32;
			unsigned dest:32;
		} v;
		char payload[20]; // extra header and data would follow
	} packet;
	int packet_id;
	struct nfqnl_msg_packet_hdr hdr;
	unsigned char queue[TEST_QUEUE_MAX];
};

struct nfq_handle
{
	// q must be the first element, see get_queue_num
	struct nfq_q_handle q[TEST_QUEUE_MAX];
	FILE *in, *out;
	char *cmd_arg[TEST_ARGS_MAX];
	time_t current;
	int sv[2]; // socketpair
	struct nfq_data next_pkt;
	int line;
	char sv_open, sv_sent;
	char have_next_pkt;
	char verbose;
	char fname[];
};

typedef struct test_cmd
{
	int cmd;
	int arg;
} test_cmd;

#define STRING_MACRO(x, y) test_cmd_##x,
enum test_cmd_enum
{
#include "cmd_strings.h"
test_cmd_invalid
};

#undef STRING_MACRO
#define STRING_MACRO(x, y) #x,
static const char *test_cmd_strings[] =
{
#include "cmd_strings.h"
};

#undef STRING_MACRO
#define STRING_MACRO(x, y) y,
static const char test_cmd_has_arg[] =
{
#include "cmd_strings.h"
};

#undef STRING_MACRO


int nfq_errno;
extern int verbose; // in ibd-judge.c

static void close_socketpair(struct nfq_handle *h)
{
	if (h->sv_open)
	{
		close(h->sv[0]);
		close(h->sv[1]);
		h->sv_open = 0;
	}	
}

static int parse_command(char **p)
{
	char buf[64];
	size_t len = 0, i;
	int ch;
	unsigned char const *s = *(unsigned char**)p;
	
	if (!isalpha(*s))
		return -1;

	while ((ch = *s) == '_' || isalnum(ch))
	{
		if (isupper(ch))
		{
			ch = _tolower(ch);
		}
		buf[len] = ch;
		if (++len >= sizeof buf)
			return test_cmd_invalid;
		s += 1;
	}
	
	buf[len] = 0;
	for (i = 0; i < sizeof test_cmd_strings/ sizeof test_cmd_strings[0]; ++i)
		if (strcmp(test_cmd_strings[i], buf) == 0)
		{
			*p += len;
			return i;
		}
	
	return test_cmd_invalid;
}

#if 0 // not used
static time_t parse_time(char **s)
// convert from yyyy-mm-dd:hh:mm:ss
// return -1 for mktime error,
{
	char *t = NULL;
	char const *p = *s;
	unsigned long ul = strtoul(p, &t, 10);
	if (ul >= 1900 && ul < INT_MAX && *t == '-')
	{
		struct tm tm;
		char *p = t + 1;
		memset(&tm, 0, sizeof tm);
		tm.tm_isdst = -1;
		tm.tm_year = (int)ul - 1900;

		ul = strtoul(p, &t, 10);
		if (ul < INT_MAX && *t == '-')
		{
			tm.tm_mon = (int)ul - 1;
			p = t + 1;
			ul = strtoul(p, &t, 10);
			if (ul < INT_MAX && *t)
			{
				tm.tm_mday = (int)ul;
				p = t + 1;
				ul = strtoul(p, &t, 10);
				if (ul < INT_MAX && *t == ':')
				{
					tm.tm_hour = (int)ul;
					p = t + 1;
					ul = strtoul(p, &t, 10);
					if (ul < INT_MAX && *t == ':')
					{
						tm.tm_min = (int)ul;
						p = t + 1;
						ul = strtoul(p, &t, 10);
						if (ul < INT_MAX)
						{
							tm.tm_sec = (int)ul;
							time_t rtc = mktime(&tm);
							if (rtc != (time_t)-1)
							{
								*s = t;
								return rtc;
							}
						}
					}
				}
			}
		}
	}
	return (time_t)-1;
}
#endif // 0

static int parse_addr(char **p, uint32_t *addr)
{
	char *s = *p, *t = NULL;
	ip4_range range;
	int rtc = parse_ip_address(s, &range, &t);
	if (rtc || range.first != range.last ||
		t == NULL || (*t && !isspace(*(unsigned char*)t)))
			return -1;
	
	*addr = range.first;
	*p = t;
	return 0;
}

static char const* parse_pkt(char *s, struct nfq_data *pkt)
/*
* read packet format:
* id [queue...] from-addr to-addr [time increase]
* return NULL if ok, an the failed field name if not
*/
{
	assert(isdigit(*(unsigned char*)s));
	
	int id;
	unsigned char queue[TEST_QUEUE_MAX];
	uint32_t from, to;
	time_t timestamp = 0;	
	
	char *t = NULL;
	unsigned int l = strtoul(s, &t, 0);
	int i;

	if (l >= INT_MAX || t == NULL || !isspace(*(unsigned char*)t))
		return "id";

	id = (int)l;
	memset(queue, 0, sizeof queue);
	
	for (i = 0; i < TEST_QUEUE_MAX; ++i)
	{
		s = t + 1;
		while (isspace(*(unsigned char*)s))
			s += 1;
		
		l = strtoul(s, &t, 10);
		if (*t == '.')
			break;

		if (l >= TEST_QUEUE_MAX)
			return "queue number";

		queue[l] = 1;
	}
	
	if (parse_addr(&s, &from))
		return "source IP";
	
	while (isspace(*(unsigned char*)s))
		s += 1;
		
	if (parse_addr(&s, &to))
		return "dest IP";

	while (isspace(*(unsigned char*)s))
		s += 1;

	if (*s)
	{
		timestamp = strtoul(s, &t, 0);
		if (*t)
			return "time";
	}
	
	memset(pkt, 0, sizeof *pkt);
	pkt->timestamp = timestamp;
	pkt->packet.v.version = 4;
	pkt->packet.v.hlen = 5;
	pkt->packet.v.id = (uint16_t)id;
	pkt->packet.v.ttl = 8;
	pkt->packet.v.proto = 6;
	pkt->packet.v.source = from;
	pkt->packet.v.dest = to;
	pkt->packet_id = id;
	pkt->hdr.packet_id = htonl(id);
	// pkt->hdr.hw_protocol
	// pkt->hdr.hook	
	memcpy(pkt->queue, queue, sizeof pkt->queue);
	
	return 0;
}

static char *simple_expand(char *arg)
// assume arg is malloc'ed, formatted as "XYZ=ABCD$EFG,etc."
// replace $EFG and return the same or a different string
// simple means no ${funny} stuff :-/
// return NULL on memory failure
{
	size_t len = strlen(arg);
	char *start = strchr(arg, '='), *repl;
	while (arg != NULL && start != NULL &&
		(repl = strchr(start, '$')) != NULL)
	{
		char *var = repl + 1, *v = var;
		int ch;
		while ((isalnum(ch = *(unsigned char*)v) || ch == '_'))
			++v;

		*v = 0;
		char *val = getenv(var);
		size_t const ini = repl - arg,
			vlen = val? strlen(val): 0,
			nlen = v - var,
			rest = len - nlen - ini; // incl. term 0
		*v = ch;
		if (vlen <= nlen)
		{
			if (val)
				memcpy(repl, val, vlen);
			memmove(start = repl + vlen, v, rest);
		}
		else
		{
			char *newarg = (char*)malloc(len + vlen - nlen);
			if (newarg)
			{
				memcpy(newarg, arg, ini);
				start = newarg + ini;
				if (val)
				{
					memcpy(start, val, vlen);
					start += vlen;
				}
				memcpy(start, v, rest);
			}
			free(arg);
			arg = newarg;
		}
		len += vlen - nlen - 1;
		assert(arg == NULL || strlen(arg) == len);
	}
	return arg;
}

static int get_free_arg(struct nfq_handle *h)
{
	unsigned i;
	for (i = 0; i < sizeof h->cmd_arg/sizeof h->cmd_arg[0]; ++i)
		if (h->cmd_arg[i] == NULL)
			return i;

	fprintf(stderr,
			MY_PREFIX ": max arguments exceeded at line %d in %s\n",
				h->line, h->fname);
	return -1;
}

static int read_test_input(struct nfq_handle *h)
/*
* read in-file and send 1 buffer of commands until one pkt is ready,
* or return -1 for failure.
*/
{
	assert(h && h->have_next_pkt == 0);

	char buf[1024], *s;
	char cmdbuf[TEST_CMD_MAX * sizeof(test_cmd)];
	size_t in_cmdbuf = 0, have_group = 0;
	
	while ((s = fgets(buf, sizeof buf, h->in)) != NULL)
	{
		h->line += 1;
		
		size_t len = strlen(buf);
		if (len + 1 >= sizeof buf)
		{
			fprintf(stderr,
				MY_PREFIX ": line %d too long in %s\n",
				h->line, h->fname);
			return -1;
		}
				
		while (isspace(*(unsigned char*)s))
			++s;
		
		// discard empty lines and comments
		if (*s == '#' || *s == 0)
			continue;
		
		while (len > 0 && isspace(*(unsigned char*)&buf[len-1]))
			len -= 1;
		buf[len] = 0;

		if (isdigit(*(unsigned char*)s)) // packet data
		{
			char const *field = parse_pkt(s, &h->next_pkt);
			if (field == NULL)
			{
				h->have_next_pkt = 1;
				break;
			}

			fprintf(stderr,
				MY_PREFIX ": invalid %s at line %d in %s\n",
				field, h->line, h->fname);
		}
		else
		// parse command and possibly add it to cmdbuf
		// ignore lines with errors when possible
		{
			char *cmd = s;
			int rtc = parse_command(&s);
			if (rtc < 0 || rtc == test_cmd_invalid)
			{
				fprintf(stderr,
					MY_PREFIX ": invalid %s at line %d in %s\n",
					rtc < 0? "data": "command", h->line, h->fname);
			}
			else
			{
				int err = 0, needarg = test_cmd_has_arg[rtc];
				char *unex = NULL;
				test_cmd tc;
				memset(&tc, 0, sizeof tc);
				tc.cmd = rtc;
				*s++ = 0;
				
				while (isspace(*(unsigned char*)s))
					++s;
				
				switch (needarg)
				{
					case 0: // no args
						if (*s)
							unex = s;
						break;
					
					case 1: // integer arg
						if (isdigit(*(unsigned char*)s))
						{
							char *t = NULL;
							long l = strtol(s, &t, 0);
							if (t && *t)
								unex = t;
							if (l > INT_MIN && l < INT_MAX)
								tc.arg = (int)l;
							else
							{
								err = 1;
								fprintf(stderr,
									MY_PREFIX
									": invalid argument %s at line %d in %s\n",
									s, h->line, h->fname);
							}
							
							needarg = 0;
						}
						break;

					case 2: // string arg
						if (*s)
						{
							int const ndx = get_free_arg(h);
							if (ndx < 0)
								err = 1;
							else if ((h->cmd_arg[ndx] = strdup(s)) == NULL)
							{
								fputs("OUT OF MEMORY\n", stderr);
								close_socketpair(h);
								return -1;
							}
							else
								tc.arg = ndx;
							
							needarg = 0;
						}
						break;

					default:
						assert(0); // unsupported number in cmd_strings.h
						return -1;
						break;
				}
				
				if (needarg)
				{
					err = 1;
					fprintf(stderr,
						MY_PREFIX
						": command %s requires a%s argument"
						" at line %d in %s\n",
						needarg == 1? " numeric": "n",
						cmd, h->line, h->fname);
				}
				
				if (unex)
				{
					err = 1;
					fprintf(stderr,
						MY_PREFIX
						": unexpected character(s) \"%s\""
						" at line %d in %s\n",
						unex, h->line, h->fname);
				}
				
				if (err == 0)
				{
					// always keep room for an extra test_cmd in cmdbuf
					memcpy(&cmdbuf[in_cmdbuf], &tc, sizeof tc);
					in_cmdbuf += sizeof tc;

					if (in_cmdbuf + sizeof tc > sizeof cmdbuf)
					{
						fprintf(stderr,
							MY_PREFIX
							": too many commands without packet data"
							" at line %d in %s\n",
							h->line, h->fname);
						break;
					}
					
					if (rtc == test_cmd_group)
						have_group = 1;
				}
			}	
		}
	}

	if (!h->have_next_pkt || !h->sv_open)
	{
		close_socketpair(h);
		return -1;
	}

	// no group command: default to a group of 1 pkt
	if (!have_group)
	{
		test_cmd tc;
		memset(&tc, 0, sizeof tc);
		tc.cmd = test_cmd_group;
		tc.arg = 1;
		memcpy(&cmdbuf[in_cmdbuf], &tc, sizeof tc);
		in_cmdbuf += sizeof tc;
	}
	
	if (write(h->sv[0], cmdbuf, in_cmdbuf) == -1)
	{
		fprintf(stderr,
			MY_PREFIX ": cannot send through socketpair: %s\n",
			strerror(errno));
		close_socketpair(h);
		return -1;
	}

	h->sv_sent = 1;
	return 0;
}

#if defined __GNUC__
__attribute__ ((format(printf, 2, 3)))
#endif
static int print_out(struct nfq_handle *h, char const *fmt, ...)
{
	int rtc = 0;
	if (h->verbose && h->out)
	{
		va_list al;
		va_start(al, fmt);
		rtc = vfprintf(h->out, fmt, al);
		va_end(al);
	}
	
	return rtc;
}

static void send_noop_packet(struct nfq_handle *h)
{
	assert(h && h->sv_open);

	test_cmd tc;
	memset(&tc, 0, sizeof tc);
	tc.cmd = test_cmd_noop;
	if (write(h->sv[0], &tc, sizeof tc) == -1)
		close_socketpair(h);
	else
		h->sv_sent = 1;
}

/* public interface */

struct nfnl_handle *nfq_nfnlh(struct nfq_handle *h)
{
	(void)h;
	return NULL; // not used
}

int nfq_fd(struct nfq_handle *h)
{
	return h->sv_open? h->sv[1]: -123;
}

struct nfq_handle *nfq_open(void)
{
	return nfq_open_nfnl(NULL);
}

struct nfq_handle *nfq_open_nfnl(struct nfnl_handle *nfnlh)
{
	struct nfq_handle *h;
	char *fname = getenv(my_prefix);
	size_t fname_size = fname? strlen(fname) + 1: 0;

	// don't ruin installed stuff: check permissions and database path
	if (getuid() == 0 || geteuid() == 0 || access("/", W_OK) == 0)
	{
		fputs(MY_PREFIX ": tests MUST NOT be run with root privileges\n",
			stderr);
		return NULL;
	}
	
	if (fname == NULL)
	{
		fputs(MY_PREFIX
			": Missing environment variable \"" MY_PREFIX
			"\" pointing to input file\n",
			stderr);
		return NULL;
	}
		
	h = malloc(sizeof *h + fname_size);
	if (!h)
		return NULL;

	memset(h, 0, sizeof *h);
	strcpy(h->fname, fname);
	if ((h->in = fopen(fname, "r")) == NULL)
	{
		fprintf(stderr,
			MY_PREFIX ": cannot open %s for reading: %s\n",
			fname, strerror(errno));
		goto error_exit;
	}
		
	if (socketpair(PF_UNIX, SOCK_SEQPACKET, 0, h->sv) == 0)
	{
		h->sv_open = 1;
		if (!read_test_input(h) == 0)
			goto error_exit;

	} else
	{
		fprintf(stderr,
			MY_PREFIX ": socketpair error: %s\n", strerror(errno));
		goto error_exit;
	}

	h->out = stdout;
	h->verbose = verbose != 1;
	time(&h->current);
	return h;

	error_exit:
	{
		if (h->in) fclose(h->in);
		free(h);
	}

	return NULL;
	(void)nfnlh;
}

int nfq_close(struct nfq_handle *h)
{
	close_socketpair(h);
	if (h->out && h->out != stdout) fclose(h->out);
	if (h->in) fclose(h->in);
	free(h);
	return 0;
}

/* bind nf_queue from a specific protocol family */
int nfq_bind_pf(struct nfq_handle *h, u_int16_t pf)
{
	(void)h;
	(void)pf; // should check it is AF_INET
	return 0;
}

/* unbind nf_queue from a specific protocol family */
int nfq_unbind_pf(struct nfq_handle *h, u_int16_t pf)
{
	(void)h;
	(void)pf; // should check it is AF_INET
	return 0;
}

/* bind this socket to a specific queue number */
struct nfq_q_handle *nfq_create_queue(struct nfq_handle *h, 
		u_int16_t num,
		nfq_callback *cb,
		void *data)
{
	if (num >= TEST_QUEUE_MAX)
	{
		fprintf(stderr,
			MY_PREFIX ": cannot create queue %d: only support up to %d\n",
			num, TEST_QUEUE_MAX);
		return NULL;
	}
	
	if (h->q[num].h == h) // already opened. Is this an error?
	{
		return NULL;
	}
	
	h->q[num].h = h;
	h->q[num].cb = cb;
	h->q[num].data = data;
	
	return &h->q[num];
}

static int get_queue_num(struct nfq_q_handle *qh)
{
	struct nfq_handle *h = qh->h;
	unsigned num = qh - (struct nfq_q_handle *)h;
	
	if (num >= TEST_QUEUE_MAX || h != h->q[num].h)
	{
		fputs(
			MY_PREFIX ": bad handle!!\n",
			stderr);
		assert(0);
		return -1;
	}

	return num;
}

/* unbind this socket from a specific queue number */
int nfq_destroy_queue(struct nfq_q_handle *qh)
{
	if (get_queue_num(qh) < 0)
		return -1;

	memset(qh, 0, sizeof *qh);
	return 0;
}

int nfq_handle_packet(struct nfq_handle *h, char *buf, int len)
{	
	test_cmd cmd;
	int rtc = 0, i;
	
	struct nfq_data *pkt = &h->next_pkt;
	
	h->sv_sent = 0;
	
	while (len > 0 && (unsigned)len >= sizeof cmd && rtc >= 0)
	{
		int pkts = 0;
		
		memcpy(&cmd, buf, sizeof cmd);
		buf += sizeof cmd;
		len -= sizeof cmd;
			
		switch (cmd.cmd)
		{
			case test_cmd_group:
				assert(cmd.arg > 0);
				pkts = cmd.arg;
				break;

			case test_cmd_sleep:
				print_out(h, "sleep %d\n", cmd.arg);
				sleep(cmd.arg);
				break;

			case test_cmd_settime:
				time(&h->current);
				// print_out(h, "------------->reset time to ...... %ld\n",
				//	(long)h->current);
				break;

			case test_cmd_putenv:
			{
				assert(cmd.arg >= 0 && cmd.arg < TEST_ARGS_MAX);
				assert(h->cmd_arg[cmd.arg] != NULL);
				
				char *arg = simple_expand(h->cmd_arg[cmd.arg]);
				if (arg == NULL)
				{
					close_socketpair(h);
					rtc = -1;
				}

				// print_out(h, "putenv %s\n", arg);
				if (putenv(arg) != 0)
					fprintf(stderr,
						"%s: putenv %s failed: %s\n",
						my_prefix, arg, strerror(errno));

				// don't free arg
				h->cmd_arg[cmd.arg] = NULL;
				break;
			}

			case test_cmd_system:
			{
				assert(cmd.arg >= 0 && cmd.arg < TEST_ARGS_MAX);
				assert(h->cmd_arg[cmd.arg] != NULL);
				print_out(h, "system %s\n", h->cmd_arg[cmd.arg]);

				int rc = system(h->cmd_arg[cmd.arg]);
				if (rc == -1)
					fprintf(stderr,
						"%s: system %s failed: %s\n",
						my_prefix,
						h->cmd_arg[cmd.arg],
						strerror(errno));
				else if (WIFSIGNALED(rc) &&
					(WTERMSIG(rc) == SIGINT || WTERMSIG(rc) == SIGQUIT))
				{
					close_socketpair(h);
					rtc = -1;
				}
				else if (WEXITSTATUS(rc))
					fprintf(stderr,
						"%s: system %s exited with status %d\n",
						my_prefix,
						h->cmd_arg[cmd.arg],
						WEXITSTATUS(rc));

				free(h->cmd_arg[cmd.arg]);
				h->cmd_arg[cmd.arg] = NULL;
				break;
			}

			case test_cmd_noop:
			default: // unimplemented commands
				break;
		}

		for (i = 0; i < pkts && rtc >= 0; ++i)
		{			
			if (!h->have_next_pkt)
				if (read_test_input(h))
					break;

			assert(h->have_next_pkt);
			h->have_next_pkt = 0;
			
			// increase current --ibd-judge gets the delta specified on pkt
			h->current += h->next_pkt.timestamp;
			h->next_pkt.timestamp = h->current;
			
			// call all registered callbacks		
			for (int q = 0; q < TEST_QUEUE_MAX && rtc >= 0; ++q)
				if (h->next_pkt.queue[q] && h->q[q].cb)
					rtc = h->q[q].cb(&h->q[q], NULL /* ?? */, pkt, h->q[q].data);

			memset(pkt, 0, sizeof *pkt);
		}		
	}

	/*
	* make sure that either socketpair is closed
	* or there is something to be read from it
	*/	
	if (h->sv_open && !h->sv_sent)
	{
		if (h->have_next_pkt)
			send_noop_packet(h);
		else
			read_test_input(h);
	}
		
	return rtc < 0? rtc: 0;
}

int nfq_set_mode(struct nfq_q_handle *qh, u_int8_t mode, u_int32_t range)
{
	if (get_queue_num(qh) < 0)
		return -1;

	print_out(qh->h, "nfq_set_mode: %d, %d\n", mode, range);
	return 0;
}

int nfq_set_queue_maxlen(struct nfq_q_handle *qh, u_int32_t queue_maxlen)
{
	if (get_queue_num(qh) < 0)
		return -1;

	print_out(qh->h, "nfq_set_queue_maxlen: %u\n", queue_maxlen);
	return 0;
}

int nfq_set_verdict_mark(struct nfq_q_handle *qh, u_int32_t id,
		u_int32_t verdict, u_int32_t mark,
		u_int32_t datalen, unsigned char *buf)
{
	if (get_queue_num(qh) < 0)
		return -1;
	print_out(qh->h,
		"nfq_set_verdict_mark: %d, %d, %d\n",
		id, verdict, mark);
	return 0;
	(void)datalen;
	(void)buf;
}

int nfq_set_verdict(struct nfq_q_handle *qh, u_int32_t id,
		u_int32_t verdict, u_int32_t data_len, 
		unsigned char *buf)
{
	if (get_queue_num(qh) < 0)
		return -1;
	print_out(qh->h,
		"nfq_set_verdict: %d, %d\n",
		id, verdict);
	return 0;
	(void)data_len;
	(void)buf;
}

/*************************************************************
 * Message parsing functions 
 *************************************************************/

struct nfqnl_msg_packet_hdr *nfq_get_msg_packet_hdr(struct nfq_data *pkt)
{
	return &pkt->hdr;
}

uint32_t nfq_get_nfmark(struct nfq_data *pkt)
{
	(void)pkt;
	return 0;
}

int nfq_get_timestamp(struct nfq_data *pkt, struct timeval *tv)
{
	if (pkt->timestamp == (time_t)-1)
		return -1;

	tv->tv_sec = pkt->timestamp;
	tv->tv_usec = 0;

	return 0;
}

/* all nfq_get_*dev() functions return 0 if not set, since linux only allows
 * ifindex >= 1, see net/core/dev.c:2600  (in 2.6.13.1) */
u_int32_t nfq_get_indev(struct nfq_data *pkt)
{
	(void)pkt;
	return 0;
}

u_int32_t nfq_get_physindev(struct nfq_data *pkt)
{
	(void)pkt;
	return 0;
}

u_int32_t nfq_get_outdev(struct nfq_data *pkt)
{
	(void)pkt;
	return 0;
}

u_int32_t nfq_get_physoutdev(struct nfq_data *pkt)
{
	(void)pkt;
	return 0;
}

struct nfqnl_msg_packet_hw *nfq_get_packet_hw(struct nfq_data *pkt)
{
	(void)pkt;
	return 0;
}

int nfq_get_payload(struct nfq_data *pkt, char **data)
{
	*data = pkt->packet.payload;
	return sizeof pkt->packet.payload;
}

unsigned int nfnl_rcvbufsiz(const struct nfnl_handle *h, unsigned int size)
{
	(void)h;
	return size*2;
}

