/*
* Ipq Berkeley db Daemon  BAN - ibd-ban
* written by ale in milano on 13 sep 2008

Copyright (C) 2008-2019 Alessandro Vesely

This file is part of Ipqbdb.

Ipqbdb is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Ipqbdb is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Ipqbdb.  If not, see <http://www.gnu.org/licenses/>.

*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <unistd.h>
#include <limits.h>
#include <signal.h>
#include <syslog.h>

#include <sys/types.h>

// Berkeley DB v4.4
#include <db.h>

#include <popt.h>

// format of data packet
#include "config_names.h"
#include "dbstruct.h"

#include <time.h>

// for inet_ntoa
#define _GNU_SOURCE 1
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "ipv4_util2.h"
#include "initial_count.h"
#include "percent_prob.h"
#include "ext_exec.h"


static app_private ap;
static char const err_prefix[] = "ibd-ban";
#include "logline_written_block.h"

int caught_signal;

#if !defined NDEBUG
#define TRACE(x...) while (ap.mode == 0) { fprintf(stderr, ## x); break; }
#else
#define TRACE(x...)
#endif

static void sig_catcher(int sig)
{
#if !defined(NDEBUG)
	if (ap.mode == 0)
	{
		char buf[80];
		unsigned s = snprintf(buf, sizeof buf,
			"%s [%d]: received signal %s\n",
			err_prefix, (int)getpid(), strsignal(sig));
		if (s >= sizeof buf)
		{
			s = sizeof buf;
			buf[s - 1] = '\n';
		}
		write(2, buf, s);
	}
#endif
	if (sig == SIGABRT)
		syslog(LOG_CRIT, "aborting %s\n", err_prefix);
	caught_signal = sig;
}

static int setsigs(void)
{
	int rtc;
	struct sigaction act;
	memset(&act, 0, sizeof act);
	sigemptyset(&act.sa_mask);
	act.sa_flags = 0;
	act.sa_handler = sig_catcher;

	rtc =
		sigaction(SIGPIPE, &act, NULL) ||
		sigaction(SIGILL, &act, NULL)  ||
		sigaction(SIGHUP, &act, NULL)  ||
		sigaction(SIGINT, &act, NULL)  ||
		sigaction(SIGQUIT, &act, NULL) ||
		sigaction(SIGPWR,  &act, NULL) ||
		sigaction(SIGABRT, &act, NULL) ||
		sigaction(SIGTERM, &act, NULL);

	act.sa_handler = SIG_IGN;
	rtc |=
		sigaction(SIGALRM, &act, NULL) ||
		sigaction(SIGIO,   &act, NULL) ||
		sigaction(SIGUSR1, &act, NULL) ||
		sigaction(SIGUSR2, &act, NULL);
//		sigaction(SIGEMT,  &act, NULL);
	return rtc;
}

static int initial_count = IPQBDB_INITIAL_COUNT;
static double initial_decay = IPQBDB_INITIAL_DECAY;
static char *db_block_name = IPQBDB_DATABASE_NAME;
static char *db_white_name = IPQBDB_WHITE_DATABASE_NAME;
static char *db_descr_name = IPQBDB_DESCR_DATABASE_NAME;
static char *reason_string = "0";
static char **ip_address;
static int version_opt, help_opt, syslog_opt, cleanup_opt, verbose = -1;
static int force_probability_opt, force_decay_opt, force_reason_opt;
static int exec_connkill_opt;
static struct poptOption opttab[] =
{
	{"ip-addr", 'i', POPT_ARG_ARGV, &ip_address, 0,
	"An address to ban, repeatable", "ip_addr"},
	{"reason", 'r', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &reason_string, 0,
	"The reason string (or its record number) for the block record", "int-or-string"},
	{"force-reason", '\0', POPT_ARG_NONE, &force_reason_opt, 0,
	"Change the reason even if probability and decay are not changed", NULL},
	{"initial-decay", 't', POPT_ARG_DOUBLE|POPT_ARGFLAG_SHOW_DEFAULT, &initial_decay, 0,
	"The time taken for the block probability to halve (seconds)", "float"},
	{"force-decay", '\0', POPT_ARG_NONE, &force_decay_opt, 0,
	"Change the decay even if the current value is higher", NULL},
	{"initial-count", 'c', POPT_ARG_INT|POPT_ARGFLAG_SHOW_DEFAULT, &initial_count, 0,
	"Set initial probability count", "integer"},
	{"force-probability", '\0', POPT_ARG_NONE, &force_probability_opt, 0,
	"Change the probability even if the current value is higher", NULL},
	{"exec-connkill", 'e', POPT_ARG_NONE, &exec_connkill_opt, 0,
	"Execute connection kill if probability reaches 100%", NULL},
	{"verbose", 'v', POPT_ARG_INT|POPT_ARGFLAG_OPTIONAL, &verbose, 0,
	"Be verbose", "level"},
	{"log-syslog", 'l', POPT_ARG_NONE, &syslog_opt, 0,
	"Log to syslog rather than std I/O", NULL},
	{"version", 'V', POPT_ARG_NONE, &version_opt, 0,
	"Print version number and exit", NULL},
	{"help", 'h', POPT_ARG_NONE, &help_opt, 0,
	"This help.", NULL},
	{"db-block", 'b', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &db_block_name, 0,
	"The database of blocked IPv4 addresses", "filename"},
	{"db-white", 'w', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &db_white_name, 0,
	"The whitelist database.", "filename"},
	{"db-descr", 'd', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &db_descr_name, 0,
	"The reason description table.", "filename"},
	{"db-cleanup", '\0', POPT_ARG_NONE, &cleanup_opt, 0,
	"On exit cleanup environment (__db.00? files) if not still busy", NULL},
	POPT_TABLEEND
};

static int get_ip_address(char *ip_address, uint32_t* ip_addr)
{
	struct in_addr in;
	int rtc = my_inet_pton(ip_address, &in);
	if (rtc == 0 || in.s_addr == INADDR_ANY) // 0 is not valid
	{
		report_error(&ap, LOG_ERR, "invalid ip_address %s\n", ip_address);
		return 1;
	}

	memcpy(ip_addr, &in, sizeof *ip_addr);
	return 0;
}

static int get_all_ip_address(uint32_t **ip_addr)
{
	size_t count = 0;
	char **a = ip_address;
	while (*a++)
		++count;

	if (count == 0)
	{
		*ip_addr = NULL;
		return 0;
	}

	uint32_t *addr = *ip_addr = calloc(count + 1, sizeof(uint32_t));
	if (addr == NULL)
		return -1;

	for (a = ip_address; *a; ++a, ++addr)
		if (get_ip_address(*a, addr))
			return -1;

	return 0;
}

static int display_check_whitelist(DB *db, uint32_t ip_addr, double *decay)
{
	int const rtc = check_whitelist(db, ip_addr, decay);
	int rrc = rtc;
	if (rrc == 1)
	{
		rrc = 0;
		*decay = initial_decay;
	}

	if (verbose)
	{
		double const d = *decay;
		struct in_addr in;
		memcpy(&in, &ip_addr, sizeof in);
		if (rtc == 0)
		{
			printf("%s is whitelisted", inet_ntoa(in));
			if (d > 0)
				printf(" with initial decay of %g seconds\n", d);
			else
				fputs(" and won't be blocked\n", stdout);
		}
		else if (rtc == 1)
		{
			printf("%s is not whitelisted, initial decay of %g seconds\n",
				inet_ntoa(in), d);
		}
	}

	return rrc;
}

static const char *month_abbr(unsigned ndx)
{
	static const char *abbr[] =
	{"Jan", "Feb", "Mar", "Apr", "May", "Jun",
	 "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};
	if (ndx >= sizeof abbr / sizeof abbr[0]) return "---";
	return abbr[ndx];
}

static void tab_printtime(char const *hdr, time_t ref, time_t now)
{
	struct tm tm = *localtime(&ref);
	unsigned elapsed = now - ref;
	unsigned dd, hh, mm;
	char buf[64], *p = &buf[0], buf2[16];;

	memset(buf, ' ', sizeof buf);
	dd = elapsed / (24UL * 3600UL);
	if (dd)
	{
		elapsed -= dd * 24UL * 3600UL;
		p += sprintf(p, "(%u day(s) ", dd);
	}
	hh = elapsed / 3600UL;
	if (hh || dd)
	{
		elapsed -= hh * 3600UL;
		p += sprintf(p, "%s%2uh ", dd? "": "(", hh);
	}
	mm = elapsed / 60UL;
	elapsed -= mm * 60UL;
	sprintf(p, "%s%02um %2us ago)", hh||dd? "": "(", mm, elapsed);

	sprintf(buf2, "%d %s %d",
		tm.tm_mday, month_abbr(tm.tm_mon), tm.tm_year + 1900);

	printf("\t%-15s %18s %29s\n", hdr, buf2, buf);
}

static void display_written_block(uint32_t ip_addr,
	ip_data_t *ip_data, ip_data_t *old_data, DB *db_d)
{
	// if syslog_opt, log a line anyway (unlikely given -v).
	if (ap.mode)
		logline_written_block(ip_addr, ip_data, old_data, db_d);

	struct in_addr in;
	char descr[19 /* IPQBDB_DESCR_MAX_LENGTH */];
	memcpy(&in, &ip_addr, sizeof in);
	time_t now = ip_data->last_update;

	get_descr(db_d, ip_data->reason_id,
		get_descr_quoted | get_descr_add_id, descr, sizeof descr);
	if (old_data)
	{
		printf("changed probability record for %s", inet_ntoa(in));
		if (ip_data->reason_id == old_data->reason_id)
			printf(", %s", descr);
		fputc('\n', stdout);
		tab_printtime("created:", old_data->created, now);
		tab_printtime("last updated:", old_data->last_update, now);
		if (old_data->last_block)
			tab_printtime("last blocked:", old_data->last_block, now);
		printf("\t%-15s %18g -> %26g\n", "decay:",
			old_data->decay, ip_data->decay);
		printf("\t%-15s %10d=%6.2f%% -> %18d=%6.2f%%\n", "probability:",
			old_data->probability, percent_prob(old_data->probability),
			ip_data->probability, percent_prob(ip_data->probability));
		if (ip_data->reason_id != old_data->reason_id)
		{
			char old_descr[sizeof descr];
			get_descr(db_d, old_data->reason_id,
				get_descr_quoted | get_descr_add_id,
				old_descr, sizeof old_descr);
			printf("\t%-15s %18.18s -> %26.18s\n", "reason:",
				old_descr, descr);
		}
	}
	else
	{
		printf("inserted probability record for %s (reason %d)\n",
			inet_ntoa(in), ip_data->reason_id);
		printf("\tdecay: %g, probability: %d=%.2f%%\n",
			ip_data->decay,
			ip_data->probability, percent_prob(ip_data->probability));
	}
}

static int
write_block_display(DB *db, uint32_t ip_addr, double decay, int reason_id, DB *db_d)
/*
* Do initial_count and force options, then write and possibly display block.
* Return -1 on error, 0 if all ok, 1 if all ok and probability reached 100%.
*/
{
	int probability;

	if (initial_count == 999)
		probability = 0;
	else
	{
		if ((probability = count2prob(initial_count)) <= 0)
		{
			report_error(&ap, LOG_NOTICE,
				"count %d is too high: initial probability set to 1\n",
				initial_count);
		}
		probability += 1; // rounded
	}

	int force = 0;
	if (force_probability_opt) force |= write_block_force_probability;
	if (force_decay_opt) force |= write_block_force_decay;
	if (force_reason_opt) force |= write_block_force_reason;

	if (write_block(db, ip_addr, decay, reason_id, &probability, force,
		verbose? display_written_block:
		ap.mode? logline_written_block: NULL, db_d))
			return -1;

	return probability >= RAND_MAX;
}

static void run_exec_ip(connkill_cmd *cmd, uint32_t ip_addr)
{
	char *ip_address;
	struct in_addr in;
	memcpy(&in, &ip_addr, sizeof in);

	ip_address = strdup(inet_ntoa(in));
	run_exec(cmd, ip_address, &ap);
}

int main(int argc, char const *argv[])
{
	static const char optaliases[] = IPQBDB_OPTION_FILE;
	int rtc = 0, errs = 0;
	uint32_t *ip_addr = NULL;

	poptContext opt = poptGetContext(err_prefix, argc, argv, opttab, 0);

	if (access(optaliases, F_OK) == 0 &&
		(rtc = poptReadConfigFile(opt, optaliases)) < 0)
	{
		fprintf(stderr, "%s: cannot read %s: %s\n",
			err_prefix, optaliases, poptStrerror(rtc));
		errs = 3;
	}

	rtc = poptGetNextOpt(opt);
	if (rtc != -1)
	{
		fprintf(stderr, "%s: %s\n",
			err_prefix, poptStrerror(rtc));
		errs = 1;
	}
	else
	{
		if (version_opt)
		{
			fprintf(stdout, "%s: version " PACKAGE_VERSION "\n"
			DB_VERSION_STRING "\n", err_prefix);
			errs = 2;
		}

		if (help_opt)
		{
			poptPrintHelp(opt, stdout, 0);
			fputs_database_help();
			fputs_initial_count_help();
			fputs("The special value 999 for the initial count"
				" with the --force-probability\nflag can be used to zero out"
				" the probability and grant access to the given IP.\n", stdout);
			errs = 2;
		}

		// popt sets verbose to 0 if no arg is given
		verbose += 1;

		ap.mode = error_report_stderr; // 0
		ap.err_prefix = err_prefix;
		if (syslog_opt)
		{
			openlog(err_prefix, LOG_PID, LOG_DAEMON);
			ap.mode = LOG_DAEMON;
		}

		if (poptPeekArg(opt) != NULL)
		{
			report_error(&ap, LOG_ERR, "unexpected argument: %s\n",
				poptGetArg(opt));
			errs = 1;
		}

		if (get_all_ip_address(&ip_addr))
			errs = 3;

		if (initial_decay < 0.0)
		{
			report_error(&ap, LOG_WARNING, "initial decay %g is negative\n",
				initial_decay);
			errs = 3;
		}
	}

	if (errs == 1 && !syslog_opt)
		poptPrintUsage(opt, stdout, 0);
	poptFreeContext(opt);
	rtc = 0;

	if (errs)
		rtc = 1;
	else
	{
		char *block_fname = NULL;
		char *descr_fname = NULL;
		char *white_fname = NULL;

		int reason_id = -1;
		char *t = NULL;
		unsigned long l = strtoul(reason_string, &t, 0);
		if (t && *t == 0 && l < INT_MAX)
			reason_id = (int)l;

		if (ip_addr)
		{
			block_fname = database_fname(db_block_name, &ap);
			white_fname = database_fname(db_white_name, &ap);
			if (block_fname == NULL || white_fname == NULL)
				rtc = 1;
		}

		if (reason_id == -1 || verbose)
		{
			descr_fname = database_fname(db_descr_name, &ap);
			if (descr_fname == NULL)
				rtc = 1;
		}

		if (rtc)
		{
			if (verbose)
				report_error(&ap, LOG_INFO, "Bailing out...\n");
		}
		else if (block_fname == NULL && descr_fname == NULL)
		{
			if (verbose)
			{
				report_error(&ap, LOG_WARNING,
					"No ip address to ban, no string description to check\n");
			}
		}
		else
		{
			DB_ENV *db_env = NULL;
			DB *db_block = NULL, *db_white = NULL, *db_de1 = NULL, *db_de2 = NULL;
			uint32_t exec_connkill = 0;

			setsigs();

			rtc = open_database(block_fname? block_fname: descr_fname,
					&ap, &db_env, block_fname? &db_block: NULL);

			if (rtc == 0 && white_fname && caught_signal == 0)
				rtc = open_database(white_fname, &ap, &db_env, &db_white);

			if (rtc == 0 && descr_fname && caught_signal == 0)
				rtc = open_descrdb(descr_fname, db_env, &db_de1, &db_de2);

			if (rtc == 0 && reason_id == -1 && caught_signal == 0)
				rtc = get_reason_id(&reason_id, reason_string,
					&ap, db_de1, db_de2, verbose);

			if (rtc == 0 && ip_addr && caught_signal == 0)
			{
				double decay;
				uint32_t *next_ip = ip_addr;
				while (rtc == 0 && *next_ip && caught_signal == 0)
				{
					rtc = display_check_whitelist(db_white, *next_ip, &decay);

					if (rtc == 0 && caught_signal == 0 && decay > 0.0)
					{
						int a_rc = write_block_display(
							db_block, *next_ip, decay, reason_id, db_de1);
						if (a_rc > 0)
							exec_connkill = *next_ip;
						else if (a_rc != 0)
							rtc = -1; // bail out on error
					}
					++next_ip;
				}
			}

			if (rtc)
				rtc = 2;

			close_db(db_de2);
			close_db(db_de1);
			close_db(db_white);
			close_db(db_block);

			close_dbenv(db_env, cleanup_opt);

			if (exec_connkill && exec_connkill_opt)
			{
				connkill_cmd *cmd = read_connkill_cmd(&ap);
				if (cmd)
					run_exec_ip(cmd, exec_connkill);

				rtc = 2; // either exec or cmd failed
			}
		}
	}

	return rtc;
}

