/*
* Ipq Berkeley db Daemon  WHITE - ibd-white
* written by ale in milano on 17 dec 2008
* edit entries of white.db

Copyright (C) 2008-2019 Alessandro Vesely

This file is part of Ipqbdb.

Ipqbdb is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Ipqbdb is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Ipqbdb.  If not, see <http://www.gnu.org/licenses/>.

*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <unistd.h>
#include <limits.h>
#include <ctype.h>
#include <signal.h>
#include <syslog.h>
#include <errno.h>

#include <sys/types.h>

// Berkeley DB v5.3
#include <db.h>

#include <popt.h>

// format of data packet
#include "config_names.h"
#include "dbstruct.h"

#include <time.h>

// for inet_aton
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "ipv4_util.h"


static app_private ap;
static char const err_prefix[] = "ibd-white";
int caught_signal;

static void sig_catcher(int sig)
{
#if !defined(NDEBUG)
	if (ap.mode == 0)
	{
		char buf[80];
		unsigned s = snprintf(buf, sizeof buf,
			"%s [%d]: received signal %s\n",
			err_prefix, (int)getpid(), strsignal(sig));
		if (s >= sizeof buf)
		{
			s = sizeof buf;
			buf[s - 1] = '\n';
		}
		write(2, buf, s);
	}
#endif
	if (sig == SIGABRT)
		syslog(LOG_CRIT, "aborting %s\n", err_prefix);
	caught_signal = sig;
}

static int setsigs(void)
{
	int rtc;
	struct sigaction act;
	memset(&act, 0, sizeof act);
	sigemptyset(&act.sa_mask);
	act.sa_flags = 0;
	act.sa_handler = sig_catcher;

	rtc =
		sigaction(SIGPIPE, &act, NULL) ||
		sigaction(SIGILL, &act, NULL)  ||
		sigaction(SIGHUP, &act, NULL)  ||
		sigaction(SIGINT, &act, NULL)  ||
		sigaction(SIGQUIT, &act, NULL) ||
		sigaction(SIGPWR,  &act, NULL) ||
		sigaction(SIGABRT, &act, NULL) ||
		sigaction(SIGTERM, &act, NULL);

	act.sa_handler = SIG_IGN;
	rtc |=
		sigaction(SIGALRM, &act, NULL) ||
		sigaction(SIGIO,   &act, NULL) ||
		sigaction(SIGUSR1, &act, NULL) ||
		sigaction(SIGUSR2, &act, NULL);
//		sigaction(SIGEMT,  &act, NULL);
	return rtc;
}

static char *db_white_name = IPQBDB_WHITE_DATABASE_NAME;
static double default_decay = (IPQBDB_INITIAL_DECAY/2);
static char *ip_address = "0.0.0.0/0";
static int version_opt, help_opt, syslog_opt, cleanup_opt, force_opt;
static int verbose = -1;
static int list_opt, trunc_opt;
static struct poptOption opttab[] =
{
	{"db-white", 'w', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &db_white_name, 0,
	"The whitelist database.", "filename"},
	{"ls", 'L', POPT_ARG_NONE, &list_opt, 0,
	"List existing entries (before possibly truncate)", NULL},
	{"truncate", '\0', POPT_ARG_NONE, &trunc_opt, 0,
	"Truncate the database before inserting new entries", NULL},
	{"default-decay", 't', POPT_ARG_DOUBLE|POPT_ARGFLAG_SHOW_DEFAULT, &default_decay, 0,
	"The default decay value (seconds)", "float"},
	{"ip-addr", 'i', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &ip_address, 0,
	"The single address or range to list", "ip[-last|/cidr]"},
	{"verbose", 'v', POPT_ARG_INT|POPT_ARGFLAG_OPTIONAL, &verbose, 0,
	"Be verbose", "level"},
	{"log-syslog", 'l', POPT_ARG_NONE, &syslog_opt, 0,
	"Log to syslog rather than std I/O", NULL},
	{"db-cleanup", '\0', POPT_ARG_NONE, &cleanup_opt, 0,
	"On exit cleanup environment (__db.00? files) if not still busy", NULL},
	{"force-insane", '\0', POPT_ARG_NONE, &force_opt, 0,
	"Allow inserting more than 65536 records per input line", NULL},
	{"version", 'V', POPT_ARG_NONE, &version_opt, 0,
	"Print version number and exit", NULL},
	{"help", 'h', POPT_ARG_NONE, &help_opt, 0,
	"This help.", NULL},
	POPT_TABLEEND
};

static int get_ip_address(ip4_range *ip)
{
	int err = parse_ip_address(ip_address, ip, NULL);
	if (err)
		report_error(&ap, LOG_ERR, "invalid %s: %s (err=%d)\n",
			parse_ip_invalid_what(err), ip_address, err);
	return err;
}

static void print_line(ip4_range *cur, double decay, int *have_printed)
{
	char buf[32];

	if (cur->first == cur->last)
		buf[0] = 0;
	else
	{
		uint32_t mask = htonl(cur->first ^ cur->last);

		if ((mask & (mask + 1)) == 0 /* mask + 1 is a power of 2 */ &&
			(mask & htonl(cur->first)) == 0 /* cur->first is the network */)
		{
			unsigned cidr = 32U;
			do --cidr; while (mask >>= 1);
			sprintf(buf, "/%u", cidr);
		}
		else
			sprintf(buf, "-%s", inet_ntoa(*(struct in_addr*)&cur->last));
	}

	if (verbose && !*have_printed)
	{
		printf("%15s%-16s %12s\n", "IP", " RANGE/CIDR", "DECAY");
		*have_printed = 1;
	}

	printf("%15s%-16s %12g\n",
		inet_ntoa(*(struct in_addr*)&cur->first), buf, decay);
}

static int do_list(DB *db, ip4_range *ip)
{
	DBC *curs;
	DBT key, data;
	uint32_t ip_last = ip->last;
	uint32_t cur_key = ip->first;
	int rtc, rtc2;
	unsigned total_found = 0;
	time_t const now = time(NULL);

	double decay = 0.0;
	ip4_range cur;
	int have_cur = 0, have_printed = 0;

	memset(&cur, 0, sizeof cur);

	memset(&key, 0, sizeof key);
	memset(&data, 0, sizeof data);

	key.data = &cur_key;
	key.ulen = key.size = sizeof cur_key;
	key.flags = data.flags = DB_DBT_USERMEM;
	rtc = db->get_pagesize(db, &data.ulen);
	data.ulen *= 32;
	data.flags = DB_DBT_USERMEM;
	if (rtc ||
		(data.data = malloc(data.ulen)) == NULL)
	{
		report_error(&ap, LOG_ERR, "memory fault\n");
		return -1;
	}

	rtc = db->cursor(db, NULL, &curs, 0);
	if (rtc)
	{
		db->err(db, rtc, "cannot create cursor");
		free(data.data);
		return rtc;
	}

	for (;;)
	{
		void *p, *retkey, *retdata;
		size_t retklen, retdlen;

		rtc = curs->c_get(curs, &key, &data, DB_MULTIPLE_KEY|DB_SET_RANGE);
		if (rtc)
		{
			if (rtc != DB_NOTFOUND)
				db->err(db, rtc, "cannot read cursor");
			break;
		}

		for (DB_MULTIPLE_INIT(p, &data);;)
		{
			DB_MULTIPLE_KEY_NEXT(p, &data, retkey, retklen, retdata, retdlen);
			if (p == NULL ||
				(rtc =
					memcmp(&ip_last, retkey, sizeof ip_last) < 0?
						DB_NOTFOUND: 0) != 0)
							break;

			ip_white_t *const ip_white = (ip_white_t*)retdata;
			if (caught_signal ||
				retdlen < sizeof *ip_white || retklen < sizeof cur_key ||
				ip_white->chk != IPQBDB_CHK_SIGNATURE)
			{
				report_error(&ap, LOG_ERR, "reading data: %s%s%s%sstop.\n",
					caught_signal? "signal, ": "",
					retdlen < sizeof *ip_white? "bad data length, ": "",
					retklen < sizeof cur_key? "bad key length, ": "",
					ip_white->chk != IPQBDB_CHK_SIGNATURE? "bad record data, ": "");
				
				rtc = 1;
				break;
			}

			cur_key = *(uint32_t*)retkey;

			if (have_cur)
			{
				if (decay == ip_white->decay)
					cur.last = cur_key;
				else
				{
					print_line(&cur, decay, &have_printed);
					have_cur = 0;
				}
			}

			if (!have_cur)
			{
				decay = ip_white->decay;
				cur.first = cur.last = cur_key;
				have_cur = 1;
			}

			total_found += 1;
		}

		if (rtc)
			break;

		cur_key = htonl(ntohl(cur_key) + 1);
	}

	if (have_cur)
		print_line(&cur, decay, &have_printed);

	rtc2 = curs->c_close(curs);
	if (rtc2)
	{
		db->err(db, rtc2, "cannot close cursor");
		if (rtc == 0 || rtc == DB_NOTFOUND)
			rtc = rtc2;
	}

	free(data.data);

	if (verbose)
	{
		char buf[64];
		time_t const end = time(NULL);
		if (end != now)
			sprintf(buf, " in %d sec(s)", (int)(end - now));
		else
			buf[0] = 0;
		printf("%u record(s) found%s\n",
			total_found, buf);
	}

	return rtc != 0 && rtc != DB_NOTFOUND;
}

static int read_line(int line, ip4_range *ip, double *decay)
// return -1=hard error, 1=bad input, 2=interactive not wanted
{
	char buf[1024];
	int rtc = -1;
	int const prompt = isatty(fileno(stdin));
	
	if (prompt)
	{
		printf("%s> ", err_prefix);
		fflush(stdin);
	}
	char *s = fgets(buf, sizeof buf, stdin);
	if (s)
	{
		size_t len = strlen(s);
		char *ip_range;
		int ch, err;

		rtc = 0;
		if (len + 1 >= sizeof buf && s[len-1] != '\n')
		{
			unsigned extra = 1;
			ch = fgetc(stdin);
			while (ch != '\n' && ch != EOF)
			{
				ch = fgetc(stdin);
				++extra;
			}
			report_error(&ap, LOG_ERR,
				"line %d is %u chars too long\n", line, extra);
			rtc = 1;
		}

		while (isspace(ch = *(unsigned char*)s))
			++s;

		if (ch == 0 || ch == '#')
			return prompt? 2: 1;
		
		if (prompt && !isdigit(ch))
			return 2;
			
		ip_range = s++;
		while ((ch = *(unsigned char*)s) != 0 && !isspace(ch))
			++s;

		*decay = 0.0;
		if (ch)
		{
			*s++ = 0;
			while (isspace(ch = *(unsigned char*)s))
				++s;
			if (ch)
			{
				char *t = NULL;
				*decay = strtod(s, &t);
				if (t > s)
					while (isspace(ch = *(unsigned char*)t))
						++t;
				if (t <= s || *t != 0 || *decay < 0.0 ||
					!(isnormal(*decay) || *decay == 0.0))
				{
					report_error(&ap, LOG_ERR,
						"invalid float %s at line %d\n", s, line);
					rtc = 1;
				}
			}
		}

		err = parse_ip_address(ip_range, ip, NULL);
		if (err)
		{
			report_error(&ap, LOG_ERR, "invalid %s: %s (err=%d) at line %d\n",
				parse_ip_invalid_what(err), ip_range, err, line);
			rtc = 1;
		}
	}

	return rtc;
}

static int do_line(DB *db, int line)
/*
* return the number of records inserted (>= 0), -1 on EOF, -2 on error
*/
{
	double decay = default_decay;
	ip4_range ip, host;
	int rtc = read_line(line, &ip, &decay);

	if (rtc == 2) return -1;

	if (rtc == 0 && caught_signal == 0)
	{
		DBT key, data;
		int inserted = 0;
		ip_white_t white;

		memset(&key, 0, sizeof key);
		memset(&data, 0, sizeof data);
		memset(&white, 0, sizeof white);

		key.data = &ip.first;
		key.size = key.ulen = sizeof ip.first;
		data.data = &white;
		data.size = data.ulen = sizeof white;
		key.flags = data.flags = DB_DBT_USERMEM;
		
		white.decay = decay;
		white.chk = IPQBDB_CHK_SIGNATURE;

		host.first = ntohl(ip.first);
		host.last = ntohl(ip.last);

		if (host.last - host.first >= 0xffff && !force_opt)
			report_error(&ap, LOG_WARNING,
				"Use --force-insane to insert a class B or more\n");
		else
			do
			{
				rtc = db->put(db, NULL, &key, &data, 0);
				if (rtc)
				{
					db->err(db, rtc, "db->put");
					return -2;
				}
				ip.first = htonl(++host.first);
				++inserted;
			} while (host.first <= host.last && caught_signal == 0);

		return inserted; // good return
	}

	return rtc < 0? -1: 0; // discard syntax errors and continue
}


int main(int argc, char const *argv[])
{
	static const char optaliases[] = IPQBDB_OPTION_FILE;
	int rtc = 0, errs = 0;
	ip4_range ip;

	poptContext opt = poptGetContext(err_prefix, argc, argv, opttab, 0);

	if (access(optaliases, F_OK) == 0 &&
		(rtc = poptReadConfigFile(opt, optaliases)) < 0)
	{
		fprintf(stderr, "%s: cannot read %s: %s\n",
			err_prefix, optaliases, poptStrerror(rtc));
		errs = 3;
	}

	rtc = poptGetNextOpt(opt);
	if (rtc != -1)
	{
		fprintf(stderr, "%s: %s\n",
			err_prefix, poptStrerror(rtc));
		errs = 1;
	}
	else
	{
		if (poptPeekArg(opt) != NULL)
		{
			fprintf(stderr, "%s: unexpected argument: %s\n",
				err_prefix, poptGetArg(opt));
			errs = 1;
		}

		if (version_opt)
		{
			fprintf(stdout, "%s: version " PACKAGE_VERSION "\n", err_prefix);
			errs = 2;
		}

		if (help_opt)
		{
			poptPrintHelp(opt, stdout, 0);
			fputs_database_help();
			errs = 2;
		}

		if (get_ip_address(&ip))
			errs = 3;

		// popt sets verbose to 0 if no arg is given
		verbose += 1;

		ap.mode = error_report_stderr; // 0
		ap.err_prefix = err_prefix;
		if (syslog_opt)
		{
			openlog(err_prefix, LOG_PID, LOG_DAEMON);
			ap.mode = LOG_DAEMON;
		}
	}

	if (errs == 1)
		poptPrintUsage(opt, stdout, 0);
	poptFreeContext(opt);
	rtc = 0;

	if (errs)
		rtc = 1;
	else
	{
		char *white_fname = database_fname(db_white_name, &ap);

		if (white_fname == NULL)
			rtc = 1;

		if (rtc)
		{
			if (verbose)
				report_error(&ap, LOG_INFO, "Bailing out...\n");
		}
		else
		{
			DB_ENV *db_env = NULL;
			DB *db_white = NULL;

			setsigs();

			rtc = open_database(white_fname, &ap, &db_env, &db_white);

			if (rtc == 0 && list_opt && caught_signal == 0)
			{
				if (verbose)
				{
					printf("list entries in IP range %d.%d.%d.%d-%d.%d.%d.%d\n",
						((unsigned char*)&ip.first)[0],
						((unsigned char*)&ip.first)[1],
						((unsigned char*)&ip.first)[2],
						((unsigned char*)&ip.first)[3],
						((unsigned char*)&ip.last)[0],
						((unsigned char*)&ip.last)[1],
						((unsigned char*)&ip.last)[2],
						((unsigned char*)&ip.last)[3]);
				}
				rtc = do_list(db_white, &ip);
			}

			if (rtc == 0 && trunc_opt && caught_signal == 0)
			{
				u_int32_t count = 0;
				rtc = db_white->truncate(db_white, NULL, &count, 0);
				if (rtc)
					db_white->err(db_white, rtc, "truncate");
				else if (verbose)
					printf("%u record(s) discarded from %s\n",
						count, white_fname);
			}

			if (rtc == 0 && caught_signal == 0)
			{
				unsigned count = 0;
				unsigned line = 0;
				while (!feof(stdin) && !ferror(stdin) && caught_signal == 0)
				{
					rtc = do_line(db_white, ++line);
					if (rtc < 0)
						break;
					count += rtc;
				}

				if (ferror(stdin))
					report_error(&ap, LOG_ERR, "Error reading stdin: %s`n",
						strerror(errno));
				else if (rtc == -1)
					rtc = 0;
				if (verbose)
					printf("%u record(s) written to %s\n", count, white_fname);
			}

			if (rtc)
				rtc = 2;

			close_db(db_white);
			close_dbenv(db_env, cleanup_opt);
			if (caught_signal)
				report_error(&ap, LOG_NOTICE,
					"exiting after signal %s\n", strsignal(caught_signal));
		}
	}

	return rtc;
}
