/*
* Ipq Berkeley db Daemon  DEL - ibd-del
* written by ale in milano on 15 nov 2008
* delete and/or list records in a block.db

Copyright (C) 2008-2019 Alessandro Vesely

This file is part of Ipqbdb.

Ipqbdb is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Ipqbdb is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Ipqbdb.  If not, see <http://www.gnu.org/licenses/>.

*/
#define _GNU_SOURCE 1
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <unistd.h>
#include <limits.h>
#include <ctype.h>
#include <signal.h>
#include <syslog.h>
#include <errno.h>
#include <locale.h>
#include <sys/types.h>

// Berkeley DB v5.3
#include <db.h>

#include <popt.h>

// format of data packet
#include "config_names.h"
#include "dbstruct.h"

#include <time.h>

// for inet_aton
#define _GNU_SOURCE 1
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "ipv4_util.h"
#include "percent_prob.h"
#include "rehabilitated_prob.h"
#define INITIAL_COUNT_H_BOTH
#include "initial_count.h"

#include <pcre.h> // for grepping the reason table
#include <assert.h>

/* date conversion utilities */
static const char *month_abbr(unsigned ndx)
{
	static const char *abbr[] =
	{"Jan", "Feb", "Mar", "Apr", "May", "Jun",
	 "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};
	if (ndx >= sizeof abbr / sizeof abbr[0]) return "---";
	return abbr[ndx];
}

#define TIME_STRING_BUFSIZE 9
static char *
time_string(time_t datum, time_t now, char *dest)
{
	char buf[256]; // avoid overflows in case of error

	if (datum == 0) // never set: blank
		buf[0] = 0;
	else if (now - datum <= 900) // less than a quarter: NNNs ago
		sprintf(buf, "%ds ago", (int)(now - datum));
	else
	{
		struct tm tm;
		localtime_r(&datum, &tm);
		if (now - datum < 20L*3600) // less than 20 hour: hh:mm:ss
			sprintf(buf, "%02d:%02d:%02d", tm.tm_hour, tm.tm_min, tm.tm_sec);
		else if (now - datum < 20L*86400L) // less than 20 days: dd-hh:mm
			sprintf(buf, "%d-%02d:%02d", tm.tm_mday, tm.tm_hour, tm.tm_min);
		else // Mon-YYYY
			sprintf(buf, "%s-%d", month_abbr(tm.tm_mon), tm.tm_year + 1900);
	}

	buf[TIME_STRING_BUFSIZE - 1] = 0;
	memcpy(dest, buf, TIME_STRING_BUFSIZE);
	return dest;
}

/*
* adaptive deletion
* Problem: Delete records older than a given date, but only if this is needed
* to keep the database within the given number of records.
*
* If records were sorted by their increasing age, the problem would be solved
* by deleting all records after either the given max size or the given time,
* whichever comes last.  Instead, we read the matching block.db records in
* their natural order, and build an histogram of the number of records in each
* period of time.  From the histogram we determine the boundary date such that
* the problem is solved by deleting the records older than that.  This implies
* reading block.db for a second time, but is still quicker than sorting it.  It
* is also less precise.
*
* If the user has given a period P > 0, we can be more precise around that
* order of magnitude by mapping linearly times less than P onto the first N
* entries, that is index(t) = N*t/P; and using logarithms thereafter, that is
* index(t) = log(k*t)/log(f), where the factor f and the scale k are constants
* that we determine like so:  For continuity, we need log(k*P)/log(f) = N; that
* is, log(f) = log(k*P)/N.  For smoothness, we need the derivatives of index(t)
* to match at t=P, i.e. N/(P*log(k*P)) = N/P; that implies log(k*P) = 1, or
* k = e/P, if using natural logarithms.  Thus, we find log(f) = 1/N, and the
* mapping to get the index of a histogram bar from the age is
*
*             index(t) = log(k*t)*N   for t > P
*                        N*t/P        for t <= P
*
* The inverse mapping is
*
*             time(i) = exp(i/N)/k    for i > N
*                       P*i/N         for i <= N
*
* We need to determine the size of the histogram and the value of N.  Let M be
* the maximum age that we want to map linearly, and L the time span covered by
* a linearly mapped bar of the histogram.
*
* To ensure time(L*N) > M, we need exp(L-1) > M/P; that is L > 1+log(M/P).  So
* we can set N to a reasonable value by choosing, say, M = 4 years and an array
* of, say, 256 entries; then we calculate N as floor(256/(1+log(M/P))).  With
* these settings we get the following, for different values of the period:

period     N    time(256)/M
1 hour    22    1.18700507875353039495
6 hours   26    1.18884598165453721407
1 day     30    1.27948406388186572843
2 days    33    1.17803350415775168857
7 days    40    1.06081102904126965515
30 days   52    1.03810599376929316910
6 month   83    1.00490781176609466834
1 year   107    1.00624844395160134207
*
* N is called "mesh" below.  The period is determined as 1/4 of the min_time,
* that is when selecting records that must be at least that old.  The sense of
* max_time is rather different: it is not related with the maximum number of
* records, so we just select all records that are not older than that.
*/
#define HISLOG_SIZE 256
typedef struct histog
{
	int *his;
	double k, log_k, dmesh;
	time_t mesh, period, increase, now;
	int total_selected, total_recs, out_of_range;
} histog;

static int init_histog(histog *h, time_t secs)
{
	memset(h, 0, sizeof *h);
	if ((h->his = (int*)malloc(HISLOG_SIZE * sizeof(int))) == NULL)
		return -1;

	memset(h->his, 0, HISLOG_SIZE * sizeof(int));

	if (secs <= 0) secs = 86400;
	else
	{
		secs /= 4;
		if (secs < 3600)
			secs = 3600;
	}

	double const period = (double)secs;
	int mesh = (int)floor((double)HISLOG_SIZE /
		(1.0 + log((double)((365U*4U + 1U)*86400U)/period)));

	if (mesh < 20) mesh = 20;
	else if (mesh > HISLOG_SIZE/2) mesh = HISLOG_SIZE/2;

	h->mesh = mesh;
	h->dmesh = (double)mesh;
	h->period = secs;
	h->k = M_E/period;
	h->log_k = 1.0 - log(period);

	return 0;
}

static inline double histog_index_d(histog const *h, double t)
/*
* internal function to compute the index
*/
{
	assert(h && t + 1.0 >= (double)h->period);
	return (h->log_k + log(t)) * h->dmesh + 0.5;
}

static char const *histog_index(histog const *h, time_t t, unsigned *result)
/*
* compute the index corresponding to the time (called often)
*/
{
	if (t <= h->period)
		*result = (unsigned)((h->mesh * t)/h->period);
	else
	{
		double dndx = floor(histog_index_d(h, (double)t));
		if (!isnormal(dndx) || dndx > 10000.0 || dndx < 0.0)
			return "unnormal or too big time";

		*result = (unsigned)dndx;
	}

	return NULL;
}

static char const *histog_time(histog const *h, int ndx, time_t *result)
/*
* compute the time corresponding to the index (called sparingly)
*/
{
	if (ndx <= h->mesh)
		*result = ((time_t)ndx * h->period + h->mesh - 1)/h->mesh;
	else
	{
		double const i = (double)ndx;
		double tt = exp((ndx)/(h->dmesh))/h->k;

		/*
		* Find the smallest (more recent, when mapped to real time) side of
		* the interval whose t values are mapped to ndx. Note that the interval
		* is larger than 1, because we are in the logarithmic zone.
		*
		* The derivative of histog_index_d(h, t) at t=tt is dmesh/tt. Since
		* 0 < dmesh/tt < 1, Newton iterations around tt are stable and quick.
		*/
		double ii = histog_index_d(h, tt);
		double t = ((i - ii)/h->dmesh + 1.0) * tt;
		// printf("  for %d: %g->%g", ndx, tt, t);

		while (isnormal(t) && fabs(t - tt) > 0.4)
		{
			tt = t;
			ii = histog_index_d(h, tt);
			t = ((i - ii)/h->dmesh + 1.0) * tt;
			// printf("->%g", t);
		}
		// putchar('\n');

		int fp = fpclassify(t);
		if (fp == FP_NORMAL)
			// since we sought the boundary, rounding can go on either side,
			// adding 1.0 ensures we get the integer on the right side.
			*result = (time_t)floor(t + 1.0);
		else
			return "non-normal or too big";
	}

	return NULL;
}

static const char* count_histog(histog *h, time_t secs)
{
	assert(h);
	unsigned ndx;

	char const *err = histog_index(h, secs, &ndx);
	if (err)
		return err;

	if (ndx >= HISLOG_SIZE)
		h->out_of_range += 1;
	else
		h->his[ndx] += 1;
	h->total_selected += 1;
	return NULL;
}

static const char *increase_secs(histog *h, int target, time_t secs)
/*
* Set the value of h->increase to the number of seconds that can be
* added to the selection, according to the histogram.
*/
{
	assert(h && h->increase == 0);

#if defined NDEBUG
	(void)secs; // only used  to compute intervals
#else
	{
		unsigned i;
		printf("==histogram debug:\n"
			"  k=%g, log_k=%g, dmesh=%g\n"
			"  mesh=%jd, period=%jd, now=%jd\n"
			"  selected=%d, recs=%d, out_of_range=%d\n",
			h->k, h->log_k, h->dmesh,
			(intmax_t)h->mesh, (intmax_t)h->period, (intmax_t)h->now,
			h->total_selected, h->total_recs, h->out_of_range);

		for (i = 0; i < HISLOG_SIZE; ++i)
			if (h->his[i])
			{
				char const *er_f, *er_t = NULL;
				char from[TIME_STRING_BUFSIZE], to[TIME_STRING_BUFSIZE];
				time_t from_time, to_time, mark = h->now - secs;

				er_f = histog_time(h, i + 1, &from_time);
				if (er_f == NULL)
					er_f = time_string(mark - from_time + 1, h->now, from);

				er_t = histog_time(h, i, &to_time);
				if (er_t == NULL)
					er_t = time_string(mark - to_time, h->now, to);

				printf("  his[%3d]=%8d, t=%9jd: [%s, %s]\n",
					i, h->his[i], (intmax_t)to_time, er_f, er_t);
			}

		for (i = 0; i < HISLOG_SIZE; ++i)
		{
			time_t t = 0;
			unsigned reconv = -1;
			char const *er1 = histog_time(h, i, &t);
			char const *er2 = histog_index(h, t, &reconv);
			if (er1 || er2 || reconv != i)
				printf("  %u -%s-> %jd -%s-> %u !!!\n",
					i,
					er1? er1: "",
					(intmax_t)t,
					er2? er2: "",
					reconv);
		}
		printf("  target = %d\n==\n", target);
	}
#endif

	unsigned i;
	target -= h->out_of_range;
	for (i = HISLOG_SIZE; i > 0 && target > 0;)
		target -= h->his[--i];

	if (target <= 0)
	{
		char const *err = histog_time(h, i, &h->increase);
		if (err)
			return err;
	}
	return NULL;
}

/* (end adaptive deletion --perhaps to be moved to its own file) */


static app_private ap;
static char const err_prefix[] = "ibd-del";
int caught_signal;


static void sig_catcher(int sig)
{
#if !defined(NDEBUG)
	if (ap.mode == 0)
	{
		char buf[80];
		unsigned s = snprintf(buf, sizeof buf,
			"%s [%d]: received signal %s\n",
			err_prefix, (int)getpid(), strsignal(sig));
		if (s >= sizeof buf)
		{
			s = sizeof buf;
			buf[s - 1] = '\n';
		}
		write(2, buf, s);
	}
#endif
	if (sig == SIGABRT)
		syslog(LOG_CRIT, "aborting %s\n", err_prefix);
	caught_signal = sig;
}

static int setsigs(void)
{
	int rtc;
	struct sigaction act;
	memset(&act, 0, sizeof act);
	sigemptyset(&act.sa_mask);
	act.sa_flags = 0;
	act.sa_handler = sig_catcher;

	rtc =
		sigaction(SIGPIPE, &act, NULL) ||
		sigaction(SIGILL, &act, NULL)  ||
		sigaction(SIGHUP, &act, NULL)  ||
		sigaction(SIGINT, &act, NULL)  ||
		sigaction(SIGQUIT, &act, NULL) ||
		sigaction(SIGPWR,  &act, NULL) ||
		sigaction(SIGABRT, &act, NULL) ||
		sigaction(SIGTERM, &act, NULL);

	act.sa_handler = SIG_IGN;
	rtc |=
		sigaction(SIGALRM, &act, NULL) ||
		sigaction(SIGIO,   &act, NULL) ||
		sigaction(SIGUSR1, &act, NULL) ||
		sigaction(SIGUSR2, &act, NULL);
//		sigaction(SIGEMT,  &act, NULL);
	return rtc;
}

static char *db_block_name = IPQBDB_DATABASE_NAME;
static char *db_descr_name = IPQBDB_DESCR_DATABASE_NAME;
static char *ip_address = "0.0.0.0/0";
static char *min_time = "0";
static char *max_time;
static char *max_probability = "100%";
static char *min_probability = "0%";
static char *reason_string = "";
static char *max_records_opt = "0";
static long max_records = 0;
static int version_opt, help_opt, syslog_opt, cleanup_opt;
static int only_blocked, non_blocked;
static int verbose = -1;
static int list_opt, raw_opt, del_opt, ban_opt, stat_opt, simple_opt;
static struct poptOption opttab[] =
{
	{"only-blocked", '\0', POPT_ARG_NONE, &only_blocked, 0,
	"Only select records with blocked packets > 0", NULL},
	{"only-never-blocked", '\0', POPT_ARG_NONE, &non_blocked, 0,
	"Only select records with blocked packets = 0", NULL},
	{"max-records", 'M', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &max_records_opt, 0,
	"Limit selection to older records that exceed the given number", "int[K|M|G]"},
	{"min-time", 't', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &min_time, 0,
	"The minimum time elapsed since record was last updated or blocked", "int[D|H|M|S]"},
	{"max-time", '\0', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &max_time, 0,
	"Select records that have been blocked more recently than this", "int[D|H|M|S]"},
	{"min-probability", '\0', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &min_probability, 0,
	"Select records with probability higher than given(*)", "int[%]"},
	{"max-probability", '\0', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &max_probability, 0,
	"Select records with probability lower than given(*)", "int[%]"},
	{"ip-addr", 'i', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &ip_address, 0,
	"The single address or range to delete or list, if any", "ip[-last|/cidr]"},
	{"reason", 'r', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &reason_string, 0,
	"Only consider records matching this", "int|string|regex"},
	{"ls", 'L', POPT_ARG_NONE, &list_opt, 0,
	"List selected records", NULL},
	{"ls-ban", '\0', POPT_ARG_NONE, &ban_opt, 0,
	"List according to ibd-ban format", NULL},
	{"ls-raw", '\0', POPT_ARG_NONE, &raw_opt, 0,
	"List selected records displaying plain numbers", NULL},
	{"ls-simple", '\0', POPT_ARG_NONE, &simple_opt, 0,
	"List selected records in a simple fashion", NULL},
	{"del", '\0', POPT_ARG_NONE, &del_opt, 0,
	"Delete selected records", NULL},
	{"stats", 'S', POPT_ARG_NONE, &stat_opt, 0,
	"List by descr stat counters", NULL},
	{"verbose", 'v', POPT_ARG_INT|POPT_ARGFLAG_OPTIONAL, &verbose, 0,
	"Be verbose", "level"},
	{"log-syslog", 'l', POPT_ARG_NONE, &syslog_opt, 0,
	"Log to syslog rather than std I/O", NULL},
	{"version", 'V', POPT_ARG_NONE, &version_opt, 0,
	"Print version number and exit", NULL},
	{"help", 'h', POPT_ARG_NONE, &help_opt, 0,
	"This help.", NULL},
	{"db-block", 'b', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &db_block_name, 0,
	"The database of blocked IPv4 addresses", "filename"},
	{"db-descr", 'd', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &db_descr_name, 0,
	"The reason description table.", "filename"},
	{"db-cleanup", '\0', POPT_ARG_NONE, &cleanup_opt, 0,
	"On exit cleanup environment (__db.00? files) if not still busy", NULL},
	POPT_TABLEEND
};

static time_t time_invoked;

static int *reason_id, count_reason_id, alloc_reason_id;

static int add_reason_id(int ndx)
{
	if (reason_id == NULL || alloc_reason_id <= count_reason_id)
	{
		int new_alloc_reason_id = 2*count_reason_id + 10;
		int *new_reason_id = (int*)malloc(new_alloc_reason_id * sizeof(int));

		if (new_reason_id == NULL)
			return -1;

		if (reason_id)
		{
			memcpy(new_reason_id, reason_id, count_reason_id * sizeof(int));
			free(reason_id);
		}

		reason_id = new_reason_id;
		alloc_reason_id = new_alloc_reason_id;
	}

	reason_id[count_reason_id++] = ndx;
	return 0;
}

#if 0
// not tested, probably slower in most cases
static int cmp_reason_id(void *r1, void*r2)
{
	// use memcmp as bdb does: items are sorted after c_get
	return memcmp(r1, r2, sizeof(int));
}

static int have_reason_id(int ndx)
{
	return NULL !=
		bsearch(&key, reason_id, count_reason_id, sizeof ndx, cmp_reason_id);
}
#endif

static int have_reason_id(int ndx)
{
	assert(reason_id != NULL);
	for (int i = 0; i < count_reason_id; ++i)
		if (reason_id[i] == ndx)
			return 1;
	return 0;
}

static int all_reason_id(DB *db1, DB *db2)
/*
* prepare the reasons we operate on, according to user input: a number,
* a fixed string, or a regex.
*/
{
	if (reason_string == NULL || *reason_string == 0)
		return 0;

	char *t = NULL;
	unsigned long l = strtoul(reason_string, &t, 0);
	if (t && *t == 0 && l < INT_MAX)
		return add_reason_id((int)l);

	// dup to avoid truncation, pass NULL for db1 to avoid inserting this
	char *r2 = strdup(reason_string);
	int ndx;
	int rtc = r2? get_reason_id(&ndx, r2, &ap, NULL, db2, 0): -1;
	free(r2);
	if (rtc == 0)
		return add_reason_id(ndx);

	if (rtc != DB_NOTFOUND)
		return -1;

	// assume it is an expression, compile it
	const char *error;
	int err_off;
	pcre *re = pcre_compile(reason_string, 0, &error, &err_off, NULL);
	if (re == NULL)
	{
		report_error(&ap, LOG_ERR,
			"%s \"%.*s<< ABOUT HERE>>%s\" in reason regex\n",
			error, err_off, reason_string, reason_string + err_off);
		return -1;
	}

	pcre_extra *reex = pcre_study(re, 0, &error);
	if (error)
		report_error(&ap, LOG_WARNING,
				"W: regex %s reported as %s\n", reason_string, error);

	// get a reading cursor on the primary db (bynumber)
	DBC *curs = NULL;
	rtc = db1->cursor(db1, NULL, &curs, 0);
	if (rtc == 0)
	{
		DBT key, data;
		union
		{
			bynumber_descr_t drec;
			char fill[sizeof(bynumber_descr_t) + IPQBDB_DESCR_MAX_LENGTH + 1];
		} u;
		int ndx;
		memset(&key, 0, sizeof key);
		memset(&data, 0, sizeof data);
		key.data = &ndx;
		key.ulen = sizeof ndx;
		data.data = &u;
		data.ulen = sizeof u;
		key.flags = data.flags = DB_DBT_USERMEM;
		while ((rtc = curs->c_get(curs, &key, &data, DB_NEXT)) == 0 &&
			caught_signal == 0)
		{
			int ovec[30]; // traditional 10 back references
			int rtc2 = pcre_exec(re, reex, u.drec.descr,
				data.size - offsetof(bynumber_descr_t, descr), 0, 0, ovec, 30);
			if (rtc2 >= 0)
				add_reason_id(ndx);
			else if (rtc2 != PCRE_ERROR_NOMATCH)
				report_error(&ap, LOG_ERR,
					"pcre_exec returned %d for expr %s, subject=%s\n",
					rtc2, reason_string, u.drec.descr);
		}

		if (rtc == DB_NOTFOUND)
			rtc = 0;
	}
	if (rtc)
		db1->err(db1, rtc, curs? "c_get": "cursor");
	if (curs)
	{
		int rtc2 = curs->c_close(curs);
		if (rtc2)
		{
			db2->err(db2, rtc2, "c_close");
			rtc = -1;
		}
	}
	pcre_free(re);
	pcre_free(reex);
	return rtc;
}

static int get_seconds(time_t *secs, char *opt_time)
{
	errno = 0;
	char *t = NULL;
	unsigned long long l = strtoull(opt_time, &t, 10);
	if (t && *t && errno == 0)
	{
		switch (toupper(*(unsigned char*)t))
		{
			case 'D':
				l *= 24ULL;
				// all through...
			case 'H':
				l *= 60ULL;
			case 'M':
				l *= 60ULL;
			case 'S':
				++t;
			default:
				break;
		}
	}

	if (t == NULL || *t != 0 || (time_t)l < 0)
	{
		report_error(&ap, LOG_ERR, "invalid time %s\n", opt_time);
		return 1;
	}

	*secs = (time_t)l;
	return 0;
}

static int get_max_records(void)
{
	errno = 0;
	char *t = NULL;
	unsigned long long l = strtoull(max_records_opt, &t, 10);
	if (t && *t && errno == 0)
	{
		switch (toupper(*(unsigned char*)t))
		{
			case 'G':
				l *= 1024ULL;
				// all through...
			case 'M':
				l *= 1024ULL;
			case 'K':
				l *= 1024ULL;
				++t;
			default:
				break;
		}
	}

	if (t == NULL || *t != 0 || l > LONG_MAX)
	{
		report_error(&ap, LOG_ERR, "invalid max-records %s\n", max_records_opt);
		return 1;
	}

	max_records = (long)l;
	return 0;
}

static int get_probability(int *prob, char *probability)
{
	char *t = NULL;
	unsigned long long deno = 1;
	unsigned long long l = strtoull(probability, &t, 10);
	if (t && *t == '.')
	{
		char *dec = t + 1;
		unsigned long long l1 = strtoull(dec, &t, 10);
		deno = llround(exp10((double)(t - dec)));
		l *= deno;
		l += l1;
	}

	if (t && *t == '%')
	{
		l *= RAND_MAX;
		l /= 100;
		++t;
	}

	l /= deno;

	if (t == NULL || *t != 0 || l > RAND_MAX)
	{
		report_error(&ap, LOG_ERR, "invalid probability %s\n", probability);
		return 1;
	}

	*prob = (int)l;
	return 0;
}

static int get_ip_address(ip4_range *ip)
{
	int err = parse_ip_address(ip_address, ip, NULL);
	if (err)
		report_error(&ap, LOG_ERR, "invalid %s: %s (err=%d)\n",
			parse_ip_invalid_what(err), ip_address, err);
	return err;
}

static int
do_del(DB *db, DB*db_d, ip4_range* ip, time_t *secs, int *prob, histog *his)
/*
* If his != NULL, compute the histogram of records exceeding the given min-time
* and perform no other operation.
* Despite its name, this function also does simple listing of the block database.
*/
{
	DBC *curs;
	DBT key, data;
	uint32_t ip_last = ip->last;
	uint32_t cur_key = ip->first;
	int rtc, rtc2;
	unsigned total_selected = 0, total_del = 0, total_delfail = 0, total_recs = 0;
	time_t const now = time_invoked, last_time = now - secs[0],
		first_time = now - secs[1]; // 0 by default, or implies only-blocked

	int max_count = ban_opt? max_initial_count(): 0;

	char descr[IPQBDB_DESCR_MAX_LENGTH];

	memset(&key, 0, sizeof key);
	memset(&data, 0, sizeof data);

	key.data = &cur_key;
	key.ulen = key.size = sizeof cur_key;
	key.flags = data.flags = DB_DBT_USERMEM;
	rtc = db->get_pagesize(db, &data.ulen);
	data.ulen *= 32;
	data.flags = DB_DBT_USERMEM;
	if (rtc ||
		(data.data = malloc(data.ulen)) == NULL)
	{
		report_error(&ap, LOG_ERR, "memory fault\n");
		return -1;
	}

	/*
	* acquire a read cursor: this can possibly delete a record that has
	* been modified since we read it, even if it was requested a min
	* modified time (secs).
	*/
	rtc = db->cursor(db, NULL, &curs, 0);
	if (rtc)
	{
		db->err(db, rtc, "cannot create cursor");
		free(data.data);
		return rtc;
	}

	/*
	* outer loop: fetch a bufferful of records, ordered by IP. The SET_RANGE
	* flag selects the first record that is not less than the key.
	*/
	while (caught_signal == 0)
	{
		void *p, *retkey, *retdata;
		size_t retklen, retdlen;

		rtc = curs->c_get(curs, &key, &data, DB_MULTIPLE_KEY|DB_SET_RANGE);
		if (rtc)
		{
			if (rtc != DB_NOTFOUND)
				db->err(db, rtc, "cannot read cursor");
			break;
		}

		/*
		* inner loop: examine each record in memory.
		*/
		for (DB_MULTIPLE_INIT(p, &data); caught_signal == 0;)
		{
			DB_MULTIPLE_KEY_NEXT(p, &data, retkey, retklen, retdata, retdlen);

			/* test for end-of-buffer and end-of-range */
			if (p == NULL ||
				(rtc =
					memcmp(&ip_last, retkey, sizeof ip_last) < 0?
						DB_NOTFOUND: 0) != 0)
							break;

			/* test for sanity */
			ip_data_t *const ip_data = (ip_data_t*)retdata;
			if (retdlen < sizeof *ip_data || retklen < sizeof cur_key ||
				ip_data->chk != IPQBDB_CHK_SIGNATURE)
			{
				report_error(&ap, LOG_ERR, "Unmatched database record: "
					"data sz=%zu (expected %zu), "
					"key sz=%zu (expected %zu), chk=%#x (expected %#x)\n",
					retdlen, sizeof *ip_data, retklen, sizeof cur_key,
					ip_data->chk, IPQBDB_CHK_SIGNATURE);
				rtc = 1;
				break;
			}

			/* mark key as done */
			cur_key = *(uint32_t*)retkey;
			total_recs += 1;

			/* test for specific selection criteria */
			int selected = (secs[0] == 0 ||
					(ip_data->last_update <= last_time &&
						ip_data->last_block <= last_time)) &&
				ip_data->last_block >= first_time && // ==> was blocked || first==0
				(only_blocked == 0 || ip_data->block_cnt > 0) &&
				(non_blocked == 0 || ip_data->block_cnt == 0) &&
				(reason_id == NULL || have_reason_id(ip_data->reason_id));

			time_t delta = now - ip_data->last_update;
			int adjusted_prob = ip_data->probability;

			if (selected)
			{
				if (adjusted_prob > 0 &&
					IPQBDB_UPDATE_TICK_TIME < delta &&
					ip_data->decay > 0.0)
						adjusted_prob =
							rehabilitated_prob(adjusted_prob, delta, ip_data->decay);

				// adjusted prob should never be negative, but just in case...
				if (!((adjusted_prob >= prob[0] || prob[0] == 0) &&
					adjusted_prob <= prob[1]))
						selected = 0;
			}

			if (selected)
			{
				if (his)
				{
					time_t const ndxsecs = last_time - ip_data->last_update;
					char const *err = count_histog(his, ndxsecs);
					if (err)
					{
						report_error(&ap, LOG_ERR,
							"histogram fails: %s, secs=%jd\n",
							err, (intmax_t)ndxsecs);
						rtc = -1;
						break;
					}
					total_selected += 1;
					continue;
				}

				if (del_opt)
				{
					rtc2 = db->del(db, NULL, &key, 0);
					if (rtc2)
					{
						total_delfail += 1;
						db->err(db, rtc2, "delete");
					}
					else
					{
						total_del += 1;
						set_descr_cnt(db_d, ip_data->reason_id, 0, ip_data, &now);
					}
				}

				char ntopbuf[INET_ADDRSTRLEN];

				if (raw_opt)
				{
					if (verbose && !total_selected)
						printf("%15s %11s %11s %11s %11s %11s %11s %11s %11s %11s\n",
							"IP", "CREATED", "PROBABILITY", "BLOCKED", "PACKETS",
							"UPDATED", "DECAY", "THRESHOLD", "CAUGHT", "DESCRIPTION");

					printf("%15s %11d %11d %11d %11d %11d %11g %11d %11d %11d\n",
						inet_ntop(AF_INET, (struct in_addr*)&cur_key,
							ntopbuf, sizeof ntopbuf),
						(int)ip_data->created,
						ip_data->probability,
						(int)ip_data->last_block,
						ip_data->block_cnt,
						(int)ip_data->last_update,
						ip_data->decay,
						ip_data->decay_add_cnt,
						ip_data->caught_cnt,
						ip_data->reason_id);
				}
				else if (list_opt || ban_opt)
				{
					char buf1[TIME_STRING_BUFSIZE],
						buf2[TIME_STRING_BUFSIZE],
						buf3[TIME_STRING_BUFSIZE];

					if (verbose && list_opt && !total_selected)
						printf("%15s %8s %7s %8s %10s %8s %10s %10s %10s %s\n",
							"IP", "CREATED", "PROB.", "BLOCKED", "PACKETS",
							"UPDATED", "DECAY", "THRESHOLD", "CAUGHT", "DESCRIPTION");

					get_descr(db_d, ip_data->reason_id, 0, descr, sizeof descr);
					if (list_opt)
					{
						printf("%15s %8s %6.2f%% %8s %10d %8s %10g %10d %10d %s\n",
							inet_ntop(AF_INET, (struct in_addr*)&cur_key,
								ntopbuf, sizeof ntopbuf),
							time_string(ip_data->created, now, buf1),
							percent_prob(adjusted_prob),
							time_string(ip_data->last_block, now, buf2),
							ip_data->block_cnt,
							time_string(ip_data->last_update, now, buf3),
							ip_data->decay,
							ip_data->decay_add_cnt,
							ip_data->caught_cnt,
							descr);
					}

					/*
					* creation and update times, as well as caught, blocked, and
					* threshold counts are lost in this way.  However, ip, reasons,
					* and decays are saved; probabilities will be nearly the same
					* if results are sourced more or less at the same time.
					*/
					if (ban_opt)
						printf("ibd-ban --ip-addr=%s --initial-count=%d"
							" --initial-decay=%g --reason='%s'\n",
							inet_ntop(AF_INET, (struct in_addr*)&cur_key,
								ntopbuf, sizeof ntopbuf),
							guess_initial_count(adjusted_prob, max_count),
							ip_data->decay,
							descr);
				}
				else if (simple_opt)
					printf("%s was caught %d time%s since %s",
						inet_ntop(AF_INET, (struct in_addr*)&cur_key,
							ntopbuf, sizeof ntopbuf),
						ip_data->caught_cnt,
						ip_data->caught_cnt > 1? "s": "",
						ctime(&ip_data->created)/* ctime ends with \n */);

				total_selected += 1;
			}
		}

		if (rtc)
			break;

		cur_key = htonl(ntohl(cur_key) + 1);
	}

	rtc2 = curs->c_close(curs);
	if (rtc2)
	{
		db->err(db, rtc2, "cannot close cursor");
		if (rtc == 0 || rtc == DB_NOTFOUND)
			rtc = rtc2;
	}

	free(data.data);

	if (his)
	{
		his->now = now;
		his->total_recs = total_recs;
		his->total_selected = total_selected;
	}
	else if (verbose)
	{
		time_t const end = time(NULL);
		if (end != now)
			sprintf(descr, " in %d sec(s)", (int)(end - now));
		else
			descr[0] = 0;
		printf("%u record(s) selected, %u deleted, %u failed deletion(s)%s\n",
			total_selected, total_del, total_delfail, descr);
	}

	return (rtc != 0 && rtc != DB_NOTFOUND) || total_delfail;
}

static int do_stat(DB *db)
{
	DBC *curs;
	DBT key, data;
	int rtc, descr_id;
	union
	{
		bynumber_descr_t drec;
		char fill[sizeof(bynumber_descr_t) + IPQBDB_DESCR_MAX_LENGTH + 1];
	} u;

	unsigned total_selected = 0, total_recs = 0;
	time_t const now = time(NULL);

	uint64_t caught_cnt = 0, rec_cnt = 0, block_cnt = 0, decay_add_cnt = 0;

	memset(&key, 0, sizeof key);
	memset(&data, 0, sizeof data);

	key.data = &descr_id;
	key.ulen = key.size = sizeof descr_id;
	key.flags = data.flags = DB_DBT_USERMEM;
	data.data = &u;
	data.ulen = data.size = sizeof u;

	rtc = db->cursor(db, NULL, &curs, 0);
	if (rtc)
	{
		db->err(db, rtc, "no descr cursor");
		return rtc;
	}

	setlocale(LC_NUMERIC, ""); // for thousands separators
	while (caught_signal == 0 &&
		(rtc = curs->get(curs, &key, &data, DB_NEXT)) == 0)
	{
		if (u.drec.chk != IPQBDB_CHK_SIGNATURE)
		{
			report_error(&ap, LOG_ERR,
				"Bad descr record format for key %d\n",
				descr_id);
			continue;
		}

		int const selected = reason_id == NULL || have_reason_id(descr_id);
		total_recs += 1;
		if (selected)
		{
			char buf1[TIME_STRING_BUFSIZE],
				buf2[TIME_STRING_BUFSIZE],
				buf3[TIME_STRING_BUFSIZE];

			if (verbose && !total_selected)
				printf("%11s %11s %11s %8s %8s %11s %8s %s\n",
					"CAUGHT", "BLOCKED", "ADD REC", "LAST ADD", "LAST DEL",
					"THRESHOLD", "CREATED", "DESCRIPTION");
			printf("%'11" PRIi64 " %'11" PRIi64 " %'11" PRIi64
				" %8s %8s" " %'11" PRIi64 " %8s %s [%d]\n",
				u.drec.caught_cnt, u.drec.block_cnt, u.drec.rec_cnt,
				time_string(u.drec.last_added, now, buf1),
				time_string(u.drec.last_removed, now, buf2), u.drec.decay_add_cnt,
				time_string(u.drec.created, now, buf3), u.drec.descr, descr_id);
			total_selected += 1;
			caught_cnt += u.drec.caught_cnt;
			block_cnt += u.drec.block_cnt;
			rec_cnt += u.drec.rec_cnt;
			decay_add_cnt += u.drec.decay_add_cnt;
		}
	}

	curs->close(curs);

	if (rtc != DB_NOTFOUND)
		db->err(db, rtc, "read descr (%d)", descr_id);
	else
		rtc = 0;

	if (total_selected > 1)
		printf("%'11" PRIi64 " %'11" PRIi64 " %'11" PRIi64
			" %8s %8s" " %'11" PRIi64 " %8s Total\n",
			caught_cnt, block_cnt, rec_cnt,
			"",
			"", decay_add_cnt, "");

	return rtc;
}

int main(int argc, char const *argv[])
{
	static const char optaliases[] = IPQBDB_OPTION_FILE;
	static char max_max_time[32];
	int rtc = 0, errs = 0, prob[2] = {0, 0};
	time_t secs[2] = {0, time(&time_invoked)};
	ip4_range ip;

	sprintf(max_max_time, "%ldS", (long)secs[1]);
	max_time = max_max_time;
	poptContext opt = poptGetContext(err_prefix, argc, argv, opttab, 0);

	if (access(optaliases, F_OK) == 0 &&
		(rtc = poptReadConfigFile(opt, optaliases)) < 0)
	{
		fprintf(stderr, "%s: cannot read %s: %s\n",
			err_prefix, optaliases, poptStrerror(rtc));
		errs = 3;
	}

	rtc = poptGetNextOpt(opt);
	if (rtc != -1)
	{
		fprintf(stderr, "%s: %s\n",
			err_prefix, poptStrerror(rtc));
		errs = 1;
	}
	else
	{
		if (poptPeekArg(opt) != NULL)
		{
			fprintf(stderr, "%s: unexpected argument: %s\n",
				err_prefix, poptGetArg(opt));
			errs = 1;
		}

		if (version_opt)
		{
			fprintf(stdout, "%s: version " PACKAGE_VERSION "\n", err_prefix);
			errs = 2;
		}

		if (help_opt)
		{
			poptPrintHelp(opt, stdout, 0);
			fputs("\n"
"[*] Note that the min and max probabilities are compared accounting for\n"
"rehabilitation, not the bare values stored into the database. Thus the values\n"
"displayed in raw listing may differ from the values given for these options.\n",
				stdout);
			fputs_database_help();
			errs = 2;
		}

		else
		{
			if (raw_opt)
				list_opt = 1;

			if (!list_opt && !del_opt && !ban_opt && !stat_opt && !simple_opt && errs == 0)
			{
				fprintf(stdout,
					"%s: at least one of"
					" --ls, --ls-raw, --ls-ban, ls-simple, --stats, or --del"
					" must be specified\n",
					err_prefix);
				errs = 1;
			}

			if (only_blocked && non_blocked)
			{
				fprintf(stdout,
					"%s: only one of --only-blocked and --only-never-blocked "
					"may make sense\n",
					err_prefix);
				errs = 1;
			}
		}

		if (get_ip_address(&ip) ||
			get_seconds(&secs[0], min_time) ||
			get_seconds(&secs[1], max_time) ||
			get_max_records() ||
			get_probability(&prob[0], min_probability) ||
			get_probability(&prob[1], max_probability))
				errs = 3;

		if (secs[0] >= secs[1] && errs != 3)
		{
			report_error(&ap, LOG_WARNING, "no record can be"
				" at least as old as min-time=%s and at most as max-time=%s\n",
				min_time, max_time);
			errs = 1;
		}

		// popt sets verbose to 0 if no arg is given
		verbose += 1;

		ap.mode = error_report_stderr; // 0
		ap.err_prefix = err_prefix;
		if (syslog_opt)
		{
			openlog(err_prefix, LOG_PID, LOG_DAEMON);
			ap.mode = LOG_DAEMON;
		}
	}

	if (errs == 1)
		poptPrintUsage(opt, stdout, 0);
	poptFreeContext(opt);
	rtc = 0;

	if (errs)
		rtc = 1;
	else
	{
		char *block_fname = database_fname(db_block_name, &ap);
		char *descr_fname = database_fname(db_descr_name, &ap);

		if (block_fname == NULL || descr_fname == NULL)
			rtc = 1;

		if (rtc)
		{
			if (verbose)
				report_error(&ap, LOG_INFO, "Bailing out...\n");
		}
		else
		{
			DB_ENV *db_env = NULL;
			DB *db_block = NULL, *db_de1 = NULL, *db_de2 = NULL;

			setsigs();

			rtc = open_database(block_fname, &ap, &db_env, &db_block);

			if (rtc == 0)
				rtc = open_descrdb(descr_fname, db_env, &db_de1, &db_de2);

			if (rtc == 0 && caught_signal == 0)
				rtc = all_reason_id(db_de1, db_de2);

			if (rtc == 0 && caught_signal == 0 && max_records > 0)
			{
				histog his;
				if (init_histog(&his, secs[0]) == 0)
				{
					rtc = do_del(db_block, db_de1, &ip, secs, prob, &his);
					if (rtc == 0)
					{
						if (his.total_recs <= max_records)
						{
							if (verbose)
								printf(
									"total records = %d <= %ld: no deletion needed!\n",
									his.total_recs, max_records);
							del_opt = 0;
						}
						else if (his.total_recs - his.total_selected < max_records)
						{
							char const* err =
								increase_secs(&his, max_records, secs[0]);
							if (err)
								report_error(&ap, LOG_ERR,
									"increase_secs: %s\n", err);
							else
							{
								if (verbose)
								{
									printf("total records = %d, selected = %d,",
										his.total_recs, his.total_selected);
									if (his.increase)
									{
										char before[TIME_STRING_BUFSIZE],
											after[TIME_STRING_BUFSIZE];
										time_string(his.now - secs[0], his.now, before);
										time_string(his.now - secs[0] - his.increase,
											his.now, after);
										if (strcmp(before, after))
											printf(" min age bumped by %jd: %s to %s!\n",
												(intmax_t)his.increase, before, after);
										else
											printf(
												" min age increased by %jd before %s!\n",
													(intmax_t)his.increase, before);
									}
									else
										printf(" min age did not vary!\n");
								}
								secs[0] += his.increase; // may get >= secs[1]
							}
						}
						else
						{
							if (verbose)
								printf("total records = %d, selected = %d,"
									" expected = %d >= %ld: no adjustment possible!\n",
									his.total_recs, his.total_selected,
									his.total_recs - his.total_selected,
										max_records);
						}
					}
					free(his.his);
				}
				else rtc = 0; // ignore errors during adaptive count
			}

			if (rtc == 0 && caught_signal == 0 &&
				(list_opt || raw_opt || ban_opt || del_opt || simple_opt))
			{
				if (verbose)
				{
					printf("%s%s%s records in IP range %d.%d.%d.%d-%d.%d.%d.%d,\n"
						"\t%s%smin age %jd secs, max age %jd secs,"
						" min prob %d=%.2f%%, max prob %d=%.2f%%.\n",
						del_opt? "delete": "",
						del_opt && (list_opt || raw_opt || ban_opt)? " and ": "",
						(list_opt || raw_opt || ban_opt || simple_opt)? "list": "",
						((unsigned char*)&ip.first)[0],
						((unsigned char*)&ip.first)[1],
						((unsigned char*)&ip.first)[2],
						((unsigned char*)&ip.first)[3],
						((unsigned char*)&ip.last)[0],
						((unsigned char*)&ip.last)[1],
						((unsigned char*)&ip.last)[2],
						((unsigned char*)&ip.last)[3],
						only_blocked? "only blocked, ": "",
						non_blocked? "non blocked, ": "",
						(intmax_t)secs[0], (intmax_t)secs[1],
						prob[0], percent_prob(prob[0]),
						prob[1], percent_prob(prob[1]));
				}
				rtc = do_del(db_block, db_de1, &ip, secs, prob, NULL);
			}

			if (rtc == 0 && caught_signal == 0 && stat_opt)
			{
				rtc = do_stat(db_de1);
			}

			if (rtc)
				rtc = 2;

			close_db(db_de2);
			close_db(db_de1);
			close_db(db_block);

			close_dbenv(db_env, cleanup_opt);
		}
	}

	return rtc;
}
