/*
* ipq berkeley db daemon emitting verdicts - ibd-judge
* written by ale in milano on 10 sep 2008

Copyright (C) 2008-2021 Alessandro Vesely

This file is part of Ipqbdb.

Ipqbdb is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Ipqbdb is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Ipqbdb.  If not, see <http://www.gnu.org/licenses/>.

*/
#define _GNU_SOURCE 1
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <syslog.h>
#include <math.h>
#include <errno.h>
#include <ctype.h>
#include <limits.h>

#include <sys/types.h>

// misc. ip stuff
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

// netfilter for iptables v1.3.6
#include <linux/netfilter.h>
#include <linux/netfilter_ipv4.h>
#include <netinet/ip.h>

// this works with ip_queue.ko (CONFIG_IP_NF_QUEUE)
//#include <libipq/libipq.h> // from package iptables-dev

/*
how it works.... (attempt)
in net/netfilter/nf_queue.c, fun nf_queue()
  the packet is sent to queue_handler[pf]->outfn
  (todo: make sure this is what gets called for the NFQUEUE target)

  the [nf_]queue_handler (2nd parm) is set with fun nf_register_queue_handler
  it returns EEXIST, EBUSY, 0 resp. if the same, another, NULL handler is
  registered at that number (pf=1st parm). unregister sets to NULL.
  When NULL, packets are dropped. Pf is the protocol family (eg pf-inet?).

in net/netfilter/nfnetlink_queue.c the queue_handler is the same for all.
  it is a static struct, whose outfn is nfqnl_enqueue_packet. Since the
  queuenum comes from the originator, the nfqnl_instance is looked up here.

  The comment in unregister says:
  "This is a bug and a feature.  We can unregister other handlers(!)"

Using the test procedure in nfqnl_test.c (libnetfilter_queue) the
xt_NFQUEUE module gets automatically loaded on adding an iptables
rule, e.g., iptables -I interna_in 2 -p tcp --destination-port 13 -j NFQUEUE
on deleting the rule, the module remains loaded but its usage count
drops to 0 so that it can be removed with modprobe. The userspace program
does not receive a stop of some sort.
*/

// this works with .ko
// from package libnetfilter-queue-dev
#include <libnetfilter_queue/libnetfilter_queue.h>

// Berkeley DB v4.4
#include <db.h>

#include <popt.h>

// format of data packet
#include "config_names.h"
#include "dbstruct.h"

#include "percent_prob.h"
#include "rehabilitated_prob.h"

#include <assert.h>

static app_private ap;
static char const err_prefix[] = "ibd-judge";
#include "setsig_func.h"

// visible for TESTjudge
int verbose = -1;

// options for netfilter libraries
static int netlink_bufsize, netlink_no_enobufs;

// #if !defined NDEBUG
#define TRACE(v, x...) \
	while (ap.mode == 0 && verbose >= (v)) \
		{ fprintf(stderr, ## x); break; }
// #else
// #define TRACE(x...)
// #endif


struct ipaddr_cnt;
typedef struct queue_descr
{
	struct ipaddr_cnt *cnt;
	struct nfq_q_handle *qh;
	uint32_t deadbeef;

	// values set by parse_queue_descr
	uint32_t mark_value;  // host order
	unsigned int queue_maxlen;
	uint16_t queue;
	unsigned char dest;
	unsigned char mark;
	unsigned char nu[14];
} queue_descr;

typedef struct pkt_data
{
	queue_descr *qu;
	time_t now;
	uint32_t ipaddr;
	uint32_t id;
	uint32_t old_mark;
} pkt_data;

// seems a callback may be called multiple times for a given packet...(?)
typedef struct ipaddr_cnt
{
	pkt_data pkt[64];
	uint32_t cnt;
} ipaddr_cnt;


static int parse_queue_descr(char const *descr, queue_descr* out)
{
	queue_descr q;
	int ch, rtc = 0, source = 0;
	
	memset(&q, 0, sizeof q);

	while (rtc == 0 && isalpha(ch = *(unsigned char const*)descr++))
	{
		char *t = NULL;
		unsigned long l;
		switch (tolower(ch))
		{
			case 'd': // dest
				if (source)
					rtc = -1;
				q.dest = 1;
				break;
			case 'l': //length
				l = strtoul(descr, &t, 0);
				if (l <= UINT_MAX)
					q.queue_maxlen = (unsigned int)l;
				else
					rtc = -1;
				descr = t;
				break;
			case 'm': // mark
				l = strtoul(descr, &t, 0);
				if (l <= UINT32_MAX)
				{
					q.mark_value = (uint32_t)l; // was htonl in 0.4
					q.mark = 1;
				}
				else
					rtc = -1;
				descr = t;
				break;
			case 'q': // queue
				l = strtoul(descr, &t, 0);
				if (l <= UINT16_MAX)
					q.queue = (uint16_t)l;
				else
					rtc = -1;
				descr = t;
				break;
			case 's': // source
				if (q.dest)
					rtc = -1;
				source = 1;
				break;
			default:
				rtc = -1;
				break;
		}
	}
	if (*descr)
		rtc = -1;

	if (out)
		*out = q;

	return rtc;
}

static int
packet_cb(struct nfq_q_handle *qh, struct nfgenmsg *nfmsg,
	      struct nfq_data *nfa, void *data)
{
	(void)qh; (void)nfmsg;
	queue_descr *qu = (queue_descr*)data;
	ipaddr_cnt *cnt /* compiler happy */ = NULL;
	
	if (qu == NULL || qu->deadbeef != 0xdeadbeef ||
		(cnt = qu->cnt) == NULL ||
		cnt->cnt >= sizeof cnt->pkt / sizeof cnt->pkt[0])
	{
		report_error(&ap, LOG_CRIT, "cb fucked up (%p->%x/%u)\n",
			qu, qu? qu->deadbeef: 0, cnt? cnt->cnt: 0);
		return -1;
	}

	unsigned char *payload = NULL;
	int plen = nfq_get_payload(nfa, &payload);
	struct nfqnl_msg_packet_hdr *ph = nfq_get_msg_packet_hdr(nfa);
	TRACE(3, "RECV in queue %d, %s %d, len %d, cnt %u\n",
		qu->queue, ph? "hook": "_!*!BAD!*!_", ph? ph->hook: -1, plen, cnt->cnt);

	if (ph)
	{
		int const addr_off = qu->dest? 16: 12;

		if (plen >= addr_off + 4 /*membersize(pkt_data, ipaddr) */)
		{
			pkt_data *const pkt = &cnt->pkt[cnt->cnt++];

			// queue description
			pkt->qu = qu;

			// time if given, or 0
			struct timeval tv;
			if (nfq_get_timestamp(nfa, &tv))
				pkt->now = 0;
			else
				pkt->now = tv.tv_sec + (tv.tv_usec > 500000);

			// IP address, in network order
			memcpy(&pkt->ipaddr, payload + addr_off, sizeof pkt->ipaddr);

			// id for verdict
			pkt->id = ntohl(ph->packet_id);

			// existing mark
			pkt->old_mark = nfq_get_nfmark(nfa);
			TRACE(4, "pkt set: cnt=%d, id=%d, IP=%s, time=%ld\n", cnt->cnt,
				pkt->id, inet_ntoa(*(struct in_addr*)&pkt->ipaddr), pkt->now);
		}
	}

	return 0;
}

static queue_descr default_queue[1], *queue_data = default_queue;
static size_t queue_data_cnt = 1;

static int get_buffer_size(int fd)
{
	socklen_t bufsize = 0, socklen = sizeof bufsize;
	if (getsockopt(fd, SOL_SOCKET, SO_RCVBUF, &bufsize, &socklen))
	{
		report_error(&ap, LOG_CRIT, "getsockopt fail for fd %d (%s)\n",
			fd, strerror(errno));
		return -1;
	}
	return bufsize;
}

/*
Possible verdicts and guessed behavior:

#define NF_DROP 0
the packet is discarded

#define NF_ACCEPT 1
the packet passes, continue iterations

#define NF_STOLEN 2
gone away,

#define NF_QUEUE 3
inject into a different queue
(the queue number is in the high 16 bits of verdict)

#define NF_REPEAT 4
iterate the same cycle once more

#define NF_STOP 5
accept, but don't continue iteration

*/
static int
daemon_loop(struct nfq_handle *h, DB *db)
{
	DBT key, data;
	ip_data_t ip_data;
	uint32_t ip_addr;

	memset(&ip_data, 0, sizeof ip_data);
	memset(&key, 0, sizeof key);
	memset(&data, 0, sizeof data);
	key.flags = data.flags = DB_DBT_USERMEM;
	key.ulen = key.size = sizeof ip_addr;
	key.data = &ip_addr;
	data.ulen = data.size = sizeof ip_data;
	data.data = &ip_data;

	ipaddr_cnt cb_cnt;
	memset(&cb_cnt, 0, sizeof cb_cnt);

	int const fd = nfq_fd(h);
	int rv = 0; // compiler happy
	char buf[8192];
	
	if (fd < 0)
	{
		report_error(&ap, LOG_CRIT, "invalid fd %d (%s)\n",
					fd, strerror(errno));
		return 1;
	}

	if (verbose)
	{
		int bufsize = get_buffer_size(fd);
		if (bufsize >= 0)
			report_error(&ap, LOG_INFO, "netlink socket recv size %0x\n",
				bufsize);
	}

	if (netlink_bufsize)
	{
		socklen_t bufsize = netlink_bufsize;
		if (setsockopt(fd, SOL_SOCKET, SO_RCVBUF, &bufsize, sizeof bufsize))
			report_error(&ap, LOG_CRIT,
				"setsockopt fail for fd %d size = %0x (%s)\n",
					fd, netlink_bufsize, strerror(errno));
		else if (verbose)
		{
			int newsize = get_buffer_size(fd);
			if (newsize >= 0)
				report_error(&ap, LOG_INFO,
					"netlink socket recv size set to %0x, now %0x\n",
					netlink_bufsize, newsize);
		}

		/* was:
		unsigned int const read_size =
			nfnl_rcvbufsiz(nfq_nfnlh(h), netlink_bufsize);
		report_error(&ap, LOG_INFO, "netlink socket recv size size set to %u\n",
			read_size);
		*
		* by replacing the above code, can compile without -lnfnetlink
		* (which is set by pkg-config anyway)
		*/
	}

	if (netlink_no_enobufs)
	{
		int on = 1;
		if (setsockopt(fd, SOL_NETLINK, NETLINK_NO_ENOBUFS, &on, sizeof on) != 0)
		{
			report_error(&ap, LOG_ERR, "cannot suppress ENOBUFS: %s\n",
				strerror(errno));
			// ignore the error
		}
	}

	for (size_t i = 0; i < queue_data_cnt; ++i)
	{
		queue_descr *const qu = &queue_data[i];
		struct nfq_q_handle *qh =
			nfq_create_queue(h, qu->queue, &packet_cb, qu);
		if (qh == NULL)
		{
			report_error(&ap, LOG_CRIT,
				"cannot nfq_create_queue #%d\n", qu->queue);
			return 1;
		}

		// NFQNL_COPY_META and NFQNL_COPY_NONE imply 0 payload size
		// we just need source and dest address of ipv4
		if (nfq_set_mode(qh, NFQNL_COPY_PACKET, 20) < 0)
		{
			report_error(&ap, LOG_CRIT, "can't set packet_copy mode\n");
			nfq_destroy_queue(qh);
			return 1;
		}

		if (qu->queue_maxlen &&
			nfq_set_queue_maxlen(qh, qu->queue_maxlen) < 0)
		{
			report_error(&ap, LOG_CRIT,
				"can't set queue_maxlen to %u for queue #%d\n",
				qu->queue_maxlen, qu->queue);
			nfq_destroy_queue(qh);
			return 1;
		}

		qu->cnt = &cb_cnt;
		qu->qh = qh;
		qu->deadbeef = 0xdeadbeef;
	}

	while (caught_signal == 0)
	{
		if ((rv = recv(fd, buf, sizeof(buf), 0)) < 0)
		{
			if (errno != ENOBUFS)
				break;
			
			report_error(&ap, LOG_ERR, "packet(s) lost\n");
			continue;
		}

		TRACE(4, "\nPACKET RECV size=%d\n", rv);
		nfq_handle_packet(h, buf, rv); // deliver to queue handlers

		TRACE(4, "%d slot(s) filled...\n", cb_cnt.cnt);
		while (cb_cnt.cnt > 0)
		{
			pkt_data *const pkt = &cb_cnt.pkt[--cb_cnt.cnt];
			unsigned int verdict = NF_ACCEPT;
			time_t now = pkt->now? pkt->now: time(NULL);
			ip_addr = pkt->ipaddr;

			assert(pkt->qu && pkt->qu->deadbeef == 0xdeadbeef);

			int rc = db->get(db, NULL, &key, &data, 0);
			if (rc == 0 &&
				data.size == sizeof ip_data &&
				ip_data.chk == IPQBDB_CHK_SIGNATURE)
			{
				int need_write = 0;
				if (ip_data.probability <= 0)
				{
					TRACE(2, "ACCEPT %s: probability=%d\n",
						inet_ntoa(*(struct in_addr*)&ip_addr),
						ip_data.probability);
				}
				else if ((ip_data.last_block + IPQBDB_RECENT_DROP_TIME >= now &&
					ip_data.rep_ruling < IPQBDB_REPETITION_MAX) ||
					ip_data.decay >= IPQBDB_NEVER_DECAY) // static block
				{
					verdict = NF_DROP;
					ip_data.last_block = now;
					ip_data.block_cnt += 1;
					ip_data.rep_ruling += 1;
					need_write = 1;
					TRACE(2, "BLOCK  %s %s\n",
						inet_ntoa(*(struct in_addr*)&ip_addr),
						ip_data.decay >= IPQBDB_NEVER_DECAY? "forever": "again");
				}
				else
				{
					// rehabilitate
					time_t delta = now - ip_data.last_update;
					if (IPQBDB_UPDATE_TICK_TIME < delta && ip_data.decay > 0.0)
					{
						unsigned int const prob =
							rehabilitated_prob(ip_data.probability,
								delta, ip_data.decay);
						if (prob < (unsigned) ip_data.probability)
						{
							ip_data.probability = prob;
							ip_data.last_update = now;
							need_write = 1;
						}
					}

					// rule
					int const toss = rand();
					if (toss < ip_data.probability)
					{
						verdict = NF_DROP;
						ip_data.last_block = now;
						ip_data.block_cnt += 1;
						ip_data.rep_ruling = 0;
						need_write = 1;
					}
					// else possibly allow rep_ruling to stay above max

					TRACE(2, "%s %s: probability=%.2f%%: toss=%.2f, delta=%ld, reason=%d\n",
						verdict == NF_DROP? "BLOCK ": "ACCEPT",
						inet_ntoa(*(struct in_addr*)&ip_addr),
						percent_prob(ip_data.probability),
						percent_prob(toss),
						(long)delta, ip_data.reason_id);
				}

				/*
				* update the record (re-insert it if it was deleted meanwhile)
				*/
				if (need_write &&
					(rc = db->put(db, NULL, &key, &data, 0)) != 0)
				{
					report_error(&ap, LOG_CRIT,
						"cannot write record for %#x: %s\n",
							ip_addr, db_strerror(rc));
				}
			}
			else if (rc != DB_NOTFOUND)
			{
				report_error(&ap, LOG_CRIT, "cannot read key %#x: %s\n",
					ip_addr, rc? db_strerror(rc): "bad data record");
			}
			else
			{
				TRACE(2, "ACCEPT %s: not in db\n",
					inet_ntoa(*(struct in_addr*)&ip_addr));
			}

			struct nfq_q_handle *const qh = pkt->qu->qh;
			if (verdict != NF_DROP || pkt->qu->mark == 0)
			{
				rc = nfq_set_verdict(qh, pkt->id, verdict, 0, NULL);
				TRACE(1, "%s %s\n",
					verdict == NF_DROP? "BLOCK ": "ACCEPT",
					inet_ntoa(*(struct in_addr*)&ip_addr));
			}
			else
			{
				rc = nfq_set_verdict2(qh, pkt->id, NF_ACCEPT,
					pkt->qu->mark_value, 0, NULL);
				TRACE(1, "MARK %#x (was %#x) %s\n",
					pkt->qu->mark_value,
					pkt->old_mark,
					inet_ntoa(*(struct in_addr*)&ip_addr));
			}
			if (rc < 0)
			{
				time_t later = time(NULL);
				report_error(&ap, LOG_CRIT,
					"cannot set verdict %s for %#x: nfq_err=%d (time diff=%ld)\n",
					verdict == NF_DROP? "BLOCK ": "ACCEPT",
					ip_addr, nfq_errno, (long)(later - now));
			}
		}
	}
	
	/*
	* rv < 0 on EINTR or other error; EINTR implies a signal
	*/
	if (caught_signal)
	{
		if (caught_signal == SIGHUP)
		{
			report_error(&ap, LOG_INFO, "reopening on SIGHUP\n");
			caught_signal = 0;
			rv = -1; // enter daemon_loop again
		}
		else
		{
			report_error(&ap, LOG_INFO, "exiting on signal %s\n",
				strsignal(caught_signal));
			rv = 0;  // program exit code
		}
	}
	else
	{
		if (errno == EBADF && nfq_fd(h) == -123) // test in-file closed
		{
			rv = 0;
		}
		else
		{
			report_error(&ap, LOG_CRIT, "break loop after recv returns %d (%s)\n",
				rv, strerror(errno));
			rv = -1; // enter daemon_loop again, hopefully the error is transient
		}
	}

	for (size_t i = 0; i < queue_data_cnt; ++i)
	{
		queue_descr *const qu = &queue_data[i];
		if (qu->deadbeef == 0xdeadbeef)
		{
			nfq_destroy_queue(qu->qh);
			qu->cnt = NULL;
			qu->qh = NULL;
			qu->deadbeef = 0;
		}
	}

	return rv;
}

static char *db_name = IPQBDB_DATABASE_NAME;
static int version_opt, help_opt, no_daemon_opt, no_cleanup_opt;
static struct poptOption opttab[] =
{
	{"db-block", 'b', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &db_name, 0,
	"The database where IPv4 addresses are looked up", "filename"},
	{"no-daemon", 'D', POPT_ARG_NONE, &no_daemon_opt, 0,
	"Stay foreground and use stderr", NULL},
	{"verbose", 'v', POPT_ARG_INT|POPT_ARGFLAG_OPTIONAL, &verbose, 0,
	"Be verbose", "level"},
	{"no-db-cleanup", '\0', POPT_ARG_NONE, &no_cleanup_opt, 0,
	"On exit don't cleanup environment (__db.00? files) if not still busy", NULL},
	{"netlink-bufsize", '\0', POPT_ARG_INT, &netlink_bufsize, 0,
	"Set nfnetlink buffer size", "bytes"},
	{"netlink-no-enobufs", '\0', POPT_ARG_NONE, &netlink_no_enobufs, 0,
	"Suppress error notifications for dropped packets", NULL},
	{"version", 'V', POPT_ARG_NONE, &version_opt, 0,
	"Print version number and exit", NULL},
	{"help", 'h', POPT_ARG_NONE, &help_opt, 0,
	"This help.", NULL},
	POPT_TABLEEND
};

int main(int argc, char const *argv[])
{
	static const char optaliases[] = IPQBDB_OPTION_FILE;
	int rtc = 0, errs = 0;

	poptContext opt = poptGetContext(err_prefix, argc, argv, opttab, 0);

	if (access(optaliases, F_OK) == 0 &&
		(rtc = poptReadConfigFile(opt, optaliases)) < 0)
	{
		fprintf(stderr, "%s: cannot read %s: %s\n",
			err_prefix, optaliases, poptStrerror(rtc));
		errs = 3;
	}

	rtc = poptGetNextOpt(opt);
	if (rtc != -1)
	{
		fprintf(stderr, "%s: %s\n",
			err_prefix, poptStrerror(rtc));
		errs = 1;
	}
	else
	{
		char const **q_argv = poptGetArgs(opt);
		size_t q_argc = 0;
		
		if (q_argv)
			while (q_argv[q_argc] != NULL)
			{
				if (parse_queue_descr(q_argv[q_argc], NULL))
				{
					fprintf(stderr, "%s: invalid queue description: %s\n",
						err_prefix, q_argv[q_argc]);
					errs = 1;
				}
				++q_argc;
			}

		if (version_opt)
		{
			fprintf(stdout, "%s: version " PACKAGE_VERSION "\n", err_prefix);
			errs = 2;
		}

		// popt sets verbose to 0 if no arg is given
		// otherwise it stays -1
		verbose += 1;

		if (help_opt)
		{
			// popt bug: cannot access opt->otherHelp
			printf("Usage: %s [OPTION...] [QUEUEARG...]\n", err_prefix);
			poptPrintHelp(opt, stdout, 0);
			fputs_database_help();
			fputs("\n"
"Queue arguments can be specified as Qn[Mn][S|D][Ln], with no blanks, where\n"
"QMSDL are (case insensitive) specifiers, and n are suitable decimal integers.\n"
"Only one of S, for source, or D, for destination address, must be specified;\n"
"S is assumed by default. When the selected address is found to deserve being\n"
"blocked, if M, for mark, is specified, the packet is marked and accepted, else\n"
"it is dropped. L specifies the configured max length of the queue, in packets.\n"
"\n"
"For example, \"Q2M16DL10240\" will set queue_maxlen to 10240 for queue #2, and\n"
"then check the destination addresses of those packets, marking guilty packets\n"
"with the value 16, which can be tested with -m mark --mark 16 on a subsequent\n"
"iptables table.\n"
"\n"
"Zero or more queue argumens can be specified. \"Q0S\" is assumed by default.\n",
				stdout);
			errs = 2;
		}

		if (errs == 0 && q_argc > 0)
		{
			queue_descr *queue =
				(queue_descr*)malloc(q_argc * sizeof(queue_descr));
			if (queue == NULL)
				errs = 3;
			else
			{
				for (size_t i = 0; i < q_argc; ++i)
					parse_queue_descr(q_argv[i], &queue[i]);

				// check duplicates
				for (size_t i = 0; i < q_argc; ++i)
					for (size_t j = 0; j < i; ++j)
						if (queue[j].queue == queue[i].queue)
						{
							fprintf(stderr,
								"%s: queues %s [%zd] and %s [%zd]"
								" have the same queue number %d\n",
								err_prefix, q_argv[j], j, q_argv[i], i, queue[i].queue);
							errs = 1;
						}
			}
			
			if (errs == 1)
			{
				free(queue);
				fputs("queue arg: Q<queue-num>[M<mark-num>][S|D]\n", stderr);
				errs = 3;
			}
			else
			{
				queue_data = queue;
				queue_data_cnt = q_argc;
			}
		}
	}

	if (errs == 1)
		poptPrintUsage(opt, stderr, 0);
	poptFreeContext(opt);
	rtc = 0;

	if (errs)
	{
		rtc = 1;
	}
	else
	{
		ap.mode = error_report_stderr; // 0
		ap.err_prefix = err_prefix;

		char *fname = database_fname(db_name, &ap);
		if (fname == NULL)
			rtc = 1;
		else
		{
			char const *what = NULL;
			if (no_daemon_opt == 0)
			{
				char *p = strrchr(fname, '/');
				if (what == NULL && daemon(p != NULL /* nochdir */, 0))
					what = "daemon";
				if (what == NULL && setsigs())
					what = "sigaction";
				if (p)
				{
					*p = 0;
					if (chdir(fname))
						what = "chdir";
					*p = '/';
				}
				openlog(err_prefix, LOG_PID, LOG_DAEMON);
				ap.mode = LOG_DAEMON;
			}
			else if (setsigs())
				what = "sigaction";

			if (what)
			{
				report_error(&ap, LOG_CRIT, "cannot %s: %s - exiting\n",
					what, strerror(errno));
				rtc = 1;
			}
		}

		while (rtc == 0)
		{
			DB_ENV *db_env = NULL;
			DB *db = NULL;
			TRACE(3, "Open database %s\n", fname);
			rtc = open_database(fname, &ap, &db_env, &db);

			if (rtc == 0)
			{
				// from nfqnl_test.c
				struct nfq_handle *h;

				rtc = 2;

				// printf("opening library handle\n");
				h = nfq_open();
				if (!h)
				{
					report_error(&ap, LOG_CRIT, "error during nfq_open()\n");
				}
				else
				{
					// printf("unbinding existing nf_queue handler for AF_INET (if any)\n");
					if (nfq_unbind_pf(h, AF_INET) < 0)
					{
						report_error(&ap, LOG_CRIT, "error during nfq_unbind_pf()\n");
					}

					// printf("binding nfnetlink_queue as nf_queue handler for AF_INET\n");
					else if (nfq_bind_pf(h, AF_INET) < 0)
					{
						report_error(&ap, LOG_CRIT, "error during nfq_bind_pf()\n");
					}

					else
					{
						rtc = daemon_loop(h, db);
						TRACE(2, "Daemon loop exited with rtc=%d\n", rtc);
					}


	#ifdef INSANE
					/* normally, applications SHOULD NOT issue this command, since
					 * it detaches other programs/sockets from AF_INET, too ! */
					// printf("unbinding from AF_INET\n");
					nfq_unbind_pf(h, AF_INET);
	#endif

					// printf("closing library handle\n");
					nfq_close(h);
				}


				close_db(db);
				close_dbenv(db_env, !no_cleanup_opt);
				TRACE(3, "Database closed\n");
				if (rtc >= 0)
					break;

				/*
				* when daemon_loop returns a negative number,
				* enter the loop again.
				*/
				TRACE(2, "Restarting loop in 2 secs...\n");
				rtc = 0;
				sleep(2);
			}
		}
	}

	TRACE(2, "Exit with status %d\n", rtc);
	return rtc;
}
