/*
* dbstruct.h


Copyright (C) 2008-2021 Alessandro Vesely

This file is part of Ipqbdb.

Ipqbdb is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Ipqbdb is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Ipqbdb.  If not, see <http://www.gnu.org/licenses/>.

*/

#ifndef DBSTRUCT_H_INCLUDED
#define DBSTRUCT_H_INCLUDED

#include <sys/time.h>
#include <time.h>
#include <string.h>
#include <stdlib.h>
#include <db.h>

#define IPQBDB_CHK_SIGNATURE 0X5a
#define IPBQDB_VERSION 1

/*
* This is the data part of the main database.
* For IPv4, the key part is the IPv4 address, uint32_t, in network order.
* For IPv6, the key part is the IPv6, 16 bytes, in network order, where
* the host part is filled with 1s; this allows to retrieve the range.
* Record layout is optimized assuming sizeof: time_t=8, int=4, double=8.
*/
typedef struct ip_data_t
{
	double decay;               // >= IPQBDB_NEVER_DECAY for always blocked
	time_t last_update, last_block, created;
	int probability;            // 0-RAND_MAX probability of being blocked
	int reason_id;              // index into the reason file
	int block_cnt;              // number of packets blocked
	int decay_add_cnt;          // number of threshold upward passes
// --- added since v.0
	int caught_cnt;             // number of times caught (from ban or parse)
	uint32_t not_used;          // reserved --must be 0
	uint16_t rep_ruling;        // repeated ruling, added in v. 1.05
	unsigned char flag;          // reserved --must be 0
	unsigned char plen;         // prefix len, only used for IPv6, added in 2.0
								// 0 means 128, i.e. a single IP
	unsigned char official_plen; // prefix len found in RDAP, added in 2.0
	unsigned char nu[2];   // reserved --must be 0
	unsigned char chk;     // CHK_SIGNATURE, for loose type checking
} ip_data_t;


/*
* This is the data part of the "bynumber" (primary) description table
* (the key part is the reason_id number)
* The last field of this structure is a variable length string containing
* the textual description, and is the secondary key "bydescr" of the table.
*/
typedef struct bynumber_descr_t
{
	uint64_t caught_cnt, rec_cnt,  // set on block record creation/update
		block_cnt, decay_add_cnt;  // set on block record deletion
	time_t created,                // this record creation
		last_added,                // set on block record creation/update
		last_removed;              // set on block record deletion
	unsigned char report_category; // external category added in v. 1.07
	unsigned char nu[6];           // reserved --must be 0
	unsigned char chk;             // CHK_SIGNATURE, for loose type checking
	char descr[/*IPQBDB_DESCR_MAX_LENGTH*/];
} bynumber_descr_t;

/*
* This is the data part of the whitelist
* (the key part is the IPv4 address)
*/
typedef struct ip_white_t
{
	double decay;         // 0 = never block this ip, otherwise set initial decay
	int probability;      // reserved --must be 0
	int reason_id;        // reserved --must be 0
	time_t expiration;    // record won't be valid thereafter
	unsigned char plen;   // prefix len, only used for IPv6, added in 2.0
	                      // 0 means 128, i.e. a single IP
	unsigned char nu[6];  // reserved --must be 0
	unsigned char chk;    // CHK_SIGNATURE, for loose type checking
} ip_white_t;

// used to be here
#include "report_error.h"

/*
* Structure to switch names by adding '6'
*/
typedef struct switchable_fname
{
	char *home;   // possibly NULL
	unsigned six; // write '6' at this offset, shift the rest
	char fname[];
} switchable_fname;

#include "ip_util.h" // for ip_u

void set_ipqbdb_index_factor(double f);
double get_ipqbdb_index_factor(void);
int get_ipqbdb_index_factor_precision(void);

void fputs_database_help(void);
switchable_fname *database_fname(char const *name, app_private*);
switchable_fname *database_fname_with_home(char const *name, app_private *ap,
	char **home, int add_slash_if);
int open_database(switchable_fname *name, app_private *ap, DB_ENV**, DB**, DB**);
int close_db(DB *db);
int close_dbenv(DB_ENV *db_env, int cleanup);


enum
{ get_descr_quoted = 1, get_descr_add_id = 2, get_descr_add_category = 16 };
int get_descr(DB *db, int id, int opt, char *dest, size_t size);
int last_descr_ndx(DB *db1);
int get_reason_id(int *reason_id, char *reason_string,
	app_private *ap, DB *db1, DB *db2, int verbose);
enum
{ set_desc_old_record = 0, set_descr_new_record = 1 };
int set_descr_cnt(DB *db1, int descr_id, int is_new_record,
	ip_data_t const *deldata, time_t const *tloc);
int open_descrdb(char *fname, DB_ENV *db_env, DB** db1, DB** db2);
int check_whitelist(void *, ip_u *ip_addr, double *decay);
int check_whitelist_e(void *, ip_u *ip_addr, double *decay, unsigned char e[16]);

typedef void (*write_block_cb)(ip_u *ip_addr, void*key, int reason_id,
	int write_block_force,
	ip_data_t *ip_data, ip_data_t *old_data, DB *db_d);
enum
{ write_block_force_decay = 1, write_block_force_probability = 2,
	write_block_force_reason = 4,
	write_block_force_add_category = 16 };
int write_block(DB *db, ip_u *ip_addr, double decay, int reason_id,
	int* /* I/O */ probability, int force, write_block_cb verbose_fn, DB *db_d);
	
#if defined SIMULATION_TEST
void set_simulation_clock(time_t clock);
#endif

unsigned make_six(switchable_fname *sf);
unsigned make_four(switchable_fname *sf);

void fputs_initial_count_help(void);

#endif // DBSTRUCT_H_INCLUDED
