#! /bin/bash
#
# Report blocked addresses
#
# This script reads daemon log and prints a list of blocked addresses.
#
# A one or two digit numeric arguments limits the report to addresses
# that were caught at least that many times.
# An argument of ".0" is taken to read /var/log/daemon.log.0 instead,
# otherwise a the full path of the log file is accepted.
#
# Several assumptions are made about path and format of the log files.


dotzero=""
minlin=2
command=cat
filename=/var/log/daemon.log

while test $# -gt 0; do
	case "$1" in
		.[0-9]) dotzero=$1;;
		[0-9]|[0-9][0-9]) minlin="$1";;
		*) filename=$1
	esac
	shift
done

if [ -r "$filename$dotzero" ]; then
	:
elif [ -r "$filename$dotzero.gz" ]; then
	filename="$filename$dotzero.gz"
	dotzero=""
	command=zcat
fi

# assume the log starts with 15 characters date (May 29 18:34:16)
$command $filename$dotzero |\
sed -rn '/ibd-(parse|ban)/s/(^.{15})[^:]*: *(old|new) record for ([^,]*), *(.*)/\3 \1 \2 \4/p' |\
sort --key 1 --stable |\
(
	read ip line
	while test -n "$ip"; do
		declare -a report
		key=$ip
		if [[ "$line" =~ " old " ]]; then
			first_is_old=1
		else
			first_is_old=0
		fi
		while true; do
			report+=("$line")
			read ip line
			test "$key" = "$ip" || break
		done
		if [ ${#report[@]} -gt $minlin ]; then
			curl -s https://freegeoip.app/csv/$key
			readarray -t <<< "$(abuserdap -s http://rdap.arin.net/bootstrap/ip/$key)"
			if test -z "${MAPFILE[0]}"; then
				printf '%s not controlled by an abuse team, see %s\n' \
					"$key" "${MAPFILE[1]}"
				# if banning for 1 month in end-of-day, don't use -l!
				# ibd-ban -i $key -c 0 -t 2592000 -r "IP without abuse team"
			fi
			printf "    %s\n" "${report[@]}"
			echo
			if [ $first_is_old -eq 1 ]; then
				printf "    %s\n\n" "$(ibd-del --ls-simple -i $key)"
			fi
		fi
		unset report
	done
)

