/* FAKED libnfqnetlink.c: generic library for FAKE access to nf_queue
 *
 * (C) 2005 by Harald Welte <laforge@gnumonks.org>
 * Copyright (C) 2010-2011 Alessandro Vesely
 *
This file is part of Ipqbdb.

Ipqbdb is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Ipqbdb is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Ipqbdb.  If not, see <http://www.gnu.org/licenses/>.


NOTICE:
=======

This only serves to link a test program that so that it can run
without root privileges and without disturbing the kernel.

HOW THIS WORKS:
===============

The main function runs in ibd-judge. It is assumed it calls 
mnl_socket_open as a first thing. Sanity checks are run there.  The 
function also retrieves an environment variable that has the name of an 
in-file with test details, and creates the mnl_socket structure filled 
with the data for the test.

The in-file may contain packets, commands, and comments. Commands are 
defined in cmd_strings.h; the "group" command groups packets. In-file 
is first read on opening mnl_socket, and continues until there are 
enough packets to be sent.

A call to mnl_socket_recvfrom is expected next. That function executes 
any command and then calls all registered callbacks. The noop command 
is used to yield control to ibd-judge. Other commands are useful to set 
test values, e.g. the database, in order for ibd-judge to take certain 
actions.

When the in-file terminates, the socketpair is closed, letting 
ibd-judge's call to recv terminate with EBADF.

*/

#if ! defined _GNU_SOURCE
#define _GNU_SOURCE
#endif
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>
#include <time.h>
#include <errno.h>
#include <limits.h>
#include <wait.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <signal.h>


// this works with .ko
// from package libnetfilter-queue-dev
#include <libmnl/libmnl.h>
#include <linux/netfilter.h>
#include <linux/netfilter/nfnetlink.h>

#include <linux/types.h>
#include <linux/netfilter/nfnetlink_queue.h>
#include <libnetfilter_queue/libnetfilter_queue.h>
#include <linux/netfilter/nfnetlink_conntrack.h>


#include "ip_util.h"

#include <assert.h>

#define TEST_QUEUE_MAX 8
#define TEST_CMD_MAX 100
#define TEST_ARGS_MAX TEST_CMD_MAX
#define MY_PREFIX "TESTJUDGE"
static const char my_prefix[] = MY_PREFIX;

struct mnl_socket;

struct nfq_data
{
	struct mnl_socket *nl;
	time_t timestamp;
	union ip
	{
		unsigned version_n_what;
		struct fields_v4
		{
			unsigned tos:8;
			unsigned len:16;
			unsigned id:16;
			unsigned flag:3;
			unsigned frag:13;
			unsigned ttl:8;
			unsigned proto:8;
			unsigned chk:16;
			unsigned source:32;
			unsigned dest:32;
		} v4;
		struct fields_v6
		{
			unsigned class:4;  // rest of it, not used
			unsigned label:20;
			unsigned len:16;
			unsigned next:8;
			unsigned ttl:8;
			unsigned char source[16];
			unsigned char dest[16];
		} v6;
		unsigned char payload[40]; // extra header and data would follow
	} packet;
	int packet_id;
	int current_q;
	struct nfgenmsg my_nfgenmsg; // for mnl_nlmsg_get_payload()

	struct nfqnl_msg_packet_hdr my_nfqnl_msg_packet_hdr; // NFQA_PACKET_HDR

	struct nfqnl_msg_packet_timestamp my_nfqnl_msg_packet_timestamp; //NFQA_TIMESTAMP

	unsigned char queue[TEST_QUEUE_MAX];
};

struct mnl_socket // fubar
{
	FILE *in, *out;
	char *cmd_arg[TEST_ARGS_MAX];
	time_t current;
	struct nfq_data next_pkt;
	void *nextbuf;
	size_t nextbuf_sz;
	int line;
	char sv_sent;
	char have_next_pkt;
	char verbose;
	unsigned char q[TEST_QUEUE_MAX];
	char fname[];
};

typedef struct test_cmd
{
	int cmd;
	int arg;
} test_cmd;

#define STRING_MACRO(x, y) test_cmd_##x,
enum test_cmd_enum
{
#include "cmd_strings.h"
test_cmd_invalid
};

#undef STRING_MACRO
#define STRING_MACRO(x, y) #x,
static const char *test_cmd_strings[] =
{
#include "cmd_strings.h"
};

#undef STRING_MACRO
#define STRING_MACRO(x, y) y,
static const char test_cmd_has_arg[] =
{
#include "cmd_strings.h"
};

#undef STRING_MACRO


extern int verbose; // in ibd-judge.c

static int parse_command(char **p)
{
	char buf[64];
	size_t len = 0, i;
	int ch;
	unsigned char const *s = *(unsigned char**)p;
	
	if (!isalpha(*s))
		return -1;

	while ((ch = *s) == '_' || isalnum(ch))
	{
		if (isupper(ch))
		{
			ch = _tolower(ch);
		}
		buf[len] = ch;
		if (++len >= sizeof buf)
			return test_cmd_invalid;
		s += 1;
	}
	
	buf[len] = 0;
	for (i = 0; i < sizeof test_cmd_strings/ sizeof test_cmd_strings[0]; ++i)
		if (strcmp(test_cmd_strings[i], buf) == 0)
		{
			*p += len;
			return i;
		}
	
	return test_cmd_invalid;
}

static int parse_addr(char **p, ip_u *addr)
{
	char *s = *p, *t = NULL;
	ip_range range;
	int rtc = parse_ip_address(s, &range, &t);
	if (rtc || range.args != 1 || t == NULL ||
		(*t && !isspace(*(unsigned char*)t)))
			return -1;

	if (range.ipv4_mapped) // is an IPv6 packet
	{
		memmove(&range.u.ipv6[12], &range.u.ipv4[0], 4);
		range.u.ipv6[10] = range.u.ipv6[11] = 0xff;
		memset(&range.u.ipv6[0], 0, 10);
		range.ip = 6;
	}

	memcpy(addr, &range, sizeof *addr);
	*p = t;
	return 0;
}

static char const* parse_pkt(char *s, struct nfq_data *pkt)
/*
* read packet format:
* id [queue...] from-addr to-addr [time increase]
* return NULL if ok, an the failed field name if not
*/
{
	assert(isdigit(*(unsigned char*)s));

	int id;
	unsigned char queue[TEST_QUEUE_MAX];
	ip_u from, to;
	time_t timestamp = 0;

	char *t = NULL;
	unsigned int l = strtoul(s, &t, 0);
	int i;

	if (l >= INT_MAX || t == NULL || !isspace(*(unsigned char*)t))
		return "id";

	id = (int)l;
	memset(queue, 0, sizeof queue);
	
	for (i = 0; i < TEST_QUEUE_MAX; ++i)
	{
		s = t + 1;
		while (isspace(*(unsigned char*)s))
			s += 1;
		
		l = strtoul(s, &t, 10);
		if (*t == '.' || *t == ':')  // IP address
			break;

		if (l >= TEST_QUEUE_MAX)
			return "queue number";

		queue[l] = 1;
	}

	if (parse_addr(&s, &from))
		return "source IP";
	
	while (isspace(*(unsigned char*)s))
		s += 1;

	if (parse_addr(&s, &to))
		return "dest IP";

	if (to.ip != from.ip)
		return "mixed IPv4 and IPv6";

	while (isspace(*(unsigned char*)s))
		s += 1;

	if (*s)
	{
		timestamp = strtoul(s, &t, 0);
		if (*t)
			return "time";
	}

	memset(pkt, 0, sizeof *pkt);
	pkt->timestamp = timestamp;
	pkt->packet.version_n_what = (from.ip << 4) + 5;
	if (from.ip == 4)
	{
		pkt->packet.v4.id = (uint16_t)id;
		pkt->packet.v4.ttl = 8;
		pkt->packet.v4.proto = 6;
		pkt->packet.v4.source = from.u.ipv4l;
		pkt->packet.v4.dest = to.u.ipv4l;
		// pkt->hdr.packet_id = htonl(id);
		// pkt->hdr.hw_protocol
		// pkt->hdr.hook
	}
	else
	{
		pkt->packet.v6.ttl = 8;
		memcpy(pkt->packet.v6.source, from.u.ipv6, 16);
		memcpy(pkt->packet.v6.dest, to.u.ipv6, 16);
		// pkt->packet.v6.class
		// pkt->packet.v6.label
		// pkt->packet.v6.len
		// pkt->packet.v6.next
	}
	pkt->packet_id = id;
	memcpy(pkt->queue, queue, sizeof pkt->queue);
	
	return 0;
}

static char *simple_expand(char *arg)
// assume arg is malloc'ed, formatted as "XYZ=ABCD$EFG,etc."
// replace $EFG and return the same or a different string
// simple means no ${funny} stuff :-/
// return NULL on memory failure
{
	size_t len = strlen(arg);
	char *start = strchr(arg, '='), *repl;
	while (arg != NULL && start != NULL &&
		(repl = strchr(start, '$')) != NULL)
	{
		char *var = repl + 1, *v = var;
		int ch;
		while ((isalnum(ch = *(unsigned char*)v) || ch == '_'))
			++v;

		*v = 0;
		char *val = getenv(var);
		size_t const ini = repl - arg,
			vlen = val? strlen(val): 0,
			nlen = v - var,
			rest = len - nlen - ini; // incl. term 0
		*v = ch;
		if (vlen <= nlen)
		{
			if (val)
				memcpy(repl, val, vlen);
			memmove(start = repl + vlen, v, rest);
		}
		else
		{
			char *newarg = (char*)malloc(len + vlen - nlen);
			if (newarg)
			{
				memcpy(newarg, arg, ini);
				start = newarg + ini;
				if (val)
				{
					memcpy(start, val, vlen);
					start += vlen;
				}
				memcpy(start, v, rest);
			}
			free(arg);
			arg = newarg;
		}
		len += vlen - nlen - 1;
		assert(arg == NULL || strlen(arg) == len);
	}
	return arg;
}

static int get_free_arg(struct mnl_socket *h)
{
	unsigned i;
	for (i = 0; i < sizeof h->cmd_arg/sizeof h->cmd_arg[0]; ++i)
		if (h->cmd_arg[i] == NULL)
			return i;

	fprintf(stderr,
			MY_PREFIX ": max arguments exceeded at line %d in %s\n",
				h->line, h->fname);
	return -1;
}

static int read_test_input(struct mnl_socket *h)
/*
* read in-file and send 1 buffer of commands until one pkt is ready,
* or return -1 for failure.
*/
{
	assert(h && h->have_next_pkt == 0);
	if (h->in == NULL)
		return -1;

	char buf[1024], *s;
	char cmdbuf[TEST_CMD_MAX * sizeof(test_cmd)];
	size_t in_cmdbuf = 0, have_group = 0;
	
	while ((s = fgets(buf, sizeof buf, h->in)) != NULL)
	{
		h->line += 1;
		
		size_t len = strlen(buf);
		if (len + 1 >= sizeof buf)
		{
			fprintf(stderr,
				MY_PREFIX ": line %d too long in %s\n",
				h->line, h->fname);
			return -1;
		}
				
		while (isspace(*(unsigned char*)s))
			++s;
		
		// discard empty lines and comments
		if (*s == '#' || *s == 0)
			continue;
		
		while (len > 0 && isspace(*(unsigned char*)&buf[len-1]))
			len -= 1;
		buf[len] = 0;

		if (isdigit(*(unsigned char*)s)) // packet data
		{
			char const *field = parse_pkt(s, &h->next_pkt);
			if (field == NULL)
			{
				h->have_next_pkt = 1;
				break;
			}

			fprintf(stderr,
				MY_PREFIX ": invalid %s at line %d in %s\n",
				field, h->line, h->fname);
		}
		else
		// parse command and possibly add it to cmdbuf
		// ignore lines with errors when possible
		{
			char *cmd = s;
			int rtc = parse_command(&s);
			if (rtc < 0 || rtc == test_cmd_invalid)
			{
				fprintf(stderr,
					MY_PREFIX ": invalid %s at line %d in %s\n",
					rtc < 0? "data": "command", h->line, h->fname);
			}
			else
			{
				int err = 0, needarg = test_cmd_has_arg[rtc];
				char *unex = NULL;
				test_cmd tc;
				memset(&tc, 0, sizeof tc);
				tc.cmd = rtc;
				*s++ = 0;
				
				while (isspace(*(unsigned char*)s))
					++s;
				
				switch (needarg)
				{
					case 0: // no args
						if (*s)
							unex = s;
						break;
					
					case 1: // integer arg
						if (isdigit(*(unsigned char*)s))
						{
							char *t = NULL;
							long l = strtol(s, &t, 0);
							if (t && *t)
								unex = t;
							if (l > INT_MIN && l < INT_MAX)
								tc.arg = (int)l;
							else
							{
								err = 1;
								fprintf(stderr,
									MY_PREFIX
									": invalid argument %s at line %d in %s\n",
									s, h->line, h->fname);
							}
							
							needarg = 0;
						}
						break;

					case 2: // string arg
						if (*s)
						{
							int const ndx = get_free_arg(h);
							if (ndx < 0)
								err = 1;
							else if ((h->cmd_arg[ndx] = strdup(s)) == NULL)
							{
								fputs("OUT OF MEMORY\n", stderr);
								return -1;
							}
							else
								tc.arg = ndx;
							
							needarg = 0;
						}
						break;

					default:
						assert(0); // unsupported number in cmd_strings.h
						return -1;
						break;
				}
				
				if (needarg)
				{
					err = 1;
					fprintf(stderr,
						MY_PREFIX
						": command %s requires a%s argument"
						" at line %d in %s\n",
						needarg == 1? " numeric": "n",
						cmd, h->line, h->fname);
				}
				
				if (unex)
				{
					err = 1;
					fprintf(stderr,
						MY_PREFIX
						": unexpected character(s) \"%s\""
						" at line %d in %s\n",
						unex, h->line, h->fname);
				}
				
				if (err == 0)
				{
					// always keep room for an extra test_cmd in cmdbuf
					memcpy(&cmdbuf[in_cmdbuf], &tc, sizeof tc);
					in_cmdbuf += sizeof tc;

					if (in_cmdbuf + sizeof tc > sizeof cmdbuf)
					{
						fprintf(stderr,
							MY_PREFIX
							": too many commands without packet data"
							" at line %d in %s\n",
							h->line, h->fname);
						break;
					}
					
					if (rtc == test_cmd_group)
						have_group = 1;
				}
			}	
		}
	}

	if (feof(h->in))
	{
		fclose(h->in);
		h->in = NULL;
	}

	if (!h->have_next_pkt)
	{
		return -1;
	}

	// no group command: default to a group of 1 pkt
	if (!have_group)
	{
		test_cmd tc;
		memset(&tc, 0, sizeof tc);
		tc.cmd = test_cmd_group;
		tc.arg = 1;
		memcpy(&cmdbuf[in_cmdbuf], &tc, sizeof tc);
		in_cmdbuf += sizeof tc;
	}

	h->nextbuf = malloc(h->nextbuf_sz = in_cmdbuf);
	if (h->nextbuf == NULL)
	{
		fprintf(stderr,
			MY_PREFIX ": out of memory (%zu)\n", in_cmdbuf);
		return -1;
	}

	memcpy(h->nextbuf, cmdbuf, in_cmdbuf);
	h->sv_sent = 1;
	return 0;
}

static void suicide(void)
{
	kill(getpid(), SIGTERM);
}

#if defined __GNUC__
__attribute__ ((format(printf, 2, 3)))
#endif
static int print_out(struct mnl_socket *h, char const *fmt, ...)
{
	int rtc = 0;
	if (h->verbose && h->out)
	{
		va_list al;
		va_start(al, fmt);
		rtc = vfprintf(h->out, fmt, al);
		va_end(al);
	}
	
	return rtc;
}

/* use instead of nlmsghdr for configuration */
struct my_params
{
	uint32_t nlmsg_len; // must be first, for cast to nlmsg in nfq_nlmsg_put()
	uint32_t queue_num;
	struct mnl_socket* nl;
	int range;
	int verdict;
	int id;
	uint32_t queue_maxlen;
	uint32_t mark;
	uint8_t cmd;
};

enum my_cmd
{
	my_NFQNL_CFG_CMD_BIND,
	my_NFQNL_CFG_CMD_UNBIND,
	my_NFQNL_COPY_PACKET,
	my_NFQNL_MSG_VERDICT,
	my_NFQNL_MSG_CONFIG_MAXLEN,
	my_NFQNL_IGNORE_COMMAND
};



/* public interface */

struct mnl_socket *mnl_socket_open(int bus)
// was struct nfq_handle *nfq_open_nfnl(struct nfnl_handle *nfnlh)
{
	if (bus != NETLINK_NETFILTER)
	{
		fprintf(stderr,
			MY_PREFIX ": unexpected value %d, should be %d\n",
			bus, NETLINK_NETFILTER);
		return NULL;
	}

	char *fname = getenv(my_prefix);
	size_t fname_size = fname? strlen(fname) + 1: 0;

	// don't ruin installed stuff: check permissions and database path
	if (getuid() == 0 || geteuid() == 0 || access("/", W_OK) == 0)
	{
		fputs(MY_PREFIX ": tests MUST NOT be run with root privileges\n",
			stderr);
		return NULL;
	}
	
	if (fname == NULL)
	{
		fputs(MY_PREFIX
			": Missing environment variable \"" MY_PREFIX
			"\" pointing to input file\n",
			stderr);
		return NULL;
	}

	struct mnl_socket *h;
	h = malloc(sizeof *h + fname_size);
	if (!h)
		return NULL;

	memset(h, 0, sizeof *h);
	strcpy(h->fname, fname);
	if ((h->in = fopen(fname, "r")) == NULL)
	{
		fprintf(stderr,
			MY_PREFIX ": cannot open %s for reading: %s\n",
			fname, strerror(errno));
		goto error_exit;
	}

	if (!read_test_input(h) == 0)
		goto error_exit;

	h->out = stdout;
	h->verbose = verbose != 1;
	time(&h->current);
	return h;

	error_exit:
	{
		if (h->in) fclose(h->in);
		free(h);
		suicide();
	}

	return NULL;
}

int mnl_socket_close(struct mnl_socket *h)
{
	if (h->out && h->out != stdout) fclose(h->out);
	if (h->in) fclose(h->in);
	free(h);
	return 0;
}

/* bind nf_queue from a specific protocol family */
int mnl_socket_bind(struct mnl_socket *nl, unsigned int groups,
				  pid_t pid){
	(void)nl;
	(void)groups;
	(void)pid;
	return 0;
}

struct nlmsghdr *nfq_nlmsg_put(char *buf, int type, uint32_t queue_num)
{
	struct my_params par;
	par.nlmsg_len = sizeof par;
	par.queue_num = queue_num;
	memcpy(buf, &par, sizeof par);

	return (struct nlmsghdr *)buf;
	(void)type;
}

void nfq_nlmsg_cfg_put_cmd(struct nlmsghdr *nlh, uint16_t pf, uint8_t cmd)
{
	struct my_params *par = (struct my_params*)nlh;
	switch (cmd)
	{
		case NFQNL_CFG_CMD_BIND:
			par->cmd = my_NFQNL_CFG_CMD_BIND;
			break;
		case NFQNL_CFG_CMD_UNBIND:
			par->cmd = my_NFQNL_CFG_CMD_UNBIND;
			break;
	}
	(void)pf;
}

void nfq_nlmsg_cfg_put_params(struct nlmsghdr *nlh, uint8_t mode, int range)
{
	struct my_params *par = (struct my_params*)nlh;
	par->range = range;
	par->cmd = my_NFQNL_COPY_PACKET;
	(void)mode;
}

void nfq_nlmsg_cfg_put_qmaxlen(struct nlmsghdr *nlh, uint32_t queue_maxlen)
{
	struct my_params *par = (struct my_params*)nlh;
	par->cmd = my_NFQNL_MSG_CONFIG_MAXLEN;
	par->queue_maxlen = queue_maxlen;
}

void mnl_attr_put_u32(struct nlmsghdr *nlh, uint16_t type,
				    uint32_t data)
{
	struct my_params *par = (struct my_params*)nlh;

	if (type == CTA_MARK)
		par->mark = data;

	par->cmd = my_NFQNL_IGNORE_COMMAND;
}

void nfq_nlmsg_verdict_put(struct nlmsghdr *nlh, int id, int verdict)
{
	struct my_params *par = (struct my_params*)nlh;
	par->id = id;
	par->cmd = my_NFQNL_MSG_VERDICT;
	par->mark = 0;
	par->verdict = verdict;

}

struct nlattr *mnl_attr_nest_start(struct nlmsghdr *nlh,
						 uint16_t type)
{
	return NULL;
	(void)nlh;
	(void)type;
}

void mnl_attr_nest_end(struct nlmsghdr *nlh,
				     struct nlattr *start)
{
	(void)nlh;
	(void)start;
}

int nfq_nlmsg_parse(const struct nlmsghdr *nlh, struct nlattr **attr)
{
	for (int i = 0; i <= NFQA_MAX; ++i)
		attr[i] = NULL;

	struct nfq_data *pkt = (struct nfq_data*)nlh;
	attr[NFQA_PAYLOAD] = (struct nlattr*)&pkt->packet;
	attr[NFQA_PACKET_HDR] = (struct nlattr*)&pkt->my_nfqnl_msg_packet_hdr; // payload
	memset(&pkt->my_nfqnl_msg_packet_hdr, 0, sizeof pkt->my_nfqnl_msg_packet_hdr);
	pkt->my_nfqnl_msg_packet_hdr.packet_id = htonl(pkt->packet_id);
	if (pkt->timestamp)
	{
		attr[NFQA_TIMESTAMP] =(struct nlattr*) &pkt->my_nfqnl_msg_packet_timestamp;
		pkt->my_nfqnl_msg_packet_timestamp.sec = htobe64(pkt->timestamp);
		pkt->my_nfqnl_msg_packet_timestamp.usec = 0;
	}
	return 0;
}

void *mnl_nlmsg_get_payload(const struct nlmsghdr *nlh)
{
	struct nfq_data *pkt = (struct nfq_data*)nlh;
	pkt->my_nfgenmsg.res_id = htons(pkt->current_q);
	return (void *)&pkt->my_nfgenmsg;
}

uint16_t mnl_attr_get_payload_len(const struct nlattr *attr)
{
	return 40;
	(void)attr;
}

void *mnl_attr_get_payload(const struct nlattr *attr)
{
	return (void*)attr;
}

int mnl_socket_setsockopt(const struct mnl_socket *nl, int type,
					void *buf, socklen_t len)
 // called if netlink_no_enobufs
{
	return 0;
	(void)nl;
	(void)type;
	(void)buf;
	(void)len;
}

unsigned int mnl_socket_get_portid(const struct mnl_socket *nl)
{
	return 0;
	(void)nl;
}

// configuration packet ready
ssize_t mnl_socket_sendto(const struct mnl_socket *hconst,
					const void *buf, size_t len)
{
	struct my_params *par = (struct my_params*)buf;
	int num = par->queue_num;
	if (num >= TEST_QUEUE_MAX)
	{
		fprintf(stderr,
			MY_PREFIX ": cannot create queue %d: only support up to %d\n",
			num, TEST_QUEUE_MAX);
		return -1;
	}

	struct mnl_socket *h = (struct mnl_socket*) hconst;
	if (par->cmd == my_NFQNL_CFG_CMD_BIND)
	{
		if (h->q[num] == num) // already opened. Is this an error?
		{
			return -1;
		}

		h->q[num] = num;
		return len;
	}

	if (par->cmd == my_NFQNL_CFG_CMD_UNBIND)
	{
	}

	if (par->cmd == my_NFQNL_COPY_PACKET) // range
	{
		
	}

	if (par->cmd == my_NFQNL_MSG_VERDICT)
	{
		if (par->mark)
			print_out(h,
				"nfq_set_verdict_mark: %d, %d, %d\n",
				par->id, par->verdict, par->mark);
		else
			print_out(h,
				"nfq_set_verdict: %d, %d\n",
				par->id, par->verdict);
		return len;
	}

	if (par->cmd == my_NFQNL_MSG_CONFIG_MAXLEN)
	{
		print_out(h, "nfq_set_queue_maxlen: %u\n", par->queue_maxlen);
	}
	return len;
}



int mnl_cb_run(const void *buf_void, size_t numbytes, unsigned int seq,
			     unsigned int portid, mnl_cb_t cb_data, void *data)
{

	unsigned char *buf = (unsigned char*)buf_void;
	int rtc = -1;
	while (numbytes >= sizeof(struct nfq_data))
	{
		struct nfq_data *pkt = (struct nfq_data*)buf;
		for (int q = 0; q < TEST_QUEUE_MAX; ++q)
			if (pkt->queue[q])
			{
				if (pkt->nl->q[q] != q) // not opened?
				{
					fprintf(stderr,
						MY_PREFIX ": queue %d not opened\n", q);
					return -1;
				}

				pkt->current_q = q;
				rtc = (*cb_data)((const struct nlmsghdr *)buf, data);
				if (rtc <= 0)
				{
					suicide();
					return rtc;
				}
			}

		buf += sizeof(struct nfq_data);
		numbytes -= sizeof(struct nfq_data);
	}

	return rtc;
	(void)seq;
	(void)portid;
}

#define MAX_NEXTBUFS 10
static void *to_be_freed[MAX_NEXTBUFS];
static unsigned int cnt_to_be_freed;
static void add_to_be_freed(void*n)
{
	if (cnt_to_be_freed < MAX_NEXTBUFS)
		to_be_freed[cnt_to_be_freed++] = n;
}

static void free_nextbuf(void)
{
	while (cnt_to_be_freed > 0)
		free(to_be_freed[--cnt_to_be_freed]);
}

ssize_t mnl_socket_recvfrom(const struct mnl_socket *nlconst,
					  void *pktbuf_void, size_t pktlen)
{	
	test_cmd cmd;
	int rtc = 0, i;

	struct mnl_socket *nl = (struct mnl_socket*) nlconst;
	struct nfq_data *pkt = &nl->next_pkt;

	if (nl->sv_sent == 0)
	{
		suicide();
		return 0;
	}

	unsigned char *pktbuf = pktbuf_void;
	ssize_t pktlen_copied = 0;

	unsigned char *buf = nl->nextbuf;
	add_to_be_freed(buf);
	size_t len = nl->nextbuf_sz;
	nl->sv_sent = 0;
	nl->nextbuf = NULL;
	nl->nextbuf_sz = 0;

	while (len > 0 && len >= sizeof cmd && rtc >= 0)
	{
		int pkts = 0;

		memcpy(&cmd, buf, sizeof cmd);
		buf += sizeof cmd;
		len -= sizeof cmd;

		switch (cmd.cmd)
		{
			case test_cmd_group:
				assert(cmd.arg > 0);
				pkts = cmd.arg;
				break;

			case test_cmd_sleep:
				print_out(nl, "sleep %d\n", cmd.arg);
				sleep(cmd.arg);
				break;

			case test_cmd_settime:
				time(&nl->current);
				// print_out(nl, "------------->reset time to ...... %ld\n",
				//	(long)nl->current);
				break;

			case test_cmd_putenv:
			{
				assert(cmd.arg >= 0 && cmd.arg < TEST_ARGS_MAX);
				assert(nl->cmd_arg[cmd.arg] != NULL);

				char *arg = simple_expand(nl->cmd_arg[cmd.arg]);
				if (arg == NULL)
				{
					rtc = -1;
				}

				// print_out(nl, "putenv %s\n", arg);
				if (putenv(arg) != 0)
					fprintf(stderr,
						"%s: putenv %s failed: %s\n",
						my_prefix, arg, strerror(errno));

				// don't free arg
				nl->cmd_arg[cmd.arg] = NULL;
				break;
			}

			case test_cmd_system:
			{
				assert(cmd.arg >= 0 && cmd.arg < TEST_ARGS_MAX);
				assert(nl->cmd_arg[cmd.arg] != NULL);
				print_out(nl, "system %s\n", nl->cmd_arg[cmd.arg]);

				int rc = system(nl->cmd_arg[cmd.arg]);
				if (rc == -1)
					fprintf(stderr,
						"%s: system %s failed: %s\n",
						my_prefix,
						nl->cmd_arg[cmd.arg],
						strerror(errno));
				else if (WIFSIGNALED(rc) &&
					(WTERMSIG(rc) == SIGINT || WTERMSIG(rc) == SIGQUIT))
				{
					rtc = -1;
				}
				else if (WEXITSTATUS(rc))
					fprintf(stderr,
						"%s: system %s exited with status %d\n",
						my_prefix,
						nl->cmd_arg[cmd.arg],
						WEXITSTATUS(rc));

				free(nl->cmd_arg[cmd.arg]);
				nl->cmd_arg[cmd.arg] = NULL;
				break;
			}

			case test_cmd_noop:
			default: // unimplemented commands
				break;
		}

		for (i = 0; i < pkts && rtc >= 0; ++i)
		{
			if (!nl->have_next_pkt)
				if (read_test_input(nl))
					break;

			assert(nl->have_next_pkt);
			nl->have_next_pkt = 0;
			
			// increase current --ibd-judge gets the delta specified on pkt
			nl->current += nl->next_pkt.timestamp;
			nl->next_pkt.timestamp = nl->current;

			// copy the packet to ibd-judge's buffer
			pkt->nl = nl;
			if (pktlen > sizeof *pkt)
			{
				memcpy(pktbuf, pkt, sizeof *pkt);
				pktbuf += sizeof *pkt;
				pktlen -= sizeof *pkt;
				pktlen_copied += sizeof *pkt;

				memset(pkt, 0, sizeof *pkt);
			}
			else
			{
				fprintf(stderr,
					MY_PREFIX ": buffer too small: %zu, needed %zu\n",
					len, nl->nextbuf_sz);
				return -1;
			}
		}
	}

	free_nextbuf();

	if (nl->in == NULL)
		suicide();
	else if (!nl->sv_sent)
	{
		if (!nl->have_next_pkt)
			read_test_input(nl);
	}

	return rtc < 0? rtc: pktlen_copied;
}


