/*
* Picture for test ipv6
* written by ale in milano on 31 Oct 2023

Copyright (C) 2023 Alessandro Vesely

This file is part of Ipqbdb.

Ipqbdb is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Ipqbdb is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Ipqbdb.  If not, see <http://www.gnu.org/licenses/>.

*/
#include <config.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <arpa/inet.h>
#include <png.h>
#include "pic.h"
#include "dbstruct.h"
#include "ip_util.h"
#include "rehabilitated_prob.h"

#include <assert.h>

struct pic_handle
{
	FILE *fp;
	day_stats *stats;
	png_structp png_ptr;
	png_infop info_ptr;
	size_t size, caught_size;
	int pic_height;
	int row;
	unsigned char *caught; // bit array
	unsigned char linebuf[];
};

#define STR1(x) #x
#define STR2(x) STR1(x)

pic_handle *pic_open(pic_params *pp, day_stats* stats)
{
	char fname[256];
	if ((unsigned)snprintf(fname, sizeof fname,
		"TESTipv6-factor_%.*f-a_%d-c_%d-t_%.1f-m_%d-s_%d.png",
		get_ipqbdb_index_factor_precision(), get_ipqbdb_index_factor(),
		pp->attacks_per_minute,
		pp->initial_count, pp->initial_decay,
		pp->minutes_per_pixel, pp->seed) > sizeof fname)
			return NULL;

	for (unsigned i = 0; fname[i] != 0; ++i)
		if (isspace((unsigned char)fname[i]))
			fname[i] = '_';

	FILE *fp = fopen(fname, "wb");
	if (fp == NULL)
		return NULL;

	size_t size = 3*PIC_WIDTH;
	size_t caught_size = PIC_WIDTH/8 + 1;
	pic_handle *pic = malloc(sizeof *pic + size + caught_size);
	if (pic == NULL)
	{
		fclose(fp);
		return NULL;
	}

	memset(pic, 0, sizeof *pic);
	pic->fp = fp;
	pic->size = size;
	pic->caught_size = caught_size;
	pic->pic_height = pp->pic_height;
	pic->stats = stats;
	pic->caught = &pic->linebuf[size];
	memset(pic->caught, 0, caught_size);

	pic->png_ptr = png_create_write_struct
		(PNG_LIBPNG_VER_STRING, (png_voidp)NULL, NULL, NULL);

	if (!pic->png_ptr)
	{
		fclose(fp);
		free(pic);
		return NULL;
	}

	pic->info_ptr = png_create_info_struct(pic->png_ptr);
	if (!pic->info_ptr)
	{
		png_destroy_write_struct(&pic->png_ptr, (png_infopp)NULL);
		fclose(fp);
		free(pic);
		return NULL;
	}

	if (setjmp(png_jmpbuf(pic->png_ptr)))
	{
		png_destroy_write_struct(&pic->png_ptr, &pic->info_ptr);
		fclose(fp);
		free(pic);
		return NULL;
	}

	png_init_io(pic->png_ptr, fp);

	png_set_IHDR(pic->png_ptr, pic->info_ptr, PIC_WIDTH, PIC_HEIGHT,
		8 /* bit_depth */, PNG_COLOR_TYPE_RGB, PNG_INTERLACE_NONE,
		PNG_COMPRESSION_TYPE_DEFAULT, PNG_FILTER_TYPE_DEFAULT);

	/* comments...? */
	png_write_info(pic->png_ptr, pic->info_ptr);
	png_set_flush(pic->png_ptr, 12);


	return pic;
}

void pic_done(pic_handle *pic)
{
	assert(pic);

	if (setjmp(png_jmpbuf(pic->png_ptr)))
	{
		png_destroy_write_struct(&pic->png_ptr, &pic->info_ptr);
		fclose(pic->fp);
		free(pic);
		return;
	}

	memset(pic->linebuf, 255, pic->size);
	for (int i = pic->row; i < PIC_HEIGHT; ++i)
		png_write_row(pic->png_ptr, pic->linebuf);

	png_write_end(pic->png_ptr, pic->info_ptr);
	png_destroy_write_struct(&pic->png_ptr, &pic->info_ptr);
	fclose(pic->fp);
	free(pic);
}

static int warns_given = 0;
static char ip_valid[] = { 0x20, 1, 0xd, 0xb8, 0, 0, 0};
static inline void check_ip(unsigned char const ip[16])
{
	if (warns_given < 20 && memcmp(ip_valid, ip, sizeof ip_valid))
	{
		char buf[INET6_ADDRSTRLEN];
		inet_ntop(AF_INET6, ip, buf, sizeof buf);
		printf("Invalid IP %s\n", buf);
		++warns_given;
	}
}

static inline void range_error(unsigned char const ip[16], int plen)
{
	if (warns_given < 20 && memcmp(ip_valid, ip, sizeof ip_valid))
	{
		char buf[INET6_ADDRSTRLEN];
		inet_ntop(AF_INET6, ip, buf, sizeof buf);
		printf("Range contains other data: %s/%d\n", buf, plen);
		++warns_given;
	}
}

static inline int ip2dot(unsigned char const ip[16])
{
	int rtc = (ip[7] * 256 + ip[8])/IP_PER_DOT;
	if (rtc >= PIC_WIDTH)
		rtc = PIC_WIDTH - 1;
	return rtc;
}

void pic_set_caught(pic_handle *pic, unsigned char const ip[16])
{
	int caught = ip2dot(ip);
	int ndx = caught >> 3;
	unsigned char bit = 1 << (caught & 7);
	pic->caught[ndx] |= bit;
}

static int nlz(uint32_t x)
/*
* Number of leading 0's (approx log2)
*/
{
	if (x == 0) return 32;

	int n = 1;
	if ((x >> 16) == 0) { n += 16; x <<= 16; }
	if ((x >> 24) == 0) { n +=  8; x <<=  8; }
	if ((x >> 28) == 0) { n +=  4; x <<=  4; }
	if ((x >> 30) == 0) { n +=  2; x <<=  2; }

	return n - (x >> 31);
}

int pic_draw_line(pic_handle *pic, time_t now, DB* db, int is_halfway)
{
	assert(pic);
	assert(db);

	if (setjmp(png_jmpbuf(pic->png_ptr)))
		return 2;

	DBC *dbc = NULL;
	int rc = db->cursor(db, NULL, &dbc, DB_CURSOR_BULK);
	if (rc || dbc == NULL)
		return 2;

	DBT key, data;
	ip_data_t ip_data;
	unsigned char ip[16];

	memset(&key, 0, sizeof key);
	memset(&data, 0, sizeof data);
	memset(&ip_data, 0, sizeof ip_data);
	memset(ip, 0, sizeof ip);
	data.ulen = data.size = sizeof ip_data;
	data.data = &ip_data;
	key.ulen = key.size = sizeof ip;
	key.data = ip;
	key.flags = data.flags = DB_DBT_USERMEM;

	memset(pic->linebuf, 255, pic->size);
	int prev_dot = -1;
	rc = dbc->get(dbc, &key, &data, DB_FIRST);
	if (pic->row == 0)
		printf("first dot: %d\n", ip2dot(ip));

	while (rc == 0)
	{
		int dot = ip2dot(ip), next_dot, first_dot = PIC_WIDTH;
		enum type
		{
			type_none,
			type_singleton,
			type_range,
			type_mixed
		} type = type_none;
		int probability = 0;

		for (;;)
		{
			check_ip(ip);
			if (ip_data.plen == 0)
			{
				type |= type_singleton;
				++pic->stats->singletons;
			}
			else
			{
				type |= type_range;
				++pic->stats->ranges;
				if (pic->stats->widest > ip_data.plen)
					pic->stats->widest = ip_data.plen;
				first_in_range(ip, ip_data.plen);
				int f = ip2dot(ip);
				if (first_dot > f)
					first_dot = f;
				if (first_dot < prev_dot || ip_data.plen >= 128)
					range_error(ip, ip_data.plen);
			}

			// rehabilitate
			time_t delta = now - ip_data.last_update;
			int prob;
			if (IPQBDB_UPDATE_TICK_TIME < delta && ip_data.decay > 0.0)
				prob = rehabilitated_prob(ip_data.probability,
					delta, ip_data.decay);
			else
				prob = ip_data.probability;
			if (prob > probability)
				probability = prob;

			// next record
			rc = dbc->get(dbc, &key, &data, DB_NEXT);
			if (rc != 0)
				break;

			next_dot = ip2dot(ip);
			if (next_dot != dot)
				break;
		}

		int color = 255;
		int white = 255;
		if (probability)
			white -= 7*(31 - nlz(probability));
		// if (white < 0)
		// {
		// 	white = 0;
		// 	color = 128;
		// }
		unsigned char *p = pic->linebuf + dot*3;
		if (type == type_singleton) // singleton, red
		{
				*p++ = color;
				*p++ = white;
				*p = white;
		}
		else
		{
			assert(type);
			assert(first_dot < PIC_WIDTH);
			int dilute = 125*(dot - first_dot)/PIC_WIDTH;
			white += dilute;
			if (white > 240)
				white = 240;

			unsigned char *q = pic->linebuf + first_dot*3;
			if (type == type_range) // range, blue
				p += 3;

			while (q < p)
			{
				*q++ = white;
				*q++ = white;
				*q++ = color;
			}

			if (type == type_mixed) // mixed, green
			{
				*p++ = white;
				*p++ = color;
				*p = white;
			}
		}
		prev_dot = dot;
	}

	dbc->close(dbc);
	if (rc == DB_NOTFOUND)
	{
		for (unsigned int ndx = 0; ndx < pic->caught_size; ++ndx)
		{
			int caught = pic->caught[ndx];
			if (caught)
			{
				for (int dot = 0; dot < 8; ++dot)
				{
					unsigned char const bit = 1 << dot;
					if (caught & bit)
					{
						unsigned char *p = pic->linebuf + (8*ndx + dot)*3;
						*p++ = 0;
						*p++ = 0;
						*p = 0;
					}
				}
			}
		}

		if (is_halfway)
			for (int i = 0; i < 3*5; ++i)
				pic->linebuf[i] = pic->linebuf[pic->size -1 -i] = 100;

		png_write_row(pic->png_ptr, pic->linebuf);
		memset(pic->caught, 0, pic->caught_size);
		pic->row += 1;
	}
	else
	{
		db->err(db, rc, "next curs");
		return 2;
	}

	return pic->row >= pic->pic_height;
}
