/*
* Ipq Berkeley db Daemon  category management utility- ibd-category
* written by ale in milano on Tue 12 Jan 2021

Copyright (C) 2021 Alessandro Vesely

This file is part of Ipqbdb.

Ipqbdb is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Ipqbdb is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Ipqbdb.  If not, see <http://www.gnu.org/licenses/>.

*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <syslog.h>

// Berkeley DB
#include <db.h>

#include <popt.h>

#include "config_names.h"
#include "dbstruct.h"
#include "utf8_util.h"

static app_private ap;
static char const err_prefix[] = "ibd-category";
#include "setsig_func.h"

#include <assert.h>

static char *db_descr_name = IPQBDB_DESCR_DATABASE_NAME;
static int category, category_given;
static int version_opt, help_opt, list_opt, header_printed, verbose = -1;
static int list_opt, del_opt, cleanup_opt;
static struct poptOption opttab[] =
{
	{"category", 'C', POPT_ARG_INT, &category, 1,
	"The category to be assigned", "integer in [0, 255]"},
	{"list", 'L', POPT_ARG_NONE, &list_opt, 0,
	"Bare listing of reasons", NULL},
	{"del", '\0', POPT_ARG_NONE, &del_opt, 0,
	"Delete the reasons (careful!)", NULL},
	{"verbose", 'v', POPT_ARG_INT|POPT_ARGFLAG_OPTIONAL, &verbose, 0,
	"Be verbose", "level"},
	{"version", 'V', POPT_ARG_NONE, &version_opt, 0,
	"Print version number and exit", NULL},
	{"help", 'h', POPT_ARG_NONE, &help_opt, 0,
	"This help.", NULL},
	{"db-descr", 'd', POPT_ARG_STRING|POPT_ARGFLAG_SHOW_DEFAULT, &db_descr_name, 0,
	"The reason description table.", "filename"},
	{"db-cleanup", '\0', POPT_ARG_NONE, &cleanup_opt, 0,
	"On exit cleanup environment (__db.00? files) if not still busy", NULL},
	POPT_TABLEEND
};

static int descr_action(DB *db1, int descr_id, int action)
/*
* Action can be one of:
*   - set category  action in [0, 255]
*   - delete record action = -1
*   - list record   action = -2
*/
{
	assert(db1);

	union
	{
		bynumber_descr_t drec;
		char fill[sizeof(bynumber_descr_t) + IPQBDB_DESCR_MAX_LENGTH + 1];
	} u;

	char *const descr = &u.drec.descr[0];

	DBT key, data;
	DBC *curs;
	char const *what = "dunno";
	int rtc;

	memset(&key, 0, sizeof key);
	memset(&data, 0, sizeof data);
	key.data = &descr_id;
	key.size = sizeof descr_id;
	key.flags = DB_DBT_USERMEM;
	data.data = &u;
	data.size = data.ulen = sizeof u;
	data.flags = DB_DBT_USERMEM;

	// write cursor: prevent writing the description table until close below
	rtc = db1->cursor(db1, NULL, &curs, action == -2? 0: DB_WRITECURSOR);
	if (rtc)
	{
		db1->err(db1, rtc, "no write cursor");
		return rtc;
	}

	rtc = curs->get(curs, &key, &data, DB_SET);
	if (rtc == 0)
	{
		if (data.size >= sizeof u.drec && u.drec.chk == IPQBDB_CHK_SIGNATURE)
		{
			if (action >= 0)
			{
				u.drec.report_category = action;
				if ((rtc = db1->put(db1, NULL, &key, &data, 0)) != 0)
					what = "write";
			}
			else if (action == -1) // del
			{
				if ((rtc = db1->del(db1, NULL, &key, 0)) != 0)
					what = "del";
			}
			else if (action == -2) // list
			{
				// data.size should include trailing 0
				size_t dlen = data.size - 1 - offsetof(bynumber_descr_t, descr);
				if (dlen >= IPQBDB_DESCR_MAX_LENGTH)
					dlen = IPQBDB_DESCR_MAX_LENGTH - 1;

				if (descr[dlen] != 0)
				{
					char *t = begin_utf8(&descr[0], &descr[dlen]);
					dlen = t - &descr[0];
					descr[dlen] = 0;
				}
				if (verbose && header_printed == 0)
				{
					puts("Record CAT Description");
					++header_printed;
				}
				printf("%6d %3d %s\n", descr_id, u.drec.report_category, descr);
			}
		}
		else
		{
			rtc = 1;
			what = "record size/format";
		}
	}
	else if (rtc == DB_NOTFOUND)
	{
		report_error(&ap, LOG_ERR,
			"Nonexistent reason record number: %d\n",
			descr_id);
		rtc = 1;
	}
	else what = "get";

	curs->close(curs);

	if (rtc < 0) // DB error codes
		db1->err(db1, rtc, "descr_action on %s key=%d", what, descr_id);

	return rtc;
}


int main(int argc, char const *argv[])
{
	static const char optaliases[] = IPQBDB_OPTION_FILE;
	int rtc = 0, errs = 0;

	poptContext opt = poptGetContext(err_prefix, argc, argv, opttab, 0);

	if (access(optaliases, F_OK) == 0 &&
		(rtc = poptReadConfigFile(opt, optaliases)) < 0)
	{
		fprintf(stderr, "%s: cannot read %s: %s\n",
			err_prefix, optaliases, poptStrerror(rtc));
		errs = 3;
	}

	poptSetOtherOptionHelp(opt, "[OPTION...] reason-ids...\n"
		"\twhere reason-ids are the records to operate on"
		" and options are as follows:\n");
	while ((rtc = poptGetNextOpt(opt)) == 1)
	{
		if (category < 0 || category >= 256)
		{
			fprintf(stderr, "%s: invalid category %d\n",
				err_prefix, category);
			errs = 1;
		}
		else
			++category_given;
	}

	if (rtc != -1)
	{
		fprintf(stderr, "%s: %s at %s\n",
			err_prefix, poptStrerror(rtc), poptBadOption(opt, 0));
		errs = 1;
	}
	else
	{
		if (version_opt)
		{
			fprintf(stdout, "%s: version " PACKAGE_VERSION "\n"
			DB_VERSION_STRING "\n", err_prefix);
			errs = 2;
		}

		if (help_opt)
		{
			poptPrintHelp(opt, stdout, 0);
			fputs_database_help();
			fputs("Categories are tiny integers in [0, 255], where"
				" 0 is to be used to indicate"
				" non-categorized records.\n", stdout);
			errs = 2;
		}

		// popt sets verbose to 0 if no arg is given
		verbose += 1;

		ap.mode = error_report_stderr; // 0
		ap.err_prefix = err_prefix;

		if (errs == 0 && poptPeekArg(opt) == NULL)
		{
			report_error(&ap, LOG_INFO,
				"No reason record number arguments, no action.\n");
			errs = 1;
		}

		if (errs == 0 &&
			list_opt == 0 && del_opt == 0 && category_given == 0)
		{
			report_error(&ap, LOG_INFO,
				"No option given, no action.\n");
			errs = 1;
		}

		if (category_given > 1)
			report_error(&ap, LOG_WARNING,
				"Only the last category given (%d) is used.\n", category);
	}

	if (errs == 1)
		poptPrintUsage(opt, stdout, 0);

	rtc = 0;

	if (errs)
		rtc = 1;
	else
	{
		switchable_fname *descr_fname = database_fname(db_descr_name, &ap);
		if (descr_fname == NULL)
			rtc = 1;

		if (rtc)
		{
			if (verbose)
				report_error(&ap, LOG_INFO, "Bailing out...\n");
		}
		else
		{
			DB_ENV *db_env = NULL;
			DB *db_de1 = NULL, *db_de2 = NULL;

			setsigs();

			rtc = open_database(descr_fname, &ap, &db_env, NULL, NULL);

			if (rtc == 0 && caught_signal == 0)
				rtc = open_descrdb(descr_fname->fname, db_env, &db_de1, &db_de2);

			while (rtc == 0 && caught_signal == 0 && poptPeekArg(opt))
			{
				int reason_id = -1;
				char *t = NULL;
				const char *reason_string = poptGetArg(opt);
				unsigned long l = strtoul(reason_string, &t, 0);
				if (t && *t == 0 && l < INT_MAX)
					reason_id = (int)l;
				else
				{
					report_error(&ap, LOG_ERR,
						"Invalid reason record number: %s\n",
						reason_string);
					continue;
				}

				if (list_opt && caught_signal == 0)
					rtc = descr_action(db_de1, reason_id, -2);
				if (category_given && rtc == 0 && caught_signal == 0)
					rtc = descr_action(db_de1, reason_id, category);
				if (del_opt && rtc == 0 && caught_signal == 0)
					rtc = descr_action(db_de1, reason_id, -1);

				if (rtc == 1) // not found
					rtc = 0;
			}

			if (rtc)
				rtc = 2;

			close_db(db_de2);
			close_db(db_de1);

			close_dbenv(db_env, cleanup_opt);
		}
	}

	poptFreeContext(opt);
	return rtc;
}

