(* 
    This file is a part of IsarMathLib - 
    a library of formalized mathematics written for Isabelle/Isar.

    Copyright (C) 2005  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{EquivClass1.thy}*}

theory EquivClass1 = EquivClass + func1 + ZF1:

text{*In this theory file we extend the  work on equivalence relations 
  done in the standard Isabelle's EquivClass.thy file. The problem that 
  we have with the EquivClass.thy is that the notions congruent and congruent2
  are defined for meta-functions rather then ZF - functions (subsets of 
  Cartesian products). This causes inflexibility (that is typical for typed 
  set theories) in making the notions depend on additional parameters
  For example the congruent2 there takes [i,[i,i]=>i] as parameters, that is 
  the second parameter is a meta-function that takes two sets and results in 
  a set. So, when our function depends on additional parameters, 
  (as for example in Real\_ZF1.thy, where the function we want to be 
  congruent depends on a group and we want to show that for all groups 
  the function is congruent) there is no easy way to use that notion.
  The ZF functions are sets and it there is no problem if in actual 
  application this set depends on some parameters. *}

section{*Congruent functions and projections on the quotient*}

text{*First we define the notion of function that maps equivalent 
  elements to equivalent values. We use similar names as
  in the original EquivClass.thy file to indicate the conceptual 
  correspondence of the notions. Then we define the projection of
  a function onto the quotient space. We will show that if the function is
  congruent the projection is 
  a mapping from the quotient space into itself. In standard math the 
  condion that the function is congruent allows to show that the value
  of the projection does not depend on the choice of elemnts that represent
  the equivalence classes. We set up things a little differently to avoid
  making choices as this would require an axiom of choice.*}

constdefs
  Congruent :: "[i,i,i]\<Rightarrow>o"
  "Congruent(A,r,f) ==
  (\<forall>x y. <x,y> \<in> r  \<longrightarrow> <f`(x),f`(y)> \<in> r)"

  ProjFun :: "[i,i,i] \<Rightarrow> i"
  "ProjFun(A,r,f) ==
  {<c,d> \<in> (A//r)\<times>(A//r). (\<Union>x\<in>c. r``{f`(x)}) = d}";

text{*Elements of equivalence classes belong to the set.*}

lemma EquivClass_1_L1: 
  assumes A1:"equiv(A,r)" and A2:"C \<in> A//r" and A3:"x\<in>C"
  shows "x\<in>A"
proof -;
  from A2 have "C \<subseteq> \<Union> (A//r)" by auto;
  with A1 A3 show "x\<in>A"
    using Union_quotient by auto;
qed;


text{*If an element belongs to an equivalence class, then its image
  under relation is this equivalence class.*}

lemma EquivClass_1_L2: 
  assumes A1:"equiv(A,r)" "C \<in> A//r" and A2:"x\<in>C"
  shows "r``{x} = C"
proof -;
  from A1 A2 have "x \<in> r``{x}" 
    using EquivClass_1_L1  equiv_class_self by simp;
  with A2 have T1:"r``{x}\<inter>C \<noteq> 0" by auto;
  from A1 A2 have "r``{x} \<in> A//r"
    using EquivClass_1_L1 quotientI by simp;
  with A1 T1 show ?thesis
    using quotient_disj by blast;
qed;
      
text{*Elements that belong to the same equivalence class are equivalent.*}

lemma EquivClass_1_L2A:
  assumes A1:"equiv(A,r)" "C \<in> A//r" "x\<in>C" "y\<in>C"
  shows "<x,y> \<in> r"
  proof -;
    from A1 have "r``{x} = r``{y}"
    using EquivClass_1_L2 by simp;
  with A1 show "<x,y> \<in> r"
    using EquivClass_1_L1 equiv_class_eq_iff by simp;
qed;
  
text{*If a function is congruent then the equivalence classes of the values
  that come from the arguments from the same class are the same.*}

lemma EquivClass_1_L3: 
  assumes A1:"equiv(A,r)" and A2:"Congruent(A,r,f)" 
  and A3:"C \<in> A//r" "x\<in>C" "y\<in>C" 
  shows "r``{f`(x)} = r``{f`(y)}"
proof -;
  from A1 A3 have "<x,y> \<in> r"
    using EquivClass_1_L2A by simp;
  with A2 have  "<f`(x),f`(y)> \<in> r"
    using Congruent_def by simp;
  with A1 show ?thesis using equiv_class_eq by simp
qed;


text{*The values of congruent functions are in the space.*}

lemma EquivClass_1_L4:
  assumes A1:"equiv(A,r)" and A2:"C \<in> A//r" "x\<in>C"
  and A3:"Congruent(A,r,f)"
  shows "f`(x) \<in> A"
proof -;
  from A1 A2 have "x\<in>A";
    using EquivClass_1_L1 by simp;
  with A1 have "<x,x> \<in> r"
    using equiv_def refl_def by simp;
  with A3 have  "<f`(x),f`(x)> \<in> r"
    using Congruent_def by simp;
  with A1 show ?thesis using equiv_type by auto;
qed;

text{*Equivalence classes are not empty.*}

lemma EquivClass_1_L5: 
  assumes A1:"refl(A,r)" and A2: "C \<in> A//r"
  shows "C\<noteq>0"
proof -;
  from A2 obtain x where D1:"C = r``{x}" and D2:"x\<in>A"
    using quotient_def by auto;
  from D2 A1 have "x \<in> r``{x}" using refl_def by auto;
  with D1 show ?thesis by auto;
qed;
  
text{*To avoid using an axiom of choice, we define the projection using 
  the expression $\bigcup _{x\in C} r(\{f(x)\})$. The next lemma shows that for
  congruent function this is in the quotient space $A/r$. *}

lemma EquivClass_1_L6:
  assumes A1:"equiv(A,r)" and A2:"Congruent(A,r,f)" 
  and A3:"C \<in> A//r"
  shows "(\<Union>x\<in>C. r``{f`(x)}) \<in> A//r"
proof -;
  from A1 A3 have "C\<noteq>0"
    using equiv_def EquivClass_1_L5 by auto;
  moreover from A2 A3 A1 have "\<forall>x\<in>C. r``{f`(x)} \<in> A//r"
    using EquivClass_1_L4 quotientI by auto;
  moreover from A1 A2 A3 have 
    "\<forall>x y. x\<in>C \<and> y\<in>C \<longrightarrow> r``{f`(x)} = r``{f`(y)}" 
    using EquivClass_1_L3 by blast;
  ultimately show ?thesis by (rule ZF1_1_L2);
qed;

text{*Congruent functions can be projected.*}

lemma EquivClass_1_T1: 
  assumes A1:"equiv(A,r)" "Congruent(A,r,f)"
  shows "ProjFun(A,r,f) \<in> A//r \<rightarrow> A//r"
proof -;
  from A1 show ?thesis 
    using EquivClass_1_L6 ProjFun_def func1_1_L11A by simp;
qed;
  
text{*We now define congruent functions of two variables. Congruent2 
  corresponds to congruent2 in EquivClass.thy, but uses ZF-functions rather
  than meta-functions.*}

constdefs
  Congruent2 :: "[i,i,i]\<Rightarrow>o"
  "Congruent2(A,r,f) ==
  (\<forall>x1 x2 y1 y2. <x1,x2> \<in> r \<and> <y1,y2> \<in> r  \<longrightarrow> 
  <f`<x1,y1>,f`<x2,y2> > \<in> r)"

  ProjFun2 :: "[i,i,i] \<Rightarrow> i"
  "ProjFun2(A,r,f) ==
  {<p,d> \<in> ((A//r)\<times>(A//r))\<times>(A//r) . 
  (\<Union> z \<in> fst(p)\<times>snd(p). r``{f`(z)}) = d}";


text{*The following lemma is a two-variables equivalent of EquivClass\_1\_L3.*}

lemma EquivClass_1_L7: 
  assumes A1:"equiv(A,r)" and A2:"Congruent2(A,r,f)"
  and A3:"C1 \<in> A//r" "C2 \<in> A//r" 
  and A4:"z1 \<in> C1\<times>C2" "z2 \<in> C1\<times>C2"
  shows "r``{f`(z1)} = r``{f`(z2)}"
proof -;
  from A4 obtain x1 y1 x2 y2 where 
    "x1\<in>C1" and "y1\<in>C2" and D1:"z1 = <x1,y1>" and 
    "x2\<in>C1" and "y2\<in>C2" and D2:"z2 = <x2,y2>" 
    by auto;
  with A1 A3 have "<x1,x2> \<in> r" and "<y1,y2> \<in> r"
    using EquivClass_1_L2A by auto;
  with A2 have "<f`<x1,y1>,f`<x2,y2> > \<in> r"
    using Congruent2_def by simp;
  with A1 D1 D2 show ?thesis using equiv_class_eq by simp;
qed;

text{*The values of two variable congruent functions are in the space.*}

lemma EquivClass_1_L8:
  assumes A1:"equiv(A,r)" and A2:"C1 \<in> A//r" and A3:"C2 \<in> A//r"
  and A4:"z \<in> C1\<times>C2" and A5: "Congruent2(A,r,f)"
  shows "f`(z) \<in> A"
proof -;
  from A4 obtain x y where "x\<in>C1" and "y\<in>C2" and D1:"z = <x,y>"  
    by auto;
  with A1 A2 A3 have "x\<in>A" and "y\<in>A" 
    using EquivClass_1_L1 by auto;
  with A1 A4 have "<x,x> \<in> r" and "<y,y> \<in> r"
    using equiv_def refl_def by auto;
  with A5 have "<f`<x,y>, f`<x,y> > \<in> r"
    using Congruent2_def by simp;
  with A1 D1 show ?thesis using equiv_type by auto;
qed;


text{*The following lemma is a two-variables equivalent of EquivClass\_1\_L6.*}

lemma EquivClass_1_L9:
  assumes A1:"equiv(A,r)" and A2:"Congruent2(A,r,f)" 
  and A3:"p \<in> (A//r)\<times>(A//r)"
  shows "(\<Union> z \<in> fst(p)\<times>snd(p). r``{f`(z)}) \<in> A//r"
proof -;
  from A3 have D1:"fst(p) \<in> A//r" and D2:"snd(p) \<in> A//r"
    by auto;
  with A1 A2 have 
    T1:"\<forall>z \<in> fst(p)\<times>snd(p). f`(z) \<in> A"
    using EquivClass_1_L8 by simp;
  from A3 A1 have "fst(p)\<times>snd(p) \<noteq> 0" 
    using equiv_def EquivClass_1_L5 Sigma_empty_iff
    by auto;
  moreover from A1 T1 have 
    "\<forall>z \<in> fst(p)\<times>snd(p). r``{f`(z)} \<in> A//r"
    using quotientI by simp;
  moreover from A1 A2 D1 D2 have
    "\<forall>z1 z2. z1 \<in> fst(p)\<times>snd(p) \<and> z2 \<in> fst(p)\<times>snd(p) \<longrightarrow> 
    r``{f`(z1)} = r``{f`(z2)}"
    using EquivClass_1_L7 by blast;
   ultimately show ?thesis by (rule ZF1_1_L2);
qed;

text{*Congruent functions of two variables can be projected.*}

lemma EquivClass_1_T1: 
  assumes A1:"equiv(A,r)" "Congruent2(A,r,f)"
  shows "ProjFun2(A,r,f) \<in> (A//r)\<times>(A//r) \<rightarrow> A//r"
proof -;
  from A1 show ?thesis 
    using EquivClass_1_L9 ProjFun2_def func1_1_L11A by simp;
qed;

end