(* 
    This file is a part of IsarMathLib - 
    a library of formalized mathematics written for Isabelle/Isar.

    Copyright (C) 2005  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{Topology\_ZF.thy}*}

theory Topology_ZF = Finite1 + Fol1:
(* Finite1 depends on Finite and shows some
properties of finite subcollections that we need to define 
relative (induced) topology*) 
section{*Basic definitions and properties*}

text{*We define a topology on a set $X$ as a collection $T$ of subsets 
of $X$ such that $X\in T$, $\emptyset \in T$ and $T$ is closed 
with respect to arbitrary unions and  intersection of two sets. 
This is different than the standard definiton that requires intersection of 
finite number of open sets to be open. However, our definition is equivalent 
to the standard one (we will proove that of course) and makes a couple of 
proofs much shorter.*}

constdefs
 isAtopologyOn :: "[i,i] \<Rightarrow> o" (infixl "{is a topology on}" 65)
 " T {is a topology on} X == 
  ( 
    ( T \<subseteq> Pow(X)) &
    ( X \<in> T) &
    ( 0 \<in> T) &
    ( \<forall> A. (A \<subseteq> T \<longrightarrow> ( ( \<Union> A )\<in> T ))) &
    ( \<forall> U V. ( (U\<in>T & V\<in>T)  \<longrightarrow>  ( U\<inter>V \<in> T)))
  )"
 
text{*Lets give an example of a topology.*}

lemma TopZF_1_L1: "Pow(X) {is a topology on} X"
proof -
  have "Pow(X) \<subseteq> Pow(X)" ..;
  moreover have "X\<in> Pow(X)" by simp;
  moreover have "0 \<in> Pow(X)" by simp;
  moreover have "\<forall> A. (A \<subseteq> Pow(X) \<longrightarrow> ( ( \<Union> A )\<in> Pow(X) ))" by fast;
  moreover have 
    "\<forall> U V. ((U\<in>Pow(X) & V\<in>Pow(X))  \<longrightarrow>  ( U\<inter>V \<in> Pow(X)))" by fast;
  ultimately show "Pow(X) {is a topology on} X"  
    using isAtopologyOn_def by simp; 
qed;

text{*Our first theorem states that intersection of any nonempty 
collection of topologies on a set $X$ is a topology.*}

theorem TopZF_1_T1: assumes A1: "A\<noteq>0" 
  and A2: "\<forall>T\<in>A. (T {is a topology on} X)" 
  shows "(\<Inter> A) {is a topology on} X";
proof-;
  have "( \<Inter> A ) \<subseteq> Pow(X)"; 
  proof -;
      from A2 have "\<forall> T\<in> A. ( T\<subseteq> Pow(X))" 
	using  isAtopologyOn_def by simp;
      with A1 show "( \<Inter> A ) \<subseteq> Pow(X)" by fast;
  qed;
  moreover have "X\<in> \<Inter> A";
  proof -;
    from A2 A1 show "X\<in> \<Inter> A" using isAtopologyOn_def by auto;
  qed
  moreover have "0 \<in> \<Inter> A"
  proof -
    from A2 A1 show "0 \<in> \<Inter> A" using isAtopologyOn_def by auto;
  qed;
  moreover have " \<forall> B. (B \<subseteq> \<Inter> A  \<longrightarrow> ( ( \<Union> B )\<in> \<Inter> A))"
  proof;
    fix B; 
    show "B \<subseteq> \<Inter> A  \<longrightarrow> ( ( \<Union> B )\<in> \<Inter> A)"; 
    proof;
      assume A41: "B \<subseteq> \<Inter> A" show "( \<Union> B )\<in> \<Inter> A";
      proof -;
	from A41 have T41: "\<forall> T\<in> A. B\<subseteq> T" by fast;
	from A2 T41 have T43: "\<forall> T\<in> A. (\<Union> B) \<in> T" 
	  using isAtopologyOn_def  by simp;
	from A1 T43 show "( \<Union> B )\<in> \<Inter> A" by fast;
      qed
    qed
  qed;

  moreover have "( \<forall> U V. ( (U\<in> \<Inter> A & V\<in> \<Inter> A)  \<longrightarrow>  ( U\<inter>V \<in> \<Inter> A )))"    
  proof (rule double_forall_implies);
    fix U V	
    assume "U \<in> \<Inter> A & V\<in> \<Inter> A";
    hence "\<forall> T\<in>A. (U\<in> T & V\<in> T)" by fast;
    with A2 have "\<forall> T\<in>A. U\<inter>V \<in> T " 
      using isAtopologyOn_def by simp;
    with A1 show "U\<inter>V \<in> \<Inter>A" by auto;  
  qed;
   
  ultimately show "(\<Inter> A){is a topology on} X" 
    using isAtopologyOn_def by simp
qed;

text{* We define a topology generated by a collection of sets as the 
intersection of all topologies containing that collection. 
This will be denoted as Topology$(A,X)$, where $A$ is 
the collection of sets and $X$ is the space where we generate the topology.*};

constdefs
 Topology :: "[i,i] \<Rightarrow> i"
"Topology(B,X) == 
  \<Inter> {T\<in> Pow(Pow(X)) . ( B \<subseteq> T  & (T {is a topology on} X))}"

text{* Now we show that the generated topology is indeed a topology.
The strategy here is to show that Topology$(A,X)$ is an intersection of a 
nonempty collection of topologies and then use theorem TopZF1 *}

theorem TopZF_1_T2: assumes A: "B \<subseteq> Pow(X)" 
  shows "Topology(B,X) {is a topology on} X";
proof -
  let ?A = "{T\<in> Pow(Pow(X)) . ( B \<subseteq> T  & (T {is a topology on} X))}";
  from A have "Pow(X) \<in> ?A" using TopZF_1_L1 by simp;
  hence "?A\<noteq>0" by (rule not_emptyI);
  thus "Topology(B,X) {is a topology on} X" 
    using TopZF_1_T1 Topology_def by simp 
qed;

text{*Of course topology generated by a collection of sets 
contains that collection *}

theorem TopZF_1_T3: assumes A: "B \<subseteq> Pow(X)" shows "B \<subseteq> Topology(B,X)"
proof -;
   let ?A = "{T\<in> Pow(Pow(X)) . ( B \<subseteq> T  & (T {is a topology on} X))}";
   from A have "Pow(X) \<in> ?A" using TopZF_1_L1 by simp;
   hence "?A\<noteq>0" by (rule not_emptyI);
   hence "B\<subseteq> \<Inter> ?A" by fast;
   thus "B \<subseteq> Topology(B,X)" using Topology_def by simp;
qed

text{*Now we  will create a locale that fixes $X$ and $T$ and 
assumes that $T$ is a topology on $X$. We are not trying to represent 
the notion of topological space by a locale. We introduce this locale only
as a proof and theorem context to avoid repeating this assumption 
over and over again.*}

locale top_space0 =
  fixes X::"i"
  fixes T::"i"
  assumes topSpaceAssum:"T {is a topology on} X"; 

text{*Intersection of a finite nonempty collection of open sets is open.*}
 
lemma (in top_space0) TopZF_1_L2: assumes A1:"N\<noteq>0" "N\<in>Fin(T)"
  shows "\<Inter>N \<in> T"
proof -;
  from topSpaceAssum A1 show ?thesis 
    using isAtopologyOn_def Finite1_L5 by simp;
qed;
  
text{*Having a topology on a set $X$ and a subset $A$ of $X$ we can 
define the induced topology 
as the one consisting of the intersections of $A$ with sets from $T$.
The notion of a collection restricted to a set is defoned in Finite1.thy.*}


theorem (in top_space0) TopZF_1_T4: 
  assumes A1: "A\<subseteq>X"
  shows "T {restricted to} A {is a topology on} A"
proof -
  let ?S = "T {restricted to} A";
  from A1 topSpaceAssum have "?S\<subseteq> Pow(A)" 
    using isAtopologyOn_def RestrictedTo_def by auto;
  
  moreover have "A \<in> ?S";
  proof -;
    from topSpaceAssum have "X\<in> T" using isAtopologyOn_def by simp;
    hence "A\<inter>X \<in> { A\<inter>V . V\<in> T}" by auto;
    moreover from A1 have T2:"A\<inter>X = A" by fast;
    finally show "A \<in> ?S" using RestrictedTo_def by simp;
  qed;
  
  moreover have "0\<in>?S";
  proof -;
    from topSpaceAssum  have "0\<in> T" using isAtopologyOn_def by simp;
    thus "0\<in> ?S" using  RestrictedTo_def by auto;
  qed;
  
  moreover have "\<forall> M. (M \<subseteq> ?S \<longrightarrow> ( ( \<Union> M )\<in> ?S ))";
  proof;
    fix M; show "M \<subseteq> ?S \<longrightarrow> ( ( \<Union> M )\<in> ?S )"; 
    proof;
      assume A3:"M \<subseteq> ?S"
      from  topSpaceAssum have "\<forall> V\<in> M. \<Union>{U\<in> T. V = U\<inter>A} \<in> T " 
	using isAtopologyOn_def by auto (*blast no-go*)
      hence "{\<Union>{U\<in> T. V = U\<inter>A}.V\<in> M} \<subseteq> T" by auto; 
      with  topSpaceAssum have "(\<Union>V\<in> M. \<Union>{U\<in> T. V = U\<inter>A})\<in> T" 
	using isAtopologyOn_def by auto;
      hence T7: "(\<Union>V\<in> M. \<Union>{U\<in> T. V = U\<inter>A})\<inter> A \<in> ?S" 
	using RestrictedTo_def by auto;
      from A3 have "\<forall> V\<in> M. \<exists> U\<in> T. V=U\<inter>A" 
	using RestrictedTo_def by auto;
      hence  "(\<Union>V\<in> M. \<Union>{U\<in> T. V = U\<inter>A})\<inter> A = \<Union>M" by fast;
      with T7 show "( \<Union> M )\<in> ?S" by simp;
    qed;
  qed;
  
  moreover have " \<forall> U1 U2. ( (U1\<in>?S & U2\<in>?S)  \<longrightarrow>  ( U1\<inter>U2 \<in> ?S))"
  proof;
    fix U1
    show "\<forall> U2. ( (U1\<in>?S & U2\<in>?S)  \<longrightarrow>  ( U1\<inter>U2 \<in> ?S))"
    proof;
      fix U2 show "(U1\<in>?S & U2\<in>?S)  \<longrightarrow>  ( U1\<inter>U2 \<in> ?S)";
      proof;
	assume "U1\<in>?S & U2\<in>?S"
	then obtain V1 V2 
	  where T3:"V1\<in> T & U1=V1\<inter>A" and T4:"V2\<in>T & U2=V2\<inter>A" 
	  using RestrictedTo_def by auto; 
	from T3 T4 topSpaceAssum have T5:"V1\<inter>V2\<in> T & U1\<inter>U2 = (V1\<inter>V2)\<inter>A"
	  using isAtopologyOn_def by auto; 
	thus "U1\<inter>U2 \<in> ?S" using RestrictedTo_def by auto;
      qed
    qed
  qed;
  ultimately show "?S {is a topology on} A"  
    using isAtopologyOn_def by simp
qed


section{*Interior of a set*}

text{*We define interior of a set $A$ as the union of all open sets 
contained in $A$. We use interior$(A)$ to denote the interior, 
the Int symbol seems to be already taken*}

constdefs 
interior :: "[i,i]\<Rightarrow>i"
"interior(A,T) == \<Union> {U\<in> T. U\<subseteq> A}"

text{*Interior of a set $A$ is contained in $A$.*}

lemma topZF_2_L1: shows "interior(A,T)\<subseteq> A"; 
proof -; 
  show ?thesis  using interior_def by auto;
qed;

text{*Interior of any set is open.*}

lemma (in top_space0) topZF_2_L2: shows "interior(A,T)\<in>T";
proof-;
  from topSpaceAssum have "\<forall> A. (A \<subseteq> T \<longrightarrow> ( ( \<Union> A )\<in> T ))" 
    using isAtopologyOn_def by simp;
  then show ?thesis using interior_def by auto;
qed;

text{*A set is open iff it is equal to its interior.*}

lemma (in top_space0) topZF_2_L3: shows "U\<in> T \<longleftrightarrow> interior(U,T) = U";
proof -
  have "interior(U,T)\<in> T" using topZF_2_L2 by simp
  then have "interior(U,T) = U \<longrightarrow> U\<in>T" by auto;
  moreover have "U\<in> T \<longrightarrow> interior(U,T) = U" 
    using interior_def by auto;
  ultimately show "U\<in>T \<longleftrightarrow> interior(U,T) = U" by auto;
qed;

text{*Interior of the interior is the interior.*}

lemma (in top_space0) topZF_2_L4: 
  shows "interior(interior(A,T),T) = interior(A,T)";
proof -;
  let ?U = "interior(A,T)";
  from topSpaceAssum have "?U\<in>T" using topZF_2_L2 by simp;
  then show ?thesis using topZF_2_L3 by simp;
qed;

text{*Interior of a bigger set is bigger.*}

lemma topZF_2_L5: 
  assumes A:"A\<subseteq>B" shows "interior(A,T)\<subseteq> interior(B,T)"
proof-;
  from A have "\<forall> U\<in> T. (U\<subseteq>A \<longrightarrow> U\<subseteq>B)" by auto; 
  then show ?thesis using interior_def by auto; 
qed;
  
text{*An open subset of any set is a subset of the interior of that set.*}

lemma topZF_2_L6: assumes A:"U\<subseteq>A" "U\<in>T" shows "U\<subseteq>interior(A,T)"
proof -;
  from A show ?thesis using interior_def by auto
qed;

text{*If a point of a set has an open neighboorhood contained in the set,
then the point belongs to the interior of the set.*}

lemma topZF_2_L7: assumes A1:"\<exists>U\<in>T. (x\<in>U & U\<subseteq>A)"
  shows "x \<in> interior(A,T)"
proof -;
  from A1 show ?thesis using interior_def by auto;
qed;

text{*A set is open iff its every point has a an open neighbourhood 
contained in the set. We will formulate this statement as three lemmas
(implication one way, the other way and the equivalence). This is more 
convenient in applications. It seems that Isar simplifier is not very good
at dealing with equivalences (iff).*}

text{*If a set is open then every point has a an open neighbourhood 
contained in the set. *}

lemma (in top_space0) topZF_2_L8: 
  assumes A1:"V\<in>T" shows "\<forall>x\<in>V. \<exists>U\<in>T. (x\<in>U & U\<subseteq>V)"
proof -;
  from A1 have "\<forall>x\<in>V. V\<in> T & x \<in> V \<and> V \<subseteq> V" by simp;
  then show ?thesis by auto;
qed;

text{*If every point of a set has a an open neighbourhood 
contained in the set then the set is open.*}

lemma (in top_space0) topZF_2_L9:
  assumes A1:"\<forall>x\<in>V. \<exists>U\<in>T. (x\<in>U & U\<subseteq>V)" shows "V\<in>T"
proof -;
  from A1 have "V=interior(V,T)" 
    using topZF_2_L1 topZF_2_L7 by blast; 
  then show ?thesis using topZF_2_L3 by simp;
qed;

text{*This is the conclusion from the above two lemmas.*}

lemma (in top_space0) topZF_2_L10: shows "V\<in>T \<longleftrightarrow> (\<forall>x\<in>V. \<exists>U\<in>T. (x\<in>U & U\<subseteq>V))"
proof -;
  show ?thesis using topZF_2_L8 topZF_2_L9 by blast;
qed;
 
section{*Closed sets, closure, boundary.*}

text{*Closed set is one such that its complement is open.
The closure of a set is the intersection of all closed sets that contain it.
We will often use the collection of closed sets that contain a given set $A$ 
in the proofs.  Such collection does not have a name in romantic math, 
we will call it ClosedCovers$(A)$. We also define boundary 
of a set as the intersection of its closure with 
the closure of the complement.
A set is $K$ compact if for every collection of open sets that covers $K$ 
we can choose a finite one that still covers the set. 
Recall the Fin($M$) is the collection of finite subsets of M 
(finite powerset of $M$), defined in Finite.thy.*}

constdefs
  IsClosed :: "[i,i,i] \<Rightarrow> o" 
  "IsClosed(D,X,T) == (D\<in>Pow(X) & (X - D)\<in>T)"

  IsCompact :: "[i,i,i]\<Rightarrow>o"
  "IsCompact(K,X,T) == 
  (K\<in>Pow(X)) & 
  (\<forall> M\<in>Pow(T). ((K \<subseteq> \<Union>M) \<longrightarrow> (\<exists> N\<in>Fin(M). K \<subseteq> \<Union>N)))"

  ClosedCovers :: "[i,i,i]\<Rightarrow>i"
  "ClosedCovers(A,X,T) == {D\<in> Pow(X). IsClosed(D,X,T) & A\<subseteq>D}"

  Closure :: "[i,i,i]\<Rightarrow>i"
  "Closure(A,X,T) == \<Inter> ClosedCovers(A,X,T)"

  Boundary :: "[i,i,i]\<Rightarrow>i"
  "Boundary(A,X,T) == Closure(A,X,T) \<inter> Closure(X-A,X,T)";

text{*Next we use locale to define notation for the notions defined above. 
We do it this way to to keep the notation separate from the notion definitions.
The [simp] atribute allows to avoid referencing the notation definition 
in proofs where this notation is used, the notation is replaced automatically 
by its definition by Isabelle/Isar.*}

locale top_space1 = top_space0 +
  fixes isClosed :: "i\<Rightarrow>o" ("_ {is closed}" [50] 50)
  defines isClosed_def [simp]: "D {is closed} == IsClosed(D,X,T)"

  fixes isCompact :: "i\<Rightarrow>o" ("_ {is compact}" [50] 50)
  defines isCompact_def [simp]: "K {is compact} == IsCompact(K,X,T)"  
  fixes closure :: "i\<Rightarrow>i" ("_\<^sup>-" [90] 91)
  defines closure_def [simp]: "A\<^sup>- == Closure(A,X,T)" 
  fixes  boundary :: "i\<Rightarrow>i" ("\<partial>_" [90] 91)  
  defines boundary_def [simp]: "\<partial>A == Boundary(A,X,T)";


text{* The carrier of the space is closed.*}

lemma (in top_space1) topZF_3_L1:  shows "X {is closed}"
proof-;
  have "X-X = 0" by auto
  with topSpaceAssum have "X-X \<in> T" using isAtopologyOn_def by auto;
  then show "X {is closed}" using IsClosed_def by simp;
qed;

text{*Empty set is closed.*}

lemma (in top_space1) topZF_3_L1A:  shows "0 {is closed}"
proof -
  from topSpaceAssum show ?thesis 
    using isAtopologyOn_def IsClosed_def by simp;
qed;

text{*The collection of closed covers of a set $A$ is never empty, 
as the carrier $X$ of the space belongs to it. 
We need this fact to be able take the intersection 
of this collection.
The first step in the proof justifies the usage of a lemma proven 
in a different context (locale top\_space1).*}

lemma (in top_space0) topZF_3_L2: 
  assumes A: "A\<subseteq>X" shows "ClosedCovers(A,X,T)\<noteq> 0"
proof -;
  have "top_space1(X,T)" using top_space1_def by simp;
  with A have "X\<in>ClosedCovers(A,X,T)" 
    using ClosedCovers_def top_space1.topZF_3_L1 by auto
  then show "ClosedCovers(A,X,T)\<noteq> 0" by auto
qed;

text{*Intersection of a nonempty family of closed sets is closed. *};

lemma (in top_space1) topZF_3_L3: assumes A2:"K\<noteq>0" 
  and A3: "\<forall> D\<in> K. D {is closed}" 
  shows "\<Inter> K {is closed}"
proof-;
  from A3 have T1:"\<forall> D\<in> K. (D\<in> Pow(X) & (X-D)\<in> T)"  
    using IsClosed_def by simp;
  then have "{X-D. D\<in> K} \<subseteq> T" by auto; 
  with topSpaceAssum have "(\<Union> {(X-D).D\<in> K}) \<in> T" 
    using isAtopologyOn_def by auto; (*blast no-go*)
  moreover from A2 have "\<Union>{(X-D).D\<in> K} = X - \<Inter> K" by fast;
  moreover from A2 T1 have "\<Inter> K \<in> Pow(X)" by auto;
  ultimately show "\<Inter> K {is closed}" using  IsClosed_def by simp;
qed;

text{*We need to specialize the above lemma to show that 
  intersection of two closed sets 
is closed (it is annoying, but for Isar this is not obvious)*}

lemma (in top_space1) topZF_3_L3A: 
  assumes A: "D1 {is closed}" "D2 {is closed}"
  shows "D1\<inter>D2 {is closed}"
proof-;
  have "{D1,D2}\<noteq>0" by simp
  with A have "\<Inter>{D1,D2} {is closed}" using topZF_3_L3 by fast;
  thus ?thesis by simp;
qed;

text{*Union of two closed sets is closed.*}

lemma (in top_space1) topZF_3_L4A: 
  assumes A2: "A {is closed}" "B {is closed}" 
  shows "A\<union>B {is closed}"
proof-
  from A2 topSpaceAssum have "(X-A)\<inter>(X-B)\<in> T" 
    using IsClosed_def isAtopologyOn_def by simp;
  moreover have "(X-A)\<inter>(X-B) = X-(A\<union>B)" by auto; 
  moreover from A2 have "A\<union>B \<in> Pow(X)" using IsClosed_def by auto;
  ultimately show "A\<union>B {is closed}" using IsClosed_def by simp
qed;

text{*Finite union of closed sets is closed.*} 

lemma (in top_space1) topZF_3_L4B:
  assumes A1:"N \<in> Fin({D\<in>Pow(X). D {is closed}})"
  shows "\<Union>N {is closed}"
proof -;
  let ?C = "{D\<in>Pow(X). D {is closed}}"
  have "0\<in>?C" using topZF_3_L1A by simp;
  moreover have "\<forall>A B. ((A\<in>?C & B\<in>?C) \<longrightarrow> A\<union>B\<in>?C)" 
    using topZF_3_L4A by auto;
  moreover from A1 have "N\<in>Fin(?C)" .;
  ultimately have "\<Union>N \<in> ?C" by (rule Finite1_L3)
  then show ?thesis by simp
qed

text{*Closure of a set is closed.*}

lemma (in top_space1) topZF_3_L4: assumes A2:"A\<subseteq>X"
  shows "A\<^sup>- {is closed}"
proof-;
  from A2 show ?thesis 
    using Closure_def topZF_3_L2 
      ClosedCovers_def topZF_3_L3 by simp;
qed;

text{*Closures of bigger sets are bigger.*}

lemma (in top_space1) topZF_3_L5: 
  assumes A1: "A\<in>Pow(X)" "B\<in>Pow(X)" and  A2:"A\<subseteq>B" 
  shows "A\<^sup>- \<subseteq> B\<^sup>-" 
proof -;
  from A2 have "ClosedCovers(B,X,T)\<subseteq> ClosedCovers(A,X,T)" 
    using ClosedCovers_def by auto;
  with A1 show ?thesis using topZF_3_L2 Closure_def  by auto; 
qed;

text{*Boundary of a set is closed.*}

lemma (in top_space1) topZF_3_L6:  
  assumes A2:"A\<subseteq>X" shows "\<partial>A {is closed}"
proof -;
  from A2 have "X-A\<subseteq>X" by fast;
  with A2 show ?thesis 
    using topZF_3_L4 topZF_3_L3A closure_def Boundary_def 
    by auto; 
qed;

text{*A set is closed iff it is equal to its closure.*}

lemma (in top_space1) topZF_3_L7: assumes A2:"A\<subseteq>X"
  shows "A {is closed} \<longleftrightarrow> A\<^sup>- = A";
proof;
  from A2 show "A {is closed} \<Longrightarrow> A\<^sup>- = A" 
    using Closure_def ClosedCovers_def by auto;  
  assume "A\<^sup>- = A"
  then have "X-A = X-A\<^sup>-" by simp;
  with A2 have "A {is closed}" using topZF_3_L4 IsClosed_def by simp;
  then show "(A\<^sup>- = A ) \<Longrightarrow> A {is closed}" by simp;
qed;

text{*Complement of an open set is closed.*}

lemma (in top_space1) topZF_3_L8: assumes A:"A\<in> T" shows "X-A {is closed}"
proof-
  from A topSpaceAssum have "X-(X-A)=A & X-A \<in> Pow(X)" 
    using isAtopologyOn_def by auto;
  with A show ?thesis using IsClosed_def by simp
qed;

text{*A set is contained in its closure.*}

lemma (in top_space1) topZF_3_L9: assumes A:"A\<subseteq>X" shows "A\<subseteq>A\<^sup>-"
proof-
  from A show ?thesis 
    using ClosedCovers_def topZF_3_L2 Closure_def by auto;
qed;

text{*Closure of a subset of $X$ is a subset of $X$.*}
lemma (in top_space1) topZF_3_L10: assumes A2:"A\<subseteq>X" shows "A\<^sup>-\<subseteq>X"
proof-; 
  from A2 show ?thesis 
    using topZF_3_L1 Closure_def ClosedCovers_def by auto;
qed;

text{*Closure of the complement is the complement of the interior.*}

lemma (in top_space1) topZF_3_L11: assumes A2:"A\<subseteq>X" 
  shows "(X-A)\<^sup>- = X - interior(A,T)"
proof
  from A2 have "X-A \<subseteq> X - interior(A,T)" using  topZF_2_L1 by auto
  moreover from A2 have T1:"X-interior(A,T)\<in> Pow(X)" 
    using topZF_2_L1 by auto
  moreover from A2 have T2:"X-A \<in> Pow(X)" by auto
  ultimately have "(X-A)\<^sup>- \<subseteq> (X - interior(A,T))\<^sup>-" 
    using topZF_3_L5 by auto
  moreover from T1 have 
    "(X - interior(A,T))\<^sup>- = X - interior(A,T)" 
    using topZF_2_L2 topZF_3_L8 topZF_3_L7 by auto;
  ultimately show "(X-A)\<^sup>- \<subseteq>  X - interior(A,T)" by simp
  from T2 A2 have T3:"X-(X-A)\<^sup>-\<subseteq>A" using  topZF_3_L9 by auto
  have "X-A\<subseteq>X" by auto
  with T3 have "X-((X-A)\<^sup>-)\<subseteq> interior(A,T)" 
    using topZF_3_L4 IsClosed_def topZF_2_L6 by simp
  thus "X - interior(A,T)\<subseteq> (X-A)\<^sup>-" by auto
qed;
 
text{*Boundary of a set is the closure of the set 
  minus the interior of the set.*}

lemma (in top_space1) topZF_3_L12: assumes A2:"A\<subseteq>X" 
  shows "\<partial> A = A\<^sup>- - interior(A,T)"
proof-
  from A2 have "\<partial> A = A\<^sup>-\<inter> (X-interior(A,T))" 
    using Boundary_def topZF_3_L11 by simp
  moreover from A2 have 
    "A\<^sup>-\<inter> (X-interior(A,T)) = A\<^sup>- - interior(A,T)" 
    using topZF_3_L10 by auto
  ultimately show ?thesis by simp
qed

text{*If a set $A$ is contained in a closed set $B$, then the closure of $A$ 
is contained in $B$.*}

lemma (in top_space1) topZF_3_L13: 
  assumes A1: "B {is closed}" "A\<subseteq>B"
  shows "A\<^sup>-\<subseteq>B"
proof -;
  from A1 have "B\<in>Pow(X)" using IsClosed_def by simp;
  with A1 show ?thesis using ClosedCovers_def Closure_def by auto;
qed;

text{*If two open sets are disjoint, then we can close one of them 
and they will still be disjoint.*}

lemma (in top_space1) topZF_3_L14:
  assumes A1: "U\<in>T" and A2:"V\<in>T" and A3:"U \<inter> V = 0"
  shows "U\<^sup>- \<inter> V =0"
proof -;
  from A1 topSpaceAssum have T1:"U\<subseteq>X" using isAtopologyOn_def by auto;
  with A3 have "U \<subseteq> X-V" using Diff_contains by simp;
  moreover from A2 have "X-V {is closed}" using topZF_3_L8 by simp;
  ultimately have "U\<^sup>- -(X-V) = 0" 
    using topZF_3_L13 Diff_eq_0_iff by simp;
  moreover from T1 have "U\<^sup>- -(X-V) = U\<^sup>-\<inter>V " 
    using topZF_3_L4 IsClosed_def by auto;
  ultimately show ?thesis by simp;
qed;
end;