(*
    This file is a part of IsarMathLib - 
    a library of formalized mathematics written for Isabelle/Isar.

    Copyright (C) 2005  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

HIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{Real\_ZF.thy}*}

theory Real_ZF = EquivClass1 + Fol1 + Finite1 + Group_ZF:

text{*The goal of the Real\_ZF series of theory files is to provide 
a contruction of the set of 
  real numbers. There are several ways construct real numbers. 
  Most common start from the rational numbers and use Dedekind cuts 
  or Cauchy sequences. Following
  our desire for adventure we will use neither 
  of these and instead formalize a 
  construction devised by Norbert A'Campo in 2003 (look for 
  "A natural construction for the real numbers" on www.scroogle.org  or your 
  favourite search engine). This contruction magically makes the real 
  numbers directly from the group (!) of integers by defining reals as
  eqivalence classes of slopes. 
  Slopes are certain type of maps from the set of integers into itself.
  At the end of this theory file we show that the (would be) set of real
  numbers forms a group with addition.
*}

section{*Slopes and their equivalence*}

text{*We start from a definition of the set of slopes and the slope 
  equivalence, using a generic group instead of the group of integers.
  HomDiff is an acronym for "homomorphism difference". This is the expression
  $s(mn)(s(m)s(n))^{-1}$, or $s(m+n)-s(m)-s(n)$ in the additive notation.
  It is equal to the neutral element of the group if $s$ is a homomorphism.
  Slopes are defined as those maps $s:G\rightarrow G$ such that the 
  homomorphism difference takes only finite number of values on $G\times G$.
*}

constdefs
  HomDiff :: "[i,i,i,i]\<Rightarrow>i"
  "HomDiff(G,f,s,x) == 
  f` < s`(f`<fst(x),snd(x)>) , 
  (GroupInv(G,f)`(f`<s`(fst(x)),s`(snd(x))>)) >"
 
  SetOfSlopes :: "[i,i]\<Rightarrow>i"
  "SetOfSlopes(G,f) == 
  {s \<in> G\<rightarrow>G.{HomDiff(G,f,s,x). x \<in> G\<times>G } \<in> Fin(G)}"
  
  SlopesEquivRel :: "[i,i]\<Rightarrow>i"
  "SlopesEquivRel(G,f) == 
  {<s,r> \<in> SetOfSlopes(G,f) \<times> SetOfSlopes(G,f). 
  {f`<s`(n), GroupInv(G,f)`(r`(n))> . n\<in>G} \<in> Fin(G)}";

text{*This lemma provides more readable notation for the HomDiff
  definition. Not really intended to be used in proofs, but just to see the
  definition in the notation defined in the group0 locale.*}

lemma (in group0) Real_ZF_1_L1:
  shows "HomDiff(G,f,s,<m,n>) = s`(m\<cdot>n)\<cdot>(s`(m)\<cdot>s`(n))\<inverse>"
proof -;
  show ?thesis using HomDiff_def by simp;
qed;

(*
lemma (in group0) Real_ZF_1_L1:
  shows "SetOfSlopes(G,f) = 
  {s \<in> G\<rightarrow>G. { s`(m\<cdot>n)\<cdot>(s`(m)\<cdot>s`(n))\<inverse>. <m,n> \<in> G\<times>G} \<in> Fin(G\<times>G) }"
proof -;
  show ?thesis using SetOfSlopes_def by simp;
qed;
*)

text{*This lemma provides more readable notation for the SlopesEquivRel
  definition.*}

lemma (in group0) Real_ZF_1_L2:
  shows "SlopesEquivRel(G,f) = 
  {<s,r> \<in> SetOfSlopes(G,f) \<times> SetOfSlopes(G,f).
  {s`(n)\<cdot>(r`(n))\<inverse> . n \<in> G} \<in> Fin(G)}"
proof -;
  show ?thesis using SlopesEquivRel_def by simp;
qed;

text{*SlopesEquivRel is reflexive.*}

lemma (in group0) Real_ZF_1_L3: assumes A1: "G\<noteq>0"
  shows "refl(SetOfSlopes(G,f),SlopesEquivRel(G,f))"
proof -;
  have 
    "\<forall>s \<in> SetOfSlopes(G,f). {s`(n)\<cdot>(s`(n))\<inverse> . n \<in> G} = {e}"
  proof;
    fix s assume "s \<in> SetOfSlopes(G,f)"
    with A1 show 
      "{s`(n)\<cdot>(s`(n))\<inverse> . n \<in> G} = {e}"
      using SetOfSlopes_def apply_type group0_2_L6 by auto;
  qed;
  thus "refl(SetOfSlopes(G,f),SlopesEquivRel(G,f))"
    using group0_2_L2 SlopesEquivRel_def refl_def by simp; 
qed;

text{* SlopesEquivRel is symmetric.*}

lemma (in group0) Real_ZF_1_L4: 
  shows "sym(SlopesEquivRel(G,f))"
proof -;
  have "\<forall>x y. <x,y> \<in> SlopesEquivRel(G,f) \<longrightarrow>  
    <y,x> \<in> SlopesEquivRel(G,f)"
  proof (rule double_forall_implies);
    fix s r assume "<s,r> \<in> SlopesEquivRel(G,f)"
     then have 
      T1:"s \<in> SetOfSlopes(G,f)" and T2:"r \<in> SetOfSlopes(G,f)"
      and T3:"{s`(n)\<cdot>(r`(n))\<inverse> . n \<in> G} \<in> Fin(G)"
      using SlopesEquivRel_def by auto;
    from T1 T2 have T4:"s \<in> G\<rightarrow>G" and T5:"r \<in> G\<rightarrow>G"
      using SetOfSlopes_def by auto;
    from groupAsssum T3 have
      "GroupInv(G,f)``{s`(n)\<cdot>(r`(n))\<inverse>. n\<in>G} \<in> Fin(G)"
      using group0_2_T2 Finite1_L6A by blast;
    with T4 T5 T1 T2 show "<r,s> \<in> SlopesEquivRel(G,f)"
      using group0_2_L13 SlopesEquivRel_def
      by simp;
  qed;
  then show ?thesis using sym_def by simp;
qed;

text{*Proving transitivity of SlopesEquivRel is a little more involved. 
  We want to show that if the sets $\{s(n)\cdot (r(n))^{-1}\}_{n\in G}$ and 
   $\{r(n)\cdot (q(n))^{-1}\}_{n\in G}$ are finite, then the set
   $\{s(n)\cdot (Q(n))^{-1}\}_{n\in G}$ is also finite.
  The lemma below shows that 
  $s(n)q(n)^{-1} = (s(n)r(n)^{-1})(r(n)q(n)^{-1})$.
  Recall that in the group0 context $f:G\times G\rightarrow G$ is the group
  operation: $f(x,y)=x\cdot y$ .*} 

lemma (in group0) Real_ZF_1_L5:
  assumes A1:"s \<in> G\<rightarrow>G" "r \<in> G\<rightarrow>G" "q \<in> G\<rightarrow>G"
  and A4:"n\<in>G"
  shows 
  "s`(n)\<cdot>(q`(n))\<inverse> = (s`(n)\<cdot>(r`(n))\<inverse>) \<cdot> (r`(n)\<cdot>(q`(n))\<inverse>)"
proof -;
  from A1 A4 have  "s`(n)\<in>G" "r`(n)\<in>G" "q`(n)\<in>G"
    using apply_type by auto;
  then show ?thesis by (rule group0_2_L14);
qed;

text{*SlopesEquivRel is transitive.*}

lemma (in group0) Real_ZF_1_L6: 
assumes A1: 
  "\<langle>s, r\<rangle> \<in> SlopesEquivRel(G,f)" 
  "\<langle>r, q\<rangle> \<in> SlopesEquivRel(G,f)"
  shows "\<langle>s, q\<rangle> \<in> SlopesEquivRel(G,f)" 
proof -;
  from A1 have
    T1:"s \<in> G\<rightarrow>G" "r \<in> G\<rightarrow>G" "q \<in> G\<rightarrow>G"
    and T2:
    "{s`(n)\<cdot>(r`(n))\<inverse> . n \<in> G} \<in> Fin(G)"
    "{r`(n)\<cdot>(q`(n))\<inverse> . n \<in> G} \<in> Fin(G)"
    and T3:
    "s \<in> SetOfSlopes(G,f)" "q \<in> SetOfSlopes(G,f)"
    using SlopesEquivRel_def SetOfSlopes_def by auto;
  from groupAsssum have "f \<in> G\<times>G\<rightarrow>G"
    using IsAgroup_def IsAmonoid_def IsAssociative_def 
    by simp;
  with T2 have T4:
    "{(s`(n)\<cdot>(r`(n))\<inverse>)\<cdot>(r`(n)\<cdot>(q`(n))\<inverse>) . n \<in> G} \<in> Fin(G)"
    using Finite1_L15 by simp;
  from T1 have 
    "\<forall>n\<in>G. s`(n)\<cdot>(q`(n))\<inverse> = (s`(n)\<cdot>(r`(n))\<inverse>)\<cdot>(r`(n)\<cdot>(q`(n))\<inverse>)"
    using Real_ZF_1_L5 by blast;
  then have 
    "{s`(n)\<cdot>(q`(n))\<inverse>. n\<in>G} =
    {(s`(n)\<cdot>(r`(n))\<inverse>)\<cdot>(r`(n)\<cdot>(q`(n))\<inverse>) . n \<in> G}"
    by simp
  with T4 T3 show ?thesis using SlopesEquivRel_def
    by simp;
qed;
 
text{*SlopesEquivRel is an equivalence relation on the set of slopes.*}

theorem (in group0) Real_ZF_1_T1: assumes A1: "G\<noteq>0"
  shows "equiv(SetOfSlopes(G,f),SlopesEquivRel(G,f))"
proof -;
  let ?r = "SlopesEquivRel(G,f)"
  have 
    "\<forall>x y z. (\<langle>x, y\<rangle> \<in> ?r  \<and>  \<langle>y, z\<rangle> \<in> ?r \<longrightarrow> \<langle>x, z\<rangle> \<in> ?r)"
    using Real_ZF_1_L6 by blast;
  then have "trans(SlopesEquivRel(G,f))"
     using Fol1_L2 by blast;
   with A1 show ?thesis using
      Real_ZF_1_L3 Real_ZF_1_L4 SlopesEquivRel_def equiv_def
     by auto;
qed

section{*The first operation on slopes*}

text{*Next we define the set of slope classes. 
  This will become the set of real
   numbers when we replace the generic group with the addidtive group of 
  integers. We also define an operation on slopes (as a meta-function) in the 
  natural way as induced by the group operation. This is called SlopeMetaOp1.
  SlopeOp1 is the ZF-function corresponding to SlopeMetaOp1
   SlopeClassOp1 is the projection of SlopeOp1
  on the classes of slopes (the quotient space). 
  This operation will become the addition of real numbers one 
  day.
  The future SlopeMetaOp2 will be the composition of slopes and 
  will be turned into multiplication of real numbers.
 *}

constdefs

  SlopeClasses :: "[i,i]\<Rightarrow>i"
  "SlopeClasses(G,f) == (SetOfSlopes(G,f)//SlopesEquivRel(G,f))"

  SlopeMetaOp1 :: "[i,i,i,i]\<Rightarrow>i"
  "SlopeMetaOp1(G,f,s,r) == {<n,m> \<in> G\<times>G. f`<s`(n),r`(n)> = m}"

  SlopeOp1 :: "[i,i]\<Rightarrow>i"
  "SlopeOp1(G,f) ==  
  {<c,d> \<in> (SetOfSlopes(G,f)\<times> SetOfSlopes(G,f))\<times> SetOfSlopes(G,f).
  SlopeMetaOp1(G,f,fst(c),snd(c)) = d}"

  SlopeClassOp1 :: "[i,i]\<Rightarrow>i"
  "SlopeClassOp1(G,f) == 
  ProjFun2(SetOfSlopes(G,f),SlopesEquivRel(G,f),SlopeOp1(G,f))"

  ZeroSlope :: "[i,i]\<Rightarrow>i"
  "ZeroSlope(G,f) == ConstantFunction(G,TheNeutralElement(G,f))"

text{*The result of the first slope operation is a function that
  maps $G$ into itself.*}

lemma(in group0) Real_ZF_2_L1: 
  assumes A1:"s \<in> G\<rightarrow>G" "r \<in> G\<rightarrow>G"
  shows "SlopeMetaOp1(G,f,s,r) \<in> G\<rightarrow>G"
proof -;
  from A1 have 
    "\<forall>n\<in>G. s`(n)\<cdot>r`(n) \<in> G" using
    apply_type group0_2_L1 monoid0.group0_1_L1 by simp;
  then show ?thesis using SlopeMetaOp1_def func1_1_L11A  
    by simp;
qed;

text{*SlopeMetaOp1 multiplies slopes.*}

lemma (in group0) Real_ZF_2_L2:
  assumes A1:"s \<in> SetOfSlopes(G,f)" "r \<in> SetOfSlopes(G,f)"
  and A2:"p = SlopeMetaOp1(G,f,s,r)"
  and A3:"m\<in>G"
  shows "p`(m) = s`(m)\<cdot>r`(m)"
proof -;
  from A1 have "s \<in> G\<rightarrow>G" "r \<in> G\<rightarrow>G" 
    using SetOfSlopes_def by auto;
  with A3 A2 show ?thesis 
    using Real_ZF_2_L1 SlopeMetaOp1_def func1_1_L11B 
    by simp
qed;

text{*HomDiff is a homomorphism on slopes with SlopeMetaOp1.*}

lemma (in group0) Real_ZF_2_L3:
  assumes A1:"s \<in> SetOfSlopes(G,f)" "r \<in> SetOfSlopes(G,f)"
  and A2:"p = SlopeMetaOp1(G,f,s,r)"
  and A3:"x \<in> G\<times>G"
  and A4:"f {is commutative on} G"
  shows "HomDiff(G,f,p,x) =  HomDiff(G,f,s,x)\<cdot> HomDiff(G,f,r,x)"
proof -;
  from A3 obtain m n where D1:"x = <m,n>" by auto;
  with A3 have T0: "m\<in>G" "n\<in>G" by auto;
  then have T1:"m\<cdot>n \<in> G"
    using group0_2_L1 monoid0.group0_1_L1 by simp;
  with A1 T0 have T2:
    "s`(m)\<in>G" "s`(n)\<in>G" "r`(m)\<in>G" "r`(n)\<in>G" "s`(m\<cdot>n)\<in>G" "r`(m\<cdot>n)\<in>G"
    using SetOfSlopes_def apply_type by auto;
  from A1 A2 have T3:"p \<in> G\<rightarrow>G" 
    using SetOfSlopes_def Real_ZF_2_L1 by simp;
  from D1 have
    "HomDiff(G,f,p,x) = p`(m\<cdot>n)\<cdot>(p`(m)\<cdot>p`(n))\<inverse>"
    using Real_ZF_1_L1 by simp;
  also from T3 T0 have
     "p`(m\<cdot>n)\<cdot>(p`(m)\<cdot>p`(n))\<inverse> = p`(m\<cdot>n)\<cdot>((p`(n))\<inverse>\<cdot>(p`(m))\<inverse>)"
    using apply_type group0_2_L10 by simp;
  also from A1 A2 T0 T1 have 
    "p`(m\<cdot>n)\<cdot>((p`(n))\<inverse>\<cdot>(p`(m))\<inverse>) =
    (s`(m\<cdot>n)\<cdot>r`(m\<cdot>n))\<cdot>((s`(n)\<cdot>r`(n))\<inverse>\<cdot>(s`(m)\<cdot>r`(m))\<inverse>)"
    using Real_ZF_2_L2 by simp;
  also from A4 T2 have
    "(s`(m\<cdot>n)\<cdot>r`(m\<cdot>n))\<cdot>((s`(n)\<cdot>r`(n))\<inverse>\<cdot>(s`(m)\<cdot>r`(m))\<inverse>) =
    (s`(m\<cdot>n)\<cdot>(s`(m)\<cdot>s`(n))\<inverse>)\<cdot>(r`(m\<cdot>n)\<cdot>(r`(m)\<cdot>r`(n))\<inverse>)"
    using group0_4_L3 by simp;
  also have 
    "(s`(m\<cdot>n)\<cdot>(s`(m)\<cdot>s`(n))\<inverse>)\<cdot>(r`(m\<cdot>n)\<cdot>(r`(m)\<cdot>r`(n))\<inverse>) =
    HomDiff(G,f,s,<m,n>)\<cdot> HomDiff(G,f,r,<m,n>)"
    using Real_ZF_1_L1 by simp;
  also from D1 have 
     "HomDiff(G,f,s,<m,n>)\<cdot> HomDiff(G,f,r,<m,n>) =
     HomDiff(G,f,s,x)\<cdot> HomDiff(G,f,r,x)"
    by simp;
  finally show ?thesis by simp;
qed;
  
text{*The result of the first slope meta-operation is a slope.*}

lemma(in group0) Real_ZF_2_L4: 
  (*assumes A1:"s \<in> SetOfSlopes(G,f)" "r \<in> SetOfSlopes(G,f)"*)
  assumes A1: "c \<in> SetOfSlopes(G,f)\<times>SetOfSlopes(G,f)"
  and A2:"f {is commutative on} G"
  shows "SlopeMetaOp1(G,f,fst(c),snd(c)) \<in> SetOfSlopes(G,f)"
proof -;
  def D1: s == "fst(c)"
  def D2: r == "snd(c)"
  from A1 D1 D2 have A3: 
    "s \<in> SetOfSlopes(G,f)" "r \<in> SetOfSlopes(G,f)"
    by auto;
  let ?p = "SlopeMetaOp1(G,f,s,r)"
  from A3 have T1:"?p \<in> G\<rightarrow>G"
    using SetOfSlopes_def Real_ZF_2_L1 by simp;
  moreover have 
    "{HomDiff(G,f,?p,x). x \<in> G\<times>G} \<in> Fin(G)"
  proof -;
    from A3 have 
      "{HomDiff(G,f,s,x). x \<in> G\<times>G } \<in> Fin(G)" 
      "{HomDiff(G,f,r,x). x \<in> G\<times>G } \<in> Fin(G)"
      using SetOfSlopes_def by auto;
    moreover from groupAsssum have "f \<in> G\<times>G\<rightarrow>G"
      using IsAgroup_def IsAmonoid_def IsAssociative_def 
      by simp;
    ultimately have 
      "{HomDiff(G,f,s,x)\<cdot>HomDiff(G,f,r,x). x \<in> G\<times>G } \<in> Fin(G)"
      using Finite1_L15 by simp;
    moreover from A3 A2 have
      "{HomDiff(G,f,s,x)\<cdot>HomDiff(G,f,r,x). x \<in> G\<times>G } =
      {HomDiff(G,f,?p,x). x \<in> G\<times>G}"
      using Real_ZF_2_L3 by simp; 
    ultimately show ?thesis by simp;
  qed;
  ultimately have
    "SlopeMetaOp1(G,f,s,r) \<in> SetOfSlopes(G,f)"
     using SetOfSlopes_def by simp;
  with D1 D2 show ?thesis by simp;
qed;

text{*The first slope operation is a (ZF) function of two variables
  on the set of slopes with values in the set of slopes.*}

lemma(in group0) Real_ZF_2_L5:
  assumes A1: "f {is commutative on} G"
  shows "SlopeOp1(G,f) \<in> 
  SetOfSlopes(G,f)\<times> SetOfSlopes(G,f) \<rightarrow> SetOfSlopes(G,f)"
proof -;
  from A1 have "\<forall>c\<in>SetOfSlopes(G,f)\<times> SetOfSlopes(G,f).
    SlopeMetaOp1(G,f,fst(c),snd(c)) \<in>  SetOfSlopes(G,f)"
    using Real_ZF_2_L4 by simp;
  then show ?thesis using SlopeOp1_def func1_1_L11A by simp;
qed;
  
  
text{*The first thing to prove is that SlopeOp1 can be projected to the 
  quotient space. In romantic math we need to show that the definition of 
  the projection does not depend on the choice of class representants. 
  Here we show that SlopeOp1 is Congruent2, as defined in EquivClass1.thy.*};

text{*Let's ease the pain and introduce some notation. 
  We inherit the notation and assumptions from the group0 context (locale) 
  and add some. Namely, we add the follwoing notation to the group0 
  proof context: $x\simeq y$ means that (the slopes) $x$ and $y$ are 
  equivalent in the SlopesEquivRel$(G,f)$ sense. $x\oplus y$ denotes the first
  slope operation: $(x\oplus y)(n)= x(n)\cdot y(n)$, where $\cdot$ is the 
  group operation. $\Delta $ will denote the homomorphism difference specific
  for the group (HomDiff$(G,f)$).Finally, $\sim$ is the inverse 
  (negative if the group is the group of integers) of slopes,
  $(\sim p)(n)= p(n)^{-1}$.*}

locale real0 = group0 +
  assumes isAbelian:"f {is commutative on} G"

  fixes sleq :: "[i,i]\<Rightarrow>o" (infix "\<simeq>" 65)
  defines sleq_def [simp]: "x \<simeq> y == <x,y> \<in> SlopesEquivRel(G,f)"
  
  fixes slop1 :: "[i,i]\<Rightarrow>i" (infixl "\<oplus>" 67)
  defines slop1_def [simp]: "x\<oplus>y == SlopeOp1(G,f)`<x,y>"

  fixes \<Delta> :: "[i,i]\<Rightarrow>i"
  defines \<Delta>_def [simp]: "\<Delta>(s,x) == HomDiff(G,f,s,x)"

  fixes neg :: "i\<Rightarrow>i" ("\<sim>_" [90] 91)
  defines neg_def [simp]: "\<sim>s == GroupInv(G,f) O s";

text{*The first slope operation multiplies slopes.*}

lemma(in real0) Real_ZF_2_L6: 
  assumes A1:"s \<in> SetOfSlopes(G,f)" "r \<in> SetOfSlopes(G,f)"
  and A2: "m\<in>G"
  shows "(s\<oplus>r)`(m) = s`(m)\<cdot>r`(m)"
proof -;
  from isAbelian A1 A2 show ?thesis
    using Real_ZF_2_L5 func1_1_L11B SlopeOp1_def Real_ZF_2_L2
    by simp;
qed;

text{*The result of the first slope operation is a slope.*}

lemma(in real0) Real_ZF_2_L7:
  assumes A1:"s \<in> SetOfSlopes(G,f)" "r \<in> SetOfSlopes(G,f)"
  shows "s\<oplus>r \<in> SetOfSlopes(G,f)"
proof -;
  from A1 isAbelian show ?thesis
    using  Real_ZF_2_L5 apply_type by simp;
qed;
  
text{*The first slope operation is congruent with respect to the slope 
  equivalence. This means that the equivalence class of the result of
  the slope operation does dont depend on the choice of slopes that represent 
  the slope classes.*}

text{*We first prove the essential condition of congruency as a separate 
  lemma.*}

lemma (in real0) Real_ZF_2_L8: 
  assumes A1:"s1 \<simeq> s2" and A2:"p1\<simeq>p2"
  shows "s1\<oplus>p1 \<simeq> s2\<oplus>p2"
proof -
  from A1 A2 have 
    T1:"s1 \<in> SetOfSlopes(G,f)" "s2 \<in> SetOfSlopes(G,f)"
    "p1 \<in> SetOfSlopes(G,f)" "p2 \<in> SetOfSlopes(G,f)"
    using SlopesEquivRel_def by auto;
  then have T2:
    "\<forall>n\<in>G. s1`(n)\<in>G \<and> s2`(n)\<in>G \<and> p1`(n)\<in>G \<and> p2`(n)\<in>G"
    using SetOfSlopes_def apply_type by auto;
  from T1 have 
    "s1\<oplus>p1 \<in> SetOfSlopes(G,f)" "s2\<oplus>p2 \<in> SetOfSlopes(G,f)"
    using Real_ZF_2_L7 by auto;
  moreover have 
    "{(s1\<oplus>p1)`(n)\<cdot>((s2\<oplus>p2)`(n))\<inverse> . n \<in> G} \<in> Fin(G)" 
    proof -;
      from A1 A2 have
	"{s1`(n)\<cdot>(s2`(n))\<inverse>. n\<in>G} \<in> Fin(G)"
	"{p1`(n)\<cdot>(p2`(n))\<inverse>. n\<in>G} \<in> Fin(G)"
	using SlopesEquivRel_def by auto;
      moreover from groupAsssum have "f \<in> G\<times>G\<rightarrow>G"
	using IsAgroup_def IsAmonoid_def IsAssociative_def 
	by simp;
      ultimately have 
	"{(s1`(n)\<cdot>(s2`(n))\<inverse>)\<cdot>(p1`(n)\<cdot>(p2`(n))\<inverse>). n\<in>G}\<in>Fin(G)"
	using Finite1_L15 by simp;
      moreover from T1 isAbelian T2 have
	"{(s1\<oplus>p1)`(n)\<cdot>((s2\<oplus>p2)`(n))\<inverse> . n \<in> G} =
	{(s1`(n)\<cdot>(s2`(n))\<inverse>)\<cdot>(p1`(n)\<cdot>(p2`(n))\<inverse>). n\<in>G}"
	using Real_ZF_2_L6 group0_4_L1
	by simp;
      ultimately show ?thesis by simp;
    qed;
  ultimately show ?thesis using SlopesEquivRel_def by simp;
qed;

text{*The first slope operation is congruent with respect to the slope 
  equivalence. *}

theorem (in real0) Real_ZF_2_T1: 
  shows "Congruent2(SlopesEquivRel(G,f),SlopeOp1(G,f))"
proof -
  show ?thesis using Congruent2_def Real_ZF_2_L8
    by simp;
qed;

text{*The first slope operation is associative. *}

lemma(in real0) Real_ZF_2_L9:
  assumes A1:"s \<in> SetOfSlopes(G,f)" "r \<in> SetOfSlopes(G,f)"
  "q \<in> SetOfSlopes(G,f)"
  shows "s\<oplus>r\<oplus>q = s\<oplus>(r\<oplus>q)"
proof -;
  from A1 have
    "s\<oplus>r\<oplus>q \<in> G\<rightarrow>G" "s\<oplus>(r\<oplus>q) \<in> G\<rightarrow>G"
    using Real_ZF_2_L7 SetOfSlopes_def by auto;
  moreover have 
    "\<forall>n\<in>G. (s\<oplus>r\<oplus>q)`(n) = (s\<oplus>(r\<oplus>q))`(n)"
  proof;
    fix n assume A2:"n\<in>G"
    with A1 have T1:
      "s`(n)\<in>G" "r`(n)\<in>G" "q`(n)\<in>G"
       using SetOfSlopes_def apply_type by auto;
    from A1 A2 have
      "(s\<oplus>r\<oplus>q)`(n) = s`(n)\<cdot> r`(n)\<cdot>q`(n)"
      using Real_ZF_2_L7 Real_ZF_2_L6 by simp;
    also from T1 have 
       "s`(n)\<cdot> r`(n)\<cdot>q`(n) = s`(n)\<cdot> (r`(n)\<cdot>q`(n))"
      using group0_2_L3 by simp;
    also from A1 A2 have 
      "s`(n)\<cdot> (r`(n)\<cdot>q`(n)) = (s\<oplus>(r\<oplus>q))`(n)"
      using Real_ZF_2_L7 Real_ZF_2_L6 by simp;
    finally show "(s\<oplus>r\<oplus>q)`(n) = (s\<oplus>(r\<oplus>q))`(n)"
      by simp;
  qed;
  ultimately show ?thesis using fun_extension_iff
    by simp;
qed;

text{*The first slope class operation is associative 
  on the set of slope classes.*}

lemma(in real0) Real_ZF_2_L10: assumes A1:"G\<noteq>0" shows 
  "SlopeClassOp1(G,f) {is associative on} SlopeClasses(G,f)"
proof -;
  from isAbelian have 
    "SlopeOp1(G,f) {is associative on} SetOfSlopes(G,f)"
    using Real_ZF_2_L5 Real_ZF_2_L9 IsAssociative_def
    by simp;
  with A1 show ?thesis using
    Real_ZF_1_T1 Real_ZF_2_T1 EquivClass_2_T2 
    SlopeClassOp1_def SlopeClasses_def
    by simp;
qed;
  
text{*The ZeroSlope is a slope.*}

lemma(in real0) Real_ZF_2_L11: assumes A1: "G\<noteq>0"
  shows "ZeroSlope(G,f) \<in> SetOfSlopes(G,f)"
proof -;
  let ?z = "ZeroSlope(G,f)"
  have "\<forall>x\<in>G\<times>G. HomDiff(G,f,?z,x) = e"
  proof;
    fix x assume "x \<in> G \<times> G"
    then obtain n m where D1:"n\<in>G" "m\<in>G" "x=<m,n>"
      by auto;
    then have T1:"m\<cdot>n\<in>G" 
      using group0_2_L1 monoid0.group0_1_L1 by simp;
    from D1 have 
      "HomDiff(G,f,?z,x) = ?z`(m\<cdot>n)\<cdot>(?z`(m)\<cdot>?z`(n))\<inverse>"
      using HomDiff_def by simp;
    moreover from T1 D1 have 
      "?z`(m\<cdot>n) = e" "?z`(m) = e" "?z`(n) = e"
      using ZeroSlope_def func1_5_L2 by auto;
    ultimately show "HomDiff(G,f,?z,x) = e"
      using group0_2_L2 group0_2_L7A
      by simp;
  qed;
  with A1 have "{HomDiff(G,f,?z,x). x \<in> G\<times>G } = {e}"
    by blast;
  then have "{HomDiff(G,f,?z,x). x \<in> G\<times>G } \<in> Fin(G)"
    using group0_2_L2 Finite1_L16 by simp;
  moreover have "?z \<in> G\<rightarrow>G"
    using ZeroSlope_def func1_5_L1 group0_2_L2
    by simp;
  ultimately show ?thesis using SetOfSlopes_def by simp;
qed;

 
text{*The the zero slope is neutral for the first slope operation.*}

lemma (in real0) Real_ZF_2_L12: 
  assumes A1:"s\<in>SetOfSlopes(G,f)" and A2:"G\<noteq>0"
  shows "s\<oplus>ZeroSlope(G,f) = s" "ZeroSlope(G,f)\<oplus>s = s"
proof -;
  let ?z = "ZeroSlope(G,f)"
  from A1 A2 have T1:
    "s\<oplus>?z \<in> G\<rightarrow>G" "?z\<oplus>s \<in> G\<rightarrow>G" "s \<in> G\<rightarrow>G"
    using Real_ZF_2_L11 Real_ZF_2_L7 SetOfSlopes_def
    by auto;
  have "\<forall>n\<in>G. (s\<oplus>?z)`(n) = s`(n) \<and> (?z\<oplus>s)`(n) = s`(n)"
  proof;
    fix n assume A3:"n\<in>G"
    with A1 have
      "(s\<oplus>?z)`(n) = s`(n)\<cdot>?z`(n)"
      "(?z\<oplus>s)`(n) = ?z`(n)\<cdot>s`(n)"
      using Real_ZF_2_L6 Real_ZF_2_L11 by auto;
    moreover from T1 A3 have
      "?z`(n) = e" "s`(n)\<in>G"
      using func1_5_L2 ZeroSlope_def apply_type
      by auto;
    ultimately show 
      "(s\<oplus>?z)`(n) = s`(n) \<and> (?z\<oplus>s)`(n) = s`(n)"
      using group0_2_L2 by simp;
  qed
  hence
    "\<forall>n\<in>G. (s\<oplus>?z)`(n) = s`(n)"
    "\<forall>n\<in>G. (?z\<oplus>s)`(n) = s`(n)"
    by auto;
  with T1 show 
    "s\<oplus>?z = s" "?z\<oplus>s = s"
    using fun_extension_iff by auto;
qed;

text{*The class of the zero slope is neutral for the slope class operation.*}

lemma (in real0) Real_ZF_2_L13: 
  assumes A1:"G\<noteq>0" 
  and A2:"r = SlopesEquivRel(G,f)" and A3:"Z = r``{ZeroSlope(G,f)}" 
  shows "Z \<in> SlopeClasses(G,f) \<and> 
  (\<forall> c \<in>SlopeClasses(G,f). 
  SlopeClassOp1(G,f)` <c,Z> = c \<and> 
  SlopeClassOp1(G,f)` <Z,c> =c)";
proof;
  from A1 A2 A3 show T1:"Z \<in> SlopeClasses(G,f)"
    using Real_ZF_2_L11 SlopeClasses_def quotientI
    by simp;
  show "\<forall> c \<in>SlopeClasses(G,f). 
    SlopeClassOp1(G,f)` <c,Z> = c \<and> 
    SlopeClassOp1(G,f)` <Z,c> =c"
  proof;
    fix c assume "c \<in> SlopeClasses(G,f)"
    with A2 obtain s where 
      "s \<in> SetOfSlopes(G,f)" "c = r``{s}"
      using SlopeClasses_def quotient_def 
      by auto;
    with A1 A2 A3 show 
      "SlopeClassOp1(G,f)` <c,Z> = c \<and> 
      SlopeClassOp1(G,f)` <Z,c> = c"
      using Real_ZF_1_T1 Real_ZF_2_T1 Real_ZF_2_L11
	EquivClass_1_L10 SlopeClassOp1_def Real_ZF_2_L12
      by simp;
  qed;
qed;

text{*Slope classes form a monoid with the first slope operation.*}

theorem (in real0) Real_ZF_2_T2: assumes A1:"G\<noteq>0"
  shows "IsAmonoid(SlopeClasses(G,f),SlopeClassOp1(G,f))"
proof -;
  from A1 have
    "\<exists>Z \<in> SlopeClasses(G,f). (\<forall> c \<in>SlopeClasses(G,f). 
    SlopeClassOp1(G,f)` <c,Z> = c \<and> SlopeClassOp1(G,f)` <Z,c> =c)"
    using Real_ZF_2_L13 by auto;
  with A1 show ?thesis
    using Real_ZF_2_L10 IsAmonoid_def by auto;
qed;

text{*The zero slope class is the neutral element in the Slope classes 
  monoid.*}

lemma (in real0) Real_ZF_2_L13A:
  assumes A1:"G\<noteq>0" 
  and A2:"r = SlopesEquivRel(G,f)" and A3:"Z = r``{ZeroSlope(G,f)}" 
  shows "Z = TheNeutralElement(SlopeClasses(G,f),SlopeClassOp1(G,f))"
proof -;
  from A1 have "monoid0(SlopeClasses(G,f),SlopeClassOp1(G,f))"
    using Real_ZF_2_T2 monoid0_def by simp;
  moreover from A1 A2 A3 have
    "Z \<in> SlopeClasses(G,f) \<and> 
    (\<forall> c \<in>SlopeClasses(G,f). 
    SlopeClassOp1(G,f)` <c,Z> = c \<and> 
    SlopeClassOp1(G,f)` <Z,c> =c)"
    using Real_ZF_2_L13 by simp;
  ultimately show ?thesis 
    using monoid0.group0_1_L4 by auto;
qed;


text{*To get the group structure we need to show that we can find a (right)
  inverse for every element. The obvious candidate is the class of the slope
  inverse.*}

text{*The value of the slope inverse is the inverse of the value.*}

lemma (in real0) Real_ZF_2_L13: 
  assumes A1:"s \<in> SetOfSlopes(G,f)" and A2:"n\<in>G"
  shows "(\<sim>s)`(n) = (s`(n))\<inverse>"
proof -;
  from A1 A2 show ?thesis 
    using SetOfSlopes_def comp_fun_apply 
    by auto;
qed

text{*The terms in the homomorphism difference are in the group.*}

lemma (in real0) Real_ZF_2_L14:  
  assumes A1:"s \<in> SetOfSlopes(G,f)" and A2:"x\<in>G\<times>G"
  shows "fst(x)\<cdot>snd(x) \<in> G" 
  "s`(fst(x)\<cdot>snd(x)) \<in> G"
  "s`(fst(x)) \<in> G" "s`(snd(x)) \<in> G"
proof -;
  let ?m = "fst(x)" 
  let ?n = "snd(x)";
  from A2 show "?m\<cdot>?n \<in> G"
    using group0_2_L1 monoid0.group0_1_L1
    by simp;
  with A1 A2 show
    "s`(?m\<cdot>?n) \<in> G"
    "s`(?m) \<in> G" "s`(?n) \<in> G"
    using SetOfSlopes_def apply_type by auto;
qed;

text{*Homomorphism difference commutes with the slope inverse.*}
  
lemma (in real0) Real_ZF_2_L15:  
  assumes A1:"s \<in> SetOfSlopes(G,f)" and A2:"x\<in>G\<times>G"
  shows "\<Delta>(\<sim>s,x) = (\<Delta>(s,x))\<inverse>"
proof -;
  from A1 A2 isAbelian show ?thesis
    using  Real_ZF_2_L14 HomDiff_def 
      Real_ZF_2_L13 group0_4_L4
    by simp;
qed;

text{*The value of homomorphism difference is in the group (how did we get
  so far without knowing that?).*}

lemma (in real0) Real_ZF_2_L16: 
  assumes A1:"s \<in> SetOfSlopes(G,f)"and A2:"x\<in>G\<times>G"
  shows "\<Delta>(s,x) \<in> G"
proof -;
  from A1 A2 show ?thesis 
    using HomDiff_def Real_ZF_2_L14 group0_2_L5
      group0_2_L1 monoid0.group0_1_L1 by simp;
qed;


text{*The inverse of a slope maps the group into itself.*}

lemma (in real0) Real_ZF_2_L17: 
  assumes A1:"s \<in> SetOfSlopes(G,f)"
  shows "\<sim>s : G\<rightarrow>G"
proof -;
  from A1 groupAsssum show ?thesis
    using SetOfSlopes_def group0_2_T2 comp_fun
    by auto;
qed;
  
text{*The inverse of a slope is a slope.*}

lemma (in real0) Real_ZF_2_L18: 
  assumes A1:"s \<in> SetOfSlopes(G,f)"
  shows "\<sim>s \<in> SetOfSlopes(G,f)"
proof -;
  from A1 have T1:"{\<Delta>(s,x). x \<in> G\<times>G} \<in> Fin(G)"
    using SetOfSlopes_def by simp;
  with groupAsssum have
    "GroupInv(G,f)``{\<Delta>(s,x). x \<in> G\<times>G} \<in> Fin(G)"
    using group0_2_T2 Finite1_L6A by blast;
  moreover have
     "GroupInv(G,f)``{\<Delta>(s,x). x \<in> G\<times>G} =
    {(\<Delta>(s,x))\<inverse>. x \<in> G\<times>G}"
  proof -;
    from groupAsssum have 
      "GroupInv(G,f) : G\<rightarrow>G"
      using group0_2_T2 by simp;
    moreover from A1 have 
      "\<forall>x\<in>G\<times>G. \<Delta>(s,x)\<in>G"
      using Real_ZF_2_L16 by simp;
    ultimately show ?thesis 
    using func1_1_L17 by simp;
  qed;
  ultimately have  
    "{(\<Delta>(s,x))\<inverse>. x \<in> G\<times>G} \<in> Fin(G)"
    by simp;
  with A1 show ?thesis 
    using Real_ZF_2_L15 Real_ZF_2_L17 SetOfSlopes_def 
    by simp;
qed;

text{*The inverse slope is an inverse in the set of slopes.*}

lemma (in real0) Real_ZF_2_L19:
  assumes A1:"s \<in> SetOfSlopes(G,f)" and A2:"G\<noteq>0"
  shows "s\<oplus>(\<sim>s) = ZeroSlope(G,f)"
proof -;
  from A1 A2 have 
    "s\<oplus>(\<sim>s) : G\<rightarrow>G" "ZeroSlope(G,f) :  G\<rightarrow>G"
    using Real_ZF_2_L18 Real_ZF_2_L7 
      Real_ZF_2_L11 SetOfSlopes_def
    by auto;
  moreover have
    "\<forall>n\<in>G. (s\<oplus>(\<sim>s))`(n) = ZeroSlope(G,f)`(n)"
  proof
    fix n assume A3:"n\<in>G"
    with A1 have "s`(n)\<in>G" using
      SetOfSlopes_def apply_type by auto;
    with A3 A1 show 
      "(s\<oplus>(\<sim>s))`(n) = ZeroSlope(G,f)`(n)"
      using Real_ZF_2_L18 Real_ZF_2_L6  Real_ZF_2_L13
	group0_2_L6 func1_5_L2 ZeroSlope_def
      by simp;
  qed;
  ultimately show ?thesis using fun_extension_iff
    by simp;
qed;

text{*The class of the inverse slope is the inverse of the slope class.*}

lemma (in real0) Real_ZF_2_L20: 
  assumes A1:"s \<in> SetOfSlopes(G,f)" and A2:"G\<noteq>0" 
  and A3:"F = SlopeClassOp1(G,f)" 
  and A4:"r = SlopesEquivRel(G,f)"
  shows "F`<r``{s},r``{\<sim>s}> = r``{ZeroSlope(G,f)}"
proof -;
  from A1 A2 A3 A4 show ?thesis 
     using Real_ZF_1_T1 Real_ZF_2_T1 Real_ZF_2_L18 
       EquivClass_1_L10 SlopeClassOp1_def Real_ZF_2_L19
     by simp;
qed;

text{*Every slope class has an inverse.*}

lemma (in real0) Real_ZF_2_L21:
  assumes A1:"c \<in> SlopeClasses(G,f)" and A2:"G\<noteq>0"
  and A3:"F = SlopeClassOp1(G,f)" 
  and A4:"r = SlopesEquivRel(G,f)"
  and A5:"Z = TheNeutralElement(SlopeClasses(G,f),SlopeClassOp1(G,f))"
  shows "\<exists>b\<in>SlopeClasses(G,f). F`<c,b> = Z"
proof -
  from A1 A4 obtain s where 
    D1: "s \<in> SetOfSlopes(G,f)" "c = r``{s}"
    using SlopeClasses_def quotient_def 
    by auto;
  let ?b="r``{\<sim>s}"
  from D1 A4 have "?b \<in> SlopeClasses(G,f)"
    using Real_ZF_2_L18 SlopeClasses_def quotientI
    by simp;
  moreover from D1 A2 A3 A4 A5 have
    "F`<c,?b> = Z"
    using Real_ZF_2_L20 Real_ZF_2_L13A by simp;
  ultimately show ?thesis by auto;
qed;

theorem (in real0) Real_ZF_2_T3: assumes A1:"G\<noteq>0"
  shows "IsAgroup(SlopeClasses(G,f),SlopeClassOp1(G,f))"
proof -;
  from A1 show ?thesis using 
    Real_ZF_2_T2 Real_ZF_2_L21 IsAgroup_def
    by simp;
qed;
  
 
 
    
  
   
  
    
    
 
    
  
    
  
 
 
  
  
    
end