(*    This file is a part of IsarMathLib - 
    a library of formalized mathematics for Isabelle/Isar.

    Copyright (C) 2005  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{Int\_ZF\_2.thy}*}

theory Int_ZF_2 imports Int_ZF_1 Group_ZF_3 

begin

text{*In this theory file we consider the properties of integers that are 
  needed for the real numbers construction in @{text "Real_ZF_x.thy"} series.*}

section{*Slopes*}

text{*In this section we study basic properties of slopes - the integer 
  almost homomorphisms. 
  The general definition of an almost homomorphism $f$ on a group $G$ 
  written in additive notation requires the set 
  $\{f(m+n) - f(m) - f(n): m,n\in G\}$ to be finite.
  The goal of this section is
  to establish a definition that is equivalent for integers: 
  that for all integer $m,n$ we have $|f(m+n) - f(m) - f(n)| \leq L$ for
  some $L$.  *}

text{*First we extend the standard notation for integers with notation related
  to slopes. We define slopes as almost homomorphisms on the additive group of
  integers. The set of slopes is denoted @{text "\<S>"}. We also define "positive" 
  slopes as those that take infinite number of positive values on positive integers.
  We write @{text "\<delta>(s,m,n)"} to denote the homomorphism 
  difference of $s$ at $m,n$ (i.e the expression $s(m+n) - s(m) - s(n)$).
  We denote @{text "max\<delta>(s)"} the maximum absolute value of homomorphism 
  difference of $s$ as $m,n$ range over integers.
  If $s$ is a slope, 
  then the set of homomorphism differences is finite and 
  this maximum exists.
  In @{text "Group_ZF_3.thy"} we define the equivalence relation on 
  almost homomorphisms using the notion of a quotient group relation
  and use "@{text "\<approx>"}" to denote it. As here this symbol seems to be hogged
  by standard Isabelle, we will use "@{text "\<sim>"}" instead "@{text "\<approx>"}".
  We show in this section that $s\sim r$ iff for some $L$ we have $|s(m)-r(m)| \leq L$
  for all integer $m$.
  The @{text "\<circ>"} symbol denotes the second operation on almost homomorphisms
  (see @{text "Group_ZF_3.thy"} for definition), 
  defined for the group of integers.  In short @{text "\<circ>"} 
  is the composition of slopes.*}

locale int1 = int0 +

  fixes slopes ("\<S>" )
  defines slopes_def [simp]: "\<S> \<equiv> AlmostHoms(\<int>,IntegerAddition)"

  fixes posslopes ("\<S>\<^isub>+")
  defines posslopes_def [simp]: "\<S>\<^isub>+ \<equiv> {s\<in>\<S>. \<int>\<^sub>+ \<inter> s``(\<int>\<^sub>+) \<notin> Fin(\<int>)}"
 
  fixes \<delta> 
  defines \<delta>_def [simp] : "\<delta>(s,m,n) \<equiv> s`(m\<ra>n)\<rs>s`(m)\<rs>s`(n)"

  fixes maxhomdiff ("max\<delta>" )
  defines maxhomdiff_def [simp] : 
  "max\<delta>(s) \<equiv> Maximum(IntegerOrder,{abs(\<delta>(s,m,n)). <m,n> \<in> \<int>\<times>\<int>})"

  fixes AlEq :: "[i,i]\<Rightarrow>o" (infix "\<sim>" 68)
  defines AlEq_def [simp]: "s \<sim> r \<equiv> 
  <s,r> \<in> QuotientGroupRel(\<S>,AlHomOp1(\<int>,IntegerAddition),FinRangeFunctions(\<int>,\<int>))"

  fixes slope_comp (infix "\<circ>" 70)
  defines slope_comp_def [simp]: "s \<circ> r \<equiv>  AlHomOp2(\<int>,IntegerAddition)`<s,r>"

text{*We can use theorems proven in @{text "group1"} context.*}

lemma (in int1) Int_ZF_2_1_L1: shows "group1(\<int>,IntegerAddition)"
  using Int_ZF_1_T2 group1_axioms.intro group1_def by simp;

text{*Type information related to the homomorphism difference expression.*}

lemma (in int1) Int_ZF_2_1_L2: assumes "f\<in>\<S>" and "n\<in>\<int>" "m\<in>\<int>"
  shows 
  "m\<ra>n \<in> \<int>"  
  "f`(m\<ra>n) \<in> \<int>"  
  "f`(m) \<in> \<int>"   "f`(n) \<in> \<int>"
  "f`(m) \<ra> f`(n) \<in> \<int>"  
  "HomDiff(\<int>,IntegerAddition,f,<m,n>) \<in> \<int>" 
  using prems Int_ZF_2_1_L1 group1.Group_ZF_3_2_L4A
  by auto;

text{*Type information related to the homomorphism difference expression.*}

lemma (in int1) Int_ZF_2_1_L2A: 
  assumes A1: "f:\<int>\<rightarrow>\<int>" and A2: "n\<in>\<int>" "m\<in>\<int>"
  shows 
  "m\<ra>n \<in> \<int>" 
  "f`(m\<ra>n) \<in> \<int>"   "f`(m) \<in> \<int>"   "f`(n) \<in> \<int>"
  "f`(m) \<ra> f`(n) \<in> \<int>" 
  "HomDiff(\<int>,IntegerAddition,f,<m,n>) \<in> \<int>"
proof -
  from A1 A2 show 
    "m\<ra>n \<in> \<int>" 
    "f`(m\<ra>n) \<in> \<int>" 
    "f`(m) \<in> \<int>" 
    "f`(n) \<in> \<int>"
    "f`(m) \<ra> f`(n) \<in> \<int>"
    using Int_ZF_1_1_L5 apply_funtype by auto;  
  then have "f`(m\<ra>n)\<rs>(f`(m)\<ra>f`(n)) \<in> \<int>"
    using Int_ZF_1_1_L5 by blast;
  then show "HomDiff(\<int>,IntegerAddition,f,<m,n>) \<in> \<int>"
    using HomDiff_def by simp;
qed;

text{*Slopes map integers into integers.*}

lemma (in int1) Int_ZF_2_1_L2B: 
  assumes A1: "f\<in>\<S>" and A2: "m\<in>\<int>" 
  shows "f`(m) \<in> \<int>"
proof -;
  from A1 have "f:\<int>\<rightarrow>\<int>" using AlmostHoms_def by simp
  with A2 show "f`(m)\<in>\<int>" using apply_funtype by simp;
qed;

text{*The homomorphism difference in multiplicative notation is defined as
  the expression $s(m\cdot n)\cdot(s(m)\cdot s(n))^{-1}$. The next lemma 
  shows that 
  in the additive notation used for integers the homomorphism 
  difference is $f(m+n) - f(m) - f(n)$ which we denote as @{text "\<delta>(f,m,n)"}.*}

lemma (in int1) Int_ZF_2_1_L3: 
  assumes "f:\<int>\<rightarrow>\<int>" and "m\<in>\<int>" "n\<in>\<int>"
  shows "HomDiff(\<int>,IntegerAddition,f,<m,n>) = \<delta>(f,m,n)"
  using prems Int_ZF_2_1_L2A Int_ZF_1_T2 group0.group0_4_L4A HomDiff_def
  by auto;

text{*The next lemma presents two ways the set of homomorphism differences
  can be written. *}

lemma (in int1) Int_ZF_2_1_L4: assumes A1: "f:\<int>\<rightarrow>\<int>"
  shows "{abs(HomDiff(\<int>,IntegerAddition,f,x)). x \<in> \<int>\<times>\<int>} =
  {abs(\<delta>(f,m,n)). <m,n> \<in> \<int>\<times>\<int>}"
proof -
  from A1 have "\<forall>m\<in>\<int>. \<forall>n\<in>\<int>. 
    abs(HomDiff(\<int>,IntegerAddition,f,<m,n>)) = abs(\<delta>(f,m,n))"
    using Int_ZF_2_1_L3 by simp;
  then show ?thesis by (rule ZF1_1_L4A);
qed;

text{*If $f$ maps integers into integers and 
  for all $m,n\in Z$ we have $|f(m+n) - f(m) - f(n)| \leq L$ for some $L$,
  then $f$ is a slope.*}

lemma (in int1) Int_ZF_2_1_L5: assumes A1: "f:\<int>\<rightarrow>\<int>"
  and A2: "\<forall>m\<in>\<int>.\<forall>n\<in>\<int>. abs(\<delta>(f,m,n)) \<lsq> L"
  shows "f\<in>\<S>"
proof -;
  let ?Abs = "AbsoluteValue(\<int>,IntegerAddition,IntegerOrder)"
  have "group3(\<int>,IntegerAddition,IntegerOrder)" 
    "IntegerOrder {is total on} \<int>"
    using Int_ZF_2_T1 by auto
  moreover from A1 A2 have 
    "\<forall>x\<in>\<int>\<times>\<int>. HomDiff(\<int>,IntegerAddition,f,x) \<in> \<int> \<and>
    \<langle>?Abs`(HomDiff(\<int>,IntegerAddition,f,x)),L \<rangle> \<in> IntegerOrder"
    using Int_ZF_2_1_L2A Int_ZF_2_1_L3 by auto;
  ultimately have 
    "IsBounded({HomDiff(\<int>,IntegerAddition,f,x). x\<in>\<int>\<times>\<int>},IntegerOrder)"
    by (rule group3.OrderedGroup_ZF_3_L9A);
  with A1 show "f \<in> \<S>" using Int_bounded_iff_fin AlmostHoms_def
    by simp;
qed;

text{*The absolute value of homomorphism difference 
  of a slope $s$ does not exceed @{text "max\<delta>(s)"}.*}

lemma (in int1) Int_ZF_2_1_L7: 
  assumes A1: "s\<in>\<S>" and A2: "n\<in>\<int>"  "m\<in>\<int>"
  shows 
  "abs(\<delta>(s,m,n)) \<lsq> max\<delta>(s)"   
  "\<delta>(s,m,n) \<in> \<int>"  "max\<delta>(s) \<in> \<int>"
proof -
  let ?A = "{abs(HomDiff(\<int>,IntegerAddition,s,x)). x\<in>\<int>\<times>\<int>}"
  let ?B = "{abs(\<delta>(s,m,n)). <m,n> \<in> \<int>\<times>\<int>}"
  let ?d = "abs(\<delta>(s,m,n))"
  have "IsLinOrder(\<int>,IntegerOrder)" using Int_ZF_2_T1
    by simp;
  moreover have "?A \<in> Fin(\<int>)" 
  proof -;
    have "\<forall>k\<in>\<int>. abs(k) \<in> \<int>" using Int_ZF_2_L14 by simp;
    moreover from A1 have 
      "{HomDiff(\<int>,IntegerAddition,s,x). x \<in> \<int>\<times>\<int>} \<in> Fin(\<int>)"
      using AlmostHoms_def by simp;
    ultimately show "?A \<in> Fin(\<int>)" by (rule Finite1_L6C);
  qed;
  moreover have "?A\<noteq>0" by auto;
  ultimately have "\<forall>k\<in>?A. \<langle>k,Maximum(IntegerOrder,?A)\<rangle> \<in> IntegerOrder"
    by (rule Finite_ZF_1_T2)
  moreover from A1 A2 have "?d\<in>?A" using AlmostHoms_def Int_ZF_2_1_L4
    by auto;
  ultimately have "?d \<lsq> Maximum(IntegerOrder,?A)" by auto; 
  with A1 show "?d \<lsq> max\<delta>(s)"
    using AlmostHoms_def Int_ZF_2_1_L4 by simp;
  then show "max\<delta>(s) \<in> \<int>" using Int_ZF_2_L1A by simp
  from A1 A2 show "\<delta>(s,m,n) \<in> \<int>"
    using Int_ZF_2_1_L2 Int_ZF_1_1_L5 by simp;
qed;

text{*A useful estimate for the value a the slope at $0$.*}

lemma (in int1) Int_ZF_2_1_L8: assumes A1: "s\<in>\<S>"
  shows "abs(s`(\<zero>)) \<lsq> max\<delta>(s)"  "\<zero>\<lsq>max\<delta>(s)"  "max\<delta>(s) \<in> \<int>" 
proof -;
  from A1 have "s`(\<zero>) \<in> \<int>" 
    using Int_ZF_1_1_L3 Int_ZF_2_1_L2B by simp;
  then have T1: "\<zero> \<lsq> abs(s`(\<zero>))"  
    and "abs(\<delta>(s,\<zero>,\<zero>)) = abs(s`(\<zero>))" 
    using Int_ZF_2_L22 Int_ZF_1_1_L3 Int_ZF_1_1_L4 Int_ZF_2_L17 
    by auto
  moreover from A1 have "abs(\<delta>(s,\<zero>,\<zero>)) \<lsq> max\<delta>(s)"
    using Int_ZF_1_1_L3 Int_ZF_2_1_L7 by simp;
  ultimately show "abs(s`(\<zero>)) \<lsq> max\<delta>(s)"
    by simp;
  with T1 show "\<zero>\<lsq>max\<delta>(s)" by (rule Int_order_transitive);
  then show "max\<delta>(s) \<in> \<int>" using Int_ZF_2_L1A by simp;
qed;

text{*Int @{text "Group_ZF_3.thy"} we show that finite range functions 
  valued in an abelian group 
  form a normal subgroup of almost homomorphisms. 
  This allows to define the equivalence relation 
  between almost homomorphisms as the relation resulting from dividing 
  by that normal subgroup. 
  Then we show in @{text "Group_ZF_3_4_L12"} that if the difference of $f$ and $g$ 
  has finite range (actually $f(n)\cdot g(n)^{-1}$ as we use multiplicative notation 
  in @{text "Group_ZF_3.thy"}), then $f$ and $g$ are equivalent.
  The next lemma translates that fact into the notation used in @{text "int1"} context.*}

lemma (in int1) Int_ZF_2_1_L9: assumes A1: "s\<in>\<S>"  "r\<in>\<S>"
  and A2: "\<forall>m\<in>\<int>. abs(s`(m)\<rs>r`(m)) \<lsq> L"
  shows "s \<sim> r"
proof -
  from A1 A2 have 
    "\<forall>m\<in>\<int>. s`(m)\<rs>r`(m) \<in> \<int> \<and> abs(s`(m)\<rs>r`(m)) \<lsq> L"
    using Int_ZF_2_1_L2B Int_ZF_1_1_L5 by simp;
  then have
    "IsBounded({s`(n)\<rs>r`(n). n\<in>\<int>}, IntegerOrder)"
    by (rule Int_ZF_1_3_L20);
  with A1 show "s \<sim> r" using Int_bounded_iff_fin 
    Int_ZF_2_1_L1 group1.Group_ZF_3_4_L12 by simp;
qed;

text{*What is the value of a composition of slopes?*}

lemma (in int1) Int_ZF_2_1_L10: 
  assumes "s\<in>\<S>"  "r\<in>\<S>" and "m\<in>\<int>"
  shows "(s\<circ>r)`(m) = s`(r`(m))"  "s`(r`(m)) \<in> \<int>"
  using prems Int_ZF_2_1_L1 group1.Group_ZF_3_4_L2 by auto;

text{*Composition of slopes is a slope.*}

lemma (in int1) Int_ZF_2_1_L11:
  assumes "s\<in>\<S>"  "r\<in>\<S>"
  shows "s\<circ>r \<in> \<S>"
  using prems Int_ZF_2_1_L1 group1.Group_ZF_3_4_T1 by simp;

text{*A simple but useful identity.*}

lemma (in int1) Int_ZF_2_1_L12: 
  assumes "s\<in>\<S>" and "n\<in>\<int>"  "m\<in>\<int>"
  shows "s`(n\<cdot>m) \<ra> (s`(m) \<ra> \<delta>(s,n\<cdot>m,m)) = s`((n\<ra>\<one>)\<cdot>m)"
  using prems Int_ZF_1_1_L5 Int_ZF_2_1_L2B Int_ZF_1_2_L9 Int_ZF_1_2_L7
  by simp;

section{*Composing slopes*}

text{*Composition of slopes is not commutative. However, as we show in this 
  section if $f$ and $g$ are slopes then $f\circ g - g\circ f$ 
  has a bounded range. This allows to show that the multiplication of real 
  numbers is commutative.*}

text{*Two useful estimates.*}

lemma (in int1) Int_ZF_2_2_L1: 
  assumes A1: "f:\<int>\<rightarrow>\<int>" and A2: "p\<in>\<int>" "q\<in>\<int>"
  shows 
  "abs(f`((p\<ra>\<one>)\<cdot>q)\<rs>(p\<ra>\<one>)\<cdot>f`(q)) \<lsq> abs(\<delta>(f,p\<cdot>q,q))\<ra>abs(f`(p\<cdot>q)\<rs>p\<cdot>f`(q))"
  "abs(f`((p\<rs>\<one>)\<cdot>q)\<rs>(p\<rs>\<one>)\<cdot>f`(q)) \<lsq> abs(\<delta>(f,(p\<rs>\<one>)\<cdot>q,q))\<ra>abs(f`(p\<cdot>q)\<rs>p\<cdot>f`(q))"
proof -;
  let ?R = "\<int>"
  let ?A = "IntegerAddition"
  let ?M = "IntegerMultiplication"
  let ?I = "GroupInv(?R, ?A)"
  let ?a = "f`((p\<ra>\<one>)\<cdot>q)"
  let ?b = "p"
  let ?c = "f`(q)"
  let ?d = "f`(p\<cdot>q)"
  from A1 A2 have T1:
    "ring0(?R, ?A, ?M)" "?a \<in> ?R" "?b \<in> ?R" "?c \<in> ?R" "?d \<in> ?R"
    using  Int_ZF_1_1_L2 Int_ZF_1_1_L3 Int_ZF_1_1_L5 apply_funtype 
    by auto;
  then have 
    "?A`\<langle>?a,?I`(?M`\<langle>?A`\<langle>?b, TheIdentityElement(?R, ?M)\<rangle>,?c\<rangle>)\<rangle> =
    ?A`\<langle>?A`\<langle>?A`\<langle>?a,?I`(?d)\<rangle>,?I`(?c)\<rangle>,?A`\<langle>?d, ?I`(?M`\<langle>?b, ?c\<rangle>)\<rangle>\<rangle>"
    by (rule ring0.Ring_ZF_2_L2);
  with A2 have 
    "f`((p\<ra>\<one>)\<cdot>q)\<rs>(p\<ra>\<one>)\<cdot>f`(q) = \<delta>(f,p\<cdot>q,q)\<ra>(f`(p\<cdot>q)\<rs>p\<cdot>f`(q))"
    using Int_ZF_1_1_L3 Int_ZF_1_1_L1 Int_ZF_1_1_L4 by simp;
  moreover from A1 A2 T1 have "\<delta>(f,p\<cdot>q,q) \<in> \<int>" "f`(p\<cdot>q)\<rs>p\<cdot>f`(q) \<in> \<int>"
    using Int_ZF_1_1_L5 apply_funtype by auto;
  ultimately show 
    "abs(f`((p\<ra>\<one>)\<cdot>q)\<rs>(p\<ra>\<one>)\<cdot>f`(q)) \<lsq> abs(\<delta>(f,p\<cdot>q,q))\<ra>abs(f`(p\<cdot>q)\<rs>p\<cdot>f`(q))"
    using Int_triangle_ineq by simp;
  from A1 A2 have T1: 
    "f`((p\<rs>\<one>)\<cdot>q) \<in> \<int>"   "p\<in>\<int>"  "f`(q) \<in> \<int>"   "f`(p\<cdot>q) \<in> \<int>" 
    using Int_ZF_1_1_L3 Int_ZF_1_1_L5 apply_funtype by auto;
  then have
    "f`((p\<rs>\<one>)\<cdot>q)\<rs>(p\<rs>\<one>)\<cdot>f`(q) = (f`(p\<cdot>q)\<rs>p\<cdot>f`(q))\<rs>(f`(p\<cdot>q)\<rs>f`((p\<rs>\<one>)\<cdot>q)\<rs>f`(q))"
    by (rule Int_ZF_1_2_L6);
  with A2 have "f`((p\<rs>\<one>)\<cdot>q)\<rs>(p\<rs>\<one>)\<cdot>f`(q) = (f`(p\<cdot>q)\<rs>p\<cdot>f`(q))\<rs>\<delta>(f,(p\<rs>\<one>)\<cdot>q,q)"
    using Int_ZF_1_2_L7 by simp;
  moreover from A1 A2 have 
    "f`(p\<cdot>q)\<rs>p\<cdot>f`(q) \<in> \<int>"   "\<delta>(f,(p\<rs>\<one>)\<cdot>q,q) \<in> \<int>" 
    using Int_ZF_1_1_L5 Int_ZF_1_1_L3 apply_funtype by auto;
  ultimately show 
    "abs(f`((p\<rs>\<one>)\<cdot>q)\<rs>(p\<rs>\<one>)\<cdot>f`(q)) \<lsq> abs(\<delta>(f,(p\<rs>\<one>)\<cdot>q,q))\<ra>abs(f`(p\<cdot>q)\<rs>p\<cdot>f`(q))"
    using Int_triangle_ineq1 by simp;
qed;
  
text{*If $f$ is a slope, then 
  $|f(p\cdot q)-p\cdot f(q)|\leq (|p|+1)\cdot$@{text "max\<delta>(f)"}. 
  The proof is by induction on $p$ and the next lemma is the induction step for the case when $0\leq p$.*}

lemma (in int1) Int_ZF_2_2_L2: 
  assumes A1: "f\<in>\<S>" and A2: "\<zero>\<lsq>p"  "q\<in>\<int>"
  and A3: "abs(f`(p\<cdot>q)\<rs>p\<cdot>f`(q)) \<lsq> (abs(p)\<ra>\<one>)\<cdot>max\<delta>(f)"
  shows 
  "abs(f`((p\<ra>\<one>)\<cdot>q)\<rs>(p\<ra>\<one>)\<cdot>f`(q)) \<lsq> (abs(p\<ra>\<one>)\<ra> \<one>)\<cdot>max\<delta>(f)"
proof -
  from A2 have "q\<in>\<int>"  "p\<cdot>q \<in> \<int>" 
    using Int_ZF_2_L1A Int_ZF_1_1_L5 by auto;
  with A1 have I: "abs(\<delta>(f,p\<cdot>q,q)) \<lsq> max\<delta>(f)" by (rule Int_ZF_2_1_L7);
  moreover from A3 have "abs(f`(p\<cdot>q)\<rs>p\<cdot>f`(q)) \<lsq> (abs(p)\<ra>\<one>)\<cdot>max\<delta>(f)" .
  moreover from A1 A2 have
    "abs(f`((p\<ra>\<one>)\<cdot>q)\<rs>(p\<ra>\<one>)\<cdot>f`(q)) \<lsq> abs(\<delta>(f,p\<cdot>q,q))\<ra>abs(f`(p\<cdot>q)\<rs>p\<cdot>f`(q))"
    using AlmostHoms_def Int_ZF_2_L1A Int_ZF_2_2_L1 by simp;
  ultimately have 
    "abs(f`((p\<ra>\<one>)\<cdot>q)\<rs>(p\<ra>\<one>)\<cdot>f`(q)) \<lsq> max\<delta>(f)\<ra>(abs(p)\<ra>\<one>)\<cdot>max\<delta>(f)"
    by (rule Int_ZF_2_L15);
  moreover from I A2 have 
    "max\<delta>(f)\<ra>(abs(p)\<ra>\<one>)\<cdot>max\<delta>(f) = (abs(p\<ra>\<one>)\<ra> \<one>)\<cdot>max\<delta>(f)"
    using Int_ZF_2_L1A Int_ZF_1_2_L2 by simp;
  ultimately show
    "abs(f`((p\<ra>\<one>)\<cdot>q)\<rs>(p\<ra>\<one>)\<cdot>f`(q)) \<lsq> (abs(p\<ra>\<one>)\<ra> \<one>)\<cdot>max\<delta>(f)"
    by simp;
qed;

text{*If $f$ is a slope, then 
  $|f(p\cdot q)-p\cdot f(q)|\leq (|p|+1)\cdot$@{text "max\<delta>"}. 
  The proof is by induction on $p$ and the next lemma is the induction step for the case when $p\leq 0$.*}

lemma (in int1) Int_ZF_2_2_L3: 
  assumes A1: "f\<in>\<S>" and A2: "p\<lsq>\<zero>" "q\<in>\<int>"
  and A3: "abs(f`(p\<cdot>q)\<rs>p\<cdot>f`(q)) \<lsq> (abs(p)\<ra>\<one>)\<cdot>max\<delta>(f)"
  shows  "abs(f`((p\<rs>\<one>)\<cdot>q)\<rs>(p\<rs>\<one>)\<cdot>f`(q)) \<lsq> (abs(p\<rs>\<one>)\<ra> \<one>)\<cdot>max\<delta>(f)"
proof -;
  from A2 have "q\<in>\<int>"  "(p\<rs>\<one>)\<cdot>q \<in> \<int>" 
    using Int_ZF_2_L1A Int_ZF_1_1_L3 Int_ZF_1_1_L5 by auto;
  with A1 have I: "abs(\<delta>(f,(p\<rs>\<one>)\<cdot>q,q)) \<lsq> max\<delta>(f)" by (rule Int_ZF_2_1_L7);
  moreover from A3 have "abs(f`(p\<cdot>q)\<rs>p\<cdot>f`(q)) \<lsq> (abs(p)\<ra>\<one>)\<cdot>max\<delta>(f)" .
  moreover from A1 A2 have 
    "abs(f`((p\<rs>\<one>)\<cdot>q)\<rs>(p\<rs>\<one>)\<cdot>f`(q)) \<lsq> abs(\<delta>(f,(p\<rs>\<one>)\<cdot>q,q))\<ra>abs(f`(p\<cdot>q)\<rs>p\<cdot>f`(q))"
    using AlmostHoms_def Int_ZF_2_L1A Int_ZF_2_2_L1 by simp;
  ultimately have 
    "abs(f`((p\<rs>\<one>)\<cdot>q)\<rs>(p\<rs>\<one>)\<cdot>f`(q)) \<lsq> max\<delta>(f)\<ra>(abs(p)\<ra>\<one>)\<cdot>max\<delta>(f)"
    by (rule Int_ZF_2_L15);
  with I A2 show ?thesis using Int_ZF_2_L1A Int_ZF_1_2_L5 by simp;
qed;

text{*If $f$ is a slope, then 
  $|f(p\cdot q)-p\cdot f(q)|\leq (|p|+1)\cdot$@{text "max\<delta>"}$(f)$.*} 

lemma (in int1) Int_ZF_2_2_L4: 
  assumes A1: "f\<in>\<S>" and A2: "p\<in>\<int>" "q\<in>\<int>"
  shows "abs(f`(p\<cdot>q)\<rs>p\<cdot>f`(q)) \<lsq> (abs(p)\<ra>\<one>)\<cdot>max\<delta>(f)"
proof (cases "\<zero>\<lsq>p")
  assume "\<zero>\<lsq>p"
  moreover from A1 A2 have "abs(f`(\<zero>\<cdot>q)\<rs>\<zero>\<cdot>f`(q)) \<lsq> (abs(\<zero>)\<ra>\<one>)\<cdot>max\<delta>(f)"
    using Int_ZF_1_1_L3 Int_ZF_2_1_L2B Int_ZF_1_1_L4 
      Int_ZF_2_1_L8 Int_ZF_2_L18 by simp;
  moreover from A1 A2 have 
    "\<forall>p. \<zero>\<lsq>p \<and> abs(f`(p\<cdot>q)\<rs>p\<cdot>f`(q)) \<lsq> (abs(p)\<ra>\<one>)\<cdot>max\<delta>(f) \<longrightarrow>
    abs(f`((p\<ra>\<one>)\<cdot>q)\<rs>(p\<ra>\<one>)\<cdot>f`(q)) \<lsq> (abs(p\<ra>\<one>)\<ra> \<one>)\<cdot>max\<delta>(f)"
    using Int_ZF_2_2_L2 by simp;
  ultimately show "abs(f`(p\<cdot>q)\<rs>p\<cdot>f`(q)) \<lsq> (abs(p)\<ra>\<one>)\<cdot>max\<delta>(f)" 
    by (rule Induction_on_int);
next assume "\<not>(\<zero>\<lsq>p)"
  with A2 have "p\<lsq>\<zero>" using Int_ZF_2_L19A by simp;
  moreover from A1 A2 have "abs(f`(\<zero>\<cdot>q)\<rs>\<zero>\<cdot>f`(q)) \<lsq> (abs(\<zero>)\<ra>\<one>)\<cdot>max\<delta>(f)"
    using Int_ZF_1_1_L3 Int_ZF_2_1_L2B Int_ZF_1_1_L4
      Int_ZF_2_1_L8 Int_ZF_2_L18 by simp;
  moreover from A1 A2 have 
    "\<forall>p. p\<lsq>\<zero> \<and> abs(f`(p\<cdot>q)\<rs>p\<cdot>f`(q)) \<lsq> (abs(p)\<ra>\<one>)\<cdot>max\<delta>(f) \<longrightarrow>
    abs(f`((p\<rs>\<one>)\<cdot>q)\<rs>(p\<rs>\<one>)\<cdot>f`(q)) \<lsq> (abs(p\<rs>\<one>)\<ra> \<one>)\<cdot>max\<delta>(f)"
    using Int_ZF_2_2_L3 by simp;
  ultimately show "abs(f`(p\<cdot>q)\<rs>p\<cdot>f`(q)) \<lsq> (abs(p)\<ra>\<one>)\<cdot>max\<delta>(f)" 
    by (rule Back_induct_on_int);
qed;

text{*The next elegant result is Lemma 7 in the Arthan's paper \cite{Arthan2004} .*}

lemma (in int1) Arthan_Lem_7: assumes A1: "f\<in>\<S>" and A2: "p\<in>\<int>"  "q\<in>\<int>"
  shows "abs(q\<cdot>f`(p)\<rs>p\<cdot>f`(q)) \<lsq> (abs(p)\<ra>abs(q)\<ra>\<two>)\<cdot>max\<delta>(f)"
proof -;
  from A1 A2 have T:
    "q\<cdot>f`(p)\<rs>f`(p\<cdot>q) \<in> \<int>" 
    "f`(p\<cdot>q)\<rs>p\<cdot>f`(q) \<in> \<int>"
    "f`(q\<cdot>p) \<in> \<int>" "f`(p\<cdot>q) \<in> \<int>"
    "q\<cdot>f`(p) \<in> \<int>" "p\<cdot>f`(q) \<in> \<int>" 
    "max\<delta>(f) \<in> \<int>"
    "abs(q) \<in> \<int>" "abs(p) \<in> \<int>"
    using Int_ZF_1_1_L5 Int_ZF_2_1_L2B Int_ZF_2_1_L7 Int_ZF_2_L14 by auto;
  moreover have "abs(q\<cdot>f`(p)\<rs>f`(p\<cdot>q)) \<lsq> (abs(q)\<ra>\<one>)\<cdot>max\<delta>(f)"
  proof -
    from A1 A2 have "abs(f`(q\<cdot>p)\<rs>q\<cdot>f`(p)) \<lsq> (abs(q)\<ra>\<one>)\<cdot>max\<delta>(f)"
      using Int_ZF_2_2_L4 by simp;
    with T A2 show ?thesis
      using Int_ZF_2_L20 Int_ZF_1_1_L5 by simp;
  qed;
  moreover from A1 A2 have "abs(f`(p\<cdot>q)\<rs>p\<cdot>f`(q)) \<lsq> (abs(p)\<ra>\<one>)\<cdot>max\<delta>(f)"
    using Int_ZF_2_2_L4 by simp;
  ultimately have 
    "abs(q\<cdot>f`(p)\<rs>f`(p\<cdot>q)\<ra>(f`(p\<cdot>q)\<rs>p\<cdot>f`(q))) \<lsq> (abs(q)\<ra>\<one>)\<cdot>max\<delta>(f)\<ra>(abs(p)\<ra>\<one>)\<cdot>max\<delta>(f)"
    using Int_ZF_2_L21 by simp;
  with T show ?thesis using Int_ZF_1_2_L9 Int_ZF_1_1_L3 Int_ZF_1_2_L10
    by simp;
qed;

text{*This is Lemma 8 in the Arthan's paper.*}

lemma (in int1) Arthan_Lem_8: assumes A1: "f\<in>\<S>"
  shows "\<exists>A B. A\<in>\<int> \<and> B\<in>\<int> \<and> (\<forall>p\<in>\<int>. abs(f`(p)) \<lsq> A\<cdot>abs(p)\<ra>B)"
proof -
  let ?A = "max\<delta>(f) \<ra> abs(f`(\<one>))"
  let ?B = "\<three>\<cdot>max\<delta>(f)"
  from A1 have "?A\<in>\<int>" "?B\<in>\<int>"
    using Int_ZF_1_1_L3 Int_ZF_1_1_L5 Int_ZF_2_1_L2B 
      Int_ZF_2_1_L7 Int_ZF_2_L14 by auto;
  moreover have "\<forall>p\<in>\<int>. abs(f`(p)) \<lsq> ?A\<cdot>abs(p)\<ra>?B"
  proof
    fix p assume A2: "p\<in>\<int>" 
    with A1 have T: 
      "f`(p) \<in> \<int>"  "abs(p) \<in> \<int>"  "f`(\<one>) \<in> \<int>" 
      "p\<cdot>f`(\<one>) \<in> \<int>"  "\<three>\<in>\<int>"  "max\<delta>(f) \<in> \<int>"
      using Int_ZF_2_1_L2B Int_ZF_2_L14 Int_ZF_1_1_L3 Int_ZF_1_1_L5 Int_ZF_2_1_L7
      by auto;
    from A1 A2 have 
      "abs(\<one>\<cdot>f`(p)\<rs>p\<cdot>f`(\<one>)) \<lsq> (abs(p)\<ra>abs(\<one>)\<ra>\<two>)\<cdot>max\<delta>(f)"
      using Int_ZF_1_1_L3 Arthan_Lem_7 by simp;
    with T have "abs(f`(p)) \<lsq> abs(p\<cdot>f`(\<one>))\<ra>(abs(p)\<ra>\<three>)\<cdot>max\<delta>(f)"
      using Int_ZF_2_L16A Int_ZF_1_1_L4 Int_ZF_1_2_L11 
	Int_triangle_ineq2 by simp
    with A2 T show "abs(f`(p)) \<lsq> ?A\<cdot>abs(p)\<ra>?B"
      using Int_ZF_1_3_L14 by simp;
  qed
  ultimately show ?thesis by auto;
qed;

text{*If $f$ and $g$ are slopes, then $f\circ g$ is equivalent 
  (almost equal) to $g\circ f$. This is Theorem 9 in Arthan's paper \cite{Arthan2004} .*}

theorem (in int1) Arthan_Th_9: assumes A1: "f\<in>\<S>"  "g\<in>\<S>"
  shows "f\<circ>g \<sim> g\<circ>f"
proof -
   from A1 have 
      "\<exists>A B. A\<in>\<int> \<and> B\<in>\<int> \<and> (\<forall>p\<in>\<int>. abs(f`(p)) \<lsq> A\<cdot>abs(p)\<ra>B)"
      "\<exists>C D. C\<in>\<int> \<and> D\<in>\<int> \<and> (\<forall>p\<in>\<int>. abs(g`(p)) \<lsq> C\<cdot>abs(p)\<ra>D)"
      using Arthan_Lem_8 by auto;
    then obtain A B C D where D1: "A\<in>\<int>" "B\<in>\<int>" "C\<in>\<int>" "D\<in>\<int>" and D2: 
      "\<forall>p\<in>\<int>. abs(f`(p)) \<lsq> A\<cdot>abs(p)\<ra>B"
      "\<forall>p\<in>\<int>. abs(g`(p)) \<lsq> C\<cdot>abs(p)\<ra>D"
      by auto;
    let ?E = "max\<delta>(g)\<cdot>(A\<ra>\<one>) \<ra> max\<delta>(f)\<cdot>(C\<ra>\<one>)"
    let ?F = "(B\<cdot>max\<delta>(g) \<ra> \<two>\<cdot>max\<delta>(g)) \<ra> (D\<cdot>max\<delta>(f) \<ra> \<two>\<cdot>max\<delta>(f))"
  { fix p assume A2: "p\<in>\<int>"
    with A1 have T1:
      "g`(p) \<in> \<int>"  "f`(p) \<in> \<int>"  "abs(p) \<in> \<int>"  "\<two> \<in> \<int>"
      "f`(g`(p)) \<in> \<int>"  "g`(f`(p)) \<in> \<int>"  "f`(g`(p)) \<rs> g`(f`(p)) \<in> \<int>"
      "p\<cdot>f`(g`(p)) \<in> \<int>"  "p\<cdot>g`(f`(p)) \<in> \<int>"
      "abs(f`(g`(p))\<rs>g`(f`(p))) \<in> \<int>"
      using Int_ZF_2_1_L2B Int_ZF_2_1_L10 Int_ZF_1_1_L5 Int_ZF_2_L14 Int_ZF_1_1_L5A
      by auto;
    with A1 A2 have
      "abs((f`(g`(p))\<rs>g`(f`(p)))\<cdot>p) \<lsq> 
      (abs(p)\<ra>abs(f`(p))\<ra>\<two>)\<cdot>max\<delta>(g) \<ra> (abs(p)\<ra>abs(g`(p))\<ra>\<two>)\<cdot>max\<delta>(f)"
      using Arthan_Lem_7 Int_ZF_1_2_L10A Int_ZF_1_2_L12 by simp;
    moreover have 
      "(abs(p)\<ra>abs(f`(p))\<ra>\<two>)\<cdot>max\<delta>(g) \<ra> (abs(p)\<ra>abs(g`(p))\<ra>\<two>)\<cdot>max\<delta>(f) \<lsq>
      ((max\<delta>(g)\<cdot>(A\<ra>\<one>) \<ra> max\<delta>(f)\<cdot>(C\<ra>\<one>)))\<cdot>abs(p) \<ra>
      ((B\<cdot>max\<delta>(g) \<ra> \<two>\<cdot>max\<delta>(g)) \<ra> (D\<cdot>max\<delta>(f) \<ra> \<two>\<cdot>max\<delta>(f)))"
    proof -
      from D2 A2 T1 have 
	"abs(p)\<ra>abs(f`(p))\<ra>\<two> \<lsq> abs(p)\<ra>(A\<cdot>abs(p)\<ra>B)\<ra>\<two>"
	"abs(p)\<ra>abs(g`(p))\<ra>\<two> \<lsq> abs(p)\<ra>(C\<cdot>abs(p)\<ra>D)\<ra>\<two>"
	using Int_ZF_2_L15C by auto;
      with A1 have 
	"(abs(p)\<ra>abs(f`(p))\<ra>\<two>)\<cdot>max\<delta>(g) \<lsq> (abs(p)\<ra>(A\<cdot>abs(p)\<ra>B)\<ra>\<two>)\<cdot>max\<delta>(g)"
	"(abs(p)\<ra>abs(g`(p))\<ra>\<two>)\<cdot>max\<delta>(f) \<lsq> (abs(p)\<ra>(C\<cdot>abs(p)\<ra>D)\<ra>\<two>)\<cdot>max\<delta>(f)"
	using Int_ZF_2_1_L8 Int_ZF_1_3_L13 by auto;
      moreover from A1 D1 T1 have 
	"(abs(p)\<ra>(A\<cdot>abs(p)\<ra>B)\<ra>\<two>)\<cdot>max\<delta>(g) = 
	max\<delta>(g)\<cdot>(A\<ra>\<one>)\<cdot>abs(p) \<ra> (B\<cdot>max\<delta>(g) \<ra> \<two>\<cdot>max\<delta>(g))"
	"(abs(p)\<ra>(C\<cdot>abs(p)\<ra>D)\<ra>\<two>)\<cdot>max\<delta>(f) = 
	max\<delta>(f)\<cdot>(C\<ra>\<one>)\<cdot>abs(p) \<ra> (D\<cdot>max\<delta>(f) \<ra> \<two>\<cdot>max\<delta>(f))"
	using Int_ZF_2_1_L8 Int_ZF_1_2_L13 by auto;
      ultimately have 
	"(abs(p)\<ra>abs(f`(p))\<ra>\<two>)\<cdot>max\<delta>(g) \<ra> (abs(p)\<ra>abs(g`(p))\<ra>\<two>)\<cdot>max\<delta>(f) \<lsq>
	(max\<delta>(g)\<cdot>(A\<ra>\<one>)\<cdot>abs(p) \<ra> (B\<cdot>max\<delta>(g) \<ra> \<two>\<cdot>max\<delta>(g))) \<ra> 
	(max\<delta>(f)\<cdot>(C\<ra>\<one>)\<cdot>abs(p) \<ra> (D\<cdot>max\<delta>(f) \<ra> \<two>\<cdot>max\<delta>(f)))"
	using Int_ZF_2_L15A by simp;
      moreover from A1 A2 D1 have "abs(p) \<in> \<int>"
	"max\<delta>(g)\<cdot>(A\<ra>\<one>) \<in> \<int>"  "B\<cdot>max\<delta>(g) \<ra> \<two>\<cdot>max\<delta>(g) \<in> \<int>"
	"max\<delta>(f)\<cdot>(C\<ra>\<one>) \<in> \<int>"  "D\<cdot>max\<delta>(f) \<ra> \<two>\<cdot>max\<delta>(f) \<in> \<int>" 
	using Int_ZF_2_L14 Int_ZF_2_1_L8 Int_ZF_1_1_L3 
	  Int_ZF_1_1_L5 Int_ZF_1_1_L5A by auto;
      ultimately show ?thesis using Int_ZF_1_2_L14 by simp;
    qed
    ultimately have
      "abs((f`(g`(p))\<rs>g`(f`(p)))\<cdot>p) \<lsq> ?E\<cdot>abs(p) \<ra> ?F"
      by (rule Int_order_transitive);
    with A2 T1 have
      "abs(f`(g`(p))\<rs>g`(f`(p)))\<cdot>abs(p) \<lsq> ?E\<cdot>abs(p) \<ra> ?F"
      "abs(f`(g`(p))\<rs>g`(f`(p))) \<in> \<int>"
      using Int_ZF_1_3_L5 by auto
  } then have 
      "\<forall>p\<in>\<int>. abs(f`(g`(p))\<rs>g`(f`(p))) \<in> \<int>"
      "\<forall>p\<in>\<int>. abs(f`(g`(p))\<rs>g`(f`(p)))\<cdot>abs(p) \<lsq> ?E\<cdot>abs(p) \<ra> ?F"
    by auto;
  moreover from A1 D1 have "?E \<in> \<int>"  "?F \<in> \<int>"
    using Int_ZF_1_1_L3 Int_ZF_1_1_L5A Int_ZF_2_1_L8 Int_ZF_1_1_L5
    by auto;
  ultimately have
    "\<exists>L. \<forall>p\<in>\<int>. abs(f`(g`(p))\<rs>g`(f`(p))) \<lsq> L"
    by (rule Int_ZF_1_6_L1);
  with A1 obtain L where "\<forall>p\<in>\<int>. abs((f\<circ>g)`(p)\<rs>(g\<circ>f)`(p)) \<lsq> L"
    using Int_ZF_2_1_L10 by auto;
  moreover from A1 have "f\<circ>g \<in> \<S>"  "g\<circ>f \<in> \<S>"
    using Int_ZF_2_1_L11 by auto;
  ultimately show "f\<circ>g \<sim> g\<circ>f" using Int_ZF_2_1_L9 by auto;
qed;

text{*The next lemma is Lemma 3 in the Arthan's paper.*}

lemma (in int1) Arthan_Lem_3: assumes A1: "f\<in>\<S>"
  and A2: "\<int>\<^sub>+ \<inter> {f`(n). n\<in>\<int>\<^sub>+} \<notin> Fin(\<int>)" and A3: "\<one>\<lsq>D"
  shows "\<exists>M\<in>\<int>\<^sub>+. \<forall>m. \<one>\<lsq>m \<longrightarrow> (m\<ra>\<one>)\<cdot>D \<lsq> f`(m\<cdot>M)" 
proof -
  let ?E = "max\<delta>(f) \<ra> D"
  let ?A = "\<int>\<^sub>+ \<inter> {f`(m). m\<in>\<int>\<^sub>+}"
  from A1 A3 have I: "D\<lsq>?E"
    using Int_ZF_2_1_L8 Int_ZF_2_L1A Int_ZF_2_L15D
    by simp;
  from A1 A2 A3 have "?A \<subseteq> \<int>\<^sub>+"  "?A \<notin> Fin(\<int>)"  "\<two>\<cdot>?E \<in> \<int>" 
    using Int_ZF_1_1_L5A Int_ZF_2_1_L8 Int_ZF_2_L1A Int_ZF_1_1_L5
    by auto;
  then have "\<exists>n\<in>?A. \<two>\<cdot>?E \<lsq> n"
    using Int_ZF_1_5_L2A by simp;
  then obtain M where II: "M\<in>\<int>\<^sub>+"  and III: "\<two>\<cdot>?E \<lsq> f`(M)"
    by auto;
  { fix m assume A4: "\<one>\<lsq>m"
    moreover from II III have "(\<one>\<ra>\<one>) \<cdot>?E \<lsq> f`(\<one>\<cdot>M)"
      using PositiveSet_def Int_ZF_1_1_L4 by simp;
    moreover have "\<forall>k. 
      \<one>\<lsq>k \<and> (k\<ra>\<one>)\<cdot>?E \<lsq> f`(k\<cdot>M) \<longrightarrow> (k\<ra>\<one>\<ra>\<one>)\<cdot>?E \<lsq> f`((k\<ra>\<one>)\<cdot>M)"
    proof -
      { fix k assume A5: "\<one>\<lsq>k"  and A6: "(k\<ra>\<one>)\<cdot>?E \<lsq> f`(k\<cdot>M)"
	with A1 A3 II have T:
	  "k\<in>\<int>"  "M\<in>\<int>"  "k\<ra>\<one> \<in> \<int>"  "?E\<in>\<int>"  "(k\<ra>\<one>)\<cdot>?E \<in> \<int>"  "\<two>\<cdot>?E \<in> \<int>"
	  using Int_ZF_2_L1A PositiveSet_def Int_ZF_1_1_L3 
	    Int_ZF_1_1_L5 Int_ZF_2_1_L8 by auto;
	from A1 A3 A5 II have 
	  "\<delta>(f,k\<cdot>M,M) \<in> \<int>"   "abs(\<delta>(f,k\<cdot>M,M)) \<lsq> max\<delta>(f)"   "\<zero>\<lsq>D"
	  using Int_ZF_2_L1A PositiveSet_def Int_ZF_1_1_L5 Int_ZF_2_1_L7 
	    Int_ZF_2_L16C by auto;
	with III A6 have 
	  "(k\<ra>\<one>)\<cdot>?E \<ra> (\<two>\<cdot>?E \<rs> ?E) \<lsq> f`(k\<cdot>M) \<ra> (f`(M) \<ra> \<delta>(f,k\<cdot>M,M))"
	  using Int_ZF_1_3_L19A Int_ZF_2_L15A by simp;
	with A1 T have "(k\<ra>\<one>\<ra>\<one>)\<cdot>?E \<lsq> f`((k\<ra>\<one>)\<cdot>M)"
	  using Int_ZF_1_1_L1 Int_ZF_1_1_L3 Int_ZF_1_1_L4 
	    Int_ZF_1_2_L11 Int_ZF_2_1_L12 by simp;
      } then show ?thesis by simp;
    qed;
    ultimately have "(m\<ra>\<one>)\<cdot>?E \<lsq> f`(m\<cdot>M)" by (rule Induction_on_int);
    with A4 I have "(m\<ra>\<one>)\<cdot>D \<lsq> f`(m\<cdot>M)" using Int_ZF_1_3_L13A
      by simp;
  } then have "\<forall>m. \<one>\<lsq>m \<longrightarrow> (m\<ra>\<one>)\<cdot>D \<lsq> f`(m\<cdot>M)" by simp;
  with II show ?thesis by auto;
qed;

section{*Positive slopes*}

text{*This section provides background material for defining the order relation on real numbers.*}

text{*We know  from @{text "Group_ZF_3.thy"} that finite range functions are almost homomorphisms. 
  Besides reminding that fact for slopes the next lemma shows 
  that finite range functions do not belong to @{text "\<S>\<^isub>+"}. 
  This is important, because the projection
  of the set of finite range functions defines zero in the real number construction in @{text "Real_ZF_x.thy"} 
  series, while the projection of @{text "\<S>\<^isub>+"} becomes the set of (strictly) positive reals. 
  We don't want zero to be positive, do we? The next lemma is a part of Lemma 5 in the Arthan's paper 
  \cite{Arthan2004}.*}

lemma (in int1) Int_ZF_2_3_L1: 
  assumes A1: "f \<in> FinRangeFunctions(\<int>,\<int>)"
  shows "f\<in>\<S>"   "f \<notin> \<S>\<^isub>+"
proof -
  from A1 show "f\<in>\<S>" using Int_ZF_2_1_L1 group1.Group_ZF_3_3_L1
    by auto;
  have "\<int>\<^sub>+ \<subseteq> \<int>" using PositiveSet_def by auto;
  with A1 have "f``(\<int>\<^sub>+) \<in> Fin(\<int>)"
    using Finite1_L21 by simp;
  then have "\<int>\<^sub>+ \<inter> f``(\<int>\<^sub>+) \<in> Fin(\<int>)"
    using Fin_subset_lemma by blast;
  thus "f \<notin> \<S>\<^isub>+" by auto;
qed;
        
end