(*
    This file is a part of IsarMathLib - 
    a library of formalized mathematics for Isabelle/Isar.

    Copyright (C) 2005  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{OrderedRing\_ZF.thy}*}

theory OrderedRing_ZF imports Ring_ZF OrderedGroup_ZF 

begin

text{*In this theory file we consider ordered rings.*}

section{*Definition and notation*}

text{*This section defines ordered rings and sets up appriopriate notation.*}

text{*We define ordered ring as a commutative ring with linear order 
  that is preserved by 
  translations and such that the set of nonnegative elements is closed
  under multiplication. Note that this definition does not guarantee 
  that there are no zero divisors in the ring.*}

constdefs 

  "IsAnOrdRing(R,A,M,r) \<equiv> 
  ( IsAring(R,A,M) \<and> (M {is commutative on} R) \<and> 
  r\<subseteq>R\<times>R \<and> IsLinOrder(R,r) \<and>
  (\<forall>a b. \<forall> c\<in>R. <a,b> \<in> r \<longrightarrow> \<langle>A`<a,c>,A`<b,c>\<rangle> \<in> r) \<and>
  (Nonnegative(R,A,r) {is closed under} M))"

text{*The next context (locale) defines notation used for ordered rings. 
  We do that by extending the notation defined in the 
  @{text "ring0"} locale and adding some assumptions to make sure we are 
  talking about ordered rings in this context.*}

locale ring1 = ring0 +

  assumes mult_commut: "M {is commutative on} R" 

  fixes r
  
  assumes ordincl: "r \<subseteq> R\<times>R"

  assumes linord: "IsLinOrder(R,r)"
  
  fixes lesseq (infix "\<lsq>" 68)
  defines lesseq_def [simp]: "a \<lsq> b \<equiv> <a,b> \<in> r"

  assumes ordgroup: "\<forall>a b. \<forall> c\<in>R. a\<lsq>b \<longrightarrow> a\<ra>c \<lsq> b\<ra>c"

  assumes pos_mult_closed: "Nonnegative(R,A,r) {is closed under} M"

  fixes abs ("| _ |")
  defines abs_def [simp]: "|a| \<equiv> AbsoluteValue(R,A,r)`(a)"

  fixes positiveset ("R\<^isub>+")
  defines positiveset_def [simp]: "R\<^isub>+ \<equiv> PositiveSet(R,A,r)"

text{*The next lemma assures us that we are talking about ordered rings 
  in the @{text "ring1"} context.*}

lemma (in  ring1) OrdRing_ZF_1_L1: shows "IsAnOrdRing(R,A,M,r)"
  using ring0_def  ringAssum mult_commut ordincl linord ordgroup 
    pos_mult_closed IsAnOrdRing_def by simp;

text{*We can use theorems proven in the @{text "ring1"} context whenever we
  talk about an ordered ring.*}

lemma OrdRing_ZF_1_L2: assumes "IsAnOrdRing(R,A,M,r)"
  shows "ring1(R,A,M,r)"
  using prems IsAnOrdRing_def ring1_axioms.intro ring0_def ring1_def
  by simp;

text{*In the @{text "ring1"} context $a\leq b$ implies that $a,b$ are
  elements of the ring.*}

lemma (in  ring1) OrdRing_ZF_1_L3: assumes "a\<lsq>b"
  shows "a\<in>R"  "b\<in>R"
  using prems ordincl by auto;

text{*Ordered ring is an ordered group, hence we can use theorems
  proven in the @{text "group3"} context.*}

lemma (in  ring1) OrdRing_ZF_1_L4: shows 
  "IsAnOrdGroup(R,A,r)"
  "r {is total on} R"
  "A {is commutative on} R"
  "group3(R,A,r)"
proof -;
  { fix a b g assume A1: "g\<in>R" and A2: "a\<lsq>b" 
    with ordgroup have "a\<ra>g \<lsq> b\<ra>g"
      by simp;
    moreover from ringAssum A1 A2 have 
      "a\<ra>g = g\<ra>a"  "b\<ra>g = g\<ra>b"
      using OrdRing_ZF_1_L3 IsAring_def IsCommutative_def by auto;
    ultimately have
      "a\<ra>g \<lsq> b\<ra>g"  "g\<ra>a \<lsq> g\<ra>b"
      by auto;
  } hence 
    "\<forall>g\<in>R. \<forall>a b. a\<lsq>b \<longrightarrow> a\<ra>g \<lsq> b\<ra>g \<and> g\<ra>a \<lsq> g\<ra>b"
    by simp;
  with ringAssum ordincl linord show  
    "IsAnOrdGroup(R,A,r)"
    "group3(R,A,r)"
    "r {is total on} R"
    "A {is commutative on} R"
    using IsAring_def Order_ZF_1_L2 IsAnOrdGroup_def group3_def IsLinOrder_def
    by auto;
qed;
  
text{*The next lemma just expands the condition that requires the set
  of nonnegative elements to be closed with respect to multiplication.*}

lemma (in  ring1) OrdRing_ZF_1_L5: 
  assumes "\<zero>\<lsq>a"  "\<zero>\<lsq>b"
  shows "\<zero> \<lsq> a\<cdot>b"
  using pos_mult_closed prems OrdRing_ZF_1_L4 group3.OrderedGroup_ZF_1_L2
  IsOpClosed_def by simp;

text{*A sufficient (somewhat redundant) condition for a structure to be an 
  ordered ring. It says that a commutative ring that is a totally ordered
  group with respect to the additive operation such that set of nonnegative 
  elements is closed under multiplication, is an ordered ring.*}

lemma  OrdRing_ZF_1_L6:
  assumes  
  "IsAring(R,A,M)"   
  "M {is commutative on} R"
  "Nonnegative(R,A,r) {is closed under} M"
  "IsAnOrdGroup(R,A,r)"  
  "r {is total on} R"
  shows "IsAnOrdRing(R,A,M,r)"
  using prems IsAnOrdGroup_def Order_ZF_1_L3 IsAnOrdRing_def
  by simp;

text{*$a\leq b$ iff $a-b\leq 0$. This is a fact from 
  @{text "OrderedGroup.thy"}, where it is stated in multiplicative notation.*}

lemma (in ring1) OrdRing_ZF_1_L7:
  assumes "a\<in>R"  "b\<in>R"
  shows "a\<lsq>b \<longleftrightarrow> a\<rs>b \<lsq> \<zero>"
  using prems OrdRing_ZF_1_L4 group3.OrderedGroup_ZF_1_L9
  by simp;

text{*Negative times positive is negative.*}

lemma (in ring1) OrdRing_ZF_1_L8:
  assumes A1: "a\<lsq>\<zero>"  and A2: "\<zero>\<lsq>b"
  shows "a\<cdot>b \<lsq> \<zero>"
proof -
  from A1 A2 have T1: "a\<in>R"  "b\<in>R"  "a\<cdot>b \<in> R"
    using OrdRing_ZF_1_L3 Ring_ZF_1_L4 by auto;
  from A1 A2 have "\<zero>\<lsq>(\<rm>a)\<cdot>b" 
    using OrdRing_ZF_1_L4 group3.OrderedGroup_ZF_1_L5A OrdRing_ZF_1_L5
    by simp;
  with T1 show "a\<cdot>b \<lsq> \<zero>"
    using Ring_ZF_1_L7 OrdRing_ZF_1_L4 group3.OrderedGroup_ZF_1_L5AA
    by simp;
qed;

text{*We can multiply both sides of an inequality by a nonnegative ring
  element. This property is sometimes (not here) used to define
  ordered rings. *}

lemma (in ring1) OrdRing_ZF_1_L9:
  assumes A1: "a\<lsq>b" and A2: "\<zero>\<lsq>c"
  shows 
  "a\<cdot>c \<lsq> b\<cdot>c"  
  "c\<cdot>a \<lsq> c\<cdot>b"
proof -
  from A1 A2 have T1:
    "a\<in>R"  "b\<in>R"  "c\<in>R"  "a\<cdot>c \<in> R"  "b\<cdot>c \<in> R"
    using OrdRing_ZF_1_L3 Ring_ZF_1_L4 by auto;
  with A1 A2 have "(a\<rs>b)\<cdot>c \<lsq> \<zero>"
    using OrdRing_ZF_1_L7 OrdRing_ZF_1_L8 by simp;
  with T1 show "a\<cdot>c \<lsq> b\<cdot>c"
    using Ring_ZF_1_L8 OrdRing_ZF_1_L7 by simp;
  with mult_commut T1 show "c\<cdot>a \<lsq> c\<cdot>b"
    using IsCommutative_def by simp;
qed;

text{*A square is nonnegative.*}

lemma (in ring1) OrdRing_ZF_1_L10:
  assumes A1: "a\<in>R" shows "\<zero>\<lsq>(a\<twosuperior>)"
proof (cases "\<zero>\<lsq>a");
  assume "\<zero>\<lsq>a"
  then show "\<zero>\<lsq>(a\<twosuperior>)" using OrdRing_ZF_1_L5
    by simp;
next assume "\<not>(\<zero>\<lsq>a)"
  with A1 have "\<zero>\<lsq>((\<rm>a)\<twosuperior>)"
    using OrdRing_ZF_1_L4 group3.OrderedGroup_ZF_1_L8A
      OrdRing_ZF_1_L5 by simp;
  with A1 show "\<zero>\<lsq>(a\<twosuperior>)" using Ring_ZF_1_L14
    by simp;
qed;
    
text{*Nonnegative is not negative. Property of ordered groups.*}

lemma (in ring1) OrdRing_ZF_1_L11: assumes "\<zero>\<lsq>a"
  shows "\<not>(a\<lsq>\<zero> \<and> a\<noteq>\<zero>)"
  using prems OrdRing_ZF_1_L4 group3.OrderedGroup_ZF_1_L5AB
  by simp;

text{*A negative element cannot be a square.*}

lemma (in ring1) OrdRing_ZF_3_L12: 
  assumes A1: "a\<lsq>\<zero>"  "a\<noteq>\<zero>"
  shows "\<not>(\<exists>b\<in>R. a = (b\<twosuperior>))"
proof -
  { assume "\<exists>b\<in>R. a = (b\<twosuperior>)"
    with A1 have False using OrdRing_ZF_1_L10 OrdRing_ZF_1_L11
      by auto;
  } then show ?thesis by auto;
qed;

section{*Absolute value for ordered rings*}

text{*Absolute value is defined for ordered groups as a function 
  that is the identity on the nonnegative set and the negative of the element 
  (the inverse in the multiplicative notation) on the rest. In this section
  we consider properties of absolute value related to multiplication in 
  ordered rings.*}


text{*Absolute value of a product is the product of absolute values: 
  the case when both elements of the ring are nonnegative.*}

lemma (in ring1) OrdRing_ZF_2_L1: 
  assumes "\<zero>\<lsq>a" "\<zero>\<lsq>b"
  shows "|a\<cdot>b| = |a|\<cdot>|b|"
  using prems OrdRing_ZF_1_L5 OrdRing_ZF_1_L4 
    group3.OrderedGroup_ZF_1_L2 group3.OrderedGroup_ZF_3_L2
  by simp;

text{*The absolue value of an element and its negative are the same.*}

lemma (in ring1) OrdRing_ZF_2_L2: assumes "a\<in>R"
  shows "|\<rm>a| = |a|"
  using prems OrdRing_ZF_1_L4 group3.OrderedGroup_ZF_3_L7A by simp;

text{*The next lemma states that 
  $|a\cdot (-b)| = |(-a)\cdot b| = |(-a)\cdot (-b)| = |a\cdot b|$.*}

lemma (in ring1) OrdRing_ZF_2_L3:
  assumes "a\<in>R"  "b\<in>R"
  shows 
  "|(\<rm>a)\<cdot>b| = |a\<cdot>b|"
  "|a\<cdot>(\<rm>b)| = |a\<cdot>b|"
  "|(\<rm>a)\<cdot>(\<rm>b)| = |a\<cdot>b|"
  using prems Ring_ZF_1_L4 Ring_ZF_1_L7 Ring_ZF_1_L7A 
   OrdRing_ZF_2_L2 by auto;

text{*This lemma allows to prove theorems for the case of positive and 
  negative elements of the ring separately.*}

lemma (in ring1) OrdRing_ZF_2_L4: assumes "a\<in>R" and "\<not>(\<zero>\<lsq>a)" 
  shows "\<zero> \<lsq> (\<rm>a)"  "\<zero>\<noteq>a"
  using prems OrdRing_ZF_1_L4 group3.OrderedGroup_ZF_1_L8A
  by auto;
  
text{*Absolute value of a product is the product of absolute values.*}

lemma (in ring1) OrdRing_ZF_2_L5:
  assumes A1: "a\<in>R" "b\<in>R"
  shows "|a\<cdot>b| = |a|\<cdot>|b|"
proof (cases "\<zero>\<lsq>a")
  assume A2: "\<zero>\<lsq>a" show "|a\<cdot>b| = |a|\<cdot>|b|"
  proof (cases "\<zero>\<lsq>b")
    assume "\<zero>\<lsq>b"
    with A2 show "|a\<cdot>b| = |a|\<cdot>|b|"
      using OrdRing_ZF_2_L1 by simp;
  next assume "\<not>(\<zero>\<lsq>b)" 
    with A1 A2 have "|a\<cdot>(\<rm>b)| = |a|\<cdot>|\<rm>b|"
       using OrdRing_ZF_2_L4 OrdRing_ZF_2_L1 by simp;
    with A1 show "|a\<cdot>b| = |a|\<cdot>|b|"
      using OrdRing_ZF_2_L2 OrdRing_ZF_2_L3 by simp;
  qed;
next assume "\<not>(\<zero>\<lsq>a)"
  with A1 have A3: "\<zero> \<lsq> (\<rm>a)"
    using OrdRing_ZF_2_L4 by simp;
  show "|a\<cdot>b| = |a|\<cdot>|b|"
  proof (cases "\<zero>\<lsq>b")
    assume "\<zero>\<lsq>b" 
    with A3 have "|(\<rm>a)\<cdot>b| = |\<rm>a|\<cdot>|b|"
      using OrdRing_ZF_2_L1 by simp;
    with A1 show "|a\<cdot>b| = |a|\<cdot>|b|"
      using OrdRing_ZF_2_L2 OrdRing_ZF_2_L3 by simp;
  next assume "\<not>(\<zero>\<lsq>b)"
    with A1 A3 have "|(\<rm>a)\<cdot>(\<rm>b)| = |\<rm>a|\<cdot>|\<rm>b|"
      using OrdRing_ZF_2_L4 OrdRing_ZF_2_L1 by simp;
    with A1 show "|a\<cdot>b| = |a|\<cdot>|b|"
      using OrdRing_ZF_2_L2 OrdRing_ZF_2_L3 by simp;
  qed;
qed;

text{*Triangle inequality. Property of linearly ordered abelian groups.*}

lemma (in ring1) ord_ring_triangle_ineq:  assumes "a\<in>R" "b\<in>R"
  shows "|a\<ra>b| \<lsq> |a|\<ra>|b|"
  using prems OrdRing_ZF_1_L4 group3.OrdGroup_triangle_ineq 
  by simp;

(*text{*One ring element is less or equal another iff 
  the difference is nonpositive. Property of ordered groups.*}

lemma (in ring1) OrdRing_ZF_1_L7:
  assumes "a\<in>R"  "b\<in>R"
  shows "a\<lsq>b \<longleftrightarrow> a\<rs>b \<lsq> \<zero>"
  using prems OrdRing_ZF_1_L4 group3.OrderedGroup_ZF_1_L9
  by simp;

text{*Negative times positive is negative.*}

lemma (in ring1) OrdRing_ZF_2_L7:
  assumes A1: "a\<lsq>\<zero>"  and A2: "\<zero>\<lsq>b"
  shows "a\<cdot>b \<lsq> \<zero>"
proof -
  from A1 A2 have T1: "a\<in>R"  "b\<in>R"  "a\<cdot>b \<in> R"
    using OrdRing_ZF_1_L3 Ring_ZF_1_L4 by auto;
  from A1 A2 have "\<zero>\<lsq>(\<rm>a)\<cdot>b" 
    using OrdRing_ZF_1_L4 group3.OrderedGroup_ZF_1_L5A OrdRing_ZF_1_L5
    by simp;
  with T1 show "a\<cdot>b \<lsq> \<zero>"
    using Ring_ZF_1_L7 OrdRing_ZF_1_L4 group3.OrderedGroup_ZF_1_L5AA
    by simp;
qed;

text{*We can multiply both sides of an inequality by a nonnegative ring
  element. This property is sometimes (not here) used to define
  ordered rings. *}

lemma (in ring1) OrdRing_ZF_2_L8:
  assumes A1: "a\<lsq>b" and A2: "\<zero>\<lsq>c"
  shows "a\<cdot>c \<lsq> b\<cdot>c"
proof -
  from A1 A2 have T1:
    "a\<in>R"  "b\<in>R"  "c\<in>R"  "a\<cdot>c \<in> R"  "b\<cdot>c \<in> R"
    using OrdRing_ZF_1_L3 Ring_ZF_1_L4 by auto;
  with A1 A2 have "(a\<rs>b)\<cdot>c \<lsq> \<zero>"
    using OrdRing_ZF_1_L7 OrdRing_ZF_2_L7 by simp;
  with T1 show "a\<cdot>c \<lsq> b\<cdot>c"
    using Ring_ZF_1_L8 OrdRing_ZF_1_L7 by simp;
qed;*)

section{*Positivity in ordered rings*}

text{*This section is about properties of the set of positive
  elements @{text "R\<^isub>+"}. *}

text{*The set of positive elements is closed under ring addition. 
  This is a property of ordered groups, we just reference a theorem
  from @{text "OrderedGroup_ZF"} theory in the proof.*}

lemma (in ring1) OrdRing_ZF_3_L1: shows "R\<^isub>+ {is closed under} A"
  using OrdRing_ZF_1_L4 group3.OrderedGroup_ZF_1_L13
  by simp;

text{*Every element of a ring can be either in the postitive set, equal to
  zero or its opposite (the additive inverse) is in the positive set. 
  This is a property of ordered groups, we just reference a theorem
  from @{text "OrderedGroup_ZF"} theory.*}

lemma (in ring1) OrdRing_ZF_3_L2: assumes "a\<in>R"
  shows "Exactly_1_of_3_holds (a=\<zero>, a\<in>R\<^isub>+, (\<rm>a) \<in> R\<^isub>+)"
  using prems OrdRing_ZF_1_L4 group3.OrdGroup_decomp
  by simp;

text{*@{text "R\<^isub>+"} is closed under
  multiplication iff the ring has no zero divisors. *}

lemma (in ring1) OrdRing_ZF_3_L3: 
  shows "(R\<^isub>+ {is closed under} M)\<longleftrightarrow> HasNoZeroDivs(R,A,M)"
proof;
  assume A1: "HasNoZeroDivs(R,A,M)"
  { fix a b assume "a\<in>R\<^isub>+"  "b\<in>R\<^isub>+"
    then have "\<zero>\<lsq>a"  "a\<noteq>\<zero>"  "\<zero>\<lsq>b"  "b\<noteq>\<zero>"
      using PositiveSet_def by auto;
    with A1 have "a\<cdot>b \<in> R\<^isub>+"
      using OrdRing_ZF_1_L5 Ring_ZF_1_L2 OrdRing_ZF_1_L3 Ring_ZF_1_L12
	OrdRing_ZF_1_L4 group3.OrderedGroup_ZF_1_L2A
      by simp;
  } then show  "R\<^isub>+ {is closed under} M" using IsOpClosed_def
    by simp;
next assume A2: "R\<^isub>+ {is closed under} M"
  { fix a b assume A3: "a\<in>R"  "b\<in>R"  and "a\<noteq>\<zero>"  "b\<noteq>\<zero>"
    with A2 have "|a\<cdot>b| \<in> R\<^isub>+"
      using OrdRing_ZF_1_L4 group3.OrderedGroup_ZF_3_L12 IsOpClosed_def
        OrdRing_ZF_2_L5 by simp;
    with A3 have "a\<cdot>b \<noteq> \<zero>" 
      using PositiveSet_def Ring_ZF_1_L4 
	OrdRing_ZF_1_L4 group3.OrderedGroup_ZF_3_L2A 
      by auto;
  } then show "HasNoZeroDivs(R,A,M)" using HasNoZeroDivs_def 
    by auto;
qed;
    
end