(*
    This file is a part of IsarMathLib - 
    a library of formalized mathematics for Isabelle/Isar.

    Copyright (C) 2005  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

HIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{Real\_ZF.thy}*}

theory Real_ZF imports Int_ZF Ring_ZF_1

begin

text{*The goal of the @{text "Real_ZF"} series of theory files is 
  to provide a contruction of the set of 
  real numbers. There are several ways to construct real numbers. 
  Most common start from the rational numbers and use Dedekind cuts 
  or Cauchy sequences. @{text "Real_ZF_x.thy"} series formalizes 
  an alternative 
  approach that constructs real numbers directly from the group of integers.
  Our formalization is mostly based on \cite{Arthan2004}. 
  Different variants of this contruction are also 
  described in \cite{Campo2003} and \cite{Street2003}.
  I recommend to read these papers, but for the impatient here is a short 
  description: we take a set of maps $s : Z\rightarrow Z$ such that
   the set $\{s(m+n)-s(m)-s(n)\}_{n,m \in Z}$ is finite 
  ($Z$ means the integers here). We call these maps slopes. 
  Slopes form a group with the natural addition
  $(s+r)(n) = s(n)+r(n)$. The maps such that the set $s(Z)$ is finite 
  (finite range functions) form a subgroup of slopes. 
  The additive group of real numbers is defined as the 
  quotient group of slopes by the (sub)group of finite range functions.
  The multiplication is defined as the projection of the composition of slopes
  into the resulting quotient (coset) space.
  *}

section{*The definition of real numbers*}

text{*First we define slopes and real numbers as the set of their classes.
  The definition of slopes references the notion of almost homomorphisms 
  defined in @{text "Group_ZF_2.thy"}: slopes 
  are defined as almost homomorphisms
  on integers with integer addition as the operation. Similarly the notions
  of the first and second operation on slopes (which is really the addition 
  and composition of slopes) is 
  derived as a special case of the first and second operation on almost 
  homomorphisms.
  *}

constdefs

  "Slopes \<equiv> AlmostHoms(int,IntegerAddition)"

  "SlopeOp1 \<equiv> AlHomOp1(int,IntegerAddition)"

  "SlopeOp2 \<equiv> AlHomOp2(int,IntegerAddition)"

  "BoundedIntMaps \<equiv> FinRangeFunctions(int,int)"

  "SlopeEquivalenceRel \<equiv> QuotientGroupRel(Slopes,SlopeOp1,BoundedIntMaps)"

  "RealNumbers \<equiv> Slopes//SlopeEquivalenceRel"

  "RealAddition \<equiv> ProjFun2(Slopes,SlopeEquivalenceRel,SlopeOp1)"

  "RealMultiplication \<equiv> ProjFun2(Slopes,SlopeEquivalenceRel,SlopeOp2)"

  "RealZero \<equiv> TheIdentityElement(RealNumbers,RealAddition)"
 
  "RealOne \<equiv> TheIdentityElement(RealNumbers,RealMultiplication)"

text{*We first show that we can use theorems proven in some proof contexts
  (locales). The locale @{text "group1"} requires assumption that we deal with
  an abelian group. The next lemma allows to use all theorems proven 
  in the context called @{text "group1"}.*}

lemma Real_ZF_1_L1: shows "group1(int,IntegerAddition)"
  using group1_axioms.intro group1_def Int_ZF_1_T2 by simp;

text{*Real numbers form a ring. This is a special case of the theorem
  proven in @{text "Ring_ZF_1.thy"}, where we show the same in general for 
  almost homomorphisms rather than slopes.*}

theorem Real_ZF_1_T1: "IsAring(RealNumbers,RealAddition,RealMultiplication)"
proof -
  let ?AH = "AlmostHoms(int,IntegerAddition)"
  let ?Op1 = "AlHomOp1(int,IntegerAddition)"
  let ?FR = "FinRangeFunctions(int,int)"
  let ?Op2 = "AlHomOp2(int,IntegerAddition)"
  let ?R = "QuotientGroupRel(?AH,?Op1,?FR)"
  let ?A = "ProjFun2(?AH,?R,?Op1)"
  let ?M = "ProjFun2(?AH,?R,?Op2)"
  have "IsAring(?AH//?R,?A,?M)" using Real_ZF_1_L1 group1.Ring_ZF_1_1_T1
    by simp;
  then show ?thesis using Slopes_def SlopeOp2_def SlopeOp1_def 
    BoundedIntMaps_def SlopeEquivalenceRel_def RealNumbers_def
    RealAddition_def RealMultiplication_def by simp;
qed;

text{*We can use theorems proven in @{text "group0"} and @{text "group1"}
  contexts applied to the group of real numbers.*}

lemma Real_ZF_1_L2: "group1(RealNumbers,RealAddition)"
proof -
  have "IsAgroup(RealNumbers,RealAddition)"
    "RealAddition {is commutative on} RealNumbers"
    using Real_ZF_1_T1 IsAring_def by auto;
  then show ?thesis using group1_axioms.intro group0_def group1_def
    by simp;
qed;

text{*Let's define some notation.*}

locale real0 =

  fixes real ("\<real>")
  defines real_def [simp]: "\<real> \<equiv> RealNumbers"

  fixes ra (infix "\<ra>" 69)
  defines ra_def [simp]: "a\<ra> b \<equiv> RealAddition`<a,b>"

  fixes rminus :: "i\<Rightarrow>i" ("\<rm> _" 90)
  defines rminus_def [simp]:"\<rm>a \<equiv> GroupInv(\<real>,RealAddition)`(a)"

  fixes rsub (infix "\<rs>" 69)
  defines rsub_def [simp]: "a\<rs>b \<equiv> a\<ra>(\<rm>b)"
 
  fixes rm (infix "\<cdot>" 70)
  defines rm_def [simp]: "a\<cdot>b \<equiv> RealMultiplication`<a,b>"

  fixes rzero ("\<zero>")
  defines rzero_def [simp]: 
  "\<zero> \<equiv> TheIdentityElement(RealNumbers,RealAddition)"
 
  fixes rone ("\<one>")
  defines rone_def [simp]: 
  "\<one> \<equiv> TheIdentityElement(RealNumbers,RealMultiplication)"

text{*In @{text "real0"} context all theorems proven in the @{text "ring0"}
  context are valid.*}

lemma (in real0) Real_ZF_1_L3: 
  shows "ring0(\<real>,RealAddition,RealMultiplication)"
  using Real_ZF_1_T1 ring0_def by simp;

text{*Lets try out our notation to see that zero and one are real numbers.*}

lemma (in real0) Real_ZF_1_L4: shows "\<zero>\<in>\<real>"  "\<one>\<in>\<real>"
  using Real_ZF_1_L3 ring0.Ring_ZF_1_L2 by auto;

text{*The lemma below lists some properties that
  require one real number to state.*}

lemma (in real0) Real_ZF_1_L5: assumes A1: "a\<in>\<real>"
  shows 
  "(\<rm>a) \<in> \<real>"
  "(\<rm>(\<rm>a)) = a"
  "a\<ra>\<zero> = a" 
  "\<zero>\<ra>a = a" 
  "a\<cdot>\<one> = a" 
  "\<one>\<cdot>a = a" 
  "a\<rs>a = \<zero>" 
  "a\<rs>\<zero> = a"  
  using prems Real_ZF_1_L3 ring0.Ring_ZF_1_L3 by auto;

text{*The lemma below lists some properties that
  require two real numbers to state.*}

lemma (in real0) Real_ZF_1_L6: assumes "a\<in>\<real>"  "b\<in>\<real>"
  shows 
  "a\<ra>b \<in> \<real>" 
  "a\<rs>b \<in> \<real>" 
  "a\<cdot>b \<in> \<real>" 
  "a\<ra>b = b\<ra>a"
  "(\<rm>a)\<cdot>b = \<rm>(a\<cdot>b)" 
  "a\<cdot>(\<rm>b) = \<rm>(a\<cdot>b)"
  using prems Real_ZF_1_L3 ring0.Ring_ZF_1_L4 ring0.Ring_ZF_1_L7
  by auto;

text{*Addition is distributive with respect to multiplication.*}

lemma (in real0) Real_ZF_1_L7: assumes "a\<in>\<real>"  "b\<in>\<real>"  "c\<in>\<real>" 
  shows 
  "a\<cdot>(b\<ra>c) = a\<cdot>b \<ra> a\<cdot>c" 
  "(b\<ra>c)\<cdot>a = b\<cdot>a \<ra> c\<cdot>a"
  "a\<cdot>(b\<rs>c) = a\<cdot>b \<rs> a\<cdot>c"  
  "(b\<rs>c)\<cdot>a = b\<cdot>a \<rs> c\<cdot>a"
  using prems Real_ZF_1_L3 ring0.Ring_ZF_1_L5  ring0.Ring_ZF_1_L8 
  by auto;

text{*This concludes the easy part of the construction that follows from
  the fact that slope equivalence classes form a ring. It is easy to see
  that multiplication of classes of almost homomorphisms is not 
  commutative in general.
  The remaining properties of real numbers, like commutativity of 
  multiplication and the existence of multiplicative inverses have to be 
  proven using properties of the group of integers, rather that in general
  setting of abelian groups.*}

end;