(*
    This file is a part of IsarMathLib - 
    a library of formalized mathematics for Isabelle/Isar.

    Copyright (C) 2005, 2006  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{Int\_ZF\_1.thy}*}

theory Int_ZF_1 imports Int_ZF OrderedRing_ZF 

begin;

text{*This theory file considers the set of integers as an ordered ring.*}

section{*Integers as a ring*}

text{*In this section we show that integers form a commutative ring. *}


text{*The next lemma provides the condition to show that addition is 
  distributive with respect to multiplication.*}

lemma (in int0) Int_ZF_1_1_L1: assumes A1: "a\<in>\<int>"  "b\<in>\<int>"  "c\<in>\<int>" 
  shows 
  "a\<cdot>(b\<ra>c) = a\<cdot>b \<ra> a\<cdot>c" 
  "(b\<ra>c)\<cdot>a = b\<cdot>a \<ra> c\<cdot>a"
  using prems Int_ZF_1_L2 zadd_zmult_distrib zadd_zmult_distrib2
  by auto;

text{*Integers form a commutative ring, hence we can use theorems proven 
  in @{text "ring0"} context (locale).*}

lemma (in int0) Int_ZF_1_1_L2: shows
  "IsAring(\<int>,IntegerAddition,IntegerMultiplication)"
  "IntegerMultiplication {is commutative on} \<int>"
  "ring0(\<int>,IntegerAddition,IntegerMultiplication)"
proof -
  have "\<forall>a\<in>\<int>.\<forall>b\<in>\<int>.\<forall>c\<in>\<int>. 
    a\<cdot>(b\<ra>c) = a\<cdot>b \<ra> a\<cdot>c \<and> (b\<ra>c)\<cdot>a = b\<cdot>a \<ra> c\<cdot>a"
    using Int_ZF_1_1_L1 by simp;
  then have "IsDistributive(\<int>,IntegerAddition,IntegerMultiplication)"
    using IsDistributive_def by simp;
  then show "IsAring(\<int>,IntegerAddition,IntegerMultiplication)"
    "ring0(\<int>,IntegerAddition,IntegerMultiplication)"
    using Int_ZF_1_T1 Int_ZF_1_T2 IsAring_def ring0_def 
    by auto;
  have "\<forall>a\<in>\<int>.\<forall>b\<in>\<int>. a\<cdot>b = b\<cdot>a" using Int_ZF_1_L4 by simp;
  then show "IntegerMultiplication {is commutative on} \<int>"
    using IsCommutative_def by simp;
qed;

text{*Zero and one are integers.*}

lemma (in int0) int_zero_one_are_int: shows "\<zero>\<in>\<int>"  "\<one>\<in>\<int>"
  using Int_ZF_1_1_L2 ring0.Ring_ZF_1_L2 by auto;

text{*Negative of zero is zero.*}

lemma (in int0) int_zero_one_are_intA: shows "(\<rm>\<zero>) = \<zero>"
  using Int_ZF_1_T2 group0.group0_2_L7A by simp;

text{*Properties with one integer.*}

lemma (in int0) Int_ZF_1_1_L4: assumes A1: "a \<in> \<int>"
  shows 
  "a\<ra>\<zero> = a" 
  "\<zero>\<ra>a = a" 
  "a\<cdot>\<one> = a"   "\<one>\<cdot>a = a"
  "\<zero>\<cdot>a = \<zero>"   "a\<cdot>\<zero> = \<zero>" 
  "(\<rm>a) \<in> \<int>" 
  "a\<rs>a = \<zero>"   "a\<rs>\<zero> = a"  "\<two>\<cdot>a = a\<ra>a"
proof -;
  from A1 show 
    "a\<ra>\<zero> = a"   "\<zero>\<ra>a = a"   "a\<cdot>\<one> = a" 
    "\<one>\<cdot>a = a"   "a\<rs>a = \<zero>"   "a\<rs>\<zero> = a"  
    "(\<rm>a) \<in> \<int>"  "\<two>\<cdot>a = a\<ra>a" 
    using Int_ZF_1_1_L2 ring0.Ring_ZF_1_L3 by auto;
  from A1 show "\<zero>\<cdot>a = \<zero>"   "a\<cdot>\<zero> = \<zero>"
    using Int_ZF_1_1_L2 ring0.Ring_ZF_1_L6 by auto;
qed;

text{*Properties that require two integers.*}

lemma (in int0) Int_ZF_1_1_L5: assumes A1: "a\<in>\<int>"  "b\<in>\<int>"
  shows 
  "a\<ra>b \<in> \<int>" 
  "a\<rs>b \<in> \<int>" 
  "a\<cdot>b \<in> \<int>"
  "a\<ra>b = b\<ra>a" 
  "a\<cdot>b = b\<cdot>a" 
  "(\<rm>b)\<rs>a = (\<rm>a)\<rs>b" 
  "(\<rm>(a\<ra>b)) = (\<rm>a)\<rs>b"  
  "(\<rm>(a\<rs>b)) = ((\<rm>a)\<ra>b)"
  "(\<rm>a)\<cdot>b = \<rm>(a\<cdot>b)" 
  "a\<cdot>(\<rm>b) = \<rm>(a\<cdot>b)"
  "(\<rm>a)\<cdot>(\<rm>b) = a\<cdot>b"
  using prems Int_ZF_1_1_L2 ring0.Ring_ZF_1_L4 ring0.Ring_ZF_1_L9
    ring0.Ring_ZF_1_L7 ring0.Ring_ZF_1_L7A Int_ZF_1_L4 by auto;

text{*$2$ and $3$ are integers.*}

lemma (in int0) Int_ZF_1_1_L5A: shows "\<two> \<in> \<int>"  "\<three> \<in> \<int>"
    using int_zero_one_are_int Int_ZF_1_1_L5 by auto;

text{*Properties that require three integers.*}

lemma (in int0) Int_ZF_1_1_L6: assumes A1: "a\<in>\<int>"  "b\<in>\<int>"  "c\<in>\<int>"
  shows 
  "a\<rs>(b\<ra>c) = a\<rs>b\<rs>c"
  "a\<rs>(b\<rs>c) = a\<rs>b\<ra>c"
  "a\<cdot>(b\<rs>c) = a\<cdot>b \<rs> a\<cdot>c"
  "(b\<rs>c)\<cdot>a = b\<cdot>a \<rs> c\<cdot>a"
  using prems Int_ZF_1_1_L2 ring0.Ring_ZF_1_L10  ring0.Ring_ZF_1_L8 
  by auto;

text{*One more property with three integers.*}
(* Inclusion of a\<ra>(b\<rs>c) = a\<ra>b\<rs>c causes the previous lemma
  to be unusable by simp in some cases*)
lemma (in int0) Int_ZF_1_1_L6A: assumes A1: "a\<in>\<int>"  "b\<in>\<int>"  "c\<in>\<int>"
  shows "a\<ra>(b\<rs>c) = a\<ra>b\<rs>c"
  using prems Int_ZF_1_1_L2 ring0.Ring_ZF_1_L10A by simp;

text{*Associativity of addition and multiplication.*}

lemma (in int0) Int_ZF_1_1_L7: assumes A1: "a\<in>\<int>" "b\<in>\<int>" "c\<in>\<int>"
  shows 
  "a\<ra>b\<ra>c = a\<ra>(b\<ra>c)"
  "a\<cdot>b\<cdot>c = a\<cdot>(b\<cdot>c)"
  using prems Int_ZF_1_1_L2 ring0.Ring_ZF_1_L11 by auto;
  
section{*Rearrangement lemmas*}

text{*In this section we collect lemmas about identities related to 
  rearranging the terms in expresssions*}

text{*A formula with a positive integer.*}

lemma (in int0) Int_ZF_1_2_L1: assumes "\<zero>\<lsq>a"
  shows "abs(a)\<ra>\<one> = abs(a\<ra>\<one>)"
  using prems Int_ZF_2_L16 Int_ZF_2_L12A by simp;

text{*A formula with two integers, one positive.*}

lemma (in int0) Int_ZF_1_2_L2: assumes A1: "a\<in>\<int>" and A2: "\<zero>\<lsq>b"
  shows "a\<ra>(abs(b)\<ra>\<one>)\<cdot>a = (abs(b\<ra>\<one>)\<ra>\<one>)\<cdot>a"
proof -
  from A2 have T1: "abs(b\<ra>\<one>) \<in> \<int>"
    using Int_ZF_2_L12A Int_ZF_2_L1A Int_ZF_2_L14 by blast;
  with A1 A2 show ?thesis 
    using Int_ZF_1_2_L1 Int_ZF_1_1_L2 ring0.Ring_ZF_2_L1 
    by simp;
qed;

text{*A couple of formulae about canceling opposite integers.*}

lemma (in int0) Int_ZF_1_2_L3: assumes A1: "a\<in>\<int>"  "b\<in>\<int>"
  shows 
  "a\<ra>b\<rs>a = b" 
  "a\<ra>b\<rs>b = a" 
  "a\<rs>b\<ra>b = a" 
  using prems Int_ZF_1_T2 group0.group0_4_L6A group0.group0_2_L16 
  by auto;

text{*Subtracting one does not increase integers. This may be moved to a theory
  about ordered rings one day.*}

lemma (in int0) Int_ZF_1_2_L3A: assumes A1: "a\<lsq>b"
  shows "a\<rs>\<one> \<lsq> b"
proof -
  from A1 have "b\<ra>\<one>\<rs>\<one> = b"
    using Int_ZF_2_L1A int_zero_one_are_int Int_ZF_1_2_L3 by simp;
  moreover from A1 have "a\<rs>\<one> \<lsq> b\<ra>\<one>\<rs>\<one>"
    using Int_ZF_2_L12A int_zero_one_are_int Int_ZF_1_1_L4 int_ord_transl_inv
    by simp;
  ultimately show "a\<rs>\<one> \<lsq> b" by simp;
qed;

text{*A formula with a nonpositive integer.*}

lemma (in int0) Int_ZF_1_2_L4: assumes "a\<lsq>\<zero>"
  shows "abs(a)\<ra>\<one> = abs(a\<rs>\<one>)"
  using prems int_zero_one_are_int Int_ZF_1_2_L3A Int_ZF_2_T1 
      group3.OrderedGroup_ZF_3_L3A Int_ZF_2_L1A 
      int_zero_one_are_int Int_ZF_1_1_L5 by simp;

text{*A formula with two integers, one negative.*}

lemma (in int0) Int_ZF_1_2_L5: assumes A1: "a\<in>\<int>" and A2: "b\<lsq>\<zero>"
  shows "a\<ra>(abs(b)\<ra>\<one>)\<cdot>a = (abs(b\<rs>\<one>)\<ra>\<one>)\<cdot>a"
proof -
  from A2 have "abs(b\<rs>\<one>) \<in> \<int>"
    using int_zero_one_are_int Int_ZF_1_2_L3A Int_ZF_2_L1A Int_ZF_2_L14 
    by blast;
  with A1 A2 show ?thesis 
    using Int_ZF_1_2_L4 Int_ZF_1_1_L2 ring0.Ring_ZF_2_L1
    by simp;
qed;
  
text{*A rearrangement with four integers.*}

lemma (in int0) Int_ZF_1_2_L6: 
  assumes A1: "a\<in>\<int>"  "b\<in>\<int>"  "c\<in>\<int>"  "d\<in>\<int>"
  shows "a\<rs>(b\<rs>\<one>)\<cdot>c = (d\<rs>b\<cdot>c)\<rs>(d\<rs>a\<rs>c)"
proof -;
  from A1 have T1: 
    "(d\<rs>b\<cdot>c) \<in> \<int>" "d\<rs>a \<in> \<int>" "(\<rm>(b\<cdot>c)) \<in> \<int>"
    using Int_ZF_1_1_L5 Int_ZF_1_1_L4 by auto;   
  with A1 have 
    "(d\<rs>b\<cdot>c)\<rs>(d\<rs>a\<rs>c) = (\<rm>(b\<cdot>c))\<ra>a\<ra>c"
    using Int_ZF_1_1_L6 Int_ZF_1_2_L3 by simp;
  also from A1 T1 have "(\<rm>(b\<cdot>c))\<ra>a\<ra>c = a\<rs>(b\<rs>\<one>)\<cdot>c" 
    using int_zero_one_are_int Int_ZF_1_1_L6 Int_ZF_1_1_L4 Int_ZF_1_1_L5
    by simp;
  finally show ?thesis by simp
qed;

text{*Four other rearrangements with two integers.*}

lemma (in int0) Int_ZF_1_2_L7: 
  assumes "a\<in>\<int>"  "b\<in>\<int>"
  shows 
  "a\<cdot>b = (a\<rs>\<one>)\<cdot>b\<ra>b"
  "a\<cdot>(b\<ra>\<one>) = a\<cdot>b\<ra>a"
  "(b\<ra>\<one>)\<cdot>a = b\<cdot>a\<ra>a"
  "(b\<ra>\<one>)\<cdot>a = a\<ra>b\<cdot>a"
  using prems Int_ZF_1_1_L1 Int_ZF_1_1_L5 int_zero_one_are_int 
    Int_ZF_1_1_L6 Int_ZF_1_1_L4 Int_ZF_1_T2 group0.group0_2_L16 
  by auto;
  
text{*Another rearrangement with two integers.*}

lemma (in int0) Int_ZF_1_2_L8: 
  assumes A1: "a\<in>\<int>" "b\<in>\<int>"
  shows "a\<ra>\<one>\<ra>(b\<ra>\<one>) = b\<ra>a\<ra>\<two>"
  using prems int_zero_one_are_int Int_ZF_1_T2 group0.group0_4_L8
  by simp;

text{*A couple of rearrangement with three integers.*}

lemma (in int0) Int_ZF_1_2_L9: 
  assumes "a\<in>\<int>"  "b\<in>\<int>"  "c\<in>\<int>"
  shows 
  "(a\<rs>b)\<ra>(b\<rs>c) = a\<rs>c"
  "(a\<rs>b)\<rs>(a\<rs>c) = c\<rs>b"
  "a\<ra>(b\<ra>(c\<rs>a\<rs>b)) = c"
  using prems Int_ZF_1_T2 
    group0.group0_4_L4B group0.group0_4_L6D by auto;

text{*Another rearrangement with three integers.*}

lemma (in int0) Int_ZF_1_2_L10: 
  assumes A1: "a\<in>\<int>" "b\<in>\<int>" "c\<in>\<int>"
  shows "(a\<ra>\<one>)\<cdot>b \<ra> (c\<ra>\<one>)\<cdot>b = (c\<ra>a\<ra>\<two>)\<cdot>b"
proof -
  from A1 have "a\<ra>\<one> \<in> \<int>" "c\<ra>\<one> \<in> \<int>" 
    using int_zero_one_are_int Int_ZF_1_1_L5 by auto; 
  with A1 have 
    "(a\<ra>\<one>)\<cdot>b \<ra> (c\<ra>\<one>)\<cdot>b = (a\<ra>\<one>\<ra>(c\<ra>\<one>))\<cdot>b"
    using Int_ZF_1_1_L1 by simp;
  also from A1 have "\<dots> = (c\<ra>a\<ra>\<two>)\<cdot>b"
    using Int_ZF_1_2_L8 by simp;
  finally show ?thesis by simp;
qed;

text{*A technical rearrangement involing inequalities with absolute value.*}

lemma (in int0) Int_ZF_1_2_L10A: 
  assumes A1: "a\<in>\<int>"  "b\<in>\<int>"  "c\<in>\<int>"  "e\<in>\<int>"
  and A2: "abs(a\<cdot>b\<rs>c) \<lsq> d"  "abs(b\<cdot>a\<rs>e) \<lsq> f"
  shows "abs(c\<rs>e) \<lsq>  f\<ra>d"
proof -  
  from A1 A2 have T1:
    "d\<in>\<int>"  "f\<in>\<int>"  "a\<cdot>b \<in> \<int>"  "a\<cdot>b\<rs>c \<in> \<int>"  "b\<cdot>a\<rs>e \<in> \<int>"
    using Int_ZF_2_L1A Int_ZF_1_1_L5 by auto;
  with A2 have
    "abs((b\<cdot>a\<rs>e)\<rs>(a\<cdot>b\<rs>c)) \<lsq> f \<ra>d"
    using Int_ZF_2_L21 by simp;
  with A1 T1 show "abs(c\<rs>e) \<lsq> f\<ra>d" 
    using Int_ZF_1_1_L5 Int_ZF_1_2_L9 by simp;
qed;

text{*Some arithmetics.*}

lemma (in int0) Int_ZF_1_2_L11: assumes A1: "a\<in>\<int>"
  shows 
  "a\<ra>\<one>\<ra>\<two> = a\<ra>\<three>"
  "a = \<two>\<cdot>a \<rs> a"
proof -
  from A1 show "a\<ra>\<one>\<ra>\<two> = a\<ra>\<three>"
    using int_zero_one_are_int Int_ZF_1_1_L5A Int_ZF_1_T2 group0.group0_4_L4C
    by simp
  from A1 show "a = \<two>\<cdot>a \<rs> a"
    using int_zero_one_are_int Int_ZF_1_1_L1 Int_ZF_1_1_L4 Int_ZF_1_T2 group0.group0_2_L16
    by simp;
qed;

text{*A simple rearrangement with three integers.*}

lemma (in int0) Int_ZF_1_2_L12: 
  assumes A1: "a\<in>\<int>"  "b\<in>\<int>"  "c\<in>\<int>"
  shows "(b\<rs>c)\<cdot>a = a\<cdot>b \<rs> a\<cdot>c"
  using prems Int_ZF_1_1_L6 Int_ZF_1_1_L5 by simp;

text{*A big rearrangement with five integers.*}

lemma (in int0) Int_ZF_1_2_L13: 
  assumes A1: "a\<in>\<int>"  "b\<in>\<int>"  "c\<in>\<int>" "d\<in>\<int>"  "x\<in>\<int>"
  shows "(x\<ra>(a\<cdot>x\<ra>b)\<ra>c)\<cdot>d = d\<cdot>(a\<ra>\<one>)\<cdot>x \<ra> (b\<cdot>d\<ra>c\<cdot>d)"
proof -
  from A1 have T1: 
    "a\<cdot>x \<in> \<int>"   "(a\<ra>\<one>)\<cdot>x \<in> \<int>"  
    "(a\<ra>\<one>)\<cdot>x \<ra> b \<in> \<int>" 
    using Int_ZF_1_1_L5 int_zero_one_are_int by auto
  with A1 have "(x\<ra>(a\<cdot>x\<ra>b)\<ra>c)\<cdot>d = ((a\<ra>\<one>)\<cdot>x \<ra> b)\<cdot>d \<ra> c\<cdot>d"
    using Int_ZF_1_1_L7 Int_ZF_1_2_L7 Int_ZF_1_1_L1
    by simp;
  also from A1 T1 have "\<dots> = (a\<ra>\<one>)\<cdot>x\<cdot>d \<ra> b \<cdot> d \<ra> c\<cdot>d"
    using Int_ZF_1_1_L1 by simp;
  finally have "(x\<ra>(a\<cdot>x\<ra>b)\<ra>c)\<cdot>d = (a\<ra>\<one>)\<cdot>x\<cdot>d \<ra> b\<cdot>d \<ra> c\<cdot>d"
    by simp;
  moreover from A1 T1 have "(a\<ra>\<one>)\<cdot>x\<cdot>d = d\<cdot>(a\<ra>\<one>)\<cdot>x"
    using int_zero_one_are_int Int_ZF_1_1_L5 Int_ZF_1_1_L7 by simp;
  ultimately have "(x\<ra>(a\<cdot>x\<ra>b)\<ra>c)\<cdot>d = d\<cdot>(a\<ra>\<one>)\<cdot>x \<ra> b\<cdot>d \<ra> c\<cdot>d"
    by simp;
  moreover from A1 T1 have 
    "d\<cdot>(a\<ra>\<one>)\<cdot>x \<in> \<int>"  "b\<cdot>d \<in> \<int>"  "c\<cdot>d \<in> \<int>"
    using int_zero_one_are_int Int_ZF_1_1_L5 by auto;
  ultimately show ?thesis using Int_ZF_1_1_L7 by simp;
qed;

text{*Rerrangement about adding linear functions.*}

lemma (in int0) Int_ZF_1_2_L14:
  assumes "a\<in>\<int>"  "b\<in>\<int>"  "c\<in>\<int>" "d\<in>\<int>"  "x\<in>\<int>"
  shows "(a\<cdot>x \<ra> b) \<ra> (c\<cdot>x \<ra> d) = (a\<ra>c)\<cdot>x \<ra> (b\<ra>d)"
  using prems Int_ZF_1_1_L2 ring0.Ring_ZF_2_L3 by simp;

text{*A rearrangement with four integers. 
  Again we have to use the generic set notation to use a theorem proven in 
  different context.*}

lemma (in int0) Int_ZF_1_2_L15: assumes A1: "a\<in>\<int>"  "b\<in>\<int>"  "c\<in>\<int>" "d\<in>\<int>"
  and A2: "a = b\<rs>c\<rs>d"
  shows "d = b\<rs>a\<rs>c"
proof -
  let ?G = "int"
  let ?f = "IntegerAddition"
  from A1 A2 have 
    "group0(?G, ?f)"   "?f {is commutative on} ?G"
    "a \<in> ?G"  "b \<in> ?G" "c \<in> ?G"  "d \<in> ?G"
    "a = ?f`\<langle>?f`\<langle>b,GroupInv(?G, ?f)`(c)\<rangle>,GroupInv(?G, ?f)`(d)\<rangle>"
    using Int_ZF_1_T2 by auto;
  then have  
    "d = ?f`\<langle>?f`\<langle>b,GroupInv(?G, ?f)`(a)\<rangle>,GroupInv(?G,?f)`(c)\<rangle>"
    by (rule group0.group0_4_L9);
  then show "d = b\<rs>a\<rs>c" by simp;
qed;

section{*Integers as an ordered ring*}

text{*We already know from @{text "Int_ZF"} that integers with addition 
  form a linearly ordered group. To show that integers form 
  an ordered ring we need the fact that the set of nonnegative integers
  is closed under multiplication. Since we don't have the theory
  of oredered rings we temporarily put some facts about integers as an ordered
  ring in this section.*}

text{* We start with the property that a product of 
  nonnegative integers is nonnegative. The proof is by induction and the next
  lemma is the induction step.*}

lemma (in int0) Int_ZF_1_3_L1: assumes A1: "\<zero>\<lsq>a"  "\<zero>\<lsq>b"
  and A3: "\<zero> \<lsq> a\<cdot>b"
  shows "\<zero> \<lsq> a\<cdot>(b\<ra>\<one>)"
proof -
  from A1 A3 have "\<zero>\<ra>\<zero> \<lsq> a\<cdot>b\<ra>a"
    using Int_ZF_2_L15A by simp;
  with A1 show "\<zero> \<lsq> a\<cdot>(b\<ra>\<one>)"
    using int_zero_one_are_int Int_ZF_1_1_L4 Int_ZF_2_L1A Int_ZF_1_2_L7 
    by simp;
qed;

text{* Product of nonnegative integers is nonnegative.*}
  
lemma (in int0) Int_ZF_1_3_L2: assumes A1: "\<zero>\<lsq>a"  "\<zero>\<lsq>b" 
  shows "\<zero>\<lsq>a\<cdot>b"
proof -
  from A1 have "\<zero>\<lsq>b" by simp
  moreover from A1 have "\<zero> \<lsq> a\<cdot>\<zero>" using
    Int_ZF_2_L1A Int_ZF_1_1_L4 int_zero_one_are_int Int_ZF_2_L2 refl_def
    by simp;
  moreover from A1 have 
    "\<forall>m. \<zero>\<lsq>m \<and> \<zero>\<lsq>a\<cdot>m \<longrightarrow> \<zero> \<lsq> a\<cdot>(m\<ra>\<one>)"
    using Int_ZF_1_3_L1 by simp;
  ultimately show "\<zero>\<lsq>a\<cdot>b" by (rule Induction_on_int);
qed;

text{*The set of nonnegative integers is closed with respect
  to multiplication.*}

lemma (in int0) Int_ZF_1_3_L2A: shows 
  "\<int>\<^sup>+ {is closed under} IntegerMultiplication"
proof -
  { fix a b assume "a\<in>\<int>\<^sup>+"  "b\<in>\<int>\<^sup>+"
    then have "a\<cdot>b \<in>\<int>\<^sup>+"
      using Int_ZF_1_3_L2 Int_ZF_2_T1 group3.OrderedGroup_ZF_1_L2 
      by simp;
  } then have "\<forall>a\<in>\<int>\<^sup>+.\<forall>b\<in>\<int>\<^sup>+.a\<cdot>b \<in>\<int>\<^sup>+" by simp; 
  then show ?thesis using IsOpClosed_def by simp
qed;

text{*Integers form an ordered ring. All theorems proven in the 
  @{text "ring1"} context are valid in @{text "int0"} context.*}

theorem (in int0) Int_ZF_1_3_T1: shows 
  "IsAnOrdRing(\<int>,IntegerAddition,IntegerMultiplication,IntegerOrder)"
  "ring1(\<int>,IntegerAddition,IntegerMultiplication,IntegerOrder)"
  using Int_ZF_1_1_L2 Int_ZF_2_L1B Int_ZF_1_3_L2A Int_ZF_2_T1
    OrdRing_ZF_1_L6 OrdRing_ZF_1_L2 by auto;
  
text{*Product of integers that are greater that one is greater than one. 
  The proof is by induction and 
  the next step is the induction step.*}

lemma (in int0) Int_ZF_1_3_L3_indstep:  
  assumes A1: "\<one>\<lsq>a"  "\<one>\<lsq>b"
  and A2: "\<one> \<lsq> a\<cdot>b"
  shows "\<one> \<lsq> a\<cdot>(b\<ra>\<one>)"
proof -
   from A1 A2 have "\<one>\<lsq>\<two>" and "\<two> \<lsq> a\<cdot>(b\<ra>\<one>)"
    using Int_ZF_2_L1A Int_ZF_2_L15A Int_ZF_2_L16B Int_ZF_1_2_L7 
    by auto;
  then show "\<one> \<lsq> a\<cdot>(b\<ra>\<one>)" by (rule Int_order_transitive);
qed;

text{*Product of integers that are greater that one is greater than one. *}

lemma (in int0) Int_ZF_1_3_L3: 
  assumes A1: "\<one>\<lsq>a" "\<one>\<lsq>b"
  shows "\<one> \<lsq> a\<cdot>b"
proof -
  from A1 have "\<one>\<lsq>b"  "\<one>\<lsq>a\<cdot>\<one>"
    using Int_ZF_2_L1A Int_ZF_1_1_L4 by auto;
  moreover from A1 have 
    "\<forall>m. \<one>\<lsq>m \<and> \<one> \<lsq> a\<cdot>m \<longrightarrow> \<one> \<lsq> a\<cdot>(m\<ra>\<one>)"
    using Int_ZF_1_3_L3_indstep by simp;
  ultimately show "\<one> \<lsq> a\<cdot>b" by (rule Induction_on_int);
qed;

text{*$|a\cdot (-b)| = |(-a)\cdot b| = |(-a)\cdot (-b)| = |a\cdot b|$
  This is a property of ordered rings..*}

lemma (in int0) Int_ZF_1_3_L4: assumes "a\<in>\<int>"  "b\<in>\<int>"
  shows 
  "abs((\<rm>a)\<cdot>b) = abs(a\<cdot>b)"
  "abs(a\<cdot>(\<rm>b)) = abs(a\<cdot>b)"
  "abs((\<rm>a)\<cdot>(\<rm>b)) = abs(a\<cdot>b)"
  using prems Int_ZF_1_1_L5 Int_ZF_2_L17 by auto;

text{*Absolute value of a product is the product of absolute values.
  Property of ordered rings.*}

lemma (in int0) Int_ZF_1_3_L5:
  assumes A1: "a\<in>\<int>"  "b\<in>\<int>"
  shows "abs(a\<cdot>b) = abs(a)\<cdot>abs(b)"
  using prems Int_ZF_1_3_T1 ring1.OrdRing_ZF_2_L5 by simp;

text{*Double nonnegative is nonnegative. Property of ordered rings.*}

lemma (in int0) Int_ZF_1_3_L5A: assumes "\<zero>\<lsq>a"
  shows "\<zero>\<lsq>\<two>\<cdot>a"
  using prems Int_ZF_1_3_T1 ring1.OrdRing_ZF_1_L5A by simp;

text{*The next lemma shows what happens when one integer is not
  greater or equal than another.*}

lemma (in int0) Int_ZF_1_3_L6: 
  assumes A1: "a\<in>\<int>"  "b\<in>\<int>" 
  shows "\<not>(b\<lsq>a) \<longleftrightarrow> a\<ra>\<one> \<lsq> b"
proof;
  assume A3: "\<not>(b\<lsq>a)"
  with A1 have  "a\<lsq>b" by (rule Int_ZF_2_L19);
  then have "a = b   \<or>  a\<ra>\<one> \<lsq> b"
    using Int_ZF_4_L1B by simp; 
  moreover from A1 A3 have "a\<noteq>b" by (rule Int_ZF_2_L19);
  ultimately show "a\<ra>\<one> \<lsq> b" by simp;
next assume A4: "a\<ra>\<one> \<lsq> b"
  { assume "b\<lsq>a" 
    with A4 have "a\<ra>\<one> \<lsq> a" by (rule Int_order_transitive);
    moreover from A1 have "a \<lsq> a\<ra>\<one>"
      using Int_ZF_2_L12B by simp;
    ultimately have "a\<ra>\<one> = a"
      by (rule Int_ZF_2_L3); 
    with A1 have False using Int_ZF_1_L14 by simp; 
  } then show "\<not>(b\<lsq>a)" by auto
qed;

text{*An integer is not less or equal zero iff it is greater or equal one.*}

lemma (in int0) Int_ZF_1_3_L7: assumes "a\<in>\<int>" 
  shows "\<not>(a\<lsq>\<zero>) \<longleftrightarrow> \<one> \<lsq> a"
  using prems int_zero_one_are_int Int_ZF_1_3_L6 Int_ZF_1_1_L4
  by simp;

text{*Product of positive integers is positive.*}

lemma (in int0) Int_ZF_1_3_L8: 
  assumes "a\<in>\<int>"  "b\<in>\<int>" 
  and "\<not>(a\<lsq>\<zero>)"  "\<not>(b\<lsq>\<zero>)"
  shows "\<not>((a\<cdot>b) \<lsq> \<zero>)"
  using prems Int_ZF_1_3_L7 Int_ZF_1_3_L3 Int_ZF_1_1_L5 Int_ZF_1_3_L7
  by simp;

text{*If $a\cdot b$ is nonnegative and $b$ is positive, then $a$ is 
  nonnegative. Proof by contradiction.*}

lemma (in int0) Int_ZF_1_3_L9:
  assumes A1: "a\<in>\<int>"  "b\<in>\<int>"
  and A2:  "\<not>(b\<lsq>\<zero>)" and A3: "a\<cdot>b \<lsq> \<zero>" 
  shows "a\<lsq>\<zero>"
proof -
  { assume "\<not>(a\<lsq>\<zero>)"
    with A1 A2 have "\<not>((a\<cdot>b) \<lsq> \<zero>)" using Int_ZF_1_3_L8
      by simp;
  } with A3 show "a\<lsq>\<zero>" by auto;
qed;

text{*One integer is less or equal another iff the difference is nonpositive.*}

lemma (in int0) Int_ZF_1_3_L10:
  assumes "a\<in>\<int>"  "b\<in>\<int>"
  shows "a\<lsq>b \<longleftrightarrow> a\<rs>b \<lsq> \<zero>"
  using prems Int_ZF_2_T1 group3.OrderedGroup_ZF_1_L9
  by simp;

text{*We can simplify out a positive element on both sides of an 
  inequality.*}

lemma (in int0) Int_ineq_simpl_positive: 
  assumes A1: "a\<in>\<int>"  "b\<in>\<int>"  "c\<in>\<int>" 
  and A2: "a\<cdot>c \<lsq> b\<cdot>c" and A4: "\<not>(c\<lsq>\<zero>)"
  shows "a \<lsq> b"
proof -
  from A1 A4 have "a\<rs>b \<in>  \<int>"  "c\<in>\<int>"  "\<not>(c\<lsq>\<zero>)"
    using Int_ZF_1_1_L5 by auto;
  moreover from A1 A2 have "(a\<rs>b)\<cdot>c \<lsq> \<zero>"
    using Int_ZF_1_1_L5 Int_ZF_1_3_L10 Int_ZF_1_1_L6
    by simp;
  ultimately have "a\<rs>b \<lsq> \<zero>" by (rule Int_ZF_1_3_L9);
  with A1 show "a \<lsq> b" using Int_ZF_1_3_L10 by simp;
qed;

text{*A technical lemma about conclusion from an inequality between
  absolute values. This is a property of ordered rings.*}

lemma (in int0) Int_ZF_1_3_L11:
  assumes A1: "a\<in>\<int>"  "b\<in>\<int>"
  and A2: "\<not>(abs(a) \<lsq> abs(b))"
  shows "\<not>(abs(a) \<lsq> \<zero>)"
proof -
  { assume "abs(a) \<lsq> \<zero>"
    moreover from A1 have "\<zero> \<lsq> abs(a)" using Int_ZF_2_L22
      by simp;
    ultimately have "abs(a) = \<zero>" by (rule Int_ZF_2_L3);
    with A1 A2 have False using Int_ZF_2_L22 by simp;
  } then show  "\<not>(abs(a) \<lsq> \<zero>)" by auto;
qed;

text{*Negative times positive is negative. This a property of ordered rings.*}

lemma (in int0) Int_ZF_1_3_L12:
  assumes "a\<lsq>\<zero>"  and "\<zero>\<lsq>b"
  shows "a\<cdot>b \<lsq> \<zero>"
  using prems Int_ZF_1_3_T1 ring1.OrdRing_ZF_1_L8 
  by simp;

text{*We can multiply an inequality by a nonnegative number. 
  This a property of ordered rings.*}

lemma (in int0) Int_ZF_1_3_L13:
  assumes A1: "a\<lsq>b" and A2: "\<zero>\<lsq>c"
  shows 
  "a\<cdot>c \<lsq> b\<cdot>c"
  "c\<cdot>a \<lsq> c\<cdot>b"
  using prems Int_ZF_1_3_T1 ring1.OrdRing_ZF_1_L9 by auto;

text{*A technical lemma about decreasing a factor in an inequality.*}

lemma (in int0) Int_ZF_1_3_L13A:
  assumes "\<one>\<lsq>a" and "b\<lsq>c" and "(a\<ra>\<one>)\<cdot>c \<lsq> d"
  shows "(a\<ra>\<one>)\<cdot>b \<lsq> d"
proof -
  from prems have 
    "(a\<ra>\<one>)\<cdot>b \<lsq> (a\<ra>\<one>)\<cdot>c"
    "(a\<ra>\<one>)\<cdot>c \<lsq> d"
    using Int_ZF_2_L16C Int_ZF_1_3_L13 by auto;
  then show "(a\<ra>\<one>)\<cdot>b \<lsq> d" by (rule Int_order_transitive);
qed;

text{*A rearrangement with four integers and absolute value.*}

lemma (in int0) Int_ZF_1_3_L14:
  assumes  A1: "a\<in>\<int>"  "b\<in>\<int>"  "c\<in>\<int>"  "d\<in>\<int>" 
  shows "abs(a\<cdot>b)\<ra>(abs(a)\<ra>c)\<cdot>d = (d\<ra>abs(b))\<cdot>abs(a)\<ra>c\<cdot>d"
proof -;
  from A1 have T1: 
    "abs(a) \<in> \<int>"  "abs(b) \<in> \<int>" 
    "abs(a)\<cdot>abs(b) \<in> \<int>" 
    "abs(a)\<cdot>d \<in> \<int>" 
    "c\<cdot>d \<in> \<int>"
    "abs(b)\<ra>d \<in> \<int>" 
    using Int_ZF_2_L14 Int_ZF_1_1_L5 by auto;
  with A1 have "abs(a\<cdot>b)\<ra>(abs(a)\<ra>c)\<cdot>d = abs(a)\<cdot>(abs(b)\<ra>d)\<ra>c\<cdot>d"
    using Int_ZF_1_3_L5 Int_ZF_1_1_L1 Int_ZF_1_1_L7 by simp;
  with A1 T1 show ?thesis using Int_ZF_1_1_L5 by simp;
qed;

text{*A technical lemma about what happens when one absolute value is
  not greater or equal than another.*}

lemma (in int0) Int_ZF_1_3_L15: assumes A1: "m\<in>\<int>" "n\<in>\<int>"
  and A2: "\<not>(abs(m) \<lsq> abs(n))"
  shows "n \<lsq> abs(m)"  "m\<noteq>\<zero>" 
proof -
  from A1 have T1: "n \<lsq> abs(n)" 
    using Int_ZF_2_L19C by simp;
  from A1 have "abs(n) \<in> \<int>"  "abs(m) \<in> \<int>"
    using Int_ZF_2_L14 by auto;
  moreover from A2 have "\<not>(abs(m) \<lsq> abs(n))" .
  ultimately have "abs(n) \<lsq> abs(m)"
    by (rule Int_ZF_2_L19);
  with T1 show  "n \<lsq> abs(m)" by (rule Int_order_transitive);
  from A1 A2 show "m\<noteq>\<zero>" using Int_ZF_2_L18 Int_ZF_2_L22 by auto;
qed;

text{*Negative of a nonnegative is nonpositive.*}

lemma (in int0) Int_ZF_1_3_L16: assumes A1: "\<zero> \<lsq> m"
  shows "(\<rm>m) \<lsq> \<zero>"
proof -
  from A1 have "(\<rm>m) \<lsq> (\<rm>\<zero>)"
    using Int_ZF_2_L10 by simp;
  then show "(\<rm>m) \<lsq> \<zero>" using Int_ZF_1_L11
    by simp;
qed;

text{*Some statements about intervals centered at $0$.*}

lemma (in int0) Int_ZF_1_3_L17: assumes A1: "m\<in>\<int>"
  shows 
  "(\<rm>abs(m)) \<lsq> abs(m)"
  "(\<rm>abs(m))..abs(m) \<noteq> 0"
proof -
  from A1 have "(\<rm>abs(m)) \<lsq> \<zero>"  "\<zero> \<lsq> abs(m)" 
    using Int_ZF_2_L22 Int_ZF_1_3_L16 by auto;
  then show "(\<rm>abs(m)) \<lsq> abs(m)" by (rule Int_order_transitive);
  then have "abs(m) \<in> (\<rm>abs(m))..abs(m)"
    using Int_ZF_2_L2 Int_ZF_2_L1A Order_ZF_2_L2 by simp;
  thus "(\<rm>abs(m))..abs(m) \<noteq> 0" by auto;
qed

text{*The greater of two integers is indeed greater than both, 
  and the smaller one is smaller that both.*}

lemma (in int0) Int_ZF_1_3_L18: assumes A1: "m\<in>\<int>"  "n\<in>\<int>"
  shows 
  "m \<lsq> GreaterOf(IntegerOrder,m,n)"
  "n \<lsq> GreaterOf(IntegerOrder,m,n)"
  "SmallerOf(IntegerOrder,m,n) \<lsq> m"
  "SmallerOf(IntegerOrder,m,n) \<lsq> n"
  using prems Int_ZF_2_T1 Order_ZF_3_L2 by auto;

text{*If $|m| \leq n$, then $m \in -n..n$.*}

lemma (in int0) Int_ZF_1_3_L19: 
  assumes A1: "m\<in>\<int>" and A2: "abs(m) \<lsq> n"
  shows 
  "(\<rm>n) \<lsq> m"  "m \<lsq> n"
  "m \<in> (\<rm>n)..n"
  using prems Int_ZF_2_T1 group3.OrderedGroup_ZF_3_L8  
    group3.OrderedGroup_ZF_3_L8A Order_ZF_2_L1 
  by auto;

text{* A slight generalization of the above lemma. *}

lemma (in int0) Int_ZF_1_3_L19A: 
  assumes A1: "m\<in>\<int>" and A2: "abs(m) \<lsq> n" and A3: "\<zero>\<lsq>k"
  shows "(\<rm>(n\<ra>k)) \<lsq> m"
  using prems Int_ZF_2_T1 group3.OrderedGroup_ZF_3_L8B
  by simp;

text{*Sets of integers that have absolute value bounded are bounded.*}

lemma (in int0) Int_ZF_1_3_L20:
  assumes A1: "\<forall>x\<in>X. b(x) \<in> \<int> \<and> abs(b(x)) \<lsq> L"
  shows "IsBounded({b(x). x\<in>X},IntegerOrder)"
proof -;
  let ?G = "\<int>"
  let ?P = "IntegerAddition"
  let ?r = "IntegerOrder"
  from A1 have
    "group3(?G, ?P, ?r)"
    "?r {is total on} ?G"
    "\<forall>x\<in>X. b(x) \<in> ?G \<and> \<langle>AbsoluteValue(?G, ?P, ?r) ` b(x), L\<rangle> \<in> ?r"
    using Int_ZF_2_T1 by auto;
  then show "IsBounded({b(x). x\<in>X},IntegerOrder)"
    by (rule group3.OrderedGroup_ZF_3_L9A);
qed;

text{*If a set is bounded, then the absolute values of the elements of that
  set are bounded.*}

lemma (in int0) Int_ZF_1_3_L20A: assumes "IsBounded(A,IntegerOrder)"
  shows "\<exists>L. \<forall>a\<in>A. abs(a) \<lsq> L"
  using prems Int_ZF_2_T1 group3.OrderedGroup_ZF_3_L10A 
  by simp;

text{*Absolute vaues of integers from a finite image of integers are bounded
  by an integer.*}

lemma (in int0) Int_ZF_1_3_L20AA: 
  assumes A1: "{b(x). x\<in>\<int>} \<in> Fin(\<int>)"
  shows "\<exists>L\<in>\<int>. \<forall>x\<in>\<int>. abs(b(x)) \<lsq> L"
  using prems int_not_empty Int_ZF_2_T1 group3.OrderedGroup_ZF_3_L11A
  by simp;

text{*If absolute values of values of some integer function are bounded,
  then the image a set from the domain is a bounded set.*}

lemma (in int0) Int_ZF_1_3_L20B:
  assumes "f:X\<rightarrow>\<int>" and "A\<subseteq>X" and "\<forall>x\<in>A. abs(f`(x)) \<lsq> L"
  shows  "IsBounded(f``(A),IntegerOrder)"
proof -
  let ?G = "\<int>"
  let ?P = "IntegerAddition"
  let ?r = "IntegerOrder"
  from prems have 
    "group3(?G, ?P, ?r)"
    "?r {is total on} ?G"
    "f:X\<rightarrow>?G"
    "A\<subseteq>X"
    "\<forall>x\<in>A. \<langle>AbsoluteValue(?G, ?P, ?r)`(f`(x)), L\<rangle> \<in> ?r"
    using Int_ZF_2_T1 by auto;
  then show "IsBounded(f``(A), ?r)" 
    by (rule group3.OrderedGroup_ZF_3_L9B);
qed;

text{*A special case of the previous lemma for a function from integers to 
  integers.*}

corollary (in int0) Int_ZF_1_3_L20C:
  assumes "f:\<int>\<rightarrow>\<int>" and "\<forall>m\<in>\<int>. abs(f`(m)) \<lsq> L"
  shows "f``(\<int>) \<in> Fin(\<int>)"
proof -
  from prems have "f:\<int>\<rightarrow>\<int>" "\<int> \<subseteq> \<int>"  "\<forall>m\<in>\<int>. abs(f`(m)) \<lsq> L"
    by auto;
  then have "IsBounded(f``(\<int>),IntegerOrder)"
    by (rule Int_ZF_1_3_L20B);
  then show "f``(\<int>) \<in> Fin(\<int>)" using Int_bounded_iff_fin
    by simp;
qed;

text{*A triangle inequality with three integers. Property of ordered rings.*}

lemma (in int0) int_triangle_ineq3: 
  assumes A1: "a\<in>\<int>"  "b\<in>\<int>"  "c\<in>\<int>"
  shows "abs(a\<rs>b\<rs>c) \<lsq> abs(a) \<ra> abs(b) \<ra> abs(c)"
proof -
  from A1 have T: "a\<rs>b \<in> \<int>"  "abs(c) \<in> \<int>"
    using Int_ZF_1_1_L5 Int_ZF_2_L14 by auto;
  with A1 have "abs(a\<rs>b\<rs>c) \<lsq> abs(a\<rs>b) \<ra> abs(c)"
    using Int_triangle_ineq1 by simp;
  moreover from A1 T have 
    "abs(a\<rs>b) \<ra> abs(c) \<lsq>  abs(a) \<ra> abs(b) \<ra> abs(c)"
    using Int_triangle_ineq1 int_ord_transl_inv by simp;
  ultimately show ?thesis by (rule Int_order_transitive);
qed;

text{*If $a\leq c$ and $b\leq c$, then $a+b\leq 2\cdot c$. 
  Property of ordered rings.*}

lemma (in int0) Int_ZF_1_3_L21:
  assumes A1: "a\<lsq>c"  "b\<lsq>c" shows "a\<ra>b \<lsq> \<two>\<cdot>c"
  using prems Int_ZF_1_3_T1 ring1.OrdRing_ZF_2_L6 by simp;

section{*Maximum and minimum of a set of integers*}

text{*In this section we provide some sufficient conditions for
  integer subsets to have extrema (maxima and minima).*}

text{*Finite nonempty subsets of integers attain maxima and minima.*}

theorem (in int0) Int_fin_have_max_min:
  assumes A1: "A \<in> Fin(\<int>)" and A2: "A\<noteq>0"
  shows 
  "HasAmaximum(IntegerOrder,A)"
  "HasAminimum(IntegerOrder,A)"  
  "Maximum(IntegerOrder,A) \<in> A"
  "Minimum(IntegerOrder,A) \<in> A"
  "\<forall>x\<in>A. x \<lsq> Maximum(IntegerOrder,A)"
  "\<forall>x\<in>A. Minimum(IntegerOrder,A) \<lsq> x"
  "Maximum(IntegerOrder,A) \<in> \<int>"
  "Minimum(IntegerOrder,A) \<in> \<int>"
proof -
  from A1 have 
    "A=0 \<or> HasAmaximum(IntegerOrder,A)" and
    "A=0 \<or> HasAminimum(IntegerOrder,A)"
    using Int_ZF_2_T1 Int_ZF_2_L6 Finite_ZF_1_1_T1A Finite_ZF_1_1_T1B
    by auto;
  with A2 show 
    "HasAmaximum(IntegerOrder,A)"
    "HasAminimum(IntegerOrder,A)"  
    by auto;
  from A1 A2 show 
    "Maximum(IntegerOrder,A) \<in> A"
    "Minimum(IntegerOrder,A) \<in> A"
    "\<forall>x\<in>A. x \<lsq> Maximum(IntegerOrder,A)"
    "\<forall>x\<in>A. Minimum(IntegerOrder,A) \<lsq> x"
    using Int_ZF_2_T1 Finite_ZF_1_T2 by auto;
  moreover from A1 have "A\<subseteq>\<int>" using FinD by simp;
  ultimately show 
    "Maximum(IntegerOrder,A) \<in> \<int>"
    "Minimum(IntegerOrder,A) \<in> \<int>"
    by auto;
qed;

text{*Bounded nonempty integer subsets attain maximum and minimum.*}

theorem (in int0) Int_bounded_have_max_min:
  assumes A1: "IsBounded(A,IntegerOrder)" and A2: "A\<noteq>0"
  shows 
  "HasAmaximum(IntegerOrder,A)"
  "HasAminimum(IntegerOrder,A)"  
  "Maximum(IntegerOrder,A) \<in> A"
  "Minimum(IntegerOrder,A) \<in> A"
  "\<forall>x\<in>A. x \<lsq> Maximum(IntegerOrder,A)"
  "\<forall>x\<in>A. Minimum(IntegerOrder,A) \<lsq> x"
  "Maximum(IntegerOrder,A) \<in> \<int>"
  "Minimum(IntegerOrder,A) \<in> \<int>"
  using prems Int_fin_have_max_min Int_bounded_iff_fin
  by auto;

text{*A set defined by separation over a bounded set attains its maximum
  and minimum.*}

lemma (in int0) Int_ZF_1_4_L1:
  assumes A1: "IsBounded(A,IntegerOrder)" and A2: "A\<noteq>0"
  and A3: "\<forall>q\<in>\<int>. F(q) \<in> \<int>"
  and A4: "K = {F(q). q \<in> A}"
  shows
  "HasAmaximum(IntegerOrder,K)"
  "HasAminimum(IntegerOrder,K)"  
  "Maximum(IntegerOrder,K) \<in> K"
  "Minimum(IntegerOrder,K) \<in> K"
  "Maximum(IntegerOrder,K) \<in> \<int>"
  "Minimum(IntegerOrder,K) \<in> \<int>"
  "\<forall>q\<in>A. F(q) \<lsq> Maximum(IntegerOrder,K)"
  "\<forall>q\<in>A. Minimum(IntegerOrder,K) \<lsq> F(q)"
  "IsBounded(K,IntegerOrder)"
proof -
  from A1 have "A \<in> Fin(\<int>)" using Int_bounded_iff_fin
    by simp;
  with A3 have "{F(q). q \<in> A} \<in> Fin(\<int>)"
    by (rule Finite1_L6);
  with A2 A4 have T1: "K \<in> Fin(\<int>)"  "K\<noteq>0" by auto;
  then show T2: 
    "HasAmaximum(IntegerOrder,K)"
    "HasAminimum(IntegerOrder,K)"  
    and "Maximum(IntegerOrder,K) \<in> K"
    "Minimum(IntegerOrder,K) \<in> K"
    "Maximum(IntegerOrder,K) \<in> \<int>"
    "Minimum(IntegerOrder,K) \<in> \<int>"
    using Int_fin_have_max_min by auto;
  { fix q assume "q\<in>A" 
    with A4 have "F(q) \<in> K" by auto;
    with T1 have 
      "F(q) \<lsq> Maximum(IntegerOrder,K)"
      "Minimum(IntegerOrder,K) \<lsq> F(q)"
      using Int_fin_have_max_min by auto;
  } then show 
      "\<forall>q\<in>A. F(q) \<lsq> Maximum(IntegerOrder,K)"
      "\<forall>q\<in>A. Minimum(IntegerOrder,K) \<lsq> F(q)"
    by auto;
  from T2 show "IsBounded(K,IntegerOrder)"
    using Order_ZF_4_L7 Order_ZF_4_L8A IsBounded_def
    by simp;
qed;

text{*Integer functions attain maxima and minima over intervals.*}

lemma (in int0) Int_ZF_1_4_L2:
  assumes A1: "f:\<int>\<rightarrow>\<int>" and A2: "a\<lsq>b"
  shows
  "maxf(f,a..b) \<in> \<int>"
  "\<forall>c \<in> a..b. f`(c) \<lsq> maxf(f,a..b)"
  "\<exists>c \<in> a..b. f`(c) = maxf(f,a..b)"
  "minf(f,a..b) \<in> \<int>"
  "\<forall>c \<in> a..b. minf(f,a..b) \<lsq> f`(c)"
  "\<exists>c \<in> a..b. f`(c) = minf(f,a..b)"
proof -
  from A2 have T: "a\<in>\<int>"  "b\<in>\<int>"  "a..b \<subseteq> \<int>"
    using Int_ZF_2_L1A Int_ZF_2_L1B Order_ZF_2_L6 
    by auto;
  with A1 A2 have
    "Maximum(IntegerOrder,f``(a..b)) \<in> f``(a..b)" 
    "\<forall>x\<in>f``(a..b). x \<lsq> Maximum(IntegerOrder,f``(a..b))"
    "Maximum(IntegerOrder,f``(a..b)) \<in> \<int>"
    "Minimum(IntegerOrder,f``(a..b)) \<in> f``(a..b)" 
    "\<forall>x\<in>f``(a..b). Minimum(IntegerOrder,f``(a..b)) \<lsq> x"
    "Minimum(IntegerOrder,f``(a..b)) \<in> \<int>"
    using Int_ZF_4_L8 Int_ZF_2_T1 group3.OrderedGroup_ZF_2_L6
      Int_fin_have_max_min by auto;
  with A1 T show
    "maxf(f,a..b) \<in> \<int>"
    "\<forall>c \<in> a..b. f`(c) \<lsq> maxf(f,a..b)"
    "\<exists>c \<in> a..b. f`(c) = maxf(f,a..b)"
    "minf(f,a..b) \<in> \<int>"
    "\<forall>c \<in> a..b. minf(f,a..b) \<lsq> f`(c)"
    "\<exists>c \<in> a..b. f`(c) = minf(f,a..b)"
    using func_imagedef by auto;
qed;
  
section{*The set of nonnegative integers*}

text{*The set of nonnegative integers looks like the set of natural numbers.
  We explore that in this section. We also rephrasse some lemmas abou the set
  of positive integers known from the theory of oredered grups.*}

text{*@{text "\<int>\<^sup>+"} is bounded below.*}

lemma (in int0) Int_ZF_1_5_L1: shows 
  "IsBoundedBelow(\<int>\<^sup>+,IntegerOrder)"
  "IsBoundedBelow(\<int>\<^sub>+,IntegerOrder)"
  using Nonnegative_def PositiveSet_def IsBoundedBelow_def by auto;

text{*Subsets of @{text "\<int>\<^sup>+"} are bounded below.*}

lemma (in int0) Int_ZF_1_5_L1A: assumes A1: "A \<subseteq> \<int>\<^sup>+" 
  shows "IsBoundedBelow(A,IntegerOrder)"
  using A1 Int_ZF_1_5_L1 Order_ZF_3_L12 by blast;

text{*Subsets of @{text "\<int>\<^sub>+"} are bounded below.*}

lemma (in int0) Int_ZF_1_5_L1B: assumes A1: "A \<subseteq> \<int>\<^sub>+" 
  shows "IsBoundedBelow(A,IntegerOrder)"
  using A1 Int_ZF_1_5_L1 Order_ZF_3_L12 by blast;

text{*Infinite subsets of $Z^+$ do not have a maximum - If $A\subseteq Z^+$
  then for every integer we can find one in the set that is not smaller.*}

lemma (in int0) Int_ZF_1_5_L2:
  assumes A1: "A \<subseteq> \<int>\<^sup>+"  and A2: "A \<notin> Fin(\<int>)" and A3: "D\<in>\<int>"
  shows "\<exists>n\<in>A. D\<lsq>n"
proof -
  { assume "\<forall>n\<in>A. \<not>(D\<lsq>n)" 
    moreover from A1 A3 have "D\<in>\<int>"  "\<forall>n\<in>A. n\<in>\<int>" 
      using Nonnegative_def by auto;
    ultimately have "\<forall>n\<in>A. n\<lsq>D"
      using Int_ZF_2_L19 by blast;
    hence "\<forall>n\<in>A. \<langle>n,D\<rangle> \<in> IntegerOrder" by simp;
    then have "IsBoundedAbove(A,IntegerOrder)"
      by (rule Order_ZF_3_L10);
    with A1 A2 have False using Int_ZF_1_5_L1A IsBounded_def 
      Int_bounded_iff_fin by auto;
  } thus ?thesis by auto;
qed;

text{*Infinite subsets of $Z_+$ do not have a maximum - If $A\subseteq Z_+$
  then for every integer we can find one in the set that is not smaller.  This is very similar to @{text "Int_ZF_1_5_L2"}, except we have @{text "\<int>\<^sub>+"}
  instead of @{text "\<int>\<^sup>+"} here.*}

lemma (in int0) Int_ZF_1_5_L2A:
  assumes A1: "A \<subseteq> \<int>\<^sub>+"  and A2: "A \<notin> Fin(\<int>)" and A3: "D\<in>\<int>"
  shows "\<exists>n\<in>A. D\<lsq>n"
proof -
{ assume "\<forall>n\<in>A. \<not>(D\<lsq>n)" 
    moreover from A1 A3 have "D\<in>\<int>"  "\<forall>n\<in>A. n\<in>\<int>" 
      using PositiveSet_def by auto;
    ultimately have "\<forall>n\<in>A. n\<lsq>D"
      using Int_ZF_2_L19 by blast;
    hence "\<forall>n\<in>A. \<langle>n,D\<rangle> \<in> IntegerOrder" by simp;
    then have "IsBoundedAbove(A,IntegerOrder)"
      by (rule Order_ZF_3_L10);
    with A1 A2 have False using Int_ZF_1_5_L1B IsBounded_def 
      Int_bounded_iff_fin by auto;
  } thus ?thesis by auto;
qed;

text{*An integer is either positive, zero, or its opposite is postitive.*}

lemma (in int0) Int_decomp: assumes "m\<in>\<int>"
  shows "Exactly_1_of_3_holds (m=\<zero>,m\<in>\<int>\<^sub>+,(\<rm>m)\<in>\<int>\<^sub>+)"
  using prems Int_ZF_2_T1 group3.OrdGroup_decomp 
  by simp;


text{*An integer is in the positive set iff it is greater or equal one.*}

lemma (in int0) Int_ZF_1_5_L3: shows "m\<in>\<int>\<^sub>+ \<longleftrightarrow> \<one>\<lsq>m"
proof
  assume "m\<in>\<int>\<^sub>+" then have "\<zero>\<lsq>m"  "m\<noteq>\<zero>"
    using PositiveSet_def by auto;
  then have "\<zero>\<ra>\<one> \<lsq> m" 
    using Int_ZF_4_L1B by auto;
  then show "\<one>\<lsq>m" 
    using int_zero_one_are_int Int_ZF_1_T2 group0.group0_2_L2
    by simp;
next assume "\<one>\<lsq>m"
  then have "m\<in>\<int>"  "\<zero>\<lsq>m"  "m\<noteq>\<zero>"
    using Int_ZF_2_L1A Int_ZF_2_L16C by auto;
  then show "m\<in>\<int>\<^sub>+" using PositiveSet_def by auto;
qed;

text{*One and two are in @{text "\<int>\<^sub>+"}.*}

lemma (in int0) Int_ZF_1_5_L4: shows "\<one> \<in> \<int>\<^sub>+"  "\<two> \<in> \<int>\<^sub>+"
  using int_zero_one_are_int Int_ZF_2_L2 refl_def Int_ZF_1_5_L3
  Int_ZF_2_L16B by auto;

text{*The image of @{text "\<int>\<^sub>+"} by a function defined on integers 
  is not empty.*}

lemma (in int0) Int_ZF_1_5_L5: assumes A1: "f : \<int>\<rightarrow>X"
  shows "f``(\<int>\<^sub>+) \<noteq> 0"
proof -
  have "\<int>\<^sub>+ \<subseteq> \<int>" using PositiveSet_def by auto;
  with A1 show "f``(\<int>\<^sub>+) \<noteq> 0"
    using Int_ZF_1_5_L4 func_imagedef by auto;
qed;

text{*If $n$ is positive, then $n-1$ is nonnegative.*}

lemma (in int0) Int_ZF_1_5_L6: assumes A1: "n \<in> \<int>\<^sub>+"
  shows 
  "\<zero> \<lsq> n\<rs>\<one>"
  "\<zero> \<in> \<zero>..(n\<rs>\<one>)"
  "\<zero>..(n\<rs>\<one>) \<subseteq> \<int>"
proof -
  from A1 have "\<one> \<lsq> n"  "(\<rm>\<one>) \<in> \<int>"
    using Int_ZF_1_5_L3 int_zero_one_are_int Int_ZF_1_1_L4 
    by auto;
  then have "\<one>\<rs>\<one> \<lsq> n\<rs>\<one>"
    using int_ord_transl_inv by simp;
  then show "\<zero> \<lsq> n\<rs>\<one>"
    using int_zero_one_are_int Int_ZF_1_1_L4 by simp;
  then show "\<zero> \<in> \<zero>..(n\<rs>\<one>)"
    using int_zero_one_are_int Int_ZF_2_L2 refl_def Order_ZF_2_L1B
    by simp;
  show "\<zero>..(n\<rs>\<one>) \<subseteq> \<int>"
    using Int_ZF_2_L1B Order_ZF_2_L6 by simp;
qed;

text{*Intgers greater than one in @{text "\<int>\<^sub>+"} belong to @{text "\<int>\<^sub>+"}.
  This is a property of ordered groups and follows from 
  @{text "OrderedGroup_ZF_1_L19"}, but Isabelle's simplifier has problems 
  using that result directly, so we reprove it specifically for integers.*}

lemma (in int0) Int_ZF_1_5_L7:  assumes "a \<in> \<int>\<^sub>+" and "a\<lsq>b"
  shows "b \<in> \<int>\<^sub>+"
proof-
  from prems have "\<one>\<lsq>a"  "a\<lsq>b"
    using Int_ZF_1_5_L3 by auto;
  then have "\<one>\<lsq>b" by (rule Int_order_transitive);
  then show "b \<in> \<int>\<^sub>+" using Int_ZF_1_5_L3 by simp;
qed;

text{*The opposite of an element of @{text "\<int>\<^sub>+"} cannot belong to
  @{text "\<int>\<^sub>+"}.*}

lemma (in int0) Int_ZF_1_5_L8: assumes "a \<in> \<int>\<^sub>+"
  shows "(\<rm>a) \<notin> \<int>\<^sub>+"
  using prems Int_ZF_2_T1 group3.OrderedGroup_ZF_1_L20
  by simp;

section{*Miscelaneous*}

text{*In this section we put some technical lemmas needed in various other 
  places that are hard to classify.*}

text{*Suppose we have an integer expression (a meta-function)$F$ such that 
  $F(p)|p|$ is bounded by a linear function of $|p|$, that is for some integers
  $A,B$ we have $F(p)|p|\leq A|p|+B.$ We show that $F$ is then bounded. 
  The proof is easy, we just divide both sides by $|p|$ 
  and take the limit (just kidding). *}

lemma (in int0) Int_ZF_1_6_L1:
  assumes A1: "\<forall>q\<in>\<int>. F(q) \<in> \<int>"
  and A2:  "\<forall>q\<in>\<int>. F(q)\<cdot>abs(q) \<lsq> A\<cdot>abs(q) \<ra> B"
  and A3: "A\<in>\<int>"  "B\<in>\<int>"
  shows "\<exists>L. \<forall>p\<in>\<int>. F(p) \<lsq> L"
proof -
  let ?I = "(\<rm>abs(B))..abs(B)"
  def DefK: K == "{F(q). q \<in> ?I}"
  let ?M = "Maximum(IntegerOrder,K)"
  let ?L = "GreaterOf(IntegerOrder,?M,A\<ra>\<one>)"
  from A3 A1 DefK have C1:
    "IsBounded(?I,IntegerOrder)"   
    "?I \<noteq> 0"
    "\<forall>q\<in>\<int>. F(q) \<in> \<int>"
    "K = {F(q). q \<in> ?I}"
    using Order_ZF_3_L11 Int_ZF_1_3_L17 by auto;
  then have "?M \<in> \<int>" by (rule Int_ZF_1_4_L1);
  with A3 have T1: "?M \<lsq> ?L"  "A\<ra>\<one> \<lsq> ?L"
    using int_zero_one_are_int Int_ZF_1_1_L5 Int_ZF_1_3_L18
    by auto;
  from C1 have T2: "\<forall>q\<in>?I. F(q) \<lsq> ?M"
    by (rule Int_ZF_1_4_L1);
  { fix p assume A4: "p\<in>\<int>" have "F(p) \<lsq> ?L" 
    proof (cases "abs(p) \<lsq> abs(B)")
      assume "abs(p) \<lsq> abs(B)"
      with A4 T1 T2 have "F(p) \<lsq> ?M"  "?M \<lsq> ?L"
	using Int_ZF_1_3_L19 by auto;
      then show "F(p) \<lsq> ?L" by (rule Int_order_transitive);
    next assume A5: "\<not>(abs(p) \<lsq> abs(B))"
      from A3 A2 A4 have 
	"A\<cdot>abs(p) \<in> \<int>"  "F(p)\<cdot>abs(p) \<lsq> A\<cdot>abs(p) \<ra> B"
	using Int_ZF_2_L14 Int_ZF_1_1_L5 by auto;
      moreover from A3 A4 A5 have "B \<lsq> abs(p)"
	using Int_ZF_1_3_L15 by simp;
      ultimately have
	"F(p)\<cdot>abs(p) \<lsq> A\<cdot>abs(p) \<ra> abs(p)"
	using Int_ZF_2_L15B by blast;
      with A3 A4 have "F(p)\<cdot>abs(p) \<lsq> (A\<ra>\<one>)\<cdot>abs(p)"
	using Int_ZF_2_L14 Int_ZF_1_2_L7 by simp;
      moreover from A3 A1 A4 A5 have 
	"F(p) \<in> \<int>"  "A\<ra>\<one> \<in> \<int>"  "abs(p) \<in> \<int>"
	 "\<not>(abs(p) \<lsq> \<zero>)"
	using int_zero_one_are_int Int_ZF_1_1_L5 Int_ZF_2_L14 Int_ZF_1_3_L11
	by auto;
      ultimately have "F(p) \<lsq> A\<ra>\<one>"
	using Int_ineq_simpl_positive by simp;
      moreover from T1 have  "A\<ra>\<one> \<lsq> ?L" by simp;
      ultimately show "F(p) \<lsq> ?L" by (rule Int_order_transitive);
    qed
 } then have "\<forall>p\<in>\<int>. F(p) \<lsq> ?L" by simp
 thus ?thesis by auto;
qed;
end