(*
    This file is a part of IsarMathLib - 
    a library of formalized mathematics for Isabelle/Isar.

    Copyright (C) 2005, 2006  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{Ring\_ZF.thy}*}

theory Ring_ZF imports Group_ZF 

begin

text{*This theory file covers basic facts about rings.*}

section{*Definition and basic properties*}

text{*In this section we define what is a ring and list the basic properties
  of rings. *}

text{*We say that three sets $(R,A,M)$ form a ring if $(R,A)$ is an abelian 
  group, $(R,M)$ is a monoid and $A$ is distributive with respect to $M$ on 
  $R$. $A$ represents the additive operation on $R$. 
  As such it is a subset of $(R\times R)\times R$ (recall that in ZF set theory
  functions are sets).
  Similarly $M$ represents the multiplicative operation on $R$ and is also
  a subset of $(R\times R)\times R$.
  We  don't require the multiplicative operation to be commutative in the 
  definition of a ring.
  We also define the notion of having no zero divisors.*}

constdefs 
  "IsAring(R,A,M) \<equiv> IsAgroup(R,A) \<and> (A {is commutative on} R) \<and> 
  IsAmonoid(R,M) \<and> IsDistributive(R,A,M)"

  "HasNoZeroDivs(R,A,M) \<equiv> (\<forall>a\<in>R. \<forall>b\<in>R. 
  M`<a,b> = TheNeutralElement(R,A) \<longrightarrow>
  a = TheNeutralElement(R,A) \<or> b = TheNeutralElement(R,A))";

text{*Next we define a locale that will be used when considering rings.*}

locale ring0 =

  fixes R and A and M 
 
  assumes ringAssum: "IsAring(R,A,M)"

  fixes ringa (infixl "\<ra>" 90)
  defines ringa_def [simp]: "a\<ra>b \<equiv> A`<a,b>"

  fixes ringminus ("\<rm> _" 89)
  defines ringminus_def [simp]: "(\<rm>a) \<equiv> GroupInv(R,A)`(a)"

  fixes ringsub (infixl "\<rs>" 90)
  defines ringsub_def [simp]: "a\<rs>b \<equiv> a\<ra>(\<rm>b)"

  fixes ringm (infixl "\<cdot>" 95)
  defines ringm_def [simp]: "a\<cdot>b \<equiv> M`<a,b>"

  fixes ringzero ("\<zero>")
  defines ringzero_def [simp]: "\<zero> \<equiv> TheNeutralElement(R,A)"

  fixes ringone ("\<one>")
  defines ringone_def [simp]: "\<one> \<equiv> TheNeutralElement(R,M)"

  fixes ringtwo ("\<two>")
  defines ringtwo_def [simp]: "\<two> \<equiv> \<one>\<ra>\<one>"

  fixes ringsq ("_\<twosuperior>")
  defines ringsq_def [simp]: "a\<twosuperior> \<equiv> a\<cdot>a";

text{*In the @{text "ring0"} context we can use theorems proven in some 
  other contexts.*}

lemma (in ring0) Ring_ZF_1_L1: shows 
  "monoid0(R,M)"
  "group0(R,A)" 
  "A {is commutative on} R"
  using ringAssum IsAring_def group0_def monoid0_def by auto;


text{*The additive operation in a ring is distributive with respect to the
  multiplicative operation.*}

lemma (in ring0) ring_oper_distr: assumes A1: "a\<in>R"  "b\<in>R"  "c\<in>R"
  shows 
  "a\<cdot>(b\<ra>c) = a\<cdot>b \<ra> a\<cdot>c" 
  "(b\<ra>c)\<cdot>a = b\<cdot>a \<ra> c\<cdot>a"
  using ringAssum prems IsAring_def IsDistributive_def by auto;

text{*Zero and one of the ring are elements of the ring. The negative of zero
  is zero.*}

lemma (in ring0) Ring_ZF_1_L2: 
  shows "\<zero>\<in>R"  "\<one>\<in>R"   "(\<rm>\<zero>) = \<zero>"
  using Ring_ZF_1_L1 group0.group0_2_L2 monoid0.group0_1_L3 
    group0.group0_2_L7A by auto;
  
text{*The next lemma lists some properties of a ring that require one element
  of a ring.*}

lemma (in ring0) Ring_ZF_1_L3: assumes "a\<in>R"
  shows 
  "(\<rm>a) \<in> R"
  "(\<rm>(\<rm>a)) = a"
  "a\<ra>\<zero> = a" 
  "\<zero>\<ra>a = a" 
  "a\<cdot>\<one> = a" 
  "\<one>\<cdot>a = a" 
  "a\<rs>a = \<zero>" 
  "a\<rs>\<zero> = a"
  "\<two>\<cdot>a = a\<ra>a"
  using prems Ring_ZF_1_L1 group0.inverse_in_group group0.group_inv_of_inv 
    group0.group0_2_L6 group0.group0_2_L2 monoid0.group0_1_L3 
    Ring_ZF_1_L2 ring_oper_distr
  by auto;

text{*Properties that require two elements of a ring.*}

lemma (in ring0) Ring_ZF_1_L4: assumes A1: "a\<in>R" "b\<in>R"
  shows 
  "a\<ra>b \<in> R" 
  "a\<rs>b \<in> R" 
  "a\<cdot>b \<in> R" 
  "a\<ra>b = b\<ra>a"
  using ringAssum prems Ring_ZF_1_L1 Ring_ZF_1_L3 
    group0.group0_2_L1 monoid0.group0_1_L1 
    IsAring_def IsCommutative_def
  by auto;

text{*Any element of a ring multiplied by zero is zero.*}

lemma (in ring0) Ring_ZF_1_L6: 
  assumes A1: "x\<in>R" shows "\<zero>\<cdot>x = \<zero>"   "x\<cdot>\<zero> = \<zero>"
proof -
  def D1: a \<equiv> "x\<cdot>\<one>";
  def D2: b \<equiv> "x\<cdot>\<zero>"
  def D3: c \<equiv> "\<one>\<cdot>x"
  def D4: d \<equiv> "\<zero>\<cdot>x"
  from A1 D1 D2 D3 D4 have 
    "a \<ra> b = x\<cdot>(\<one> \<ra> \<zero>)"   "c \<ra> d = (\<one> \<ra> \<zero>)\<cdot>x"
    using Ring_ZF_1_L2 ring_oper_distr by auto;
  moreover from D1 D3 have "x\<cdot>(\<one> \<ra> \<zero>) = a" "(\<one> \<ra> \<zero>)\<cdot>x = c"
    using Ring_ZF_1_L2 Ring_ZF_1_L3 by auto;
  ultimately have "a \<ra> b = a" and T1: "c \<ra> d = c" by auto;
  moreover from A1 D1 D2 D3 D4 have 
    "a \<in> R"  "b \<in> R" and T2: "c \<in> R"  "d \<in> R"
    using Ring_ZF_1_L2 Ring_ZF_1_L4 by auto;
  ultimately have "b = \<zero>" using 
    Ring_ZF_1_L1 group0.group0_2_L7 by simp;
  moreover from T2 T1 have "d = \<zero>" using 
    Ring_ZF_1_L1 group0.group0_2_L7 by simp;
  moreover from D2 D4 have "b = x\<cdot>\<zero>"  "d = \<zero>\<cdot>x" by auto; 
  ultimately show "x\<cdot>\<zero> = \<zero>"  "\<zero>\<cdot>x = \<zero>" by auto;
qed;

text{*Negative can be pulled out of a product.*}

lemma (in ring0) Ring_ZF_1_L7: 
  assumes A1: "a\<in>R"  "b\<in>R"
  shows 
  "(\<rm>a)\<cdot>b = \<rm>(a\<cdot>b)" 
  "a\<cdot>(\<rm>b) = \<rm>(a\<cdot>b)" 
proof -
  from A1 have T1: 
    "a\<cdot>b \<in> R" "(\<rm>a) \<in> R" "((\<rm>a)\<cdot>b) \<in> R" 
    "(\<rm>b) \<in> R" "a\<cdot>(\<rm>b) \<in> R"
    using Ring_ZF_1_L3 Ring_ZF_1_L4 by auto;
  moreover have "(\<rm>a)\<cdot>b \<ra> a\<cdot>b = \<zero>" 
    and T2: "a\<cdot>(\<rm>b) \<ra> a\<cdot>b = \<zero>"
  proof -
    from A1 T1 have 
      "(\<rm>a)\<cdot>b \<ra> a\<cdot>b = ((\<rm>a)\<ra> a)\<cdot>b"
      "a\<cdot>(\<rm>b) \<ra> a\<cdot>b= a\<cdot>((\<rm>b)\<ra>b)"
      using ring_oper_distr by auto;
    moreover from A1 have 
      "((\<rm>a)\<ra> a)\<cdot>b = \<zero>" 
      "a\<cdot>((\<rm>b)\<ra>b) = \<zero>"
      using Ring_ZF_1_L1 group0.group0_2_L6 Ring_ZF_1_L6
      by auto;
    ultimately show 
      "(\<rm>a)\<cdot>b \<ra> a\<cdot>b = \<zero>" 
      "a\<cdot>(\<rm>b) \<ra> a\<cdot>b = \<zero>" 
      by auto;
  qed;
  ultimately show "(\<rm>a)\<cdot>b = \<rm>(a\<cdot>b)"
    using Ring_ZF_1_L1 group0.group0_2_L9 by simp
  from T1 T2 show "a\<cdot>(\<rm>b) = \<rm>(a\<cdot>b)"
    using Ring_ZF_1_L1 group0.group0_2_L9 by simp;   
qed;

text{*Minus times minus is plus.*}

lemma (in ring0) Ring_ZF_1_L7A: assumes "a\<in>R"  "b\<in>R"
  shows "(\<rm>a)\<cdot>(\<rm>b) = a\<cdot>b"
  using prems Ring_ZF_1_L3 Ring_ZF_1_L7 Ring_ZF_1_L4
  by simp;

text{*Subtraction is distributive with respect to multiplication.*}

lemma (in ring0) Ring_ZF_1_L8: assumes "a\<in>R"  "b\<in>R"  "c\<in>R"
  shows 
  "a\<cdot>(b\<rs>c) = a\<cdot>b \<rs> a\<cdot>c"  
  "(b\<rs>c)\<cdot>a = b\<cdot>a \<rs> c\<cdot>a"
  using prems Ring_ZF_1_L3 ring_oper_distr Ring_ZF_1_L7 Ring_ZF_1_L4
  by auto;

text{*Other basic properties involving two elements of a ring.*}

lemma (in ring0) Ring_ZF_1_L9: assumes "a\<in>R"  "b\<in>R"
  shows 
  "(\<rm>b)\<rs>a = (\<rm>a)\<rs>b" 
  "(\<rm>(a\<ra>b)) = (\<rm>a)\<rs>b"  
  "(\<rm>(a\<rs>b)) = ((\<rm>a)\<ra>b)"
  using prems ringAssum IsAring_def Ring_ZF_1_L1 group0.group0_4_L4 
  by auto;

text{*Other basic properties involving three elements of a ring.*}

lemma (in ring0) Ring_ZF_1_L10: 
  assumes "a\<in>R"  "b\<in>R"  "c\<in>R"
  shows 
  "a\<ra>(b\<ra>c) = a\<ra>b\<ra>c"
  (*"a\<ra>(b\<rs>c) = a\<ra>b\<rs>c"*)
  "a\<rs>(b\<ra>c) = a\<rs>b\<rs>c"
  "a\<rs>(b\<rs>c) = a\<rs>b\<ra>c"
  using prems ringAssum Ring_ZF_1_L1 group0.group_oper_assoc 
    IsAring_def group0.group0_4_L4A by auto;

text{*Another property with three elements.*}

lemma (in ring0) Ring_ZF_1_L10A: 
  assumes A1: "a\<in>R"  "b\<in>R"  "c\<in>R"
  shows "a\<ra>(b\<rs>c) = a\<ra>b\<rs>c"
  using prems Ring_ZF_1_L3 Ring_ZF_1_L10 by simp;

text{*Associativity of addition and multiplication.*}

lemma (in ring0) Ring_ZF_1_L11: 
  assumes "a\<in>R"  "b\<in>R"  "c\<in>R"
  shows 
  "a\<ra>b\<ra>c = a\<ra>(b\<ra>c)"
  "a\<cdot>b\<cdot>c = a\<cdot>(b\<cdot>c)"
  using prems ringAssum Ring_ZF_1_L1 group0.group_oper_assoc
    IsAring_def IsAmonoid_def IsAssociative_def
  by auto;

text{*An interpretation of what it means that a ring has 
  no zero divisors.*}

lemma (in ring0) Ring_ZF_1_L12: 
  assumes "HasNoZeroDivs(R,A,M)"
  and "a\<in>R"  "a\<noteq>\<zero>"  "b\<in>R"  "b\<noteq>\<zero>"
  shows "a\<cdot>b\<noteq>\<zero>" 
  using prems HasNoZeroDivs_def by auto;

text{*If a square is nonzero, then the element is nonzero.*}

lemma (in ring0) Ring_ZF_1_L13:
  assumes "a\<in>R"  and "a\<twosuperior> \<noteq> \<zero>"
  shows "a\<noteq>\<zero>"
  using prems Ring_ZF_1_L2 Ring_ZF_1_L6 by auto;

text{*Square of an element and its opposite are the same.*}

lemma (in ring0) Ring_ZF_1_L14:
  assumes "a\<in>R" shows "(\<rm>a)\<twosuperior> = ((a)\<twosuperior>)"
  using prems Ring_ZF_1_L7A by simp;

text{*Adding zero to a set that is closed under addition results
  in a set that is also closed under addition. This is a property of groups.*}

lemma (in ring0) Ring_ZF_1_L15: 
  assumes "H \<subseteq> R" and "H {is closed under} A"
  shows "(H \<union> {\<zero>}) {is closed under} A"
  using prems Ring_ZF_1_L1 group0.group0_2_L17 by simp;

text{*Adding zero to a set that is closed under multiplication results
  in a set that is also closed under moltiplication.*}

lemma (in ring0) Ring_ZF_1_L16:
  assumes A1: "H \<subseteq> R" and A2: "H {is closed under} M"
  shows "(H \<union> {\<zero>}) {is closed under} M"
  using prems Ring_ZF_1_L2 Ring_ZF_1_L6 IsOpClosed_def
  by auto;

section{*Rearrangement lemmas*}

text{*In happens quite often that we want to show a fact like 
  that in commutative rings $(a+b)c+d = (ac+d-e)+(bc+e)$. 
  This is trivial in romantic math and probably there is a way to make 
  it trivial in formalized math. However, I don't know any other way than to
  tediously prove each such rearrangement when it is needed. This section 
  collects facts of this type.*}

text{*Rearrangements with two elements of a ring.*}

lemma (in ring0) Ring_ZF_2_L1: assumes "a\<in>R" "b\<in>R" 
  shows "a\<ra>b\<cdot>a = (b\<ra>\<one>)\<cdot>a"
  using prems Ring_ZF_1_L2 ring_oper_distr Ring_ZF_1_L3 Ring_ZF_1_L4
  by simp;


text{*In commutative rings $a-(b+1)c = (a-d-c)+(d-bc)$. For unknown reasons 
  we have to use the raw set notation in the proof, otherwise all methods 
  fail.*}

lemma (in ring0) Ring_ZF_2_L2: 
  assumes A1: "a\<in>R"  "b\<in>R"  "c\<in>R"  "d\<in>R"
  shows "a\<rs>(b\<ra>\<one>)\<cdot>c = (a\<rs>d\<rs>c)\<ra>(d\<rs>b\<cdot>c)"
proof -;
  def D1: B == "b\<cdot>c" 
  from ringAssum have "A {is commutative on} R"
    using IsAring_def by simp;
  moreover from A1 D1 have "a\<in>R" "B \<in> R" "c\<in>R" "d\<in>R"
    using Ring_ZF_1_L4 by auto;
  ultimately have "A`\<langle>a, GroupInv(R,A)`(A`\<langle>B, c\<rangle>)\<rangle> =
    A`\<langle>A`\<langle>A`\<langle>a, GroupInv(R, A)`(d)\<rangle>,GroupInv(R, A)`(c)\<rangle>,
    A`\<langle>d,GroupInv(R, A)`(B)\<rangle>\<rangle>"
    using Ring_ZF_1_L1 group0.group0_4_L8 by blast;
  with D1 A1 show ?thesis 
    using Ring_ZF_1_L2 ring_oper_distr Ring_ZF_1_L3 by simp;
qed;

text{*Rerrangement about adding linear functions.*}

lemma (in ring0) Ring_ZF_2_L3: 
  assumes A1: "a\<in>R"  "b\<in>R"  "c\<in>R"  "d\<in>R"  "x\<in>R"
  shows "(a\<cdot>x \<ra> b) \<ra> (c\<cdot>x \<ra> d) = (a\<ra>c)\<cdot>x \<ra> (b\<ra>d)"
proof -
  from A1 have 
    "group0(R,A)"
    "A {is commutative on} R"
    "a\<cdot>x \<in> R"  "b\<in>R"  "c\<cdot>x \<in> R"  "d\<in>R" 
    using Ring_ZF_1_L1 Ring_ZF_1_L4 by auto;
  then have "A`\<langle>A`<a\<cdot>x,b>,A`<c\<cdot>x,d>\<rangle> = A`\<langle>A`<a\<cdot>x,c\<cdot>x>,A`<b,d>\<rangle>"
    by (rule group0.group0_4_L8);
  with A1 show 
    "(a\<cdot>x \<ra> b) \<ra> (c\<cdot>x \<ra> d) = (a\<ra>c)\<cdot>x \<ra> (b\<ra>d)"
    using ring_oper_distr by simp;
qed;

end
