(* 
    This file is a part of IsarMathLib - 
    a library of formalized mathematics written for Isabelle/Isar.

    Copyright (C) 2005, 2006  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{func1.thy}*}
theory func1 imports func Fol1 

begin

text{*We define the notion of function that preserves a collection here.
Given two collection of sets a function preserves the collections if 
the inverse image of sets in one collection belongs to the second one.
This notion does not have a name in romantic math. It is used to define 
continuous functions in @{text "Topology_ZF_2"} theory. 
We define it here so that we can 
use it for other purposes, like defining measurable functions.
Recall that @{text "f-``(A)"} means the inverse image of the set $A$.*}

constdefs
  "PresColl(f,S,T) \<equiv> \<forall> A\<in>T. f-``(A)\<in>S";

section{*Properties of functions, function spaces and (inverse) images.*}

text{*If a function maps $A$ into another set, then $A$ is the 
  domain of the function.*}

lemma func1_1_L1: assumes "f:A\<rightarrow>C" shows "domain(f) = A"
  using prems domain_of_fun by simp;

text{*A first-order version of @{text "Pi_type"}. *}

lemma func1_1_L1A: assumes A1: "f:X\<rightarrow>Y" and A2: "\<forall>x\<in>X. f`(x) \<in> Z"
  shows "f:X\<rightarrow>Z"
proof -
  { fix x assume "x\<in>X" 
    with A2 have "f`(x) \<in> Z" by simp };
  with A1 show "f:X\<rightarrow>Z" by (rule Pi_type);
qed;

text{*There is a value for each argument.*}

lemma func1_1_L2: assumes A1: "f:X\<rightarrow>Y"  "x\<in>X" 
  shows "\<exists>y\<in>Y. <x,y> \<in> f";  
proof-
  from A1 have "f`(x) \<in> Y" using apply_type by simp;
  moreover from A1 have "<x,f`(x)>\<in> f" using apply_Pair by simp;
  ultimately show ?thesis by auto;
qed;

text{*Inverse image of any set is contained in the domain.*}

lemma func1_1_L3: assumes A1: "f:X\<rightarrow>Y" shows "f-``(D) \<subseteq> X"
proof-;
   have "\<forall>x. x\<in>f-``(D) \<longrightarrow> x\<in>domain(f)"
      using  vimage_iff domain_iff by auto;
    with A1 have "\<forall>x. (x \<in> f-``(D)) \<longrightarrow> (x\<in>X)" using func1_1_L1 by simp
    then show ?thesis by auto;
qed;

text{*The inverse image of the range is the domain.*}

lemma func1_1_L4: assumes "f:X\<rightarrow>Y" shows "f-``(Y) = X"
  using prems func1_1_L3 func1_1_L2 vimage_iff by blast;

text{*The arguments belongs to the domain and values to the range.*}

lemma func1_1_L5: 
  assumes A1: "<x,y> \<in> f" and A2: "f:X\<rightarrow>Y"  
  shows "x\<in>X \<and> y\<in>Y" 
proof
  from A1 A2 show "x\<in>X" using apply_iff by simp;
  with A2 have "f`(x)\<in> Y" using apply_type by simp;
  with A1 A2 show "y\<in>Y" using apply_iff by simp;
qed;

text{*The (argument, value) pair belongs to the graph of the function.*}

lemma func1_1_L5A: 
  assumes A1: "f:X\<rightarrow>Y" "x\<in>X" "y = f`(x)"
  shows "<x,y> \<in> f" "y \<in> range(f)" 
proof -;
  from A1 show "<x,y> \<in> f" using apply_Pair by simp;
  then show "y \<in> range(f)" using rangeI by simp;
qed;

text{*The range of function thet maps $X$ into $Y$ is contained in $Y$.*}

lemma func1_1_L5B: 
  assumes  A1:"f:X\<rightarrow>Y" shows "range(f) \<subseteq> Y"
proof;
  fix y assume "y \<in> range(f)"
  then obtain x where "<x,y> \<in> f"
    using range_def converse_def domain_def by auto;
  with A1 show "y\<in>Y" using func1_1_L5 by blast;
qed;

text{*The image of any set is contained in the range.*}

lemma func1_1_L6: assumes A1: "f:X\<rightarrow>Y" shows "f``(B)\<subseteq>Y"
proof; 
  fix y
  assume A2: "y \<in> f `` B" show "y \<in> Y";
  proof-;
    from A2 obtain x where "<x,y>\<in> f" using image_iff by auto;
    with A1 show "y \<in> Y" using func1_1_L5 by fast;
  qed;
qed;

text{*The inverse image of any set is contained in the domain.*}

lemma func1_1_L6A: assumes A1: "f:X\<rightarrow>Y" shows "f-``(A)\<subseteq>X"
proof;
  fix x
  assume A2: "x\<in>f-``(A)" then obtain y where "<x,y> \<in> f" 
    using vimage_iff by auto;
  with A1 show  "x\<in>X" using func1_1_L5 by fast;
qed;

text{*Inverse image of a greater set is greater.*}

lemma func1_1_L7: assumes "A\<subseteq>B" and "function(f)" 
  shows "f-``(A)\<subseteq> f-``(B)" using prems function_vimage_Diff by auto;


text{*Image of a greater set is greater.*}

lemma func1_1_L8: assumes A1: "A\<subseteq>B"  shows "f``(A)\<subseteq> f``(B)"
  using prems image_Un by auto;

text{* A set is contained in the the inverse image of its image.
  There is similar theorem in @{text "equalities.thy"}
  (@{text "function_image_vimage"})
  which shows that the image of inverse image of a set 
  is contained in the set.*}

lemma func1_1_L9: assumes A1: "f:X\<rightarrow>Y" and A2: "A\<subseteq>X"
  shows "A \<subseteq> f-``(f``(A))"
proof -
  from A1 A2 have "\<forall>x\<in>A. <x,f`(x)> \<in> f"  using apply_Pair by auto;
  then show ?thesis using image_iff by auto;
qed

text{*A technical lemma needed to make the @{text "func1_1_L11"} 
  proof more clear.*}

lemma func1_1_L10: 
  assumes A1: "f \<subseteq> X\<times>Y" and A2: "\<exists>!y. (y\<in>Y & <x,y> \<in> f)"
  shows "\<exists>!y. <x,y> \<in> f"
proof;
  from A2 show "\<exists>y. \<langle>x, y\<rangle> \<in> f" by auto;
  fix y n assume "<x,y> \<in> f" and "<x,n> \<in> f"
  with A1 A2 show "y=n" by auto;
qed;


text{*If $f\subseteq X\times Y$ and for every $x\in X$ there is exactly 
one $y\in Y$ such that $(x,y)\in f$ then $f$ maps $X$ to $Y$.*}

lemma func1_1_L11: 
  assumes "f \<subseteq> X\<times>Y" and "\<forall>x\<in>X. \<exists>!y. y\<in>Y & <x,y> \<in> f"
  shows "f: X\<rightarrow>Y" using prems func1_1_L10 Pi_iff_old by simp;

text{*A set defined by a lambda-type expression is a fuction. There is a 
  similar lemma in func.thy, but I had problems with lamda expressions syntax
  so I could not apply it. This lemma is a workaround this. Besides, lambda 
  expressions are not readable.
  *}

lemma func1_1_L11A: assumes A1: "\<forall>x\<in>X. b(x)\<in>Y"
  shows "{<x,y> \<in> X\<times>Y. b(x) = y} : X\<rightarrow>Y"
proof -;
  let ?f = "{<x,y> \<in> X\<times>Y. b(x) = y}"
  have "?f \<subseteq> X\<times>Y" by auto;
  moreover have "\<forall>x\<in>X. \<exists>!y. y\<in>Y & <x,y> \<in> ?f"
  proof;
    fix x assume A2: "x\<in>X"
    show "\<exists>!y. y\<in>Y \<and> \<langle>x, y\<rangle> \<in> {\<langle>x,y\<rangle> \<in> X\<times>Y . b(x) = y}"
    proof;
      def y \<equiv> "b(x)";
      with A2 A1 show 
        "\<exists>y. y\<in>Y & \<langle>x, y\<rangle> \<in> {\<langle>x,y\<rangle> \<in> X\<times>Y . b(x) = y}"
	by simp;
    next
      fix y y1
      assume "y\<in>Y \<and> \<langle>x, y\<rangle> \<in> {\<langle>x,y\<rangle> \<in> X\<times>Y . b(x) = y}"
	and "y1\<in>Y \<and> \<langle>x, y1\<rangle> \<in> {\<langle>x,y\<rangle> \<in> X\<times>Y . b(x) = y}"
      then show "y = y1" by simp;
    qed;
  qed;
  ultimately show "{<x,y> \<in> X\<times>Y. b(x) = y} : X\<rightarrow>Y" 
    using func1_1_L11 by simp;
qed;

text{*The next lemma will replace @{text "func1_1_L11A"} one day.*}

lemma ZF_fun_from_total: assumes A1: "\<forall>x\<in>X. b(x)\<in>Y"
  shows "{\<langle>x,b(x)\<rangle>. x\<in>X} : X\<rightarrow>Y"
proof -
  let ?f = "{\<langle>x,b(x)\<rangle>. x\<in>X}"
  { fix x assume A2: "x\<in>X"
    have "\<exists>!y. y\<in>Y \<and> \<langle>x, y\<rangle> \<in> ?f"
    proof;
      def y \<equiv> "b(x)"
      with A1 A2 show "\<exists>y. y\<in>Y \<and> \<langle>x, y\<rangle> \<in> ?f"
	by simp;
    next fix y y1 assume "y\<in>Y \<and> \<langle>x, y\<rangle> \<in> ?f"
	and "y1\<in>Y \<and> \<langle>x, y1\<rangle> \<in> ?f"
      then show "y = y1" by simp;
    qed;
  } then have "\<forall>x\<in>X. \<exists>!y. y\<in>Y \<and> <x,y> \<in> ?f"
    by simp;
  moreover from A1 have "?f \<subseteq> X\<times>Y" by auto;
  ultimately show ?thesis using func1_1_L11
    by simp;
qed;
 
text{*The value of a function defined by a meta-function is this 
  meta-function.*}

lemma func1_1_L11B: 
  assumes A1: "f:X\<rightarrow>Y"   "x\<in>X"
  and A2: "f = {<x,y> \<in> X\<times>Y. b(x) = y}"
  shows "f`(x) = b(x)"
proof -;
  from A1 have "<x,f`(x)> \<in> f" using apply_iff by simp;
  with A2 show ?thesis by simp;
qed;

text{*The next lemma will replace @{text "func1_1_L11B"} one day.*}

lemma ZF_fun_from_tot_val: 
  assumes A1: "f:X\<rightarrow>Y"   "x\<in>X"
  and A2: "f = {\<langle>x,b(x)\<rangle>. x\<in>X}"
  shows "f`(x) = b(x)"
proof -
  from A1 have "<x,f`(x)> \<in> f" using apply_iff by simp;
   with A2 show ?thesis by simp;
qed;

text{*We can extend a function by specifying its values on a set
  disjoint with the domain.*}

lemma func1_1_L11C: assumes A1: "f:X\<rightarrow>Y" and A2: "\<forall>x\<in>A. b(x)\<in>B"
  and A3: "X\<inter>A = 0" and Dg : "g = f \<union> {\<langle>x,b(x)\<rangle>. x\<in>A}"
  shows 
  "g : X\<union>A \<rightarrow> Y\<union>B"
  "\<forall>x\<in>X. g`(x) = f`(x)"
  "\<forall>x\<in>A. g`(x) = b(x)"
proof -
  let ?h = "{\<langle>x,b(x)\<rangle>. x\<in>A}"
  from A1 A2 A3 have 
    I: "f:X\<rightarrow>Y"  "?h : A\<rightarrow>B"  "X\<inter>A = 0"
    using ZF_fun_from_total by auto;
  then have "f\<union>?h : X\<union>A \<rightarrow> Y\<union>B"
    by (rule fun_disjoint_Un);
  with Dg show "g : X\<union>A \<rightarrow> Y\<union>B" by simp;
  { fix x assume A4: "x\<in>A"
    with A1 A3 have "(f\<union>?h)`(x) = ?h`(x)"
      using func1_1_L1 fun_disjoint_apply2
      by blast;
    moreover from I A4 have "?h`(x) = b(x)"
      using ZF_fun_from_tot_val by simp
    ultimately have "(f\<union>?h)`(x) = b(x)"
      by simp;
  } with Dg show "\<forall>x\<in>A. g`(x) = b(x)" by simp;
  { fix x assume A5: "x\<in>X"
    with A3 I have "x \<notin> domain(?h)"
      using func1_1_L1 by auto;
    then have "(f\<union>?h)`(x) = f`(x)"
      using fun_disjoint_apply1 by simp;
  } with Dg show "\<forall>x\<in>X. g`(x) = f`(x)" by simp;
qed;

text{*We can extend a function by specifying its value at a point that
  does not belong to the domain.*}

lemma func1_1_L11D: assumes A1: "f:X\<rightarrow>Y" and A2: "a\<notin>X"
  and Dg: "g = f \<union> {\<langle>a,b\<rangle>}"
  shows 
  "g : X\<union>{a} \<rightarrow> Y\<union>{b}"
  "\<forall>x\<in>X. g`(x) = f`(x)"
  "g`(a) = b"
proof -
  let ?h = "{\<langle>a,b\<rangle>}"
  from A1 A2 Dg have I:
    "f:X\<rightarrow>Y"  "\<forall>x\<in>{a}. b\<in>{b}"  "X\<inter>{a} = 0"  "g = f \<union> {\<langle>x,b\<rangle>. x\<in>{a}}"
    by auto;
  then show "g : X\<union>{a} \<rightarrow> Y\<union>{b}"
    by (rule func1_1_L11C);
  from I show "\<forall>x\<in>X. g`(x) = f`(x)"
    by (rule func1_1_L11C)
  from I have "\<forall>x\<in>{a}. g`(x) = b"
    by (rule func1_1_L11C);
  then show "g`(a) = b" by auto;
qed;

text{*A technical lemma about extending a function both by defining
  on a set disjoint with the domain and on a point that does not belong
  to any of those sets.*}

lemma func1_1_L11E:
  assumes A1: "f:X\<rightarrow>Y" and 
  A2: "\<forall>x\<in>A. b(x)\<in>B" and 
  A3: "X\<inter>A = 0" and A4: "a\<notin> X\<union>A"
  and Dg: "g = f \<union> {\<langle>x,b(x)\<rangle>. x\<in>A} \<union> {\<langle>a,c\<rangle>}"
  shows
  "g : X\<union>A\<union>{a} \<rightarrow> Y\<union>B\<union>{c}"
  "\<forall>x\<in>X. g`(x) = f`(x)"
  "\<forall>x\<in>A. g`(x) = b(x)"
  "g`(a) = c"
proof -
  let ?h = "f \<union> {\<langle>x,b(x)\<rangle>. x\<in>A}"
  from prems show "g : X\<union>A\<union>{a} \<rightarrow> Y\<union>B\<union>{c}"
    using func1_1_L11C func1_1_L11D by simp;
  from A1 A2 A3 have I:
    "f:X\<rightarrow>Y"  "\<forall>x\<in>A. b(x)\<in>B"  "X\<inter>A = 0"  "?h = f \<union> {\<langle>x,b(x)\<rangle>. x\<in>A}"
    by auto
  from prems have 
    II: "?h : X\<union>A \<rightarrow> Y\<union>B"  "a\<notin> X\<union>A"  "g = ?h \<union> {\<langle>a,c\<rangle>}"
    using func1_1_L11C by auto;
  then have III: "\<forall>x\<in>X\<union>A. g`(x) = ?h`(x)" by (rule func1_1_L11D);
  moreover from I have  "\<forall>x\<in>X. ?h`(x) = f`(x)"
    by (rule func1_1_L11C);
  ultimately show "\<forall>x\<in>X. g`(x) = f`(x)" by simp;
  from I have "\<forall>x\<in>A. ?h`(x) = b(x)" by (rule func1_1_L11C);
  with III show "\<forall>x\<in>A. g`(x) = b(x)" by simp;
  from II show "g`(a) = c" by (rule func1_1_L11D);
qed;

text{*The inverse image of an intersection of a nonempty collection of sets 
  is the intersection of the 
  inverse images. This generalizes @{text "function_vimage_Int"} 
  which is proven for the case of two sets.*}

lemma  func1_1_L12:
  assumes A1: "B\<subseteq>Pow(Y)" and A2: "B\<noteq>0" and A3: "f:X\<rightarrow>Y"
  shows "f-``(\<Inter>B) = (\<Inter>U\<in>B. f-``(U))"
proof;
  from A2 show  "f-``(\<Inter>B) \<subseteq> (\<Inter>U\<in>B. f-``(U))" by blast;
  show "(\<Inter>U\<in>B. f-``(U)) \<subseteq> f-``(\<Inter>B)"
  proof;
    fix x assume A4: "x \<in> (\<Inter>U\<in>B. f-``(U))";
    from A3 have "\<forall>U\<in>B. f-``(U) \<subseteq> X" using func1_1_L6A by simp;
    with A4 have "\<forall>U\<in>B. x\<in>X" by auto;
    with A2 have "x\<in>X" by auto;
    with A3 have "\<exists>!y. <x,y> \<in> f" using Pi_iff_old by simp;
    with A2 A4 show "x \<in> f-``(\<Inter>B)" using vimage_iff by blast;
  qed
qed;

text{*If the inverse image of a set is not empty, then the set is not empty.
  Proof by contradiction.*}

lemma func1_1_L13: assumes A1:"f-``(A)\<noteq>0" shows "A\<noteq>0"
proof (rule ccontr);
  assume A2:"\<not> A \<noteq> 0" from A2 A1 show False by simp;
qed;

text{*If the image of a set is not empty, then the set is not empty.
  Proof by contradiction.*}

lemma func1_1_L13A: assumes A1: "f``(A)\<noteq>0" shows "A\<noteq>0"
proof (rule ccontr);
  assume A2:"\<not> A \<noteq> 0" from A2 A1 show False by simp;
qed;

text{*What is the inverse image of a singleton?*}

lemma func1_1_L14: assumes "f\<in>X\<rightarrow>Y" 
  shows "f-``({y}) = {x\<in>X. f`(x) = y}" 
  using prems func1_1_L6A vimage_singleton_iff apply_iff by auto;

text{* A more familiar definition of inverse image.*}

lemma func1_1_L15: assumes A1: "f:X\<rightarrow>Y"
  shows "f-``(A) = {x\<in>X. f`(x) \<in> A}"
proof -;
  have "f-``(A) = (\<Union>y\<in>A . f-``{y})" 
    by (rule vimage_eq_UN);
  with A1 show ?thesis using func1_1_L14 by auto;
qed;

text{*A more familiar definition of image.*}
(*This can not be always replaced by image_fun from standard func.thy.*)
lemma func_imagedef: assumes A1: "f:X\<rightarrow>Y" and A2: "A\<subseteq>X"
  shows "f``(A) = {f`(x). x \<in> A}"
proof;
 from A1 show "f``(A) \<subseteq> {f`(x). x \<in> A}"
   using image_iff apply_iff by auto;
 show "{f`(x). x \<in> A} \<subseteq> f``(A)"
 proof;
   fix y assume "y \<in> {f`(x). x \<in> A}"
   then obtain x where "x\<in>A \<and> y = f`(x)"
     by auto;
   with A1 A2 show "y \<in> f``(A)"
     using apply_iff image_iff by auto;
 qed;
qed;

text{*The image of a nonempty subset of domain is nonempty.*}

lemma func_imagedefAA: 
  assumes A1: "f: X\<rightarrow>Y" and A2: "A\<subseteq>X" and A3: "A\<noteq>0"
  shows "f``(A) \<noteq> 0"
proof -
  from A3 obtain x where "x\<in>A" by auto
  with A1 A2 have "f`(x) \<in> f``(A)"
    using func_imagedef by auto;
  then show "f``(A) \<noteq> 0" by auto;
qed;

text{*The next lemma allows to prove statements about the values in the
  domain of a function given a statement about values in the range.*}

lemma func_imagedefAB: 
  assumes "f:X\<rightarrow>Y" and "A\<subseteq>X" and "\<forall>y\<in>f``(A). P(y)"
  shows "\<forall>x\<in>A. P(f`(x))"
  using prems func_imagedef by simp;
  
text{*An image of an image is the image of a composition.*}

lemma func_imagedefA: assumes  A1: "f:X\<rightarrow>Y" and A2: "g:Y\<rightarrow>Z"
  and A3: "A\<subseteq>X"
  shows 
  "g``(f``(A)) =  {g`(f`(x)). x\<in>A}"
  "g``(f``(A)) = (g O f)``(A)"
proof -
  from A1 A3 have "{f`(x). x\<in>A} \<subseteq> Y"
    using apply_funtype by auto;
  with A2 have "g``{f`(x). x\<in>A} = {g`(f`(x)). x\<in>A}"
    using func_imagedef by auto;
  with A1 A3 show I: "g``(f``(A)) =  {g`(f`(x)). x\<in>A}" 
    using func_imagedef by simp;
  from A1 A3 have "\<forall>x\<in>A. (g O f)`(x) = g`(f`(x))"
    using comp_fun_apply by auto;
  with I have "g``(f``(A)) = {(g O f)`(x). x\<in>A}"
    by simp;
  moreover from A1 A2 A3 have "(g O f)``(A) = {(g O f)`(x). x\<in>A}"
    using comp_fun func_imagedef by blast;
  ultimately show "g``(f``(A)) = (g O f)``(A)"
    by simp;
qed;

text{*If an element of the domain of a function belongs to a set, 
  then its value belongs to the imgage of that set.*}

lemma func_imagedefB: assumes "f:X\<rightarrow>Y"  "x\<in>A"  "A\<subseteq>X"
  shows "f`(x) \<in> f``(A)"
  using prems func_imagedef by auto;
  
text{*What is the image of a set defined by a meta-fuction?*}

lemma func1_1_L17: 
  assumes A1: "f \<in> X\<rightarrow>Y" and A2: "\<forall>x\<in>A. b(x) \<in> X"
  shows "f``({b(x). x\<in>A}) = {f`(b(x)). x\<in>A}"
proof -;
  from A2 have "{b(x). x\<in>A} \<subseteq> X" by auto;
  with A1 show ?thesis using func_imagedef by auto;
qed;

text{*What are the values of composition of three functions?*}

lemma func1_1_L18: assumes A1: "f:A\<rightarrow>B"  "g:B\<rightarrow>C"  "h:C\<rightarrow>D"
  and A2: "x\<in>A"
  shows
  "(h O g O f)`(x) \<in> D"
  "(h O g O f)`(x) = h`(g`(f`(x)))"  
proof -
  from A1 have "(h O g O f) : A\<rightarrow>D"
    using comp_fun by blast;
  with A2 show "(h O g O f)`(x) \<in> D" using apply_funtype
    by simp;
  from A1 A2 have "(h O g O f)`(x) = h`( (g O f)`(x))"
    using comp_fun comp_fun_apply by blast;
  with A1 A2 show "(h O g O f)`(x) = h`(g`(f`(x)))"
    using comp_fun_apply by simp;
qed;

section{*Functions restricted to a set*}
 
text{*What is the inverse image of a set under a restricted fuction?*}

lemma func1_2_L1: assumes A1: "f:X\<rightarrow>Y" and A2: "B\<subseteq>X"
  shows "restrict(f,B)-``(A) = f-``(A) \<inter> B"
proof -;
  let ?g = "restrict(f,B)";
  from A1 A2 have "?g:B\<rightarrow>Y" 
    using restrict_type2 by simp;
  with A2 A1 show "?g-``(A) = f-``(A) \<inter> B"
    using func1_1_L15 restrict_if by auto;
qed;
   
text{*A criterion for when one function is a restriction of another.
  The lemma below provides a result useful in the actual proof of the 
  criterion and applications.*}

lemma func1_2_L2: 
  assumes A1: "f:X\<rightarrow>Y" and A2: "g \<in> A\<rightarrow>Z" 
  and A3: "A\<subseteq>X" and A4: "f \<inter> A\<times>Z = g"
  shows "\<forall>x\<in>A. g`(x) = f`(x)"
proof;
  fix x assume "x\<in>A"
  with A2 have "<x,g`(x)> \<in> g" using apply_Pair by simp;
  with A4 A1 show "g`(x) = f`(x)"  using apply_iff by auto; 
qed;

text{*Here is the actual criterion.*}

lemma func1_2_L3: 
  assumes A1: "f:X\<rightarrow>Y" and A2: "g:A\<rightarrow>Z" 
  and A3: "A\<subseteq>X" and A4: "f \<inter> A\<times>Z = g"
  shows "g = restrict(f,A)"
proof;
  from A4 show "g \<subseteq> restrict(f, A)" using restrict_iff by auto;
  show "restrict(f, A) \<subseteq> g"
  proof;
    fix z assume A5:"z \<in> restrict(f,A)"
    then obtain x y where D1:"z\<in>f & x\<in>A  & z = <x,y>"
      using restrict_iff by auto;
    with A1 have "y = f`(x)" using apply_iff by auto;
    with A1 A2 A3 A4 D1 have "y = g`(x)" using func1_2_L2 by simp;
    with A2 D1 show "z\<in>g" using apply_Pair by simp;
  qed;
qed;

text{*Which function space a restricted function belongs to?*}

lemma func1_2_L4: 
  assumes A1: "f:X\<rightarrow>Y" and A2: "A\<subseteq>X" and A3: "\<forall>x\<in>A. f`(x) \<in> Z"
  shows "restrict(f,A) : A\<rightarrow>Z"
proof -;
  let ?g = "restrict(f,A)"
  from A1 A2 have "?g : A\<rightarrow>Y" 
    using restrict_type2 by simp;
  moreover { 
    fix x assume "x\<in>A" 
    with A1 A3 have "?g`(x) \<in> Z" using restrict by simp};
  ultimately show ?thesis by (rule Pi_type);
qed;

section{*Constant functions*}

text{*We define constant($=c$) functions on a set $X$  in a natural way as 
  ConstantFunction$(X,c)$. *}

constdefs
  "ConstantFunction(X,c) \<equiv> X\<times>{c}"

text{*Constant function belongs to the function space.*}

lemma func1_3_L1: 
  assumes A1: "c\<in>Y" shows "ConstantFunction(X,c) : X\<rightarrow>Y"
proof -;
   from A1 have "X\<times>{c} = {<x,y> \<in> X\<times>Y. c = y}" 
     by auto;
   with A1 show ?thesis using func1_1_L11A ConstantFunction_def
     by simp;
qed;

text{*Constant function is equal to the constant on its domain.*}

lemma func1_3_L2: assumes A1: "x\<in>X"
  shows "ConstantFunction(X,c)`(x) = c"
proof -;
  have "ConstantFunction(X,c) \<in> X\<rightarrow>{c}"
    using func1_3_L1 by simp;
  moreover from A1 have "<x,c> \<in> ConstantFunction(X,c)"
    using ConstantFunction_def by simp;
  ultimately show ?thesis using apply_iff by simp;
qed;
  
text{*This concludes func1.thy.*}

end