(* 
    This file is a part of IsarMathLib - 
    a library of formalized mathematics for Isabelle/Isar.

    Copyright (C) 2005, 2006  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.*)

header {*\isaheader{func\_ZF.thy}*}

theory func_ZF imports func1 

begin

text{*In this theory we consider properties of functions that are binary 
  operations, that is they map $X\times X$ into $X$. *}

section{*Lifting operations to a function space*}

text{*It happens quite often that we have a binary operation on some set and
  we need a similar operation that is defined for functions on that set. 
  For example once we know how to add real numbers we also know how to add
  real-valued functions: for $f,g:X \rightarrow \mathbf{R}$ we define
  $(f+g)(x) = f(x) + g(x)$. Note that formally the $+$ means something 
  different on the left hand side of this equality than on the 
  right hand side.
  This section aims at formalizing this process.
  We will call it "lifting to a function space", if you have a 
  suggestion for a better name, please let me know. *} 

constdefs 
  Lift2FcnSpce (infix "{lifted to function space over}" 65) 
  "f {lifted to function space over} X \<equiv> 
  {<p,g> \<in> ((X\<rightarrow>range(f))\<times>(X\<rightarrow>range(f)))\<times>(X\<rightarrow>range(f)).
  {<x,y> \<in> X\<times>range(f). f`<fst(p)`(x),snd(p)`(x)> = y} = g}";

text{*The result of the lift belongs to the function space.*}

lemma func_ZF_1_L1: 
  assumes A1: "f : Y\<times>Y\<rightarrow>Y" 
  and A2: "p \<in>(X\<rightarrow>range(f))\<times>(X\<rightarrow>range(f))"
  shows 
  "{<x,y> \<in> X\<times>range(f). f`<fst(p)`(x),snd(p)`(x)> = y} : X\<rightarrow>range(f)"
  proof -;
    have "\<forall>x\<in>X. f`<fst(p)`(x),snd(p)`(x)> \<in> range(f)"
    proof;
      fix x assume A3:"x\<in>X"
      let ?p = "<fst(p)`(x),snd(p)`(x)>"
      from A2 A3 have 
	"fst(p)`(x) \<in> range(f)"  "snd(p)`(x) \<in> range(f)"
	using apply_type by auto;
      with A1 have "?p \<in> Y\<times>Y"
	using func1_1_L5B by blast;
      with A1 have "<?p, f`(?p)> \<in> f"
	using apply_Pair by simp;
      with A1 show 
	"f`(?p) \<in> range(f)"
	using rangeI by simp;
    qed
    then show ?thesis using func1_1_L11A by simp;
qed;

text{*The values of the lift are defined by the value of the liftee in a 
  natural way.*}

lemma func_ZF_1_L2: 
  assumes "f : Y\<times>Y\<rightarrow>Y" 
  and "p\<in>(X\<rightarrow>range(f))\<times>(X\<rightarrow>range(f))" and "x\<in>X"
  and "P = {<x,y> \<in> X\<times>range(f). f`<fst(p)`(x),snd(p)`(x)> = y}"
  shows "P`(x) = f`\<langle>fst(p)`(x),snd(p)`(x)\<rangle>" 
  using prems func_ZF_1_L1 func1_1_L11B by simp;

text{*Function lifted to a function space results in a function space 
  operator.*}

lemma func_ZF_1_L3: 
  assumes "f \<in> Y\<times>Y\<rightarrow>Y"
  and "F = f {lifted to function space over} X"
  shows "F : (X\<rightarrow>range(f))\<times>(X\<rightarrow>range(f))\<rightarrow>(X\<rightarrow>range(f))"
  using prems Lift2FcnSpce_def func_ZF_1_L1 func1_1_L11A by simp;

text{*The values of the lift are defined by the values of the liftee in
  the natural way. For some reason we need to be extremely detailed and 
  explicit to be able to apply @{text "func1_3_L2"}. simp and auto 
  fail miserably here.*}

lemma func_ZF_1_L4: 
  assumes A1: "f : Y\<times>Y\<rightarrow>Y"
  and A2: "F = f {lifted to function space over} X"
  and A3: "s:X\<rightarrow>range(f)" "r:X\<rightarrow>range(f)"  
  and A4: "x\<in>X"
  shows "(F`<s,r>)`(x) = f`<s`(x),r`(x)>"
proof -;
  let ?P = "{<x,y> \<in> X\<times>range(f). f`<s`(x),r`(x)> = y}";
  let ?p = "<s,r>"
  from A1 have "f \<in> Y\<times>Y\<rightarrow>Y" .;
  moreover from A3 have 
    "?p \<in> (X\<rightarrow>range(f))\<times>(X\<rightarrow>range(f))"
    by simp;
  moreover from A4 have "x\<in>X" .;
  moreover have 
    "?P = {<x,y> \<in> X\<times>range(f). f`<fst(?p)`(x),snd(?p)`(x)> = y}"
    by simp;
  ultimately have "?P`(x) = f`\<langle>fst(?p)`(x),snd(?p)`(x)\<rangle>"
    by (rule func_ZF_1_L2);
  with A1 A2 A3 show ?thesis using func_ZF_1_L3 Lift2FcnSpce_def func1_1_L11B
    by simp;
qed

section{*Associative and commutative operations*}

text{*In this section we define associative and commutative oparations 
  and prove that they remain such when we lift them
  to a function space.*}

constdefs 

  IsAssociative (infix "{is associative on}" 65)
  "f {is associative on} G \<equiv> f \<in> G\<times>G\<rightarrow>G \<and> 
  (\<forall> x \<in> G. \<forall> y \<in> G. \<forall> z \<in> G. 
  ( f`(<f`(<x,y>),z>) = f`( < x,f`(<y,z>)> )))"

  IsCommutative (infix "{is commutative on}" 65)
  "f {is commutative on} G \<equiv> \<forall>x\<in>G. \<forall>y\<in>G. f`<x,y> = f`<y,x>"

text{*The lift of a commutative function is commutative.*}

lemma func_ZF_2_L1:
  assumes A1: "f : G\<times>G\<rightarrow>G"
  and A2: "F = f {lifted to function space over} X"
  and A3: "s : X\<rightarrow>range(f)" "r : X\<rightarrow>range(f)" 
  and A4: "f {is commutative on} G"
  shows "F`<s,r> = F`<r,s>" 
proof -;
  from A1 A2 have 
    "F : (X\<rightarrow>range(f))\<times>(X\<rightarrow>range(f))\<rightarrow>(X\<rightarrow>range(f))"
    using func_ZF_1_L3 by simp; 
  with A3 have 
    "F`<s,r> : X\<rightarrow>range(f)" "F`<r,s> : X\<rightarrow>range(f)"
    using apply_type by auto;
  moreover have 
    "\<forall>x\<in>X. (F`<s,r>)`(x) = (F`<r,s>)`(x)"
  proof;
    fix x assume A5:"x\<in>X"
    from A1 have "range(f)\<subseteq>G"
      using func1_1_L5B by simp;
    with A3 A5 have T1:"s`(x) \<in> G" "r`(x) \<in> G"
      using apply_type by auto;
    with A1 A2 A3 A4 A5 show 
      "(F`<s,r>)`(x) = (F`<r,s>)`(x)"
      using func_ZF_1_L4 IsCommutative_def by simp;
  qed;
  ultimately show ?thesis using fun_extension_iff
    by simp;
qed;

text{*The lift of a commutative function is commutative 
  on the function space.*}

lemma func_ZF_2_L2:
  assumes "f : G\<times>G\<rightarrow>G"
  and "f {is commutative on} G"
  and "F = f {lifted to function space over} X"
  shows "F {is commutative on} (X\<rightarrow>range(f))"
  using prems IsCommutative_def func_ZF_2_L1 by simp;
  
text{*The lift of an associative function is associative.*}

lemma func_ZF_2_L3:
  assumes A2: "F = f {lifted to function space over} X"
  and A3: "s : X\<rightarrow>range(f)" "r : X\<rightarrow>range(f)" "q : X\<rightarrow>range(f)"
  and A4: "f {is associative on} G"
  shows "F`\<langle>F`<s,r>,q\<rangle> = F`\<langle>s,F`<r,q>\<rangle>"
proof -;
  from A4 A2 have 
    "F : (X\<rightarrow>range(f))\<times>(X\<rightarrow>range(f))\<rightarrow>(X\<rightarrow>range(f))"
    using IsAssociative_def func_ZF_1_L3 by auto;
  with A3 have T1:
    "F`<s,r> : X\<rightarrow>range(f)"
    "F`<r,q> : X\<rightarrow>range(f)"
    "F`<F`<s,r>,q> : X\<rightarrow>range(f)"
    "F`<s,F`<r,q> >: X\<rightarrow>range(f)"
    using apply_type by auto;
  moreover have
    "\<forall>x\<in>X. (F`\<langle>F`<s,r>,q\<rangle>)`(x) = (F`\<langle>s,F`<r,q>\<rangle>)`(x)"
  proof;
    fix x assume A5:"x\<in>X"
    from A4 have T2:"f:G\<times>G\<rightarrow>G"
      using IsAssociative_def by simp;
    then have "range(f)\<subseteq>G"
      using func1_1_L5B by simp;
    with A3 A5 have 
      "s`(x) \<in> G" "r`(x) \<in> G" "q`(x) \<in> G"
      using apply_type by auto;
    with T2 A2 T1 A3 A5 A4 show 
      "(F`\<langle>F`<s,r>,q\<rangle>)`(x) = (F`\<langle>s,F`<r,q>\<rangle>)`(x)"
      using func_ZF_1_L4 IsAssociative_def by simp;
  qed;
  ultimately show ?thesis using fun_extension_iff
    by simp;
qed;

text{*The lift of an associative function is associative 
  on the function space.*}

lemma func_ZF_2_L4:
  assumes A1: "f {is associative on} G"
  and A2: "F = f {lifted to function space over} X"
  shows "F {is associative on} (X\<rightarrow>range(f))"
proof -;
  from A1 A2 have
    "F : (X\<rightarrow>range(f))\<times>(X\<rightarrow>range(f))\<rightarrow>(X\<rightarrow>range(f))"
    using IsAssociative_def func_ZF_1_L3 by auto;
  moreover from A1 A2 have
    "\<forall>s \<in> X\<rightarrow>range(f). \<forall> r \<in> X\<rightarrow>range(f). \<forall>q \<in> X\<rightarrow>range(f).
    F`<F`<s,r>,q> = F`<s,F`<r,q> >"
    using func_ZF_2_L3 by simp;
  ultimately show ?thesis using IsAssociative_def 
    by simp;
qed;

section{*Restricting operations*}

text{*In this section we consider when restriction of the operation to a set
  inherits properties like commutativity and associativity.*}

text{*The commutativity is inherited when restricting a function to a set.*}

lemma func_ZF_4_L1: 
  assumes A1: "f:X\<times>X\<rightarrow>Y" and A2: "A\<subseteq>X"
  and A3: "f {is commutative on} X"
  shows "restrict(f,A\<times>A) {is commutative on} A"
proof -;
  { fix x y assume A4: "x\<in>A \<and> y\<in>A"
    with A2 A3 have
      "f`<x,y> = f`<y,x>"
      using IsCommutative_def by auto;
    moreover from A4 have 
      "restrict(f,A\<times>A)`<x,y> = f`<x,y>"
      "restrict(f,A\<times>A)`<y,x> = f`<y,x>"
      using restrict_if by auto;
    ultimately have
      "restrict(f,A\<times>A)`<x,y> = restrict(f,A\<times>A)`<y,x>"
      by simp }
  then show ?thesis using IsCommutative_def by simp;
qed;
  
text{*Next we define sets closed with respect to an operation.*}

constdefs 
  IsOpClosed (infix "{is closed under}" 65) 
  "A {is closed under} f \<equiv> \<forall>x\<in>A. \<forall>y\<in>A. f`<x,y> \<in> A";

text{*Associative operation restricted to a set that is closed with
  resp. to this operation is associative.*}

lemma func_ZF_4_L2:assumes A1: "f {is associative on} X"
  and A2: "A\<subseteq>X" and A3: "A {is closed under} f"
  and A4: "x\<in>A" "y\<in>A" "z\<in>A"
  and A5: "g = restrict(f,A\<times>A)"
  shows "g`\<langle>g`<x,y>,z\<rangle> = g`\<langle>x,g`<y,z>\<rangle>"
proof -; 
  from A4 A2 have T1:
    "x\<in>X" "y\<in>X" "z\<in>X"
    by auto;
  from A3 A4 A5 have
    "g`<g`<x,y>,z> = f`<f`<x,y>,z>"
    "g`<x,g`<y,z> > = f`<x,f`<y,z> >"
    using IsOpClosed_def restrict_if by auto;
  moreover from A1 T1 have
     "f`<f`<x,y>,z> = f`<x,f`<y,z> >"
    using IsAssociative_def by simp;
  ultimately show ?thesis by simp;
qed;

text{*Associative operation restricted to a set that is closed with
  resp. to this operation is associative on the set.*}

lemma func_ZF_4_L3: assumes A1: "f {is associative on} X"
  and A2: "A\<subseteq>X" and A3: "A {is closed under} f"
  shows "restrict(f,A\<times>A) {is associative on} A"
proof -;
  let ?g = "restrict(f,A\<times>A)"
  from A1 have "f:X\<times>X\<rightarrow>X"
    using IsAssociative_def by simp;
  moreover from A2 have "A\<times>A \<subseteq> X\<times>X" by auto;
  moreover from A3 have "\<forall>p \<in> A\<times>A. ?g`(p) \<in> A"
    using IsOpClosed_def restrict_if by auto;
  ultimately have "?g : A\<times>A\<rightarrow>A"
    using func1_2_L4 by simp;
  moreover from  A1 A2 A3 have
    "\<forall> x \<in> A. \<forall> y \<in> A. \<forall> z \<in> A.
    ?g`<?g`<x,y>,z> = ?g`< x,?g`<y,z> >"
    using func_ZF_4_L2 by simp;
  ultimately show ?thesis 
    using IsAssociative_def by simp;
qed;

text{*The essential condition to show that if a set $A$ is closed 
  with respect to an operation, 
  then it is closed under this operation restricted 
  to any superset of $A$.*}

lemma func_ZF_4_L4: assumes "A {is closed under} f"
  and "A\<subseteq>B" and "x\<in>A"  "y\<in>A" and "g = restrict(f,B\<times>B)"
  shows "g`<x,y> \<in> A"
  using prems IsOpClosed_def restrict by auto;

text{*If a set $A$ is closed under an operation, 
  then it is closed under this operation restricted 
  to any superset of $A$. *}

lemma func_ZF_4_L5: 
  assumes A1: "A {is closed under} f"
  and A2: "A\<subseteq>B"
  shows "A {is closed under} restrict(f,B\<times>B)"
proof -
  let ?g = "restrict(f,B\<times>B)"
  from A1 A2 have "\<forall>x\<in>A. \<forall>y\<in>A. ?g`<x,y> \<in> A"
    using func_ZF_4_L4 by simp
  then show ?thesis using IsOpClosed_def by simp;
qed;

text{*The essential condition to show that intersection of sets that are
  closed with respect to an operation is closed with respect 
  to the operation.*}

lemma func_ZF_4_L6:
  assumes "A {is closed under} f" 
  and "B {is closed under} f"
  and "x \<in> A\<inter>B" "y\<in> A\<inter>B"
  shows "f`<x,y> \<in> A\<inter>B" using prems IsOpClosed_def by auto;

text{*Intersection of sets that are
  closed with respect to an operation is closed under 
  the operation.*}

lemma func_ZF_4_L7:
  assumes "A {is closed under} f"
  "B {is closed under} f"
  shows "A\<inter>B {is closed under} f"
  using prems IsOpClosed_def by simp;

section{*Composition*}

text{*For any set $X$ we can consider a binary operation 
  on the set of functions 
  $f:X\rightarrow X$ defined by $C(f,g) = f\circ g$. Composition of functions 
  (or relations) is defined in the standard Isabelle distribution as a higher
  order function. In this section we consider the corresponding two-argument 
  ZF-function (binary operation), that is a subset of 
  $((X\rightarrow X)\times (X\rightarrow X))\times (X\rightarrow X)$.*}

constdefs
  "Composition(X) \<equiv> 
  {<p,f> \<in> ((X\<rightarrow>X)\<times>(X\<rightarrow>X))\<times>(X\<rightarrow>X). fst(p) O snd(p) = f}"

text{*Composition operation is a function that maps 
  $(X\rightarrow X)\times (X\rightarrow X)$ into $X\rightarrow X$.*}

lemma func_ZF_5_L1: shows "Composition(X) : (X\<rightarrow>X)\<times>(X\<rightarrow>X)\<rightarrow>(X\<rightarrow>X)"
  using comp_fun Composition_def func1_1_L11A by simp;

text{*The value of the composition operation is the composition of arguments.*}

lemma func_ZF_5_L2: assumes "f:X\<rightarrow>X" "g:X\<rightarrow>X"
  shows "Composition(X)`<f,g> = f O g" 
  using prems func_ZF_5_L1 Composition_def func1_1_L11B by simp;

text{*What is the falue of a composition on an argument?*}

lemma func_ZF_5_L3: assumes "f:X\<rightarrow>X" and "g:X\<rightarrow>X" and "x\<in>X"
  shows "(Composition(X)`<f,g>)`(x) = f`(g`(x))"
  using prems func_ZF_5_L2 comp_fun_apply by simp;
  
text{*The essential condition to show that composition is associative.*}

lemma func_ZF_5_L4: assumes A1: "f:X\<rightarrow>X" "g:X\<rightarrow>X" "h:X\<rightarrow>X"
  and A2: "C = Composition(X)"
  shows "C`\<langle>C`<f,g>,h\<rangle> = C`\<langle> f,C`<g,h>\<rangle>"
proof -; 
  from A2 have "C : ((X\<rightarrow>X)\<times>(X\<rightarrow>X))\<rightarrow>(X\<rightarrow>X)"
    using func_ZF_5_L1 by simp;
  with A1 have T1:
    "C`<f,g> : X\<rightarrow>X"
    "C`<g,h> : X\<rightarrow>X"
    "C`<C`<f,g>,h> : X\<rightarrow>X"
    "C`< f,C`<g,h> > : X\<rightarrow>X"
    using apply_funtype by auto;
  moreover have 
    "\<forall> x \<in> X. C`\<langle>C`<f,g>,h\<rangle>`(x) = C`\<langle>f,C`<g,h>\<rangle>`(x)"
  proof;
    fix x assume A3:"x\<in>X"
    with A1 A2 T1 have 
      "C`<C`<f,g>,h> ` (x) = f`(g`(h`(x)))"
      "C`< f,C`<g,h> >`(x) = f`(g`(h`(x)))"
      using func_ZF_5_L3 apply_funtype by auto;
    then show "C`\<langle>C`<f,g>,h\<rangle>`(x) = C`\<langle> f,C`<g,h>\<rangle>`(x)"
      by simp;
    qed;
  ultimately show ?thesis using fun_extension_iff by simp;
qed;
  
text{*Composition is an associative operation on $X\rightarrow X$ (the space
  of functions that map $X$ into itself).*}

lemma func_ZF_5_L5: shows "Composition(X) {is associative on} (X\<rightarrow>X)"
proof -;
  let ?C = "Composition(X)"
  have "\<forall>f\<in>X\<rightarrow>X. \<forall>g\<in>X\<rightarrow>X. \<forall>h\<in>X\<rightarrow>X.
    ?C`<?C`<f,g>,h> = ?C`< f,?C`<g,h> >"
    using func_ZF_5_L4 by simp;
  then show ?thesis using func_ZF_5_L1 IsAssociative_def
    by simp;
qed;

section{*Identity function *}

text{*In this section we show some additional facts about the identity 
  function defined in the standard Isabelle's Perm.thy file.*}

text{*Composing a function with identity does not change the function.*}

lemma func_ZF_6_L1A: assumes A1: "f : X\<rightarrow>X"
  shows "Composition(X)`<f,id(X)> = f"
  "Composition(X)`<id(X),f> = f"
proof -;
  have "Composition(X) : (X\<rightarrow>X)\<times>(X\<rightarrow>X)\<rightarrow>(X\<rightarrow>X)"
    using func_ZF_5_L1 by simp;
  with A1 have "Composition(X)`<id(X),f> : X\<rightarrow>X"
    "Composition(X)`<f,id(X)> : X\<rightarrow>X"
    using id_type apply_funtype by auto;
  moreover from A1 have "f : X\<rightarrow>X" .
  moreover from A1 have 
    "\<forall>x\<in>X. (Composition(X)`<id(X),f>)`(x) = f`(x)"
    "\<forall>x\<in>X. (Composition(X)`<f,id(X)>)`(x) = f`(x)"
    using id_type func_ZF_5_L3 apply_funtype id_conv
    by auto;
  ultimately show "Composition(X)`<id(X),f> = f"
    "Composition(X)`<f,id(X)> = f"
    using fun_extension_iff by auto;
qed;

section{*Distributive operations*}

text{*In this section we deal with pairs of operations such that one is
  distributive with respect to the other, that is 
  $a\cdot (b+c) = a\cdot b + a\cdot c$ and
  $(b+c)\cdot a = b\cdot a + c\cdot a$. We show that this property is 
  preserved under restriction to a set closed with respect to both 
  operations. In EquivClass1.thy we show that this property is 
  preserved by projections to the quotient space if both operations are 
  congruent with respect to the equivalence relation.*}

text{*We define distributivity as a statement about three sets. The first 
  set is the set on which the operations act. The second set is the 
  additive operation (a ZF function) and the third is the multiplicative
  operation.*}

constdefs
  "IsDistributive(X,A,M) \<equiv> (\<forall>a\<in>X.\<forall>b\<in>X.\<forall>c\<in>X.
  M`\<langle>a,A`<b,c>\<rangle> = A`\<langle>M`<a,b>,M`<a,c>\<rangle> \<and> 
  M`\<langle>A`<b,c>,a\<rangle> = A`\<langle>M`<b,a>,M`<c,a> \<rangle>)"

text{*The essential condition to show that distributivity is preserved by 
  restrictions to sets that are closed with
  respect to both operations.*}

lemma func_ZF_7_L1: 
  assumes A1: "IsDistributive(X,A,M)"
  and A2: "Y\<subseteq>X"
  and A3: "Y {is closed under} A" 
  "Y {is closed under} M"
  and A4: "A\<^isub>r = restrict(A,Y\<times>Y)" "M\<^isub>r = restrict(M,Y\<times>Y)"
  and A5: "a\<in>Y" "b\<in>Y" "c\<in>Y"
  shows "M\<^isub>r`< a,A\<^isub>r`<b,c> > = A\<^isub>r`< M\<^isub>r`<a,b>,M\<^isub>r`<a,c> > \<and> 
  M\<^isub>r`< A\<^isub>r`<b,c>,a > = A\<^isub>r`< M\<^isub>r`<b,a>,M\<^isub>r`<c,a> >"
proof
  from A3 A5 have "A`<b,c> \<in> Y"  "M`<a,b> \<in> Y"  "M`<a,c> \<in> Y"
    "M`<b,a> \<in> Y"  "M`<c,a> \<in> Y" using IsOpClosed_def by auto;
  with A5 A4 have T1:"A\<^isub>r`<b,c> \<in> Y" "M\<^isub>r`<a,b> \<in> Y" "M\<^isub>r`<a,c> \<in> Y"
    "M\<^isub>r`<b,a> \<in> Y" "M\<^isub>r`<c,a> \<in> Y"
    using restrict by auto;
  with A1 A2 A4 A5 show "M\<^isub>r`< a,A\<^isub>r`<b,c> > = A\<^isub>r`< M\<^isub>r`<a,b>,M\<^isub>r`<a,c> >"
    "M\<^isub>r`< A\<^isub>r`<b,c>,a > = A\<^isub>r`< M\<^isub>r`<b,a>,M\<^isub>r`<c,a> >"
    using restrict IsDistributive_def by auto;
qed;
  
text{*Distributivity is preserved by restrictions to sets that are closed with
  respect to both operations.*}

lemma func_ZF_7_L2: 
  assumes "IsDistributive(X,A,M)"
  and "Y\<subseteq>X"
  and "Y {is closed under} A" 
  "Y {is closed under} M"
  and "A\<^isub>r = restrict(A,Y\<times>Y)" "M\<^isub>r = restrict(M,Y\<times>Y)"
  shows "IsDistributive(Y,A\<^isub>r,M\<^isub>r)"
proof -
  from prems have "\<forall>a\<in>Y.\<forall>b\<in>Y.\<forall>c\<in>Y. 
    M\<^isub>r`< a,A\<^isub>r`<b,c> > = A\<^isub>r`< M\<^isub>r`<a,b>,M\<^isub>r`<a,c> > \<and> 
    M\<^isub>r`< A\<^isub>r`<b,c>,a > = A\<^isub>r`< M\<^isub>r`<b,a>,M\<^isub>r`<c,a> >"
    using func_ZF_7_L1 by simp;
  then show ?thesis using IsDistributive_def by simp;
qed;

(*section{*Fixing variables in functions*}

text{*For every function of two variables we can define families of 
  functions of one variable by fixing the other variable. This section 
  establishes basic definitions and results for this concept.*}

constdefs

  Fix1stVar :: "[i,i]\<Rightarrow>i"
  "Fix1stVar(f,x) \<equiv> 
  {<y,z> \<in> range(domain(f))\<times>range(f). f`<x,y> = z}"

  
  Fix2ndVar :: "[i,i]\<Rightarrow>i"
  "Fix2ndVar(f,y) \<equiv> 
  {<x,z> \<in> domain(domain(f))\<times>range(f). f`<x,y> = z}"*)
end;