(* 
    This file is a part of IsarMathLib - 
    a library of formalized mathematics written for Isabelle/Isar.

    Copyright (C) 2005  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{Group\_ZF\_3.thy}*}

theory Group_ZF_3 = Group_ZF_2 + Finite1:

text{*In this theory we consider notions in group theory that are useful 
  for the construction of real numbers in the Real\_ZF\_x series of theories.*}


section{*Group valued finite range functions*}

text{*In this section show that the group valued functions 
  $f : X\rightarrow G$, with the property that $f(X)$ is 
  a finite subset of $G$, is a group. Such functions play an important
  role in the construction of real numbers in the Real\_ZF\_x.thy series. *}

text{*The following proves the essential condition to show that the set of 
  finite range functions is
  closed with respect to the lifted group operation.*}

lemma (in group0) Group_ZF_3_1_L1: 
  assumes A1:"F = f {lifted to function space over} X"
  and 
  A2:"s \<in> FinRangeFunctions(X,G)" "r \<in>  FinRangeFunctions(X,G)" 
  shows "F`<s,r> \<in> FinRangeFunctions(X,G)"
proof -;
  let ?q = "F`<s,r>"
  from A2 have T1:"s:X\<rightarrow>G" "r:X\<rightarrow>G"
    using FinRangeFunctions_def by auto;
  with A1 have T2:"?q : X\<rightarrow>G"
    using group0_2_L1 monoid0.Group_ZF_2_1_L0
    by simp;
  moreover have "?q``(X) \<in> Fin(G)"
  proof -;
    from A2 have
      "{s`(x). x\<in>X} \<in> Fin(G)"
      "{r`(x). x\<in>X} \<in> Fin(G)"
      using Finite1_L18 by auto;
    with A1 T1 T2 show ?thesis using 
      group0_2_L3A Finite1_L15 Group_ZF_2_1_L3 func1_1_L16
      by simp;
  qed;
  ultimately show ?thesis using FinRangeFunctions_def
    by simp;
qed;

text{*The set of group valued finite range functions is closed with respect 
  to the lifted group operation. *}

lemma (in group0) Group_ZF_3_1_L2:
  assumes A1:"F = f {lifted to function space over} X"
  shows "FinRangeFunctions(X,G) {is closed with respect to} F"
proof -;
  let ?A = "FinRangeFunctions(X,G)"
  from A1 have "\<forall>x\<in>?A. \<forall>y\<in>?A. F`<x,y> \<in> ?A"
    using Group_ZF_3_1_L1 by simp;
  then show ?thesis using IsOpClosed_def by simp;
qed;

text{*A composition of a finite range function with the group inverse is
  a finite range function.*}

lemma (in group0) Group_ZF_3_1_L3: 
  assumes A1:" s \<in> FinRangeFunctions(X,G)"
  shows "GroupInv(G,f) O s \<in> FinRangeFunctions(X,G)"
proof -;
  from groupAsssum A1 show ?thesis 
    using group0_2_T2 Finite1_L20 by simp;
qed;

text{*The set of finite range functions is s subgroup of the lifted group.*}

theorem Group_ZF_3_1_T1: 
  assumes A1:"IsAgroup(G,f)" 
  and A2:"F = f {lifted to function space over} X"
  and A3:"X\<noteq>0"
  shows "IsAsubgroup(FinRangeFunctions(X,G),F)"
proof -;
  let ?e = "TheNeutralElement(G,f)"
  let ?S = "FinRangeFunctions(X,G)"
  from A1 have T1:"group0(G,f)" using group0_def 
    by simp;
  with A1 A2 have T2:"group0(X\<rightarrow>G,F)"
    using group0.Group_ZF_2_1_T2 group0_def
    by simp;
  moreover have "?S \<noteq> 0"
  proof -;   
    from T1 A3 have 
      "ConstantFunction(X,?e) \<in> ?S"
      using group0.group0_2_L1 monoid0.group0_1_L3
	Finite1_L17 by simp;
    thus ?thesis by auto;
  qed;
  moreover have "?S \<subseteq> X\<rightarrow>G"
    using FinRangeFunctions_def by auto;
  moreover from A2 T1 have
    "?S {is closed with respect to} F"
    using group0.Group_ZF_3_1_L2
    by simp;
  moreover from A1 A2 T1 have
    "\<forall>s \<in> ?S. GroupInv(X\<rightarrow>G,F)`(s) \<in> ?S"
    using FinRangeFunctions_def Group_ZF_2_1_L6
      group0.Group_ZF_3_1_L3 by simp;
  ultimately show ?thesis
    using group0.group0_3_T3 by simp;
qed;
  
section{*Almost homomorphisms*}

text{*An almost homomorphism is a group valued function 
  defined on a monoid $M$ 
  with the property that the set $\{ f(m+n)-f(m)-f(n)\}_{m,n \in M}$ is finite.
  This term is used by R. D. Arthan in "The Eudoxus Real Numbers". We use this
  term in the general group context and use the A`Campo's term "slopes" 
  (see his "A natural construction for the real numbers") to mean
  an almost homomorphism mapping interegers into themselves. 
  We consider almost homomorphisms because we use slopes to 
  define real numbers in the Real\_ZF\_x.thy series. *}

text{*HomDiff is an acronym for "homomorphism difference". 
  This is the expression
  $s(mn)(s(m)s(n))^{-1}$, or $s(m+n)-s(m)-s(n)$ in the additive notation.
  It is equal to the neutral element of the group if $s$ is a homomorphism.
  Almost homomorphisms are defined as those maps 
  $s:G\rightarrow G$ such that the 
  homomorphism difference takes only finite number of values on $G\times G$.
  Although almost homomorphisms can be in principle defined on a monoid with 
  values in a group, we limit ourselves to the situation where the monoid and
  the group are the same. 
  The set of slopes related to a specific group is called 
  AlmostHoms$(G,f)$. AlHomOp1$(G,f)$ is the group operation on almost 
  homomorphisms defined in a natural way 
  by $(s\oplus r)(n) = s(n)\cdot r(n)$. In the terminology defined in 
  func1.thy this is the group operation $f$ (on $G$) 
  lifted to the function space $G\rightarrow G$ and restricted to the set 
  AlmostHoms$(G,f)$.*}

constdefs
  HomDiff :: "[i,i,i,i]\<Rightarrow>i"
  "HomDiff(G,f,s,x) == 
  f` < s`(f`<fst(x),snd(x)>) , 
  (GroupInv(G,f)`(f`<s`(fst(x)),s`(snd(x))>)) >"

  AlmostHoms :: "[i,i]\<Rightarrow>i"
  "AlmostHoms(G,f) == 
  {s \<in> G\<rightarrow>G.{HomDiff(G,f,s,x). x \<in> G\<times>G } \<in> Fin(G)}"

  AlHomOp1 :: "[i,i]\<Rightarrow>i"
  "AlHomOp1(G,f) == 
  restrict(f {lifted to function space over} G,
  AlmostHoms(G,f)\<times>AlmostHoms(G,f))"

text{*This lemma provides more readable notation for the HomDiff
  definition. Not really intended to be used in proofs, but just to see the
  definition in the notation defined in the group0 locale.*}

lemma (in group0) Group_ZF_3_2_L1:
  shows "HomDiff(G,f,s,<m,n>) = s`(m\<cdot>n)\<cdot>(s`(m)\<cdot>s`(n))\<inverse>"
proof -;
  show ?thesis using HomDiff_def by simp;
qed;

text{*Let's define some notation. We inherit the notation and assumptions 
  from the group0 context (locale) 
  and add some. We will use AH to denote the set of almost homomorphisms.
  $\sim$ is the inverse (negative if the group is the group of integers) of 
  almost homomorphisms, $(\sim p)(n)= p(n)^{-1}$.  
  $\delta $ will denote the homomorphism difference specific
  for the group (HomDiff$(G,f)$). We also add an assumption that the
  $G$ is abelian as many needed properties do not hold without that. *}

locale group1 = group0 +
  assumes isAbelian:"f {is commutative on} G"

  fixes AH :: "i"
  defines AH_def [simp]: "AH == AlmostHoms(G,f)"

  fixes Op1 :: "i"
  defines Op1_def [simp]: "Op1 == AlHomOp1(G,f)"

  fixes FR :: "i"
  defines FR_def [simp]: "FR == FinRangeFunctions(G,G)"

  fixes neg :: "i\<Rightarrow>i" ("\<sim>_" [90] 91)
  defines neg_def [simp]: "\<sim>s == GroupInv(G,f) O s"

  fixes \<delta> :: "[i,i]\<Rightarrow>i"
  defines \<delta>_def [simp]: "\<delta>(s,x) == HomDiff(G,f,s,x)";

text{*HomDiff is a homomorphism on the lifted group structure.*}

lemma (in group1) Group_ZF_3_2_L1: 
  assumes A1:"s:G\<rightarrow>G" "r:G\<rightarrow>G"
  and A2:"x \<in> G\<times>G"
  and A3:"F = f {lifted to function space over} G"
  shows "\<delta>(F`<s,r>,x) = \<delta>(s,x)\<cdot>\<delta>(r,x)"
proof -;
  let ?p = "F`<s,r>"
  from A2 obtain m n where
    D1: "x = <m,n>" "m\<in>G" "n\<in>G" 
    by auto;
  then have T1:"m\<cdot>n \<in> G"
    using group0_2_L1 monoid0.group0_1_L1 by simp;
  with A1 D1 have T2:
    "s`(m)\<in>G" "s`(n)\<in>G" "r`(m)\<in>G" 
    "r`(n)\<in>G" "s`(m\<cdot>n)\<in>G" "r`(m\<cdot>n)\<in>G"
    using apply_funtype by auto;
  from A3 A1 have T3:"?p : G\<rightarrow>G"
    using group0_2_L1 monoid0.Group_ZF_2_1_L0
    by simp;
  from D1 T3 have
    "\<delta>(?p,x) = ?p`(m\<cdot>n)\<cdot>((?p`(n))\<inverse>\<cdot>(?p`(m))\<inverse>)"
    using Group_ZF_3_2_L1 apply_funtype group0_2_L10 
    by simp;
  also from A3 A1 D1 T1 isAbelian T2 have
    "?p`(m\<cdot>n)\<cdot>((?p`(n))\<inverse>\<cdot>(?p`(m))\<inverse>) = \<delta>(s,x)\<cdot> \<delta>(r,x)"
    using Group_ZF_2_1_L3 group0_4_L3  Group_ZF_3_2_L1
    by simp;
  finally show ?thesis by simp;
qed;

text{*The group operation lifted to the function space over $G$ preserves
  almost homomorphisms.*}

lemma (in group1) Group_ZF_3_2_L2: assumes A1:"s \<in> AH" "r \<in> AH"
  and A2:"F = f {lifted to function space over} G"
  shows "F`<s,r> \<in> AH"
proof -
  let ?p = "F`<s,r>"
  from A1 A2 have "?p : G\<rightarrow>G"
    using AlmostHoms_def group0_2_L1 monoid0.Group_ZF_2_1_L0
    by simp;
  moreover have
    "{\<delta>(?p,x). x \<in> G\<times>G} \<in> Fin(G)"
  proof -;
    from A1 have
      "{\<delta>(s,x). x \<in> G\<times>G } \<in> Fin(G)" 
      "{\<delta>(r,x). x \<in> G\<times>G } \<in> Fin(G)"
      using AlmostHoms_def by auto;
    with groupAsssum A1 A2 show ?thesis
      using IsAgroup_def IsAmonoid_def IsAssociative_def
      Finite1_L15 AlmostHoms_def Group_ZF_3_2_L1
      by auto;
  qed;
  ultimately show ?thesis using AlmostHoms_def
    by simp;
qed;

text{*The set of almost homomorphisms is closed with respect to the 
  lifted group operation.*}

lemma (in group1) Group_ZF_3_2_L3:
  assumes A1:"F = f {lifted to function space over} G"
  shows "AH {is closed with respect to} F"
proof -;
  from A1 show ?thesis
    using IsOpClosed_def Group_ZF_3_2_L2 by simp;
qed;


text{*The terms in the homomorphism difference are in the group.*}

lemma (in group1) Group_ZF_3_2_L4:  
  assumes A1:"s \<in> AH" and A2:"x\<in>G\<times>G"
  shows "fst(x)\<cdot>snd(x) \<in> G" 
  "s`(fst(x)\<cdot>snd(x)) \<in> G"
  "s`(fst(x)) \<in> G" "s`(snd(x)) \<in> G"
  "\<delta>(s,x) \<in> G"
proof -;
  let ?m = "fst(x)" 
  let ?n = "snd(x)";
  from A2 show "?m\<cdot>?n \<in> G"
    using group0_2_L1 monoid0.group0_1_L1
    by simp;
  with A1 A2 show
    "s`(?m\<cdot>?n) \<in> G"
    "s`(?m) \<in> G" "s`(?n) \<in> G"
    using AlmostHoms_def apply_type by auto;
  then show "\<delta>(s,x) \<in> G"
    using HomDiff_def  group0_2_L5 
      group0_2_L1 monoid0.group0_1_L1 by simp;
qed;

text{*What are the values of the inverse of an almost homomorphism?*}

lemma (in group1) Group_ZF_3_2_L5:
  assumes A1:"s \<in> AH" and A2:"n\<in>G"
  shows "(\<sim>s)`(n) = (s`(n))\<inverse>"
proof -; 
  from A1 A2 show ?thesis
    using AlmostHoms_def comp_fun_apply 
    by auto;
qed;

text{*Homomorphism difference commutes with the inverse for almost
  homomorphisms.*}
  
lemma (in group1) Group_ZF_3_2_L6:  
  assumes A1:"s \<in> AH" and A2:"x\<in>G\<times>G"
  shows "\<delta>(\<sim>s,x) = (\<delta>(s,x))\<inverse>"
proof -;
  let ?m = "fst(x)"
  let ?n = "snd(x)"
  have "\<delta>(\<sim>s,x) = (\<sim>s)`(?m\<cdot>?n)\<cdot>((\<sim>s)`(?m)\<cdot>(\<sim>s)`(?n))\<inverse>"
    using HomDiff_def by simp;
  from A1 A2 isAbelian show ?thesis
    using Group_ZF_3_2_L4 HomDiff_def 
      Group_ZF_3_2_L5 group0_4_L4
    by simp;
qed;

text{*The inverse of an almost homomorphism  maps the group into itself.*}

lemma (in group1) Group_ZF_3_2_L7: 
  assumes A1:"s \<in> AH"
  shows "\<sim>s : G\<rightarrow>G"
proof -;
  from A1 groupAsssum show ?thesis
    using AlmostHoms_def group0_2_T2 comp_fun
    by auto;
qed;

text{*The inverse of an almost homomorphism is an almost homomorphism.*}

lemma (in group1) Group_ZF_3_2_L8:
  assumes A1:"F = f {lifted to function space over} G"
  and A2:"s \<in> AH"
  shows "GroupInv(G\<rightarrow>G,F)`(s) \<in> AH"
proof -;
  from A2 have "{\<delta>(s,x). x \<in> G\<times>G} \<in> Fin(G)"
    using AlmostHoms_def by simp;
  with groupAsssum  have
    "GroupInv(G,f)``{\<delta>(s,x). x \<in> G\<times>G} \<in> Fin(G)"
    using group0_2_T2 Finite1_L6A by blast;
  moreover have 
     "GroupInv(G,f)``{\<delta>(s,x). x \<in> G\<times>G} =
    {(\<delta>(s,x))\<inverse>. x \<in> G\<times>G}"
  proof -;
    from groupAsssum have 
      "GroupInv(G,f) : G\<rightarrow>G"
      using group0_2_T2 by simp;
    moreover from A2 have 
      "\<forall>x\<in>G\<times>G. \<delta>(s,x)\<in>G"
      using Group_ZF_3_2_L4 by simp;
    ultimately show ?thesis 
      using func1_1_L17 by simp;
  qed
  ultimately have "{(\<delta>(s,x))\<inverse>. x \<in> G\<times>G} \<in> Fin(G)"
    by simp;
  moreover from A2 have
    "{(\<delta>(s,x))\<inverse>. x \<in> G\<times>G} = {\<delta>(\<sim>s,x). x \<in> G\<times>G}"
    using Group_ZF_3_2_L6 by simp;
  ultimately have "{\<delta>(\<sim>s,x). x \<in> G\<times>G} \<in> Fin(G)"
    by simp;
  with A2 groupAsssum A1 show ?thesis
    using Group_ZF_3_2_L7 AlmostHoms_def Group_ZF_2_1_L6
    by simp;
qed;

text{*The function that assigns the neutral element everywhere 
  is an almost homomorphism. *}

lemma (in group1) Group_ZF_3_2_L9:
  "ConstantFunction(G,TheNeutralElement(G,f)) \<in> AH"
  "AH\<noteq>0"
proof -;
  let ?z = "ConstantFunction(G,TheNeutralElement(G,f))"
  have "G\<times>G\<noteq>0" using group0_2_L1 monoid0.group0_1_L3A
    by blast;
  moreover have "\<forall>x\<in>G\<times>G. \<delta>(?z,x) = e"
  proof;
    fix x assume A1:"x \<in> G \<times> G"
    then obtain m n where "x = <m,n>" "m\<in>G" "n\<in>G"
      by auto;
    then show "\<delta>(?z,x) = e"
      using group0_2_L1 monoid0.group0_1_L1
	func1_5_L2 HomDiff_def group0_2_L2 
	group0_2_L7A by simp;
  qed;
  ultimately show "?z \<in> AH"
    using group0_2_L2 Finite1_L16
      func1_5_L1 group0_2_L2 AlmostHoms_def
    by simp;
  then show "AH\<noteq>0" by auto;
qed;

text{*If the group is abelian, then almost homomorphisms form a 
  subgroup of the lifted group.*}

lemma Group_ZF_3_2_L10:
  assumes A1:"IsAgroup(G,f)"
  and A2:"f {is commutative on} G"
  and A3:"F = f {lifted to function space over} G"
  shows "IsAsubgroup(AlmostHoms(G,f),F)"
proof -;
  let ?AH = "AlmostHoms(G,f)"
  from A2 A1 have T1:"group1(G,f)"
    using group1_axioms.intro group0_def group1_def
    by simp;
  from A1 A3 have "group0(G\<rightarrow>G,F)"
    using group0_def group0.Group_ZF_2_1_T2 by simp;
  moreover from T1 have "?AH\<noteq>0"
    using group1.Group_ZF_3_2_L9 by simp;
  moreover have T2:"?AH \<subseteq> G\<rightarrow>G"
    using AlmostHoms_def by auto;
  moreover from T1 A3 have 
    "?AH {is closed with respect to} F"
    using group1.Group_ZF_3_2_L3 by simp;
  moreover from T1 A3 have
    "\<forall>s\<in>?AH. GroupInv(G\<rightarrow>G,F)`(s) \<in> ?AH"
    using group1.Group_ZF_3_2_L8 by simp;
  ultimately show "IsAsubgroup(AlmostHoms(G,f),F)"
    using group0.group0_3_T3 by simp;
qed;

text{*If the group is abelian, then almost homomorphisms form a group
  with the first operation.*}

lemma (in group1) Group_ZF_3_2_L10A: "IsAgroup(AH,Op1)"
proof -
  from groupAsssum isAbelian show ?thesis
    using Group_ZF_3_2_L10 IsAsubgroup_def AlHomOp1_def
    by simp;
qed;

text{*The group of almost homomorphisms is abelian*}

lemma Group_ZF_3_2_L11: assumes A1:"IsAgroup(G,f)"
  and A2:"f {is commutative on} G"
  shows "IsAgroup(AlmostHoms(G,f),AlHomOp1(G,f))"
  "AlHomOp1(G,f){is commutative on} AlmostHoms(G,f)"
proof-
  let ?AH = "AlmostHoms(G,f)"
  let ?F = "f {lifted to function space over} G"
  from A1 A2 have "IsAsubgroup(?AH,?F)"
    using Group_ZF_3_2_L10 by simp;
  then show "IsAgroup(?AH,AlHomOp1(G,f))"
    using IsAsubgroup_def AlHomOp1_def by simp;
  from A1 have "?F : (G\<rightarrow>G)\<times>(G\<rightarrow>G)\<rightarrow>(G\<rightarrow>G)"
    using IsAgroup_def monoid0_def monoid0.Group_ZF_2_1_L0A
    by simp;
  moreover have "?AH \<subseteq> G\<rightarrow>G"
    using AlmostHoms_def by auto
  moreover from A1 A2 have
    "?F {is commutative on} (G\<rightarrow>G)"
    using group0_def group0.Group_ZF_2_1_L7
    by simp;
  ultimately show 
    "AlHomOp1(G,f){is commutative on} ?AH"
    using func1_6_L1 AlHomOp1_def by simp;
qed;

section{*The classes of almost homomorphisms*}

text{*In the Real\_ZF\_x series we define real numbers as a quotient
  of the group of integer almost homomorphisms by the integer finite range
  functions. In this section we setup the background for that in the general
  group context.*}

text{*Finite range functions are almost homomorphisms.*}

lemma (in group1) Group_ZF_3_3_L1: "FR \<subseteq> AH"
proof;
  fix s assume A1:"s \<in> FR"
  then have T1:"{s`(n). n \<in> G} \<in> Fin(G)"
    "{s`(fst(x)). x\<in>G\<times>G} \<in> Fin(G)"
    "{s`(snd(x)). x\<in>G\<times>G} \<in> Fin(G)"
    using Finite1_L18 Finite1_L6B by auto;
  have "{s`(fst(x)\<cdot>snd(x)). x \<in> G\<times>G} \<in> Fin(G)"
  proof -;
    have "\<forall>x\<in>G\<times>G. fst(x)\<cdot>snd(x) \<in> G"
      using group0_2_L1 monoid0.group0_1_L1 by simp;
    moreover from T1 have "{s`(n). n \<in> G} \<in> Fin(G)" by simp;
    ultimately show ?thesis by (rule Finite1_L6B);
  qed
  moreover have 
    "{(s`(fst(x))\<cdot>s`(snd(x)))\<inverse>. x\<in>G\<times>G} \<in> Fin(G)"
  proof -;
    have "\<forall>g\<in>G. g\<inverse> \<in> G" using group0_2_L5 
      by simp
    moreover from T1 have 
      "{s`(fst(x))\<cdot>s`(snd(x)). x\<in>G\<times>G} \<in> Fin(G)"
      using group0_2_L3A  Finite1_L15 by simp;
    ultimately show ?thesis 
      by (rule Finite1_L6C);
  qed;
  ultimately have "{\<delta>(s,x). x\<in>G\<times>G} \<in> Fin(G)"
    using HomDiff_def Finite1_L15  group0_2_L3A 
    by simp;
  with A1 show "s \<in> AH" 
    using FinRangeFunctions_def AlmostHoms_def 
    by simp;
qed;

text{*Finite range functions valued in an abelian group form a normal 
  subgroup of almost homomorphisms.*}

lemma Group_ZF_3_3_L2: assumes A1:"IsAgroup(G,f)"
  and A2:"f {is commutative on} G"
  shows 
  "IsAnormalSubgroup(AlmostHoms(G,f),AlHomOp1(G,f),
  FinRangeFunctions(G,G))"
proof -;
  let ?H1 = "AlmostHoms(G,f)"
  let ?H2 = "FinRangeFunctions(G,G)"
  let ?F = "f {lifted to function space over} G"
  from A1 A2 have T1:"group0(G,f)"
    "monoid0(G,f)" "group1(G,f)"
    using group0_def group0.group0_2_L1  
      group1_axioms.intro group1_def
    by auto;
  with A1 A2 have "IsAgroup(G\<rightarrow>G,?F)"
    "IsAsubgroup(?H1,?F)" "IsAsubgroup(?H2,?F)"
    using group0.Group_ZF_2_1_T2 Group_ZF_3_2_L10
      monoid0.group0_1_L3A Group_ZF_3_1_T1
    by auto;
  then have 
    "IsAsubgroup(?H1\<inter>?H2,restrict(?F,?H1\<times>?H1))"
    using group0_3_L7 by simp;
  moreover from T1 have "?H1\<inter>?H2 = ?H2"
    using group1.Group_ZF_3_3_L1 by auto;
  ultimately have "IsAsubgroup(?H2,AlHomOp1(G,f))"
    using AlHomOp1_def by simp;
  with A1 A2 show ?thesis
    using Group_ZF_3_2_L11 Group_ZF_2_4_L6
    by simp;
qed;

text{*The group of almost homomorphisms divided by the subgroup of finite
  range functions is an abelian group.*}

theorem (in group1) Group_ZF_3_3_T1:
  "IsAgroup(QuotientGroup(AH,Op1,FR),QuotientGroupOp(AH,Op1,FR))"
  "QuotientGroupOp(AH,Op1,FR) {is commutative on}
  QuotientGroup(AH,Op1,FR)"
proof -
  from groupAsssum isAbelian show
    "IsAgroup(QuotientGroup(AH,Op1,FR),QuotientGroupOp(AH,Op1,FR))"
    using Group_ZF_3_3_L2 Group_ZF_3_2_L10A Group_ZF_2_4_T1
    by simp;
  from groupAsssum isAbelian show 
    "QuotientGroupOp(AH,Op1,FR) {is commutative on}
    QuotientGroup(AH,Op1,FR)"
    using Group_ZF_3_2_L10A Group_ZF_3_2_L11  Group_ZF_3_3_L2
    IsAnormalSubgroup_def Group_ZF_2_4_L6 by simp;
qed;
    
end


  
    
  
  
  
  
    
  
  
  
  
  
 