(*
    This file is a part of IsarMathLib - 
    a library of formalized mathematics for Isabelle/Isar.

    Copyright (C) 2005  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

HIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{Real\_ZF.thy}*}

theory Real_ZF = Group_ZF_1 + Group_ZF_3:

text{*The goal of the Real\_ZF series of theory files is to provide 
  a contruction of the set of 
  real numbers. There are several ways to construct real numbers. 
  Most common start from the rational numbers and use Dedekind cuts 
  or Cauchy sequences. Real\_ZF\_x.thy series formalizes an alternative 
  approach that constructs real numbers directly from the group of integers.
  Different variants of this contruction are described in papers 
  "A natural construction for the real numbers" (N. A'Campo),
  "The Efficient Real Numbers" (R. Street at al.), 
  "The Eudoxus Real Numbers" (R. D. Arthan).
  I recommend to read these papers, but for the impatient here is a short 
  description: we take a set of maps $s : Z\rightarrow Z$ such that
   the set $\{s(m+n)-s(m)-s(n)\}_{n,m \in Z}$ is finite 
  ($Z$ means the integers here). We call these maps slopes. 
  Slopes form a group with the natural addition
  $(s+r)(n) = s(n)+r(n)$. The maps such that the set $s(Z)$ is finite 
  (finite range functions) form a subgroup of slopes. 
  The additive group of real numbers is defined as the 
  quotient group of slopes by the (sub)group of finite range functions.
  The multiplication is defined as the projection of the composition of slopes
  into the resulting quotient (coset) space.
  *}

section{*The definition of real numbers*}

text{*First we define slopes and real numbers as the set of their classes.*}

constdefs

  Slopes :: "i"
  "Slopes == AlmostHoms(int,IntegerAddition)"

  SlopeOp1 :: "i"
  "SlopeOp1 == AlHomOp1(int,IntegerAddition)"

  BoundedIntMaps :: "i"
  "BoundedIntMaps == FinRangeFunctions(int,int)"

  RealNumbers :: "i"
  "RealNumbers == QuotientGroup(Slopes,SlopeOp1,BoundedIntMaps)"

  RealAddition :: "i"
  "RealAddition == QuotientGroupOp(Slopes,SlopeOp1,BoundedIntMaps)"

text{*Real numbers form an abelian group under real addition.*}

theorem Real_ZF_1_T1: "IsAgroup(RealNumbers,RealAddition)"
  "RealAddition {is commutative on} RealNumbers"
proof -;
  let ?AH = "AlmostHoms(int,IntegerAddition)"
  let ?Op1 = "AlHomOp1(int,IntegerAddition)"
  let ?FR = "FinRangeFunctions(int,int)"
  have 
    "IsAgroup(QuotientGroup(?AH,?Op1,?FR),QuotientGroupOp(?AH,?Op1,?FR))"
    "QuotientGroupOp(?AH,?Op1,?FR) {is commutative on}
    QuotientGroup(?AH,?Op1,?FR)"
    using group1_1_T1 group0_def group1_axioms.intro group1_def
      group1.Group_ZF_3_3_T1 by auto;
  then show "IsAgroup(RealNumbers,RealAddition)"
    "RealAddition {is commutative on} RealNumbers"
    using Slopes_def SlopeOp1_def BoundedIntMaps_def 
      RealNumbers_def RealAddition_def by auto;
qed;
end;