(* 
    This file is a part of IsarMathLib - 
    a library of formalized mathematics written for Isabelle/Isar.

    Copyright (C) 2005  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{func1.thy}*}
theory func1 = func + Fol1:

text{*We define the notion of function that preserves a collection here.
Given two collection of sets a function preserves the collections if 
the inverse image of sets in one collection belongs to the second one.
This notion does not have a name in romantic math. It is used to define 
continuous functions in Topology\_ZF\_2 theory. 
We define it here so that we can 
use it for other purposes, like defining measurable functions.
Recall that f-``(A) means the inverse image of the set $A$.*}

constdefs
  PresColl :: "[i,i,i]\<Rightarrow>o"
  "PresColl(f,S,T) == \<forall> A\<in>T. f-``(A)\<in>S";

section{*Properties of functions, function spaces and (inverse) images.*}

text{*If a function maps $A$ into another set, then $A$ is the 
  domain of the function.*}

lemma func1_1_L1: assumes A:"f:A\<rightarrow>C" shows "domain(f) = A"
proof-;
  from A show ?thesis using domain_of_fun by simp
qed;

text{*There is a value for each argument.*}

lemma func1_1_L2: assumes A:"f:X\<rightarrow>Y" "x\<in>X" 
  shows "\<exists>y\<in>Y. <x,y> \<in> f";  
proof-;
    from A have "f`(x) \<in> Y" using apply_type by simp;
    moreover from A have "<x,f`(x)>\<in> f" using apply_Pair by simp;
    ultimately show ?thesis by auto;
qed; 

text{*Inverse image of any set is contained in the domain.*}

lemma func1_1_L3: assumes A:"f:X\<rightarrow>Y" shows "f-``(D) \<subseteq> X"
proof-;
   have "\<forall>x. x \<in>  f -`` D \<longrightarrow> x \<in> domain(f)"
      using  vimage_iff domain_iff by auto;
    with A have "\<forall>x. x \<in> f -`` D \<longrightarrow> x\<in> X" using func1_1_L1 by simp
    then show ?thesis by auto;
qed;

text{*The inverse image of the range is the domain.*}

lemma func1_1_L4: assumes A:"f:X\<rightarrow>Y" shows "f-``(Y) = X"
proof;
  from A show "f -`` Y \<subseteq> X" using func1_1_L3 by simp;
  from A show "X \<subseteq> f -`` Y" using func1_1_L2 vimage_iff by auto;
qed;

text{*The arguments belongs to the domain and values to the range.*}

lemma func1_1_L5: 
  assumes A1: "<x,y> \<in> f" and A2: "f:X\<rightarrow>Y"  
  shows "x\<in>X \<and> y\<in>Y"
proof;
  from A1 A2 show "x\<in>X" using apply_iff by simp;
  with A2 have "f`(x)\<in> Y" using apply_type by simp;
  with A1 A2 show "y\<in>Y" using apply_iff by simp;
qed;

text{*The (argument, value) pair belongs to the graph of the function.*}

lemma func1_1_L5A: 
  assumes A1: "f:X\<rightarrow>Y" "x\<in>X" "y = f`(x)"
  shows "<x,y> \<in> f" "y \<in> range(f)" 
proof -;
  from A1 show "<x,y> \<in> f" using apply_Pair by simp;
  then show "y \<in> range(f)" using rangeI by simp;
qed;

text{*The range of function thet maps $X$ into $Y$ is contained in $Y$.*}

lemma func1_1_L5B: 
  assumes  A1:"f:X\<rightarrow>Y" shows "range(f) \<subseteq> Y"
proof;
  fix y assume "y \<in> range(f)"
  then obtain x where "<x,y> \<in> f"
    using range_def converse_def domain_def by auto;
  with A1 show "y\<in>Y" using func1_1_L5 by blast;
qed;


text{*The image of any set is contained in the range.*}

lemma func1_1_L6: assumes A:"f:X\<rightarrow>Y" shows "f``(B)\<subseteq>Y"
proof; 
  fix y
  assume A1:"y \<in> f `` B" show "y \<in> Y";
  proof-;
    from A1 obtain x where "<x,y>\<in> f" using image_iff by auto;
    with A show "y \<in> Y" using func1_1_L5 by fast;
  qed;
qed;

text{*The inverse image of any set is contained in the domain.*}

lemma func1_1_L6A: assumes A1:"f:X\<rightarrow>Y" shows "f-``(A)\<subseteq>X"
proof;
  fix x
  assume A2: "x\<in>f-``(A)" show "x\<in>X"
  proof -;
    from A2 obtain y where "<x,y>\<in> f" using vimage_iff by auto;
    with A1 show  "x\<in>X" using func1_1_L5 by fast;
  qed;
qed;

text{*Inverse image of a greater set is greater.*}

lemma func1_1_L7: assumes A1:"A\<subseteq>B" "function(f)" 
  shows "f-``(A)\<subseteq> f-``(B)"
proof -;
  from A1 show ?thesis using function_vimage_Diff by auto
qed;

text{*Image of a greater set is greater.*}

lemma func1_1_L8: assumes A1:"A\<subseteq>B"  shows "f``(A)\<subseteq> f``(B)"
proof -;
  from A1 show ?thesis using image_Un by auto;
qed;

text{* A set is contained in the the inverse image of its image.
There is similar theorem in equalities.thy (function\_image\_vimage)
which shows that the image of inverse image of a set 
is contained in the set.*}

lemma func1_1_L9: assumes A1:"f\<in>X\<rightarrow>Y" and A2:"A\<subseteq>X"
  shows "A \<subseteq> f-``(f``(A))"
proof -
  from A2 have "\<forall>x\<in>A. x\<in>X" by auto;
  with A1 have "\<forall>x\<in>A. <x,f`(x)> \<in> f"  using apply_Pair by simp
  then have "\<forall>x\<in>A. x \<in> f-``(f``(A))" using image_iff by auto
  thus ?thesis by auto;
qed;
 

text{*A technical lemma needed to make the func1\_1\_L11 proof more clear.*}

lemma func1_1_L10: 
  assumes A1:"f \<subseteq> X\<times>Y" and A2:"x\<in>X" and A3:"\<exists>!y. (y\<in>Y & <x,y> \<in> f)"
  shows "\<exists>!y. <x,y> \<in> f"
proof;
  from A3 show "\<exists>y. \<langle>x, y\<rangle> \<in> f" by auto;
  fix y n assume A4:"<x,y> \<in> f" and A5: "<x,n> \<in> f" show "y=n"; 
  proof -;
    from A1 A4 A5 A3 show "y=n" by auto;
  qed;
qed;


text{*If $f\subseteq X\times Y$ and for every $x\in X$ there is exactly 
one $y\in Y$ such that $(x,y)\in f$ then $f$ maps $X$ to $Y$.*}

lemma func1_1_L11: 
  assumes A1:"f \<subseteq> X\<times>Y" and A2:"\<forall>x\<in>X. \<exists>!y. y\<in>Y & <x,y> \<in> f"
  shows "f\<in>X\<rightarrow>Y"
proof -;
  from A1 A2 show ?thesis using func1_1_L10 Pi_iff_old by simp;
qed;

text{*A set defined by a lambda-type expression is a fuction. There is a 
  similar lemma in func.thy, but I had problems with lamda expressions syntax
  so I could not apply it. This lemma is a workaround this. Besides, lambda 
  expressions are not readable.
*}

lemma func1_1_L11A: assumes A1:"\<forall>x\<in>X. b(x)\<in>Y"
  shows "{<x,y> \<in> X\<times>Y. b(x) = y} \<in> X\<rightarrow>Y"
proof -;
  let ?f = "{<x,y> \<in> X\<times>Y. b(x) = y}"
  have "?f \<subseteq> X\<times>Y" by auto;
  moreover have "\<forall>x\<in>X. \<exists>!y. y\<in>Y & <x,y> \<in> ?f"
  proof;
    fix x assume A2:"x\<in>X" 
    show "\<exists>!y. y\<in>Y \<and> \<langle>x, y\<rangle> \<in> {\<langle>x,y\<rangle> \<in> X\<times>Y . b(x) = y}";
    proof;
      fix y y1
      assume A1:"y\<in>Y \<and> \<langle>x, y\<rangle> \<in> {\<langle>x,y\<rangle> \<in> X\<times>Y . b(x) = y}"
	and A2:"y1\<in>Y \<and> \<langle>x, y1\<rangle> \<in> {\<langle>x,y\<rangle> \<in> X\<times>Y . b(x) = y}"
      from A1 have "y = b(x)" by simp;
      moreover from A2 have "y1 = b(x)" by simp;
      ultimately show "y = y1" by simp;
    next;
      def y == "b(x)";
      with A2 A1 show 
        "\<exists>y. y\<in>Y & \<langle>x, y\<rangle> \<in> {\<langle>x,y\<rangle> \<in> X\<times>Y . b(x) = y}"
	by simp;
    qed;
  qed;
  ultimately show "{<x,y> \<in> X\<times>Y. b(x) = y} \<in> X\<rightarrow>Y" 
    using func1_1_L11 by simp;
qed;

text{*The value of a function defined by a meta-function is this 
  meta-function.*}

lemma func1_1_L11B: 
  assumes A1: "f \<in> X\<rightarrow>Y" "x\<in>X"
  and A2: "f = {<x,y> \<in> X\<times>Y. b(x) = y}"
  shows "f`(x) = b(x)"
proof -;
  from A1 have "<x,f`(x)> \<in> f"
    using apply_iff by simp;
  with A2 show ?thesis by simp;
qed;
  
text{*The inverse image of an intersection of a nonempty collection of sets 
is the intersection of the 
inverse images. This generalizes function\_vimage\_Int
 which is proven for the case of two sets.*}

lemma  func1_1_L12:
  assumes A1:"B\<subseteq>Pow(Y)" and A2:"B\<noteq>0" and A3:"f:X\<rightarrow>Y"
  shows "f-``(\<Inter>B) = (\<Inter>U\<in>B. f-``(U))"
proof;
  from A2 show  "f -`` (\<Inter>B) \<subseteq> (\<Inter>U\<in>B. f -`` U)" by blast;
  show "(\<Inter>U\<in>B. f -`` U) \<subseteq> f -`` (\<Inter>B)"
  proof;
    fix x assume A4: "x \<in> (\<Inter>U\<in>B. f -`` U)";
    from A3 have "\<forall>U\<in>B. f-``(U) \<subseteq> X" using func1_1_L6A by simp;
    with A4 have "\<forall>U\<in>B. x\<in>X" by auto;
    with A2 have "x\<in>X" by auto;
    with A3 have "\<exists>!y. <x,y> \<in> f" using Pi_iff_old by simp;
    with A2 A4 have "\<exists>y\<in>\<Inter>B. <x,y>\<in>f" by auto;
    then show "x \<in> f -`` (\<Inter>B)" using vimage_iff by simp;
  qed
qed;

text{*If the inverse image of a set is not empty, then the set is not empty.
Proof by contradiction.*}

lemma func1_1_L13: assumes A1:"f-``(A)\<noteq>0" shows "A\<noteq>0"
proof (rule ccontr);
  assume A2:"\<not> A \<noteq> 0" from A2 A1 show False by simp;
qed;

text{*What is the inverse image of a singleton?*}

lemma func1_1_L14: assumes A1: "f\<in>X\<rightarrow>Y" 
  shows "f-``({y}) = {x\<in>X. f`(x) = y}"
proof -
  from A1 show ?thesis
    using func1_1_L6A vimage_singleton_iff apply_iff
    by auto;
qed;

text{* A more familiar definition of inverse image.*}

lemma func1_1_L15: assumes A1: "f:X\<rightarrow>Y"
  shows "f-``(A) = {x\<in>X. f`(x) \<in> A}"
proof -;
  have "f-``(A) = (\<Union>y\<in>A . f-``{y})" 
    by (rule vimage_eq_UN);
  with A1 show ?thesis using func1_1_L14 by auto;
qed;

text{*A more familiar definition of image.*}

lemma func1_1_L16: assumes A1: "f:X\<rightarrow>Y" and A2:"A\<subseteq>X"
  shows "f``(A) = {f`(x). x \<in> A}"
proof;
 from A1 show "f``(A) \<subseteq> {f`(x). x \<in> A}"
   using image_iff apply_iff by auto;
 show "{f`(x). x \<in> A} \<subseteq> f``(A)"
 proof;
   fix y assume "y \<in> {f`(x). x \<in> A}"
   then obtain x where "x\<in>A \<and> y = f`(x)"
     by auto;
   with A1 A2 show "y \<in> f``(A)"
     using apply_iff image_iff by auto;
 qed;
qed;

text{*What is the image of a set defined by a meta-fuction?*}

lemma func1_1_L17: 
  assumes A1:"f \<in> X\<rightarrow>Y" and A2:"\<forall>x\<in>A. b(x) \<in> X"
  shows "f``({b(x). x\<in>A}) = {f`(b(x)). x\<in>A}"
proof -;
  from A2 have "{b(x). x\<in>A} \<subseteq> X" by auto;
   with A1 have 
    "f``({b(x). x\<in>A}) = {f`(y). y \<in> {b(x). x\<in>A}}"
    using func1_1_L16 by simp;
  then show ?thesis by auto;
qed;


section{*Functions restricted to a set*}
 
text{*What is the inverse image of a set under a restricted fuction?*}

lemma func1_2_L1: assumes A1: "f:X\<rightarrow>Y" and A2:"B\<subseteq>X"
  shows "restrict(f,B)-``(A) = f-``(A) \<inter> B"
proof -;
  let ?g = "restrict(f,B)";
  from A1 A2 have "?g\<in> B\<rightarrow>Y" 
    using restrict_type2 by simp;
  with A2 A1 show "?g-``(A) = f-``(A) \<inter> B"
    using func1_1_L15 restrict_if by auto;
qed;
   
text{*A criterion for when one function is a restriction of another.
  The lemma below provides a result useful in the actual proof of the 
  criterion and applications.*}

lemma func1_2_L2: 
  assumes A1:"f:X\<rightarrow>Y" and A2:"g \<in> A\<rightarrow>Z" 
  and A3:"A\<subseteq>X" and A4:"f \<inter> A\<times>Z = g"
  shows "\<forall>x\<in>A. g`(x) = f`(x)"
proof;
  fix x assume "x\<in>A"
  with A2 have "<x,g`(x)> \<in> g" using apply_Pair by simp;
  with A4 A1 show "g`(x) = f`(x)"  using apply_iff by auto; 
qed;

text{*Here is the actual criterion.*}

lemma func1_2_L3: 
  assumes A1:"f:X\<rightarrow>Y" and A2:"g:A\<rightarrow>Z" 
  and A3:"A\<subseteq>X" and A4:"f \<inter> A\<times>Z = g"
  shows "g = restrict(f,A)"
proof;
  from A4 show "g \<subseteq> restrict(f, A)" using restrict_iff by auto;
  show "restrict(f, A) \<subseteq> g"
  proof;
    fix z assume A5:"z \<in> restrict(f,A)"
    then obtain x y where D1:"z\<in>f & x\<in>A  & z = <x,y>"
      using restrict_iff by auto;
    with A1 have "y = f`(x)" using apply_iff by auto;
    with A1 A2 A3 A4 D1 have "y = g`(x)" using func1_2_L2 by simp;
    with A2 D1 show "z\<in>g" using apply_Pair by simp;
  qed;
qed;

text{*Which function space a restricted function belongs to?*}

lemma func1_2_L4: 
  assumes A1:"f:X\<rightarrow>Y" and A2:"A\<subseteq>X" and A3:"\<forall>x\<in>A. f`(x) \<in> Z"
  shows "restrict(f,A) \<in> A\<rightarrow>Z"
proof -;
  from A1 A2 have "restrict(f,A) \<in> A\<rightarrow>Y" 
    using restrict_type2 by simp;
  with A3 show ?thesis using Pi_type by simp; (*blast etc no-go*)
qed;

section{*Lifting operations to a function space*}

text{*It happens quite often that we have a binary operation on some set and
  we need a similar operation that is defined for functions on that set. 
  For example once we know how to add real numbers we also know how to add
  real-valued functions: for $f,g:X \rightarrow \mathbf{R}$ we define
  $(f+g)(x) = f(x) + g(x)$. Note that formally the $+$ means something 
  different on the left hand side of this equality than on the 
  right hand side.
  This section aims at formalizing this process.
  We will call it "lifting to a function space", if you have a 
  suggestion for a better name, please let me know. *} 

constdefs 
  Lift2FcnSpce :: 
  "[i,i]\<Rightarrow>i" (infix "{lifted to function space over}" 65) 
  "f {lifted to function space over} X == 
  {<p,g> \<in> ((X\<rightarrow>range(f))\<times>(X\<rightarrow>range(f)))\<times>(X\<rightarrow>range(f)).
  {<x,y> \<in> X\<times>range(f). f`<fst(p)`(x),snd(p)`(x)> = y} = g}";

text{*The result of the lift belongs to the function space.*}

lemma func1_3_L1: 
  assumes A1:"f : Y\<times>Y\<rightarrow>Y" 
  and A2:"p \<in>(X\<rightarrow>range(f))\<times>(X\<rightarrow>range(f))"
  shows 
  "{<x,y> \<in> X\<times>range(f). f`<fst(p)`(x),snd(p)`(x)> = y} : X\<rightarrow>range(f)"
  proof -;
    have "\<forall>x\<in>X. f`<fst(p)`(x),snd(p)`(x)> \<in> range(f)"
    proof;
      fix x assume A3:"x\<in>X"
      let ?p = "<fst(p)`(x),snd(p)`(x)>"
      from A2 A3 have 
	"fst(p)`(x) \<in> range(f)"  "snd(p)`(x) \<in> range(f)"
	using apply_type by auto;
      with A1 have "?p \<in> Y\<times>Y"
	using func1_1_L5B by blast;
      with A1 have "<?p, f`(?p)> \<in> f"
	using apply_Pair by simp;
      with A1 show 
	"f`(?p) \<in> range(f)"
	using rangeI by simp;
    qed
    then show ?thesis using func1_1_L11A by simp;
qed;

text{*The values of the lift are defined by the value of the liftee in a 
  natural way.*}

lemma func1_3_L2: 
  assumes A1:"f : Y\<times>Y\<rightarrow>Y" 
  and A2:"p\<in>(X\<rightarrow>range(f))\<times>(X\<rightarrow>range(f))"
  and A3:"x\<in>X"
  and A4:"P = {<x,y> \<in> X\<times>range(f). f`<fst(p)`(x),snd(p)`(x)> = y}"
  shows "P`(x) = f`<fst(p)`(x),snd(p)`(x)>"
proof -;
  from A1 A2 A3 A4 show ?thesis 
    using func1_3_L1 func1_1_L11B by simp;
qed;

text{*Function lifted to a function space results in a function space 
  operator.*}

lemma func1_3_L3: 
  assumes A1:"f \<in> Y\<times>Y\<rightarrow>Y"
  and A2:"F = f {lifted to function space over} X"
  shows "F : (X\<rightarrow>range(f))\<times>(X\<rightarrow>range(f))\<rightarrow>(X\<rightarrow>range(f))"
proof -;
  from A1 A2 show ?thesis 
    using Lift2FcnSpce_def func1_3_L1 func1_1_L11A
    by simp;
qed;

text{*The values of the lift are defined by the values of the liftee in
  the natural way. For some reason we need to be extremely detailed and 
  explicit to be able to apply func1\_3\_L2. simp and auto 
  fail miserably here.*}

lemma func1_3_L4: 
  assumes A1:"f : Y\<times>Y\<rightarrow>Y"
  and A2:"F = f {lifted to function space over} X"
  and A3:"s:X\<rightarrow>range(f)" "r:X\<rightarrow>range(f)"  
  and A4:"x\<in>X"
  shows "(F`<s,r>)`(x) = f`<s`(x),r`(x)>"
proof -;
  let ?P = "{<x,y> \<in> X\<times>range(f). f`<s`(x),r`(x)> = y}";
  let ?p = "<s,r>"
  from A1 have "f \<in> Y\<times>Y\<rightarrow>Y" .;
  moreover from A3 have 
    "?p \<in> (X\<rightarrow>range(f))\<times>(X\<rightarrow>range(f))"
    by simp;
  moreover from A4 have "x\<in>X" .;
  moreover have 
    "?P = {<x,y> \<in> X\<times>range(f). f`<fst(?p)`(x),snd(?p)`(x)> = y}"
    by simp;
  ultimately have
    "?P`(x) = f`<fst(?p)`(x),snd(?p)`(x)>"
    by (rule func1_3_L2);
  moreover have 
    "f`<fst(?p)`(x),snd(?p)`(x)> = f`<s`(x),r`(x)>"
    by simp;
  moreover from A1 A2 A3 have
    "F`<s,r> = ?P"
    using func1_3_L3 Lift2FcnSpce_def func1_1_L11B
    by simp;
  ultimately show ?thesis by simp;
qed;

  
section{*Associative and commutative operations*}

text{*In this section we define associative and commutative oparations 
  and prove that they remain such when we lift them
  to a function space.*}

constdefs 

  IsAssociative :: "[i,i]\<Rightarrow>o" (infix "{is associative on}" 65)
  "f {is associative on} G == f \<in> G\<times>G\<rightarrow>G \<and> 
  (\<forall> x \<in> G. \<forall> y \<in> G. \<forall> z \<in> G. 
  ( f`(<f`(<x,y>),z>) = f`( < x,f`(<y,z>)> )))"

  IsCommutative :: "[i,i]\<Rightarrow>o" (infix "{is commutative on}" 65)
  "f {is commutative on} G == \<forall>x\<in>G. \<forall>y\<in>G. f`<x,y> = f`<y,x>"

text{*The lift of a commutative function is commutative.*}

lemma func1_4_L1:
  assumes A1:"f : G\<times>G\<rightarrow>G"
  and A2:"F = f {lifted to function space over} X"
  and A3:"s : X\<rightarrow>range(f)" "r : X\<rightarrow>range(f)" 
  and A4:"f {is commutative on} G"
  shows "F`<s,r> = F`<r,s>" 
proof -;
  from A1 A2 have 
    "F : (X\<rightarrow>range(f))\<times>(X\<rightarrow>range(f))\<rightarrow>(X\<rightarrow>range(f))"
    using func1_3_L3 by simp; 
  with A3 have 
    "F`<s,r> : X\<rightarrow>range(f)" "F`<r,s> : X\<rightarrow>range(f)"
    using apply_type by auto;
  moreover have 
    "\<forall>x\<in>X. (F`<s,r>)`(x) = (F`<r,s>)`(x)"
  proof;
    fix x assume A5:"x\<in>X"
    from A1 have "range(f)\<subseteq>G"
      using func1_1_L5B by simp;
    with A3 A5 have T1:"s`(x) \<in> G" "r`(x) \<in> G"
      using apply_type by auto;
    with A1 A2 A3 A4 A5 show 
      "(F`<s,r>)`(x) = (F`<r,s>)`(x)"
      using func1_3_L4 IsCommutative_def by simp;
  qed;
  ultimately show ?thesis using fun_extension_iff
    by simp;
qed;

text{*The lift of a commutative function is commutative 
  on the function space.*}

lemma func1_4_L2:
  assumes A1:"f : G\<times>G\<rightarrow>G"
  and A2:"f {is commutative on} G"
  and A3:"F = f {lifted to function space over} X"
  shows "F {is commutative on} (X\<rightarrow>range(f))"
proof -;
  from A1 A2 A3 show ?thesis using
    IsCommutative_def func1_4_L1 by simp;
qed;
  
text{*The lift of an associative function is associative.*}

lemma func1_4_L3:
  assumes A2:"F = f {lifted to function space over} X"
  and A3:
  "s : X\<rightarrow>range(f)" "r : X\<rightarrow>range(f)" "q : X\<rightarrow>range(f)"
  and A4: "f {is associative on} G"
  shows "F`<F`<s,r>,q> = F`<s,F`<r,q> >"
proof -;
  from A4 A2 have 
    "F : (X\<rightarrow>range(f))\<times>(X\<rightarrow>range(f))\<rightarrow>(X\<rightarrow>range(f))"
    using IsAssociative_def func1_3_L3 by auto;
  with A3 have T1:
    "F`<s,r> : X\<rightarrow>range(f)"
    "F`<r,q> : X\<rightarrow>range(f)"
    "F`<F`<s,r>,q> : X\<rightarrow>range(f)"
    "F`<s,F`<r,q> >: X\<rightarrow>range(f)"
    using apply_type by auto;
  moreover have
    "\<forall>x\<in>X. (F`<F`<s,r>,q>)`(x) = (F`<s,F`<r,q> >)`(x)"
  proof;
    fix x assume A5:"x\<in>X"
    from A4 have T2:"f:G\<times>G\<rightarrow>G"
      using IsAssociative_def by simp;
    then have "range(f)\<subseteq>G"
      using func1_1_L5B by simp;
    with A3 A5 have 
      "s`(x) \<in> G" "r`(x) \<in> G" "q`(x) \<in> G"
      using apply_type by auto;
    with T2 A2 T1 A3 A5 A4 show
      "(F`<F`<s,r>,q>)`(x) = (F`<s,F`<r,q> >)`(x)"
      using func1_3_L4 IsAssociative_def by simp;
  qed;
  ultimately show ?thesis using fun_extension_iff
    by simp;
qed;

text{*The lift of an associative function is associative 
  on the function space.*}

lemma func1_4_L4:
  assumes A1:"f {is associative on} G"
  and A2:"F = f {lifted to function space over} X"
  shows "F {is associative on} (X\<rightarrow>range(f))"
proof -;
  from A1 A2 have
    "F : (X\<rightarrow>range(f))\<times>(X\<rightarrow>range(f))\<rightarrow>(X\<rightarrow>range(f))"
    using IsAssociative_def func1_3_L3 by auto;
  moreover from A1 A2 have
    "\<forall>s \<in> X\<rightarrow>range(f). \<forall> r \<in> X\<rightarrow>range(f). \<forall>q \<in> X\<rightarrow>range(f).
    F`<F`<s,r>,q> = F`<s,F`<r,q> >"
    using func1_4_L3 by simp;
  ultimately show ?thesis 
    using IsAssociative_def by simp;
qed;

section{*Constant functions*}

text{*We define constant($=c$) functions on a set $X$  in a natural way as 
  ConstantFunction$(X,c)$. *}

constdefs
  ConstantFunction :: "[i,i]\<Rightarrow>i"
  "ConstantFunction(X,c) == X\<times>{c}"

lemma func1_5_L1: 
  assumes A1:"c\<in>Y" shows "ConstantFunction(X,c) : X\<rightarrow>Y"
proof -;
   from A1 have 
    "X\<times>{c} = {<x,y> \<in> X\<times>Y. c = y}" by auto;
   with A1 show ?thesis
     using func1_1_L11A ConstantFunction_def
     by simp;
qed;

text{*Constant function is equal to the constant on its domain.*}

lemma func1_5_L2: assumes A1:"x\<in>X"
  shows "ConstantFunction(X,c)`(x) = c"
proof -;
  have "ConstantFunction(X,c) \<in> X\<rightarrow>{c}"
    using func1_5_L1 by simp;
  moreover from A1 have "<x,c> \<in> ConstantFunction(X,c)"
    using ConstantFunction_def by simp;
  ultimately show ?thesis using apply_iff by simp;
qed;

section{*Restricting operations*}

text{*In this section we consider when restriction of the operation to a set
  inherits properties like commutativity and associativity.*}

text{*The commutativity is inherited when restricting a function to a set.*}

lemma func1_6_L1: 
  assumes A1:"f\<in>X\<times>X\<rightarrow>Y" and A2:"A\<subseteq>X"
  and A3:"f {is commutative on} X"
  shows "restrict(f,A\<times>A) {is commutative on} A"
proof -;
  have 
    "\<forall>x y. (x\<in>A \<and> y\<in>A)\<longrightarrow> 
    (restrict(f,A\<times>A)`<x,y> = restrict(f,A\<times>A)`<y,x>)"
  proof (rule double_forall_implies);
    fix x y assume A4:"x\<in>A \<and> y\<in>A"
    with A2 A3 have
      "f`<x,y> = f`<y,x>"
      using IsCommutative_def by auto;
    moreover from A4 have 
      "restrict(f,A\<times>A)`<x,y> = f`<x,y>"
      "restrict(f,A\<times>A)`<y,x> = f`<y,x>"
      using restrict_if by auto;
    ultimately show 
      "restrict(f,A\<times>A)`<x,y> = restrict(f,A\<times>A)`<y,x>"
      by simp;
    qed;
  then show ?thesis
    using IsCommutative_def by simp;
qed;
  
text{*Next we define sets closed with respect to an operation.*}

constdefs 
  IsOpClosed :: "[i,i]\<Rightarrow>o" (infix "{is closed with respect to}" 65) 
  "A {is closed with respect to} f == \<forall>x\<in>A. \<forall>y\<in>A. f`<x,y> \<in> A"

text{*Associative operation restricted to a set that is closed with
  resp. to this operation is associative.*}

lemma func1_6_L2:assumes A1:"f {is associative on} X"
  and A2:"A\<subseteq>X" and A3:"A {is closed with respect to} f"
  and A4:"x\<in>A" "y\<in>A" "z\<in>A"
  and A5:"g = restrict(f,A\<times>A)"
  shows "g`<g`<x,y>,z> = g`<x,g`<y,z> >"
proof -; 
  from A4 A2 have T1:
    "x\<in>X" "y\<in>X" "z\<in>X"
    by auto;
  from A3 A4 A5 have
    "g`<g`<x,y>,z> = f`<f`<x,y>,z>"
    "g`<x,g`<y,z> > = f`<x,f`<y,z> >"
    using IsOpClosed_def restrict_if by auto;
  moreover from A1 T1 have
     "f`<f`<x,y>,z> = f`<x,f`<y,z> >"
    using IsAssociative_def by simp;
  ultimately show ?thesis by simp;
qed;

text{*Associative operation restricted to a set that is closed with
  resp. to this operation is associative on the set.*}

lemma func1_6_L2: assumes A1:"f {is associative on} X"
  and A2:"A\<subseteq>X" and A3:"A {is closed with respect to} f"
  shows "restrict(f,A\<times>A) {is associative on} A"
proof -;
  let ?g = "restrict(f,A\<times>A)"
  from A1 have "f:X\<times>X\<rightarrow>X"
    using IsAssociative_def by simp;
  moreover from A2 have "A\<times>A \<subseteq> X\<times>X" by auto;
  moreover from A3 have "\<forall>p \<in> A\<times>A. ?g`(p) \<in> A"
    using IsOpClosed_def restrict_if by auto;
  ultimately have "?g : A\<times>A\<rightarrow>A"
    using func1_2_L4 by simp;
  moreover from  A1 A2 A3 have
    "\<forall> x \<in> A. \<forall> y \<in> A. \<forall> z \<in> A.
    ?g`<?g`<x,y>,z> = ?g`< x,?g`<y,z> >"
    using func1_6_L2 by simp;
  ultimately show ?thesis 
    using IsAssociative_def by simp;
qed;

text{*The essential condition to show that if a set $A$ is closed 
  with respect to an operation, 
  then it is closed with respect to this operation restricted 
  to any superset of $A$.*}

lemma func1_6_L4: assumes A1:"A {is closed with respect to} f"
  and A2:"A\<subseteq>B"
  and A3:"x\<in>A" "y\<in>A"
  and A4:"g = restrict(f,B\<times>B)"
  shows "g`<x,y> \<in> A"
proof -;
  from A1 A2 A3 A4 show ?thesis using
    IsOpClosed_def restrict by auto;
qed;

text{*If a set $A$ is closed with respect to an operation, 
  then it is closed with respect to this operation restricted 
  to any superset of $A$. *}

lemma func1_6_L5: 
  assumes A1:"A {is closed with respect to} f"
  and A2:"A\<subseteq>B"
  shows "A {is closed with respect to} restrict(f,B\<times>B)"
proof -
  let ?g = "restrict(f,B\<times>B)"
  from A1 A2 have "\<forall>x\<in>A. \<forall>y\<in>A. ?g`<x,y> \<in> A"
    using func1_6_L4 by simp
  then show ?thesis using IsOpClosed_def by simp;
qed;

text{*The essential condition to show that intersection of sets that are
  closed with respect to an operation is closed with respect 
  to the operation.*}

lemma func1_6_L6:
  assumes A1:"A {is closed with respect to} f"
  "B {is closed with respect to} f"
  and A2:"x \<in> A\<inter>B" "y\<in> A\<inter>B"
  shows "f`<x,y> \<in> A\<inter>B"
proof -;
  from A1 A2 show ?thesis using IsOpClosed_def by auto;
qed;

text{*Intersection of sets that are
  closed with respect to an operation is closed with respect to 
  the operation.*}

lemma func1_6_L7:
  assumes A1:"A {is closed with respect to} f"
  "B {is closed with respect to} f"
  shows "A\<inter>B {is closed with respect to} f"
proof -;
  from A1 have
     "\<forall>x\<in>A\<inter>B. \<forall>y\<in>A\<inter>B. f`<x,y> \<in> A\<inter>B"
    using func1_6_L6 by simp;
  then show ?thesis using IsOpClosed_def by simp;
qed;

section{*Composition*}

text{*For any set $X$ we can consider a binary operation 
  on the set of functions 
  $f:X\rightarrow X$ defined by $C(f,g) = f\circ g$. Composition of functions 
  (or relations) is defined in the standard Isabelle distribution as a higher
  order function. In this section we consider the corresponding two-argument 
  ZF-function (binary operation), that is a subset of 
  $((X\rightarrow X)\times (X\rightarrow X))\times (X\rightarrow X)$.*}

constdefs
  Composition :: "i\<Rightarrow>i"
  "Composition(X) == 
  {<p,f> \<in> ((X\<rightarrow>X)\<times>(X\<rightarrow>X))\<times>(X\<rightarrow>X). fst(p) O snd(p) = f}"

text{*Composition operation is a function that maps 
  $(X\rightarrow X)\times (X\rightarrow X)$ into $X\rightarrow X$.*}

lemma func1_7_L1: "Composition(X) : (X\<rightarrow>X)\<times>(X\<rightarrow>X)\<rightarrow>(X\<rightarrow>X)"
proof -;
  show ?thesis using comp_fun Composition_def func1_1_L11A
    by simp;
qed;

text{*The value of the composition operation is the composition of arguments.*}

lemma func1_7_L2: assumes A1: "f:X\<rightarrow>X" "g:X\<rightarrow>X"
  shows "Composition(X)`<f,g> = f O g"
proof -;
  from A1 show ?thesis using func1_7_L1 Composition_def func1_1_L11B
    by simp;
qed;

text{*What is the falue of a composition on an argument?*}

lemma func1_7_L3: assumes A1: "f:X\<rightarrow>X" "g:X\<rightarrow>X"
  and A2: "x\<in>X"
  shows "(Composition(X)`<f,g>)`(x) = f`(g`(x))"
proof -;
  from A1 A2 show ?thesis using
    func1_7_L2 comp_fun_apply by simp;
qed;
  
text{*The essential condition to show that composition is associative.*}

lemma func1_7_L4: assumes A1:"f:X\<rightarrow>X" "g:X\<rightarrow>X" "h:X\<rightarrow>X"
  and A2:"C = Composition(X)"
  shows "C`<C`<f,g>,h> = C`< f,C`<g,h> >"
proof -; 
  from A2 have "C : ((X\<rightarrow>X)\<times>(X\<rightarrow>X))\<rightarrow>(X\<rightarrow>X)"
    using func1_7_L1 by simp;
  with A1 have T1:
    "C`<f,g> : X\<rightarrow>X"
    "C`<g,h> : X\<rightarrow>X"
    "C`<C`<f,g>,h> : X\<rightarrow>X"
    "C`< f,C`<g,h> > : X\<rightarrow>X"
    using apply_funtype by auto;
  moreover have 
    "\<forall> x \<in> X. C`<C`<f,g>,h> ` (x) = C`< f,C`<g,h> >`(x)"
  proof;
    fix x assume A3:"x\<in>X"
    with A1 A2 T1 have 
      "C`<C`<f,g>,h> ` (x) = f`(g`(h`(x)))"
      "C`< f,C`<g,h> >`(x) = f`(g`(h`(x)))"
      using func1_7_L3 apply_funtype by auto;
    then show "C`<C`<f,g>,h> ` (x) = C`< f,C`<g,h> >`(x)"
      by simp;
    qed;
  ultimately show ?thesis using fun_extension_iff by simp;
qed;
  
text{*Composition is an associative operation on $X\rightarrow X$ (the space
  of functions that map $X$ into itself).*}

lemma func1_7_L5: shows "Composition(X) {is associative on} (X\<rightarrow>X)"
proof -;
  let ?C = "Composition(X)"
  have "\<forall>f\<in>X\<rightarrow>X. \<forall>g\<in>X\<rightarrow>X. \<forall>h\<in>X\<rightarrow>X.
    ?C`<?C`<f,g>,h> = ?C`< f,?C`<g,h> >"
    using func1_7_L4 by simp;
  then show ?thesis using func1_7_L1 IsAssociative_def
    by simp;
qed;

section{*Identity function *}

text{*In this section we define the identity function on a set $X$, that 
  is a function that assigns every $x\in X$ to itself.*}

text{*We define the identity function as the set of pairs 
  $\{(x,x): x\in X\}$.*}

constdefs
  IdentityFunction :: "i\<Rightarrow>i"
  "IdentityFunction(X) == {<x,x>. x\<in>X}"

lemma func1_8_L1: shows "IdentityFunction(X) : X\<rightarrow>X"
proof -;
  have "{<x,x>. x\<in>X} = {<x,y> \<in> X\<times>X. x = y}" by auto;
  then show ?thesis using func1_1_L11A IdentityFunction_def
    by simp;
qed;

text{*Identity function assigns $x$ to itself.*}

lemma func1_8_L2: 
  assumes A1:"x\<in>X" shows "IdentityFunction(X)`(x) = x"
proof -;
  have "IdentityFunction(X) : X\<rightarrow>X"
    using func1_8_L1 by simp
  moreover from A1 have "<x,x> \<in> IdentityFunction(X)"
    using IdentityFunction_def by simp;
  ultimately show ?thesis using apply_iff by simp;
qed;

text{*Composing a function with identity does not change the function.*}

lemma func1_8_L3: assumes A1: "f : X\<rightarrow>X"
  shows "Composition(X)`<f,IdentityFunction(X)> = f"
  "Composition(X)`<IdentityFunction(X),f> = f"
proof -;
  have "Composition(X) : (X\<rightarrow>X)\<times>(X\<rightarrow>X)\<rightarrow>(X\<rightarrow>X)"
    using func1_7_L1 by simp;
  with A1 have "Composition(X)`<IdentityFunction(X),f> : X\<rightarrow>X"
    "Composition(X)`<f,IdentityFunction(X)> : X\<rightarrow>X"
    using func1_8_L1 apply_funtype by auto;
  moreover from A1 have "f : X\<rightarrow>X" .
  moreover from A1 have 
    "\<forall>x\<in>X. (Composition(X)`<IdentityFunction(X),f>)`(x) = f`(x)"
    "\<forall>x\<in>X. (Composition(X)`<f,IdentityFunction(X)>)`(x) = f`(x)"
    using func1_8_L1 func1_7_L3 apply_funtype func1_8_L2
    by auto;
  ultimately show "Composition(X)`<IdentityFunction(X),f> = f"
    "Composition(X)`<f,IdentityFunction(X)> = f"
    using fun_extension_iff by auto;
qed;
  
text{*This concludes func1.thy.*}

end