(*
    This file is a part of IsarMathLib - 
    a library of formalized mathematics for Isabelle/Isar.

    Copyright (C) 2005  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{OrderedGroup\_ZF.thy}*}

theory OrderedGroup_ZF = Group_ZF + Order_ZF + Finite1:

text{*This theory file defines and shows the basic properties of (partially
  or linearly) ordered groups *}

section{*Ordered groups*}

text{*This section defines ordered group.*}

text{*An ordered group is a group equipped with a partial order that is
  "translation invariant", that is if $a\leq b$ then $a\cdot g \leq b\cdot g$
  and $g\cdot a \leq g\cdot b$. We define the set of nonnegative elements
  in the obvious way as $\{x\in G: 1 \leq x\}$. We also define the absolute value. 
  as a ZF-function that is the 
  identity on $G^+$ and the group inverse on the rest of the group.
 *}

constdefs 

  IsAnOrdGroup :: "[i,i,i]\<Rightarrow>o"
  "IsAnOrdGroup(G,P,r) == (IsAgroup(G,P) \<and> r\<subseteq>G\<times>G \<and> IsPartOrder(G,r) \<and> 
  (\<forall>g\<in>G. \<forall>a b. 
  <a,b> \<in> r \<longrightarrow> <P`<a,g>,P`<b,g> > \<in> r \<and> <P`<g,a>,P`<g,b> > \<in> r ) )"

  Nonnegative :: "[i,i,i]\<Rightarrow>i"
  "Nonnegative(G,P,r) == {x\<in>G. <TheNeutralElement(G,P),x> \<in> r}"

  AbsoluteValue :: "[i,i,i]\<Rightarrow>i"
  "AbsoluteValue(G,P,r) == IdentityFunction(Nonnegative(G,P,r)) \<union> 
  restrict(GroupInv(G,P),G - Nonnegative(G,P,r))";


text{*We will use a similar notation for ordered groups as for the generic 
  groups.*}

locale group3 =

  fixes G :: "i"
  fixes P :: "i"
  fixes r :: "i"
  assumes ordGroupAssum:"IsAnOrdGroup(G,P,r)"

  fixes unit :: "i" ("\<one>")
  defines unit_def [simp]:"\<one> == TheNeutralElement(G,P)"

  fixes groper  :: "[i,i]\<Rightarrow>i" (infixl "\<cdot>" 70)
  defines groper_def [simp]: "a \<cdot> b == P`<a,b>"

  fixes inv :: "i\<Rightarrow>i" ("_\<inverse> " [90] 91)
  defines inv_def [simp]: "x\<inverse> == GroupInv(G,P)`(x)"

  fixes lesseq :: "[i,i]\<Rightarrow>o" (infix "\<lsq>" 68)
  defines lesseq_def [simp]: "a \<lsq> b == <a,b> \<in> r"

  fixes nonnegative :: "i" ("G\<^sup>+")
  defines nonnegative_def [simp]: "G\<^sup>+ == Nonnegative(G,P,r)";

text{*In group3 context we can use the theorems proven in group0 context.*}

lemma (in group3) OrderedGroup_ZF_1_L1: "group0(G,P)"
proof -;
  from ordGroupAssum show ?thesis using IsAnOrdGroup_def group0_def
    by simp;
qed;

text{*The next lemma is just to see the definition of nonnegative set
  in our notation.*}

lemma (in group3) OrderedGroup_ZF_1_L2: "g\<in>G\<^sup>+ \<longleftrightarrow> \<one>\<lsq>g"
proof -;
  from ordGroupAssum show ?thesis 
    using IsAnOrdGroup_def Nonnegative_def by auto;
qed; 

text{*The group order is reflexive.*}

lemma (in group3) OrderedGroup_ZF_1_L3: assumes A1:"g\<in>G"
  shows "g\<lsq>g"
proof -;
  from A1 ordGroupAssum show ?thesis using
    IsAnOrdGroup_def IsPartOrder_def refl_def by simp;
qed;

text{*$1$ is nonnegative.*}

lemma (in group3) OrderedGroup_ZF_1_L3A: "\<one>\<in>G\<^sup>+"
proof -;
 show ?thesis using OrderedGroup_ZF_1_L2  OrderedGroup_ZF_1_L3
   OrderedGroup_ZF_1_L1 group0.group0_2_L2 by simp;
qed;
  
(*text{*Absolute value maps $G$ into $G$.*}

lemma (in group3) OrderedGroup_ZF_1_L3: 
  "AbsoluteValue(G,P,r) : G\<rightarrow>G"
proof -;
  let ?f = "IdentityFunction(G\<^sup>+)"
  let ?g = "restrict(GroupInv(G,P),G-G\<^sup>+)"
  have "?f : G\<^sup>+\<rightarrow>G\<^sup>+" using func_ZF_6_L1 by simp;
  moreover have T1: "G\<^sup>+\<subseteq>G" using Nonnegative_def by auto;
  ultimately have "?f : G\<^sup>+\<rightarrow>G" by (rule fun_weaken_type);
  moreover have "?g : G-G\<^sup>+\<rightarrow>G"
  proof -;
    from ordGroupAssum have "GroupInv(G,P) : G\<rightarrow>G" 
      using IsAnOrdGroup_def group0_2_T2 by simp;
    moreover have "G-G\<^sup>+ \<subseteq> G" by auto;
    ultimately show ?thesis using restrict_type2 by simp;
  qed;
  moreover have "G\<^sup>+\<inter>(G-G\<^sup>+) = 0" by blast;
  ultimately have "?f \<union> ?g : G\<^sup>+\<union>(G-G\<^sup>+)\<rightarrow>G\<union>G" 
    by (rule fun_disjoint_Un);
  moreover from T1 have "G\<^sup>+\<union>(G-G\<^sup>+) = G" by auto;
  ultimately show "AbsoluteValue(G,P,r) : G\<rightarrow>G" 
    using AbsoluteValue_def by simp;
qed;*)

text{*In this context $a \leq b$ implies that both $a$ and $b$ belong 
  to $G$.*}

lemma (in group3) OrderedGroup_ZF_1_L4: 
  assumes A1: "a\<lsq>b" shows "a\<in>G" "b\<in>G"
proof -;
  from ordGroupAssum A1 show "a\<in>G" "b\<in>G" 
    using IsAnOrdGroup_def by auto;
qed;

text{*It is good to have transistivity handy.*}

lemma (in group3) OrderedGroup_ZF_1_L4A:
  assumes A1: "a\<lsq>b" "b\<lsq>c" shows "a\<lsq>c"
proof -
  from ordGroupAssum have "trans(r)"
    using IsAnOrdGroup_def IsPartOrder_def
    by simp
  moreover from A1 have "<a,b> \<in> r \<and> <b,c> \<in> r" by simp;
  ultimately have "<a,c> \<in> r" by (rule Fol1_L3);
  thus ?thesis by simp;
qed;

text{*If an element is smaller that the unit, then its inverse is greater.*}

lemma (in group3) OrderedGroup_ZF_1_L5: 
  assumes A1: "a\<lsq>\<one>" shows "\<one>\<lsq>a\<inverse>"
proof -;
  from A1 have T1: "a\<in>G" "\<one>\<in>G" "a\<inverse>\<in>G" 
    using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_1_L1 group0.group0_2_L5
    by auto;
  with ordGroupAssum A1 have "a\<cdot>a\<inverse>\<lsq>\<one>\<cdot>a\<inverse>" using IsAnOrdGroup_def 
    by simp;
  with T1 show "\<one>\<lsq>a\<inverse>" using OrderedGroup_ZF_1_L1 group0.group0_2_L6
    group0.group0_2_L2 by simp;
qed;
    
text{*If the order is total, the elements that do not belong
  to the positive set are negative. We also show here that the group inverse
  of an element that does not belong to the nonnegative set does belong to the
  nonnegative set.*}

lemma (in group3) OrderedGroup_ZF_1_L6: 
  assumes A1: "r {is total on} G" and A2: "a\<in>G-G\<^sup>+"
  shows "a\<lsq>\<one>" "restrict(GroupInv(G,P),G-G\<^sup>+)`(a) \<in> G\<^sup>+" 
proof -; 
  from A2 have T1: "a\<in>G" "a\<notin>G\<^sup>+" "\<one>\<in>G" 
    using OrderedGroup_ZF_1_L1 group0.group0_2_L2 by auto;
  with A1 show "a\<lsq>\<one>" using OrderedGroup_ZF_1_L2 IsTotal_def
    by auto;
  with A2 show "restrict(GroupInv(G,P),G-G\<^sup>+)`(a) \<in> G\<^sup>+"
    using OrderedGroup_ZF_1_L5 restrict OrderedGroup_ZF_1_L2
    by simp;
qed;

text{*For linearly ordered groups (where the order is total), the absolute
  value maps the group into the positive set.*}

lemma (in group3) OrderedGroup_ZF_1_L7:
  assumes A1: "r {is total on} G"
  shows "AbsoluteValue(G,P,r) : G\<rightarrow>G\<^sup>+"
proof-
  let ?f = "IdentityFunction(G\<^sup>+)"
  let ?g = "restrict(GroupInv(G,P),G -G\<^sup>+)";
  from ordGroupAssum have "GroupInv(G,P) : G\<rightarrow>G" 
    using IsAnOrdGroup_def group0_2_T2 by simp
  moreover have "G-G\<^sup>+ \<subseteq> G" by auto
  moreover from A1  have "\<forall>a\<in>G-G\<^sup>+. ?g`(a) \<in> G\<^sup>+"
    using OrderedGroup_ZF_1_L6 by simp;
  ultimately have "?g : G-G\<^sup>+\<rightarrow>G\<^sup>+" using func1_2_L4 by simp;
  moreover have "?f : G\<^sup>+\<rightarrow>G\<^sup>+" using func_ZF_6_L1 by simp;
  moreover have "(G-G\<^sup>+)\<inter>G\<^sup>+ = 0" by blast;
  ultimately have "?g \<union> ?f : (G-G\<^sup>+)\<union>G\<^sup>+\<rightarrow>G\<^sup>+\<union>G\<^sup>+" 
    by (rule fun_disjoint_Un);
  moreover have "(G-G\<^sup>+)\<union>G\<^sup>+ = G" "?g \<union> ?f = ?f \<union> ?g" 
    using Nonnegative_def by auto;
  ultimately show ?thesis using AbsoluteValue_def by simp;
qed;

text{*A bounded set can be translated to put it in  $G^+$ and then it is 
 still bounded above. *}

lemma (in group3) OrderedGroup_ZF_1_L8: 
  assumes A1: "\<forall>g\<in>A. L\<lsq>g \<and> g\<lsq>M"
  and A2: "S = RightTranslation(G,P,L\<inverse>)" 
  and A3: "a \<in> S``(A)"
  shows "a\<lsq>M\<cdot>L\<inverse>" "\<one>\<lsq>a"
proof -
  from A3 have "A\<noteq>0" using func1_1_L13A by fast;
  then obtain g where "g\<in>A" by auto
  with A1 have T1: "L\<in>G" "M\<in>G" "L\<inverse>\<in>G" 
    using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_1_L1 
    group0.group0_2_L5 by auto;
  with A2 have "S : G\<rightarrow>G" using OrderedGroup_ZF_1_L1 group0.group0_5_L1
    by simp;
  moreover from A1 have T2: "A\<subseteq>G" using OrderedGroup_ZF_1_L4 by auto;
  ultimately have "S``(A) = {S`(b). b\<in>A}" using func1_1_L16
    by simp;
  with A3 obtain b where T3: "b\<in>A" "a = S`(b)" by auto;
  with A1 ordGroupAssum T1 have "b\<cdot>L\<inverse>\<lsq>M\<cdot>L\<inverse>" "L\<cdot>L\<inverse>\<lsq>b\<cdot>L\<inverse>"
    using IsAnOrdGroup_def by auto;
  with T3 A2 T1 T2 show "a\<lsq>M\<cdot>L\<inverse>" "\<one>\<lsq>a"
    using OrderedGroup_ZF_1_L1 group0.group0_5_L2 group0.group0_2_L6 
    by auto;
qed;

text{*Every bounded set is an image of a subset of an interval that starts at 
  $1$.*}

lemma (in group3) OrderedGroup_ZF_1_L9:
  assumes A1: "IsBounded(A,r)" 
  shows "\<exists>B.\<exists>g\<in>G\<^sup>+.\<exists>T\<in>G\<rightarrow>G. A = T``(B) \<and> B \<subseteq> Interval(r,\<one>,g)";
proof (cases "A=0");
  assume A2: "A=0" 
  let ?B = "0"
  let ?g = "\<one>"
  let ?T = "ConstantFunction(G,\<one>)"
  have "?g\<in>G\<^sup>+" using OrderedGroup_ZF_1_L3A by simp;
  moreover have "?T : G\<rightarrow>G"
    using func1_3_L1 OrderedGroup_ZF_1_L1 group0.group0_2_L2 by simp;
  moreover from A2 have "A = T``(?B)" by simp;
  moreover have "?B \<subseteq> Interval(r,\<one>,?g)" by simp;
  ultimately show 
    "\<exists>B.\<exists>g\<in>G\<^sup>+.\<exists>T\<in>G\<rightarrow>G. A = T``(B) \<and> B \<subseteq> Interval(r,\<one>,g)"
    by auto;
next;
  assume A2: "A\<noteq>0"
  from A1 A2 obtain L U where D1: "\<forall>x\<in>A. L\<lsq>x \<and> x\<lsq>U "
    using IsBounded_def IsBoundedBelow_def IsBoundedAbove_def
    by auto;
  with A2 have T1: "A\<subseteq>G" using OrderedGroup_ZF_1_L4 by auto;
  from A2 obtain a where "a\<in>A" by auto;
  with D1 have T2: "L\<lsq>a" "a\<lsq>U" by auto;
  then have T3: "L\<in>G" "L\<inverse>\<in> G" "U\<in>G" 
    using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_1_L1 
      group0.group0_2_L5 by auto;
  let ?T = "RightTranslation(G,P,L)"
  let ?B = "RightTranslation(G,P,L\<inverse>)``(A)"
  let ?g = "U\<cdot>L\<inverse>"
  have "?g\<in>G\<^sup>+"
  proof -;
    from T2 have "L\<lsq>U" using OrderedGroup_ZF_1_L4A by fast;
    with ordGroupAssum T3 have "L\<cdot>L\<inverse>\<lsq>?g" 
      using IsAnOrdGroup_def by simp;
    with T3 show ?thesis using OrderedGroup_ZF_1_L1 group0.group0_2_L6
      OrderedGroup_ZF_1_L2 by simp;
  qed;
  moreover from T3 have "?T : G\<rightarrow>G"
    using OrderedGroup_ZF_1_L1 group0.group0_5_L1
    by simp;
  moreover have "A = ?T``(?B)"
  proof -;
    from T3 T1 have "?T``(?B) = {a\<cdot>L\<inverse>\<cdot>L. a\<in>A}"
      using OrderedGroup_ZF_1_L1 group0.group0_5_L6
      by simp;
    moreover from T3 T1 have "\<forall>a\<in>A. a\<cdot>L\<inverse>\<cdot>L = a\<cdot>(L\<inverse>\<cdot>L)"
      using OrderedGroup_ZF_1_L1 group0.group0_2_L3 by auto;
    ultimately have "?T``(?B) = {a\<cdot>(L\<inverse>\<cdot>L). a\<in>A}" by simp;
    with T3 have "?T``(?B) = {a\<cdot>\<one>. a\<in>A}"
      using OrderedGroup_ZF_1_L1 group0.group0_2_L6 by simp;
    moreover from T1 have "\<forall>a\<in>A. a\<cdot>\<one>=a"
      using OrderedGroup_ZF_1_L1 group0.group0_2_L2 by auto;
    ultimately show ?thesis by simp
  qed
  moreover have "?B \<subseteq> Interval(r,\<one>,?g)"
  proof;
    fix y assume A3: "y \<in> ?B"
    def D2: S == "RightTranslation(G,P,L\<inverse>)"
    from D1 have T4: "\<forall>x\<in>A. L\<lsq>x \<and> x\<lsq>U" by simp;
    moreover from D2 have T5: "S = RightTranslation(G,P,L\<inverse>)" 
      by simp; 
    moreover from A3 D2 have T6: "y \<in> S``(A)" by simp;
    ultimately have "y\<lsq>U\<cdot>L\<inverse>" using OrderedGroup_ZF_1_L8
      by blast;
    moreover from T4 T5 T6 have "\<one>\<lsq>y" by (rule OrderedGroup_ZF_1_L8);
    ultimately show "y \<in> Interval(r,\<one>,?g)" using Interval_def by auto;
  qed;
  ultimately show 
    "\<exists>B.\<exists>g\<in>G\<^sup>+.\<exists>T\<in>G\<rightarrow>G. A = T``(B) \<and> B \<subseteq> Interval(r,\<one>,g)"
    by auto;  
qed;
      
text{*If every interval starting at $1$ is finite, then every bounded set is 
  finite. I found it interesting that this does not require the group to be 
  linearly ordered (the order to be total).*}

theorem (in group3) OrderedGroup_ZF_1_T1:
  assumes A1: "\<forall>g\<in>G\<^sup>+. Interval(r,\<one>,g) \<in> Fin(G)"
  and A2: "IsBounded(A,r)" 
  shows "A \<in> Fin(G)"
proof -
  from A2 have
    "\<exists>B.\<exists>g\<in>G\<^sup>+.\<exists>T\<in>G\<rightarrow>G. A = T``(B) \<and> B \<subseteq> Interval(r,\<one>,g)"
    using OrderedGroup_ZF_1_L9 by simp;
  then obtain B g T where D1: "g\<in>G\<^sup>+" "B \<subseteq> Interval(r,\<one>,g)" 
    and D2: "T : G\<rightarrow>G" "A = T``(B)" 
    using OrderedGroup_ZF_1_L9 by auto;
  from D1 A1 have "B\<in>Fin(G)" using Fin_subset_lemma by blast;
  with D2 show ?thesis using Finite1_L6A by simp;
qed;
  
end
 