(*
    This file is a part of IsarMathLib - 
    a library of formalized mathematics for Isabelle/Isar.

    Copyright (C) 2005  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{OrderedGroup\_ZF.thy}*}

theory OrderedGroup_ZF imports Group_ZF Order_ZF Finite_ZF_1 

begin

text{*This theory file defines and shows the basic properties of (partially
  or linearly) ordered groups. We define the set of nonnegative elemnts 
  and the absolute value function. 
  We show that in linearly ordered groups finite sets are bounded 
  and provide a sufficient condition for bounded sets to be finite. This 
  allows to show in Int\_ZF.thy that subsets of integers are bounded iff 
  they are finite. *}

section{*Ordered groups*}

text{*This section defines ordered group.*}

text{*An ordered group is a group equipped with a partial order that is
  "translation invariant", that is if $a\leq b$ then $a\cdot g \leq b\cdot g$
  and $g\cdot a \leq g\cdot b$. We define the set of nonnegative elements
  in the obvious way as $G^+ =\{x\in G: 1 \leq x\}$. We also define 
  the absolute value as a ZF-function that is the 
  identity on $G^+$ and the group inverse on the rest of the group.
  We also define the maximum absolute value of a set, that is the maximum
  of the set $\{|x|.x\in A\}$. 
 *}

constdefs 

  "IsAnOrdGroup(G,P,r) \<equiv> 
  (IsAgroup(G,P) \<and> r\<subseteq>G\<times>G \<and> IsPartOrder(G,r) \<and> (\<forall>g\<in>G. \<forall>a b. 
  <a,b> \<in> r \<longrightarrow> <P`<a,g>,P`<b,g> > \<in> r \<and> <P`<g,a>,P`<g,b> > \<in> r ) )"

  "Nonnegative(G,P,r) \<equiv> {x\<in>G. <TheNeutralElement(G,P),x> \<in> r}"

  "AbsoluteValue(G,P,r) \<equiv> id(Nonnegative(G,P,r)) \<union> 
  restrict(GroupInv(G,P),G - Nonnegative(G,P,r))"

  "MaxAbs(G,P,r,A) \<equiv> Maximum(r,AbsoluteValue(G,P,r)``(A))"


text{*We will use a similar notation for ordered groups as for the generic 
  groups.*}

locale group3 =

  fixes G and P and r

  assumes ordGroupAssum: "IsAnOrdGroup(G,P,r)"

  fixes unit ("\<one>")
  defines unit_def [simp]: "\<one> \<equiv> TheNeutralElement(G,P)"

  fixes groper (infixl "\<cdot>" 70)
  defines groper_def [simp]: "a \<cdot> b \<equiv> P`<a,b>"

  fixes inv ("_\<inverse> " [90] 91)
  defines inv_def [simp]: "x\<inverse> \<equiv> GroupInv(G,P)`(x)"

  fixes lesseq (infix "\<lsq>" 68)
  defines lesseq_def [simp]: "a \<lsq> b \<equiv> <a,b> \<in> r"

  fixes nonnegative ("G\<^sup>+")
  defines nonnegative_def [simp]: "G\<^sup>+ \<equiv> Nonnegative(G,P,r)"

  fixes abs ("| _ |")
  defines abs_def [simp]: "|a| \<equiv> AbsoluteValue(G,P,r)`(a)";

text{*In group3 context we can use the theorems proven in group0 context.*}

lemma (in group3) OrderedGroup_ZF_1_L1: shows "group0(G,P)"
  using ordGroupAssum IsAnOrdGroup_def group0_def by simp;

text{*The next lemma is just to see the definition of the nonnegative set
  in our notation.*}

lemma (in group3) OrderedGroup_ZF_1_L2: shows "g\<in>G\<^sup>+ \<longleftrightarrow> \<one>\<lsq>g"
  using ordGroupAssum IsAnOrdGroup_def Nonnegative_def by auto;

text{*For total order if $g$ is not nonnegative, then it has to be negative.*}

lemma (in group3) OrderedGroup_ZF_1_L2A: 
  assumes A1: "r {is total on} G" and A2: "a\<in>G-G\<^sup>+"
  shows "a\<lsq>\<one>"
proof -;
  from A2 have "a\<in>G" "\<not>(\<one>\<lsq>a)" using OrderedGroup_ZF_1_L2 by auto;
  with A1 show ?thesis 
    using IsTotal_def OrderedGroup_ZF_1_L1 group0.group0_2_L2 by auto;
qed;

text{*The group order is reflexive.*}

lemma (in group3) OrderedGroup_ZF_1_L3: assumes "g\<in>G"
  shows "g\<lsq>g"
  using ordGroupAssum prems IsAnOrdGroup_def IsPartOrder_def refl_def
  by simp;

text{*$1$ is nonnegative.*}

lemma (in group3) OrderedGroup_ZF_1_L3A: shows "\<one>\<in>G\<^sup>+"
  using OrderedGroup_ZF_1_L2 OrderedGroup_ZF_1_L3
    OrderedGroup_ZF_1_L1 group0.group0_2_L2 by simp;
  
text{*In this context $a \leq b$ implies that both $a$ and $b$ belong 
  to $G$.*}

lemma (in group3) OrderedGroup_ZF_1_L4: 
  assumes "a\<lsq>b" shows "a\<in>G" "b\<in>G"
  using ordGroupAssum prems IsAnOrdGroup_def by auto;

text{*It is good to have transitivity handy.*}

lemma (in group3) OrderedGroup_ZF_1_L4A:
  assumes A1: "a\<lsq>b" "b\<lsq>c" shows "a\<lsq>c"
proof -
  from ordGroupAssum have "trans(r)"
    using IsAnOrdGroup_def IsPartOrder_def
    by simp
  moreover from A1 have "<a,b> \<in> r \<and> <b,c> \<in> r" by simp;
  ultimately have "<a,c> \<in> r" by (rule Fol1_L3);
  thus ?thesis by simp;
qed;

text{*For linearly ordered groups elements in the nonnegative set are
  greater than those in the complement.*}

lemma (in group3) OrderedGroup_ZF_1_L4B:
  assumes "r {is total on} G" 
  and "a\<in>G\<^sup>+" and "b\<in>G-G\<^sup>+"
  shows "b\<lsq>a"
proof -;
  from prems have "b\<lsq>\<one>" "\<one>\<lsq>a"
    using OrderedGroup_ZF_1_L2 OrderedGroup_ZF_1_L2A by auto;
  thus ?thesis by (rule OrderedGroup_ZF_1_L4A)
qed;

text{*If $a\leq 1$ and $a\neq 1$, then $a \in G\setminus G^{+}$. *}

lemma (in group3) OrderedGroup_ZF_1_L4C:
  assumes A1: "a\<lsq>\<one>" and A2: "a\<noteq>\<one>"
  shows "a\<in>G-G\<^sup>+"
proof (rule ccontr);
  assume "a \<notin> G - G\<^sup>+" 
  with ordGroupAssum A1 A2 show False 
    using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_1_L2
     OrderedGroup_ZF_1_L4 IsAnOrdGroup_def IsPartOrder_def antisym_def
    by auto;
qed;

text{*An element smaller than an element in $G\setminus G^+$ is in 
  $G\setminus G^+$.*}

lemma (in group3) OrderedGroup_ZF_1_L4D:
  assumes A1: "a\<in>G-G\<^sup>+" and A2: "b\<lsq>a"
  shows "b\<in>G-G\<^sup>+"
proof (rule ccontr);
  assume "b \<notin> G - G\<^sup>+"
  with A2 have "\<one>\<lsq>b" "b\<lsq>a"
     using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_1_L2 by auto;
  then have "\<one>\<lsq>a" by (rule OrderedGroup_ZF_1_L4A);
  with A1 show False using OrderedGroup_ZF_1_L2 by simp;
qed;

text{*The nonnegative set is contained in the group.*}

lemma (in group3) OrderedGroup_ZF_1_L2B: shows "G\<^sup>+ \<subseteq> G"
  using OrderedGroup_ZF_1_L2 OrderedGroup_ZF_1_L4 by auto;

text{*Taking the inverse on both sides inverses the inequality.*}

lemma (in group3) OrderedGroup_ZF_1_L5:
  assumes A1: "a\<lsq>b" shows "b\<inverse>\<lsq>a\<inverse>"
proof -;
  from A1 have T1: "a\<in>G" "b\<in>G" "a\<inverse>\<in>G" "b\<inverse>\<in>G" 
    using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_1_L1 
      group0.group0_2_L5 by auto;
  with A1 ordGroupAssum have "a\<cdot>a\<inverse>\<lsq>b\<cdot>a\<inverse>" using IsAnOrdGroup_def
    by simp;
  with T1 ordGroupAssum have "b\<inverse>\<cdot>\<one>\<lsq>b\<inverse>\<cdot>(b\<cdot>a\<inverse>)"
    using OrderedGroup_ZF_1_L1 group0.group0_2_L6 IsAnOrdGroup_def
    by simp;
  with T1 show ?thesis using
    OrderedGroup_ZF_1_L1 group0.group0_2_L2 group0.group0_2_L3
    group0.group0_2_L6 by simp;
qed;

text{*If an element is smaller that the unit, then its inverse is greater.*}

lemma (in group3) OrderedGroup_ZF_1_L5A: 
  assumes A1: "a\<lsq>\<one>" shows "\<one>\<lsq>a\<inverse>"
proof -
  from A1 have "\<one>\<inverse>\<lsq>a\<inverse>" using OrderedGroup_ZF_1_L5
    by simp;
  then show ?thesis using OrderedGroup_ZF_1_L1 group0.group0_2_L7A 
    by simp;
qed;
    
text{*If the order is total, the elements that do not belong
  to the positive set are negative. We also show here that the group inverse
  of an element that does not belong to the nonnegative set does belong to the
  nonnegative set.*}

lemma (in group3) OrderedGroup_ZF_1_L6: 
  assumes A1: "r {is total on} G" and A2: "a\<in>G-G\<^sup>+"
  shows "a\<lsq>\<one>" "a\<inverse> \<in> G\<^sup>+" "restrict(GroupInv(G,P),G-G\<^sup>+)`(a) \<in> G\<^sup>+" 
proof -; 
  from A2 have T1: "a\<in>G" "a\<notin>G\<^sup>+" "\<one>\<in>G" 
    using OrderedGroup_ZF_1_L1 group0.group0_2_L2 by auto;
  with A1 show "a\<lsq>\<one>" using OrderedGroup_ZF_1_L2 IsTotal_def
    by auto;
  then show "a\<inverse> \<in> G\<^sup>+" using OrderedGroup_ZF_1_L5A OrderedGroup_ZF_1_L2
    by simp;
  with A2 show "restrict(GroupInv(G,P),G-G\<^sup>+)`(a) \<in> G\<^sup>+"
    using restrict by simp;
qed;

section{*Bounded sets*}

text{*A bounded set can be translated to put it in  $G^+$ and then it is 
 still bounded above. *}

lemma (in group3) OrderedGroup_ZF_2_L1: 
  assumes A1: "\<forall>g\<in>A. L\<lsq>g \<and> g\<lsq>M"
  and A2: "S = RightTranslation(G,P,L\<inverse>)" 
  and A3: "a \<in> S``(A)"
  shows "a\<lsq>M\<cdot>L\<inverse>" "\<one>\<lsq>a"
proof -
  from A3 have "A\<noteq>0" using func1_1_L13A by fast;
  then obtain g where "g\<in>A" by auto
  with A1 have T1: "L\<in>G" "M\<in>G" "L\<inverse>\<in>G" 
    using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_1_L1 
    group0.group0_2_L5 by auto;
  with A2 have "S : G\<rightarrow>G" using OrderedGroup_ZF_1_L1 group0.group0_5_L1
    by simp;
  moreover from A1 have T2: "A\<subseteq>G" using OrderedGroup_ZF_1_L4 by auto;
  ultimately have "S``(A) = {S`(b). b\<in>A}" using func1_1_L16
    by simp;
  with A3 obtain b where T3: "b\<in>A" "a = S`(b)" by auto;
  with A1 ordGroupAssum T1 have "b\<cdot>L\<inverse>\<lsq>M\<cdot>L\<inverse>" "L\<cdot>L\<inverse>\<lsq>b\<cdot>L\<inverse>"
    using IsAnOrdGroup_def by auto;
  with T3 A2 T1 T2 show "a\<lsq>M\<cdot>L\<inverse>" "\<one>\<lsq>a"
    using OrderedGroup_ZF_1_L1 group0.group0_5_L2 group0.group0_2_L6 
    by auto;
qed;

text{*Every bounded set is an image of a subset of an interval that starts at 
  $1$.*}

lemma (in group3) OrderedGroup_ZF_2_L2:
  assumes A1: "IsBounded(A,r)" 
  shows "\<exists>B.\<exists>g\<in>G\<^sup>+.\<exists>T\<in>G\<rightarrow>G. A = T``(B) \<and> B \<subseteq> Interval(r,\<one>,g)";
proof (cases "A=0");
  assume A2: "A=0" 
  let ?B = "0"
  let ?g = "\<one>"
  let ?T = "ConstantFunction(G,\<one>)"
  have "?g\<in>G\<^sup>+" using OrderedGroup_ZF_1_L3A by simp;
  moreover have "?T : G\<rightarrow>G"
    using func1_3_L1 OrderedGroup_ZF_1_L1 group0.group0_2_L2 by simp;
  moreover from A2 have "A = T``(?B)" by simp;
  moreover have "?B \<subseteq> Interval(r,\<one>,?g)" by simp;
  ultimately show 
    "\<exists>B.\<exists>g\<in>G\<^sup>+.\<exists>T\<in>G\<rightarrow>G. A = T``(B) \<and> B \<subseteq> Interval(r,\<one>,g)"
    by auto;
next assume A3: "A\<noteq>0"
  with A1 obtain L U where D1: "\<forall>x\<in>A. L\<lsq>x \<and> x\<lsq>U "
    using IsBounded_def IsBoundedBelow_def IsBoundedAbove_def
    by auto;
  with A3 have T1: "A\<subseteq>G" using OrderedGroup_ZF_1_L4 by auto;
  from A3 obtain a where "a\<in>A" by auto;
  with D1 have T2: "L\<lsq>a" "a\<lsq>U" by auto;
  then have T3: "L\<in>G" "L\<inverse>\<in> G" "U\<in>G" 
    using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_1_L1 
      group0.group0_2_L5 by auto;
  let ?T = "RightTranslation(G,P,L)"
  let ?B = "RightTranslation(G,P,L\<inverse>)``(A)"
  let ?g = "U\<cdot>L\<inverse>"
  have "?g\<in>G\<^sup>+"
  proof -
    from T2 have "L\<lsq>U" using OrderedGroup_ZF_1_L4A by fast;
    with ordGroupAssum T3 have "L\<cdot>L\<inverse>\<lsq>?g" 
      using IsAnOrdGroup_def by simp;
    with T3 show ?thesis using OrderedGroup_ZF_1_L1 group0.group0_2_L6
      OrderedGroup_ZF_1_L2 by simp;
  qed;
  moreover from T3 have "?T : G\<rightarrow>G"
    using OrderedGroup_ZF_1_L1 group0.group0_5_L1
    by simp;
  moreover have "A = ?T``(?B)"
  proof -;
    from T3 T1 have "?T``(?B) = {a\<cdot>L\<inverse>\<cdot>L. a\<in>A}"
      using OrderedGroup_ZF_1_L1 group0.group0_5_L6
      by simp;
    moreover from T3 T1 have "\<forall>a\<in>A. a\<cdot>L\<inverse>\<cdot>L = a\<cdot>(L\<inverse>\<cdot>L)"
      using OrderedGroup_ZF_1_L1 group0.group0_2_L3 by auto;
    ultimately have "?T``(?B) = {a\<cdot>(L\<inverse>\<cdot>L). a\<in>A}" by simp;
    with T3 have "?T``(?B) = {a\<cdot>\<one>. a\<in>A}"
      using OrderedGroup_ZF_1_L1 group0.group0_2_L6 by simp;
    moreover from T1 have "\<forall>a\<in>A. a\<cdot>\<one>=a"
      using OrderedGroup_ZF_1_L1 group0.group0_2_L2 by auto;
    ultimately show ?thesis by simp
  qed
  moreover have "?B \<subseteq> Interval(r,\<one>,?g)"
  proof
    fix y assume A4: "y \<in> ?B"
    def D2: S \<equiv> "RightTranslation(G,P,L\<inverse>)"
    from D1 have T4: "\<forall>x\<in>A. L\<lsq>x \<and> x\<lsq>U" by simp;
    moreover from D2 have T5: "S = RightTranslation(G,P,L\<inverse>)" 
      by simp; 
    moreover from A4 D2 have T6: "y \<in> S``(A)" by simp;
    ultimately have "y\<lsq>U\<cdot>L\<inverse>" using OrderedGroup_ZF_2_L1
      by blast;
    moreover from T4 T5 T6 have "\<one>\<lsq>y" by (rule OrderedGroup_ZF_2_L1);
    ultimately show "y \<in> Interval(r,\<one>,?g)" using Interval_def by auto;
  qed;
  ultimately show 
    "\<exists>B.\<exists>g\<in>G\<^sup>+.\<exists>T\<in>G\<rightarrow>G. A = T``(B) \<and> B \<subseteq> Interval(r,\<one>,g)"
    by auto;  
qed;
      
text{*If every interval starting at $1$ is finite, then every bounded set is 
  finite. I find it interesting that this does not require the group to be 
  linearly ordered (the order to be total).*}

theorem (in group3) OrderedGroup_ZF_2_T1:
  assumes A1: "\<forall>g\<in>G\<^sup>+. Interval(r,\<one>,g) \<in> Fin(G)"
  and A2: "IsBounded(A,r)" 
  shows "A \<in> Fin(G)"
proof -
  from A2 have
    "\<exists>B.\<exists>g\<in>G\<^sup>+.\<exists>T\<in>G\<rightarrow>G. A = T``(B) \<and> B \<subseteq> Interval(r,\<one>,g)"
    using OrderedGroup_ZF_2_L2 by simp;
  then obtain B g T where D1: "g\<in>G\<^sup>+" "B \<subseteq> Interval(r,\<one>,g)" 
    and D2: "T : G\<rightarrow>G" "A = T``(B)" by auto;
  from D1 A1 have "B\<in>Fin(G)" using Fin_subset_lemma by blast;
  with D2 show ?thesis using Finite1_L6A by simp;
qed;

text{*In linearly ordered groups finite sets are bounded.*}

theorem (in group3) OrderedGroup_ZF_1_T2:
  assumes "r {is total on} G" and "B\<in>Fin(G)" 
  shows "IsBounded(B,r)"
  using ordGroupAssum prems IsAnOrdGroup_def IsPartOrder_def Finite_ZF_1_T1
  by simp;

section{*Absolute value and the triangle inequality*};

text{*The goal of this section is to prove the triangle inequality for 
  ordered groups.*};

text{*Absolute value maps $G$ into $G$.*}

lemma (in group3) OrderedGroup_ZF_3_L1: 
  "AbsoluteValue(G,P,r) : G\<rightarrow>G"
proof -;
  let ?f = "id(G\<^sup>+)"
  let ?g = "restrict(GroupInv(G,P),G-G\<^sup>+)"
  have "?f : G\<^sup>+\<rightarrow>G\<^sup>+" using id_type by simp;
  then have "?f : G\<^sup>+\<rightarrow>G" using OrderedGroup_ZF_1_L2B
    by (rule fun_weaken_type);
  moreover have "?g : G-G\<^sup>+\<rightarrow>G"
  proof -;
    from ordGroupAssum have "GroupInv(G,P) : G\<rightarrow>G" 
      using IsAnOrdGroup_def group0_2_T2 by simp;
    moreover have "G-G\<^sup>+ \<subseteq> G" by auto;
    ultimately show ?thesis using restrict_type2 by simp;
  qed;
  moreover have "G\<^sup>+\<inter>(G-G\<^sup>+) = 0" by blast;
  ultimately have "?f \<union> ?g : G\<^sup>+\<union>(G-G\<^sup>+)\<rightarrow>G\<union>G" 
    by (rule fun_disjoint_Un);
  moreover have "G\<^sup>+\<union>(G-G\<^sup>+) = G" using OrderedGroup_ZF_1_L2B
    by auto;
  ultimately show "AbsoluteValue(G,P,r) : G\<rightarrow>G" 
    using AbsoluteValue_def by simp;
qed;

text{*If $a\in G^+$, then $|a| = a$.*}

lemma (in group3) OrderedGroup_ZF_3_L2:
  assumes A1: "a\<in>G\<^sup>+" shows "|a| = a"
proof -
  from ordGroupAssum have "GroupInv(G,P) : G\<rightarrow>G"
    using IsAnOrdGroup_def group0_2_T2 by simp;
  with A1 show ?thesis using
    func1_1_L1 OrderedGroup_ZF_1_L2B fun_disjoint_apply1
    AbsoluteValue_def id_conv by simp;
qed;

text{*If $a\in G\setminus G^+$, then $|a| = a^{-1}$.*}

lemma (in group3) OrderedGroup_ZF_3_L3:
   assumes A1: "a \<in> G-G\<^sup>+" shows "|a| = a\<inverse>"
proof -
  have "domain(id(G\<^sup>+)) = G\<^sup>+"
    using id_type func1_1_L1 by auto;
  with A1 show ?thesis using fun_disjoint_apply2 AbsoluteValue_def
    restrict by simp;
qed;

text{*In linearly ordered groups the absolute value of any element 
  is in $G^+$.*}

lemma (in group3) OrderedGroup_ZF_3_L3A: 
  assumes A1: "r {is total on} G" and A2: "a\<in>G"
  shows "|a| \<in> G\<^sup>+"
proof (cases "a\<in>G\<^sup>+");
  assume "a \<in> G\<^sup>+" then show "|a| \<in> G\<^sup>+" 
    using OrderedGroup_ZF_3_L2 by simp;
next assume "a \<notin> G\<^sup>+" 
  with A1 A2 show "|a| \<in> G\<^sup>+" using OrderedGroup_ZF_3_L3
    OrderedGroup_ZF_1_L6 by simp;
qed;
  
text{*For linearly ordered groups (where the order is total), the absolute
  value maps the group into the positive set.*}

lemma (in group3) OrderedGroup_ZF_3_L3B:
  assumes A1: "r {is total on} G"
  shows "AbsoluteValue(G,P,r) : G\<rightarrow>G\<^sup>+"
proof-;
  have "AbsoluteValue(G,P,r) : G\<rightarrow>G" using OrderedGroup_ZF_3_L1
    by simp;
  moreover from A1 have T2: 
    "\<forall>g\<in>G. AbsoluteValue(G,P,r)`(g)  \<in> G\<^sup>+" 
    using OrderedGroup_ZF_3_L3A by simp;
  ultimately show ?thesis by (rule func1_1_L1A);
qed;

text{*If $b$ is greater than both $a$ and $a^{-1}$, then $b$ is greater than
  $|a|$.*}

lemma (in group3) OrderedGroup_ZF_3_L4: 
  assumes A1: "a\<lsq>b" and A2: "a\<inverse>\<lsq> b" 
  shows "|a|\<lsq> b"
proof (cases "a\<in>G\<^sup>+");
  assume "a\<in>G\<^sup>+" 
  with A1 show "|a|\<lsq> b" using OrderedGroup_ZF_3_L2 by simp;
next assume "a\<notin>G\<^sup>+"
  with A1 A2 show "|a|\<lsq> b" 
    using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_3_L3 by simp;
qed;

text{*In linearly ordered groups $a\leq |a|$.*}

lemma (in group3) OrderedGroup_ZF_3_L5: 
  assumes A1: "r {is total on} G" and A2: "a\<in>G"
  shows "a \<lsq> |a|"
proof (cases "a\<in>G\<^sup>+");
  assume "a \<in> G\<^sup>+"
  with A2 show "a \<lsq> |a|" 
    using OrderedGroup_ZF_3_L2 OrderedGroup_ZF_1_L3 by simp;
next assume "a \<notin> G\<^sup>+"
  with A1 A2 show "a \<lsq> |a|"
    using OrderedGroup_ZF_3_L3A OrderedGroup_ZF_1_L4B by simp;
qed;

text{*$a^{-1}\leq |a|$ (in additive notation it woul be $-a\leq |a|$.*}

lemma (in group3) OrderedGroup_ZF_3_L6: 
  assumes A1: "a\<in>G"
  shows "a\<inverse> \<lsq> |a|"
proof (cases "a\<in>G\<^sup>+")
  assume "a \<in> G\<^sup>+"
  then have T1: "\<one>\<lsq>a" and T2: "|a| = a" using OrderedGroup_ZF_1_L2  
    OrderedGroup_ZF_3_L2 by auto;
  then have "a\<inverse>\<lsq>\<one>\<inverse>" using OrderedGroup_ZF_1_L5 by simp;
  then have T3: "a\<inverse>\<lsq>\<one>" 
    using OrderedGroup_ZF_1_L1 group0.group0_2_L7A by simp;
  from T3 T1 have "a\<inverse>\<lsq>a" by (rule OrderedGroup_ZF_1_L4A);
  with T2 show "a\<inverse> \<lsq> |a|" by simp;
next assume A2: "a \<notin> G\<^sup>+"
  from A1 have "|a| \<in> G" 
    using OrderedGroup_ZF_3_L1 apply_funtype by auto;
  with ordGroupAssum have "|a| \<lsq> |a|" 
    using IsAnOrdGroup_def IsPartOrder_def refl_def by simp;
  with A1 A2 show "a\<inverse> \<lsq> |a|" using OrderedGroup_ZF_3_L3 by simp;
qed;

text{*We can multiply the sides of two inequalities.*}

lemma (in group3) OrderedGroup_ZF_3_L7:
  assumes A1: "a\<lsq>b" and A2: "c\<lsq>d"
  shows "a\<cdot>c \<lsq> b\<cdot>d"
proof -;
  from A1 A2 have "c\<in>G" "b\<in>G" using OrderedGroup_ZF_1_L4 by auto;
  with A1 A2 ordGroupAssum have "a\<cdot>c\<lsq> b\<cdot>c" "b\<cdot>c\<lsq>b\<cdot>d"
    using IsAnOrdGroup_def by auto;
  then show "a\<cdot>c \<lsq> b\<cdot>d" by (rule OrderedGroup_ZF_1_L4A);
qed;

text{*Triangle inequality for linearly ordered abelian groups. 
  It would be nice to drop commutativity or give an example that shows we
  can't do that.*}

theorem (in group3) OrderedGroup_ZF_3_T1:
  assumes A1: "P {is commutative on} G" 
  and A2: "r {is total on} G" and A3: "a\<in>G" "b\<in>G" 
  shows "|a\<cdot>b| \<lsq> |a|\<cdot>|b|"
proof -;
  from A1 A2 A3 have 
    "a \<lsq> |a|" "b \<lsq> |b|" "a\<inverse> \<lsq> |a|" "b\<inverse> \<lsq> |b|"
    using OrderedGroup_ZF_3_L5 OrderedGroup_ZF_3_L6 by auto;
  then have "a\<cdot>b \<lsq> |a|\<cdot>|b|" "a\<inverse>\<cdot>b\<inverse> \<lsq> |a|\<cdot>|b|"
    using OrderedGroup_ZF_3_L7 by auto;
  with A1 A3 show "|a\<cdot>b| \<lsq> |a|\<cdot>|b|"
    using OrderedGroup_ZF_1_L1 group0.group0_2_L10 IsCommutative_def 
    OrderedGroup_ZF_3_L4 by simp;
qed;

text{*in linearly ordered groups $|a|\leq L$ implies $L^{-1} \leq a \leq L$
  (it would be $-L \leq a \leq L$ in the additive notation) Note that
  the first inequality does not require the order to be linear.*}

lemma (in group3) OrderedGroup_ZF_3_L8:
  assumes A1: "r {is total on} G" 
  and A2: "a\<in>G"  and A3: "|a|\<lsq>L"
  shows "L\<inverse>\<lsq>a" "a\<lsq>L"
proof -;
  from A1 A2 have T1:"a \<lsq> |a|" using OrderedGroup_ZF_3_L5 by simp;
  from T1 A3 show "a\<lsq>L" by (rule OrderedGroup_ZF_1_L4A);
  from A2 have T2: "a\<inverse> \<lsq> |a|" using OrderedGroup_ZF_3_L6 by simp;
  from T2 A3 have "a\<inverse> \<lsq> L" by (rule OrderedGroup_ZF_1_L4A);
  then have "L\<inverse>\<lsq>(a\<inverse>)\<inverse>" using OrderedGroup_ZF_1_L5 by simp;
  with A2 show "L\<inverse>\<lsq>a" using OrderedGroup_ZF_1_L1 group0.group0_2_L11
    by simp
qed

text{*For linearly ordered groups if the absolute values of elements in a set
  are bounded, then the set is bounded.*}

lemma (in group3) OrderedGroup_ZF_3_L9:
  assumes A1: "r {is total on} G"
  and A2: "A\<subseteq>G" and A3: "\<forall>a\<in>A. |a| \<lsq> L"
  shows "IsBounded(A,r)"
proof -;
  from A1 A2 A3 have "\<forall>a\<in>A. a\<lsq>L" "\<forall>a\<in>A. L\<inverse>\<lsq>a" 
    using OrderedGroup_ZF_3_L8 by auto;
  then show "IsBounded(A,r)" using
    IsBoundedAbove_def IsBoundedBelow_def IsBounded_def
    by auto;
qed;

text{* A slightly more general version of the previous lemma, stating the same
  fact for a set defined by separation.*}

lemma (in group3) OrderedGroup_ZF_3_L9A:
  assumes A1: "r {is total on} G"
  and A2: "\<forall>x\<in>X. b(x)\<in>G \<and> |b(x)|\<lsq>L"
  shows "IsBounded({b(x). x\<in>X},r)"
proof -
  from A2 have "{b(x). x\<in>X} \<subseteq> G" "\<forall>a\<in>{b(x). x\<in>X}. |a| \<lsq> L" 
    by auto;
  with A1 show ?thesis using OrderedGroup_ZF_3_L9 by blast;
qed;
  
text{*For linearly ordered groups if $l\leq a\leq u$ then 
  $|a|$ is smaller than the greater of $|l|,|u|$.*}

lemma (in group3) OrderedGroup_ZF_3_L10:
  assumes A1: "r {is total on} G"
  and A2: "l\<lsq>a" "a\<lsq>u" 
  shows "|a| \<lsq> GreaterOf(r,|l|,|u|)"
proof (cases "a\<in>G\<^sup>+");
  from A2 have T1: "|l| \<in> G" "|a| \<in> G" "|u| \<in> G"
    using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_3_L1 apply_funtype
    by auto;
  assume A3: "a\<in>G\<^sup>+"
  with A2 have "\<one>\<lsq>a" "a\<lsq>u" 
    using OrderedGroup_ZF_1_L2 by auto;
  then have "\<one>\<lsq>u" by (rule OrderedGroup_ZF_1_L4A)
  with A2 A3 have "|a|\<lsq>|u|" 
    using OrderedGroup_ZF_1_L2 OrderedGroup_ZF_3_L2 by simp;
  moreover from A1 T1 have "|u| \<lsq> GreaterOf(r,|l|,|u|)"
    using Order_ZF_3_L2 by simp;
  ultimately show "|a| \<lsq> GreaterOf(r,|l|,|u|)"
    by (rule OrderedGroup_ZF_1_L4A);
next assume A4: "a\<notin>G\<^sup>+"
  with A2 have T2: 
    "l\<in>G" "|l| \<in> G" "|a| \<in> G" "|u| \<in> G" "a \<in> G-G\<^sup>+"
    using OrderedGroup_ZF_1_L4 OrderedGroup_ZF_3_L1 apply_funtype
    by auto;
  with A2 have "l \<in> G-G\<^sup>+" using OrderedGroup_ZF_1_L4D by fast;
  with T2 A2 have "|a| \<lsq> |l|" 
    using OrderedGroup_ZF_3_L3 OrderedGroup_ZF_1_L5
    by simp;
  moreover from A1 T2 have "|l| \<lsq> GreaterOf(r,|l|,|u|)"
    using Order_ZF_3_L2 by simp; 
  ultimately show "|a| \<lsq> GreaterOf(r,|l|,|u|)"
    by (rule OrderedGroup_ZF_1_L4A);
qed;

text{*For linearly ordered groups if a set is bounded then the absolute 
  values are bounded.*}

lemma (in group3) OrderedGroup_ZF_3_L10A:
  assumes A1: "r {is total on} G"
  and A2: "IsBounded(A,r)"
  shows "\<exists>L. \<forall>a\<in>A. |a| \<lsq> L"
proof (cases "A=0");
  assume "A = 0" then show ?thesis by auto;
next assume A3: "A\<noteq>0" 
  with A2 obtain u l where "\<forall>g\<in>A. l\<lsq>g \<and>  g\<lsq>u" 
    using IsBounded_def IsBoundedAbove_def IsBoundedBelow_def
    by auto;
  with A1 have "\<forall>a\<in>A. |a| \<lsq> GreaterOf(r,|l|,|u|)"
    using OrderedGroup_ZF_3_L10 by simp;
  then show ?thesis by auto;
qed;
  
text{* A slightly more general version of the previous lemma, stating the same
  fact for a set defined by separation.*}

lemma (in group3) OrderedGroup_ZF_3_L11:
  assumes A1: "r {is total on} G"
  and A2: "IsBounded({b(x).x\<in>X},r)"
  shows "\<exists>L. \<forall>x\<in>X. |b(x)| \<lsq> L"
proof -
  from A1 A2 show ?thesis using OrderedGroup_ZF_3_L10A
    by blast;
qed

section{*Maximum absolute value of a set *}

text{* Quite often when considering inequalities we prefer to talk about
  the absolute values instead of raw elements of a set. This section formalizes
  some material that is useful for that. *}

text{*If a set has a maximum and minimum, then the greater of the 
  absolute value of the maximum and minimum belongs to the image of the set 
  by the absolute value function. *}

lemma (in group3) OrderedGroup_ZF_4_L1:
  assumes "A \<subseteq> G"
  and "HasAmaximum(r,A)" "HasAminimum(r,A)"
  and "M = GreaterOf(r,|Minimum(r,A)|,|Maximum(r,A)|)"
  shows "M \<in> AbsoluteValue(G,P,r)``(A)"
  using ordGroupAssum prems IsAnOrdGroup_def IsPartOrder_def 
    Order_ZF_4_L3 Order_ZF_4_L4 OrderedGroup_ZF_3_L1 
    func1_1_L16 GreaterOf_def by auto;

text{*If a set has a maximum and minimum, then the greater of the 
  absolute value of the maximum and minimum bounds absolute values of all 
  elements of the set. *}

lemma (in group3) OrderedGroup_ZF_4_L2: 
  assumes A1: "r {is total on} G"
  and A2: "HasAmaximum(r,A)" "HasAminimum(r,A)"
  and A3: "a\<in>A"
  shows "|a|\<lsq> GreaterOf(r,|Minimum(r,A)|,|Maximum(r,A)|)" 
proof -;
  from ordGroupAssum A2 A3 have 
    "Minimum(r,A)\<lsq> a" "a\<lsq> Maximum(r,A)" 
    using IsAnOrdGroup_def IsPartOrder_def Order_ZF_4_L3 Order_ZF_4_L4
    by auto;
  with A1 show ?thesis by (rule OrderedGroup_ZF_3_L10);
qed;

text{*If a set has a maximum and minimum, then the greater of the 
  absolute value of the maximum and minimum bounds absolute values of all 
  elements of the set. In this lemma the absolute values of ekements of a 
  set are represented as the elements of the image of the set by the absolute
  value function.*}

lemma (in group3) OrderedGroup_ZF_4_L3: 
  assumes "r {is total on} G" and "A \<subseteq> G"
  and "HasAmaximum(r,A)" "HasAminimum(r,A)"
  and "b \<in> AbsoluteValue(G,P,r)``(A)"
  shows "b\<lsq> GreaterOf(r,|Minimum(r,A)|,|Maximum(r,A)|)"
  using prems OrderedGroup_ZF_3_L1 func1_1_L16 OrderedGroup_ZF_4_L2
  by auto;

text{*If a set has a maximum and minimum, then the set of absolute values 
  also has a maximum.*}

lemma (in group3) OrderedGroup_ZF_4_L4:
  assumes A1: "r {is total on} G" and A2: "A \<subseteq> G"
  and A3: "HasAmaximum(r,A)" "HasAminimum(r,A)"
  shows "HasAmaximum(r,AbsoluteValue(G,P,r)``(A))"
proof -;
  let ?M = "GreaterOf(r,|Minimum(r,A)|,|Maximum(r,A)|)"
  from A2 A3 have "?M \<in> AbsoluteValue(G,P,r)``(A)"
    using OrderedGroup_ZF_4_L1 by simp;
  moreover from A1 A2 A3 have 
    "\<forall>b \<in> AbsoluteValue(G,P,r)``(A). b \<lsq> ?M"
    using OrderedGroup_ZF_4_L3 by simp;
  ultimately show ?thesis using HasAmaximum_def by auto;
qed;

text{*If a set has a maximum and a minimum, then all absolute values are
  bounded by the maximum of the set of absolute values.*}

lemma (in group3) OrderedGroup_ZF_4_L5:
  assumes A1: "r {is total on} G" and A2: "A \<subseteq> G"
  and A3: "HasAmaximum(r,A)" "HasAminimum(r,A)"
  and A4: "a\<in>A"
  shows "|a| \<lsq> Maximum(r,AbsoluteValue(G,P,r)``(A))"
proof -;
  from A2 A4 have "|a| \<in> AbsoluteValue(G,P,r)``(A)" 
    using OrderedGroup_ZF_3_L1 func1_1_L16 by auto;
  with ordGroupAssum A1 A2 A3 show ?thesis using 
    IsAnOrdGroup_def IsPartOrder_def OrderedGroup_ZF_4_L4
    Order_ZF_4_L3 by simp;
qed;

text{*This concludes OrderedGroups\_.thy.*};

end