(* 
    This file is a part of IsarMathLib - 
    a library of formalized mathematics for Isabelle/Isar.

    Copyright (C) 2005, 2006  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{Fol1.thy}*}

theory Fol1 imports Trancl 

begin

section{*Mission statement*}

text{*Until we come up with something better let's just say
  that writing formalized proofs protects from Alzheimer's disease
  better than solving crossword puzzles.*}

section{*Release notes*}

text{* This release starts the process of importing Metamath's \cite{Metamath}
  set.mm database into IsarMathLib.*}

section{*Overview of the project*}

text{* 
  The theory files @{text "Fo11"}, @{text " ZF1"}, @{text "Nat_ZF"}, 
  @{text "func1"}, @{text "func_ZF"}, @{text "EquivClass1"}, 
  @{text "Finite1"}, @{text "Finite_ZF"}, @{text "Order_ZF"} contain some 
  background material that is needed for the remaning theories. 

  The @{text "Topology_ZF"} series covers basics of general topology: 
  interior, closure, boundary, compact sets, separation axioms and 
  continuous functions.
  
  @{text "Group_ZF"}, @{text "Group_ZF_1"}, and @{text "Group_ZF_2"}
  provide basic facts of the group theory. @{text "Group_ZF_3"} 
  considers the notion of almost homomorphisms that is nedeed for the 
  real numbers construction in @{text "Real_ZF"}. 
  
  @{text "Ring_ZF"} defines rings. @{text "Ring_ZF_1"} covers 
  the properties of  rings that are specific to the real numbers construction 
  in @{text "Real_ZF"}. 
  
  @{text "Int_ZF"} theory considers the integers as a monoid (multiplication) and 
  an abelian ordered group (addition). 
  In @{text "Int_ZF_1"} we show that integers form a commutative ring.
  @{text "Int_ZF_2"} contains some facts about slopes (almost homomorphisms 
  on integers) needed for real numbers construction, used in @{text "Real_ZF_1"}.

  @{text "Field_ZF"} and @{text "OrderedField_ZF"} contain basic facts
  about (you guessed it) fields and ordered fields. 
  
  The @{text "Real_ZF"} and @{text "Real_ZF_1"} theories 
  contain the construction of real numbers based on the paper \cite{Arthan2004}
  by R. D. Arthan (not Cauchy sequences, not Dedekind sections). The heavy lifting
  is done mostly in @{text "Group_ZF_3"}, @{text "Ring_ZF_1"} @{text "Int_ZF_2"}.
  @{text "Real_ZF"} contains the part of the construction that can be done
  starting from generic abelian groups (rather than additive group of integers).
  This allows to show that real numbers form a ring. 
  @{text "Real_ZF_1"} continues the construction using properties specific
  to the integers showing that real numbers form a complete ordered field.

  The @{text "MMI_prelude"} and @{text "MMIsar"} contain the 
  theorems imported from Metamath's set.mm database. 
  The "Importing from Metamath - how is it done" section in @{text "MMI_prelude"} 
  section provides more details on how the import works.
*}

section{*Notions and lemmas in FOL*}

text{*This section contains mostly shortcuts and workarounds 
  that allow to use more readable coding style.*}

(*temporarily removed bc. it takes too long to check
text{*The definition and lemma below implement a common idiom used in 
romantic mathematics when the author states some assumptions, says
"then the following are equivalent" and proceeds with the proof that shows 
implications between the statements. The usefulness of theorems stated this 
way is rather limited in Isar as the simp and auto methods may take a really 
long time to infer anything from equivalences if there is more than one. 
Still, we believe that being able to formulate a theorem like this 
is good for readibility.*}

constdefs
  All4areEquivalent :: "[o,o,o,o]\<Rightarrow>o"
  "All4areEquivalent(p,q,r,s) \<equiv> 
  (p\<longleftrightarrow>q) & (p\<longleftrightarrow>r) & (p\<longleftrightarrow>s) & (q\<longleftrightarrow>r) & (q\<longleftrightarrow>s) & (r\<longleftrightarrow>s)";

lemma Fol1_L1: assumes A1: "p\<longrightarrow>q" and A2: "q\<longrightarrow>r" 
  and A3: "r\<longrightarrow>s" and A4: "s\<longrightarrow>p" 
  shows "All4areEquivalent(p,q,r,s)"
proof-;  
  from A1 A2 A3 A4 have "p\<longleftrightarrow>q"by auto;
  moreover from A1 A2 A3 A4 have "p\<longleftrightarrow>r" by auto;
  moreover from A1 A2 A3 A4 have "p\<longleftrightarrow>s" by auto;
  moreover from A1 A2 A3 A4 have "q\<longleftrightarrow>r" by auto;
  moreover from A1 A2 A3 A4 have "q\<longleftrightarrow>s" by auto;
  moreover from A1 A2 A3 A4 have "r\<longleftrightarrow>s" by auto;
  ultimately show ?thesis using All4areEquivalent_def by simp;
qed; 

text{*It happens quite often that we need to prove a statement of the form
$\forall x y. P(x,y) \rightarrow Q(x,y)$. The next lemma provides a shortcut 
for proving this type of statements. To apply this shortcut we can say 
"(rule double\_forall\_implies)" after the "proof" keyword and we are free to
show the statement by doing "fix x y assume P(x,y) show Q(x,y).*}

lemma double_forall_implies: 
  assumes A1:" \<And>x y. P(x,y) \<Longrightarrow> R(x,y)"
  shows "\<forall>x y. (P(x,y) \<longrightarrow> R(x,y))"
proof -;
  from A1 show ?thesis by simp;
qed;

*)

text{*The next lemma serves as a workaround to problems with applying 
  the definition of transitivity (of a relation) in our coding style 
  (any attempt to do
  something like @{text "using trans_def"} results up Isabelle in an 
  infinite loop). We reluctantly use @{text "(unfold trans_def)"} after the 
  @{text "proof"} keyword  to workaround this.*}

lemma Fol1_L2: assumes 
  A1: "\<forall> x y z. \<langle>x, y\<rangle> \<in> r \<and> \<langle>y, z\<rangle> \<in> r \<longrightarrow> \<langle>x, z\<rangle> \<in> r"
  shows "trans(r)"
proof (unfold trans_def)
  from A1 show
    "\<forall> x y z. \<langle>x, y\<rangle> \<in> r \<longrightarrow> \<langle>y, z\<rangle> \<in> r \<longrightarrow> \<langle>x, z\<rangle> \<in> r"
    using imp_conj by blast;
qed;

text{*Another workaround for the problem of Isabelle simplifier looping when 
  the transitivity definition is used. *}

lemma Fol1_L3: assumes A1: "trans(r)" and A2: "<a,b> \<in> r  \<and> <b,c> \<in> r"
  shows "<a,c> \<in> r"
proof -;
  from A1 have  "\<forall>x y z. \<langle>x, y\<rangle> \<in> r \<longrightarrow> \<langle>y, z\<rangle> \<in> r \<longrightarrow> \<langle>x, z\<rangle> \<in> r"
    by (unfold trans_def);
  with A2 show ?thesis using imp_conj by fast;
qed;
  
text{*There is a problem with application of the definition of asymetry for
  relations. The next lemma is a workaround.*}

lemma Fol1_L4: 
  assumes A1: "antisym(r)" and A2: "<a,b> \<in> r"   "<b,a> \<in> r"  
  shows "a=b"
proof -;
  from A1 have "\<forall> x y. <x,y> \<in> r \<longrightarrow> <y,x> \<in> r \<longrightarrow> x=y"
    by (unfold antisym_def);
  with A2 show "a=b" using imp_conj by fast;
qed;

text{*The definition below implements a common idiom that states that 
  (perhaps under some assumptions) exactly one of give three statements 
  is true.*}

constdefs
  "Exactly_1_of_3_holds(p,q,r) \<equiv> 
  (p\<or>q\<or>r) \<and> (p \<longrightarrow> \<not>q \<and> \<not>r) \<and> (q \<longrightarrow> \<not>p \<and> \<not>r) \<and> (r \<longrightarrow> \<not>p \<and> \<not>q)";

text{*The next lemma allows to prove statements of the form 
  @{text "Exactly_1_of_3_holds (p,q,r)"}.*}

lemma Fol1_L5:
  assumes "p\<or>q\<or>r"
  and "p \<longrightarrow> \<not>q \<and> \<not>r"
  and "q \<longrightarrow> \<not>p \<and> \<not>r"
  and "r \<longrightarrow> \<not>p \<and> \<not>q"
  shows "Exactly_1_of_3_holds (p,q,r)"
proof -;
  from prems have
    "(p\<or>q\<or>r) \<and> (p \<longrightarrow> \<not>q \<and> \<not>r) \<and> (q \<longrightarrow> \<not>p \<and> \<not>r) \<and> (r \<longrightarrow> \<not>p \<and> \<not>q)"
    by blast;
  then show "Exactly_1_of_3_holds (p,q,r)"
    by (unfold Exactly_1_of_3_holds_def);
qed;

text{*If exactly one of $p,q,r$ holds and $p$ is not true, then
  $q$ or $r$.*}

lemma Fol1_L6: 
  assumes A1: "\<not>p" and A2: "Exactly_1_of_3_holds (p,q,r)" 
  shows "q\<or>r"
proof -
  from A2 have  
    "(p\<or>q\<or>r) \<and> (p \<longrightarrow> \<not>q \<and> \<not>r) \<and> (q \<longrightarrow> \<not>p \<and> \<not>r) \<and> (r \<longrightarrow> \<not>p \<and> \<not>q)"
    by (unfold Exactly_1_of_3_holds_def);
  then have "p\<or>q\<or>r" by blast;
  with A1 show "q\<or>r" by simp;
qed;

text{*If exactly one of $p,q,r$ holds and $q$ is true, then 
  $r$ can not be true.*}

lemma Fol1_L7:
  assumes A1: "q" and A2: "Exactly_1_of_3_holds (p,q,r)"
  shows "\<not>r"
proof -
   from A2 have  
    "(p\<or>q\<or>r) \<and> (p \<longrightarrow> \<not>q \<and> \<not>r) \<and> (q \<longrightarrow> \<not>p \<and> \<not>r) \<and> (r \<longrightarrow> \<not>p \<and> \<not>q)"
    by (unfold Exactly_1_of_3_holds_def)
  with A1 show "\<not>r" by blast;
qed;

text{* The next lemma demonstrates an elegant form of the 
  @{text "Exactly_1_of_3_holds (p,q,r)"} predicate. More on that
  at www.solcon.nl/mklooster/calc/calc-tri.html . *}

lemma Fol1_L8: 
  shows "Exactly_1_of_3_holds (p,q,r) \<longleftrightarrow> (p\<longleftrightarrow>q\<longleftrightarrow>r) \<and> \<not>(p\<and>q\<and>r)"
proof;
  assume "Exactly_1_of_3_holds (p,q,r)"
  then have 
    "(p\<or>q\<or>r) \<and> (p \<longrightarrow> \<not>q \<and> \<not>r) \<and> (q \<longrightarrow> \<not>p \<and> \<not>r) \<and> (r \<longrightarrow> \<not>p \<and> \<not>q)"
    by (unfold Exactly_1_of_3_holds_def);
  thus "(p\<longleftrightarrow>q\<longleftrightarrow>r) \<and> \<not>(p\<and>q\<and>r)" by blast;
next assume "(p\<longleftrightarrow>q\<longleftrightarrow>r) \<and> \<not>(p\<and>q\<and>r)" 
  then have 
    "(p\<or>q\<or>r) \<and> (p \<longrightarrow> \<not>q \<and> \<not>r) \<and> (q \<longrightarrow> \<not>p \<and> \<not>r) \<and> (r \<longrightarrow> \<not>p \<and> \<not>q)"
    by auto;
  thus "Exactly_1_of_3_holds (p,q,r)"
    using Exactly_1_of_3_holds_def by (unfold Exactly_1_of_3_holds_def);
qed;

text{*Exclusive or definition. There is one also defined in the standard 
  Isabelle, denoted @{text "xor"}, but it relates to boolean values, 
  which are sets. Here we define a logical functor.*}

constdefs
  Xor (infixl "Xor" 66)
  "p Xor q \<equiv> (p\<or>q) \<and> \<not>(p \<and> q)"

text{*The "exclusive or" is the same as negation of equivalence.*}

lemma Fol1_L9: shows "p Xor q \<longleftrightarrow> \<not>(p\<longleftrightarrow>q)"
  using Xor_def by auto;

(*
text{*This is an example how we can show a statement of the form 
  @{text "p Xor q"}.*}

lemma Fol1_L6: assumes "p\<or>q" and "\<not>p \<or> \<not>q"
  shows "p Xor q"
proof -
  from prems show "p Xor q" using Xor_def by simp;
qed;*)

text{*Equivalence relations are symmetric.*}

lemma equiv_is_sym: assumes A1:  "equiv(X,r)" and A2: "\<langle>x,y\<rangle> \<in> r"
  shows  "\<langle>y,x\<rangle> \<in> r"
proof -
  from A1 have "sym(r)" using equiv_def by simp;
  then have "\<forall>x y. \<langle>x,y\<rangle> \<in> r \<longrightarrow> \<langle>y,x\<rangle> \<in> r"
    by (unfold sym_def);
  with A2 show "\<langle>y,x\<rangle> \<in> r" by blast;
qed;
    

end