{- This source file is part of mmisar - a tool for converting
Metamath proofs to Isabelle/Isar.
Copyright 2006 Slawomir Kolodynski

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
-}


-- | this module defines the function that takes the internal representation
-- of a Metamath ZF formula and converts it into a string with the Isar format
-- of that formula. In other words, it exports a logical expression to Isabelle.

module Export2Isabelle
   where

import MMIDataTypes
import Maybe
import qualified Data.Map as Map


-- | a lookup table for translating logical infixes from MM to Isabelle
infixLLLMap :: Map.Map String (Int, String)
infixLLLMap = Map.fromList [("->",  (15, " \\<longrightarrow> ")),
                         ("<->", (20, " \\<longleftrightarrow> ")),
                         ("/\\", (25, " \\<and> ")),
                         ("\\/", (22, " \\<or> "))]

-- | a lookup table for translating set->set->set infixes
infixSSSMap :: Map.Map String (Int, String)
infixSSSMap = Map.fromList [("+",  (69, " \\<ca> ")),
                            ("-",  (69, " \\<cs> ")),
                            ("x.", (71, "\\<cdot>")),
                            ("/", (70, "\\<cdiv>"))]

-- | a lookup table for translating set->set->logical infixes
infixSSLMap :: Map.Map String (Int, String)
infixSSLMap = Map.fromList [("e.",  (50, " \\<in> ")),
                            ("=/=", (50, " \\<noteq> ")),
                            ("=",   (50, " = ")),
                            ("(_",  (50, " \\<subseteq>")),
                            ("<_", (68, " \\<lsq> ")),
                            ("<", (68, " \\<ls> ")),
                            ("R", (68, " R "))] -- ^ note we are cheating with the R infix

-- | a lookup table for translating symbols
symbTransTable :: Map.Map String String
symbTransTable = Map.fromList [("CC", "\\<complex>"),
                                ("ph", "\\<phi>"),
                                ("RR", "\\<real>"),
                                ("NN", "\\<nat>"),
                                ("(/)", "0"),
                                ("0", "\\<zero>"),
                                ("1",  "\\<one>"),
                                ("+oo","\\<cpnf>"),
                                ("-oo","\\<cmnf>"),
                                ("th", "\\<theta>"),
                                ("ta", "\\<tau>"),
                                ("et", "\\<eta>"),
                                ("ze", "\\<zeta>"),
                                ("ps", "\\<psi>"),
                                ("E.", " \\<exists>"),
                                ("E!", "\\<exixsts>!"),
                                ("A.", "\\<forall>")]

-- | table of infixes that cause long lines to break
breakLineTable = ["->", "<->", "="]

-- | produces an Isabelle/ZF expression from the MMIsar internal representation
-- of a Metamath ZF (logical) expression
exp2Isabelle :: Int -> LogExpr -> String

exp2Isabelle par (InfixLLL linf p q) = 
   if isNothing look then error (linf ++ " not in the logical infix lookup table")
   else (addParens par child) ( ( exp2Isabelle (child + 1) p) ++ translation ++ 
      ( ( breakLine linf) ( exp2Isabelle child q) ) )
      where
         look = Map.lookup linf infixLLLMap
         child = fst $ fromJust look
         translation = snd $ fromJust look

exp2Isabelle par (InfixSSL linf sa sb) =
   if isNothing look then error (linf ++ " not in the set->set->logical infix lookup table")
   else (addParens par child) ( ( expSet2Isabelle child sa) ++ translation ++ ( expSet2Isabelle child sb) )
      where
         look = Map.lookup linf infixSSLMap
         child = fst $ fromJust look
         translation = snd $ fromJust look

exp2Isabelle par (Quant q sym formula) =
   if isNothing look then error (q ++ " is not in the set of known symbols")
   else (addParens par 10) ( (fromJust look) ++ (expSet2Isabelle 0 sym) ++ ". " ++ (exp2Isabelle 0 formula) )
      where look = Map.lookup q symbTransTable

exp2Isabelle par (QuantRange q sym srange formula) =
   if isNothing look then error (q ++ " is not in the set of known symbols")
   else (addParens par 10) ( if q == "E!" then expr1 else expr )
      where 
         look = Map.lookup q symbTransTable
         expr = ( (fromJust look) ++ syms ++ 
                "\\<in>" ++ ( expSet2Isabelle 50 srange ) ++ ". " ++ ( exp2Isabelle 0 formula) )
         expr1 = (fromJust look) ++ syms ++ ". " ++ syms ++ "\\<in>" ++ ( expSet2Isabelle 50 srange ) ++ 
                 "\\<and>" ++ ( exp2Isabelle conjAssoc formula)
         syms  = expSet2Isabelle 0 sym
         conjAssoc = fst $ fromJust $ Map.lookup "/\\" infixLLLMap

exp2Isabelle par (IsASet sete) = isete ++ " = " ++ isete
                           where isete = expSet2Isabelle 50 sete

exp2Isabelle par (Negation le) = (addParens par 65) ("\\<not>" ++ exp2Isabelle 70 le)

exp2Isabelle par (Lvar s) = symbTrans s

exp2Isabelle _ _ = error ("not implemented type of logical expression")


-- | produces an Isabelle/ZF expression from the MMIsar internal representation
-- of a Metamath ZF (logical) expression
expSet2Isabelle :: Int -> SetExpr -> String

expSet2Isabelle par (InfixSSS sinf sa sb) =
   if isNothing look then error (sinf ++ " not in the set infix lookup table")
   else (addParens par child) ( ( expSet2Isabelle child sa )++ translation ++ (expSet2Isabelle (child+1) sb) )
      where
         look = Map.lookup sinf infixSSSMap
         child = fst $ fromJust look
         translation = snd $ fromJust look

expSet2Isabelle par (Negative s) = (addParens par 0) ( "\\<cn>" ++ (expSet2Isabelle 95 s) )

expSet2Isabelle par (If loge s1 s2) = (addParens par 55) 
   (" if(" ++ (exp2Isabelle 0 loge) ++ ", " ++ (expSet2Isabelle 0 s1) ++ ", " ++ (expSet2Isabelle 0 s2) ++ ")" )

expSet2Isabelle par (SetCompr sete loge) = "{" ++ (expSet2Isabelle 0 sete) ++ "\\<in> Pow(\\<real>). " ++ 
                                             (exp2Isabelle 0 loge) ++ " }"

expSet2Isabelle par (SetComprRange set1 set2 loge) = 
   "{" ++ (expSet2Isabelle 0 set1) ++ " \\<in> " ++ (expSet2Isabelle 50 set2) ++ ". " ++ 
   (exp2Isabelle 0 loge) ++ " }"

expSet2Isabelle par (Svar s) = symbTrans s

expSet2Isabelle par (Snum n) = symbTrans n

expSet2Isabelle _ _ = error ("not implemented type of set expression")

-- | translates a symbol
symbTrans :: String -> String
symbTrans s = if isNothing symbtrans then s
              else fromJust symbtrans
                 where symbtrans = Map.lookup s symbTransTable


-- | surrounds a string with parantheses if parent assiociation (par) is
-- as stronger as child's association  
addParens :: Int -> Int -> String -> String
addParens par child expr = if par > child then "(" ++ expr ++ ")"
                           else expr

-- | inserts a new line before some words
breakLine :: String -> String -> String
breakLine inf s = if inf `elem` breakLineTable && length s > 30 then "\n   " ++ s
                  else s 