{- This source file is part of mmisar - a tool for converting
Metamath proofs to Isabelle/Isar.
Copyright 2006 Slawomir Kolodynski

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
-}

-- | this module defines a parser for Metamath ZF formulas. 
 
module ParseMM
    where

import Text.ParserCombinators.Parsec
import MMIDataTypes

{-
some test cases:
"ph -> A e. CC"
"A e. RR -> A e. CC"
"E. x ( A + x ) = 0"
"A e. CC -> E. x e. CC ( A + x ) = 0"
"E. x e. CC ( A + x ) = 0"
"E! x e. A ph <-> ( E. x e. A ph /\\ A. x e. A A. y e. A ( ( ph /\\ ps ) -> x = y ) )"
"( A e. CC /\\ A =/= 0 ) -> E. x e. CC ( A x. x ) = 1"
"A <_ B \\/ B <_ A"
"B < A <-> -. A <_ B"
"[ z / x ] ( A e. NN -> ph )"
"if ( A e. RR , A , 0 )"
"sup ( A , B, A + C )"
"( A (_ RR /\\ -. A = (/) /\\ E. x e. RR A. y e. A y <_ x )" 
"( A (_ RR /\\ -. A = (/) /\\ E. x e. RR A. y e. A y <_ x ) /\\ ( B e. RR /\\ B < sup ( A , RR , < ) )"
"( ph /\\ ps ) -> E. z e. A B < z"
"( ph /\\ ( ps /\\ B < C ) ) -> E. z e. A B < z"
"( ps /\\ B < sup ( A , RR , < ) ) -> ph"
"( ps /\\ B < sup ( A , RR , < ) ) -> E. z e. A B < z"
"( ph /\\ ( ps /\\ B < sup ( A , RR , < ) ) ) -> E. z e. A B < z"
"( ph /\\ ( B e. RR /\\ B < sup ( A , RR , < ) ) ) -> E. z e. A B < z"
"( ( A (_ RR /\\ -. A = (/) /\\ E. x e. RR A. y e. A y <_ x ) /\\ ( B e. RR /\\ B < sup ( A , RR , < ) ) ) -> E. z e. A B < z"
"x = -u y -> ( ph <-> ps )"
"A. y e. RR ( x < y -> E. w e. A w < y )"
"( A. y e. A -. y < x /\\ A. y e. RR ( x < y -> E. w e. A w < y ) )"
"E. x e. RR ( A. y e. A -. y < x /\\ A. y e. RR ( x < y -> E. w e. A w < y ) )"
"( A _ RR /\\ -. A = (/) /\\ E. x e. RR A. y e. A x <_ y ) -> E. x e. RR ( A. y e. A -. y < x /\\ A. y e. RR ( x < y -> E. w e. A w < y ) )"
"( ( A _ RR /\\ -. A = (/) /\\ E. x e. RR A. y e. A x <_ y ) -> E. x e. RR ( A. y e. A -. y < x /\\ A. y e. RR ( x < y -> E. w e. A w < y ) ) )"
-}


-- | table of infixes for logical operations
logLogLogInfixTable = ["->", "<->", "/\\", "\\/"]

-- | table of set prefixes
setPrefixTable = ["|^|","U.","`'","-u"]

-- | table of infixes for operations that take sets and return logical values
-- Note we are cheating here with the "R" infix. 
setSetLogInfixTable = ["e." , "=/=" , "=" , "(_" , "<_" , "<" , "R" ,"`'<" , "`'R" , "Or"]

-- | table of infixes that take sets and return a set
setSetSetInfixTable = ["+", "-", "x.", "/"]

-- | table of quantifiers
quantTable = ["A.", "E.", "E!"]

-- | table of logical variable names, these are mostly greek letters
logVars = ["ph", "ps", "ch", "ta", "th"]

-- | table of set variables, note we confuse two types of variables here
-- one is x, y, u,v,t that denote real set variables that can not be substituted
-- and the other are symbols that can be substituted with set expressions: A,B, C

setVars = ["x", "y", "z", "u", "v", "w", "t"
          ,"A", "B", "C", "D", "R"
          , "CC", "NN", "RR", "(/)", "0", "1"
          ,"<", "`'<", "`'R"]

-- | the main function exported by the module - translates an MM formula into the 
-- internal representation
str2LogExpr :: [String] -> Maybe LogExpr
str2LogExpr = either2Maybe . parse parseMMform "MM formula" 

-- | parses a Metamath formula. This is essentially parseLogExpr function, except
-- that it has eof at the end to ensure that all input has been consumed
parseMMform :: GenParser String () LogExpr
parseMMform = do
   e <- parseLogExpr
   eof
   return e

-- | parses a logical expression. 
parseLogExpr :: GenParser String () LogExpr
parseLogExpr = try parse3Conj <|> try parseLogInfix <|>  try parseIsAset <|> logOperand

-- | parses a logical expression that can be an operand in another logical expression 
-- (like infix). Typically in a Metamath formula
-- operands are enclosed in parentheses, but there are exceptions to this.
logOperand :: GenParser String () LogExpr
logOperand = try parseInfixSSL <|> parseNegation <|> try parseQuantRange <|> parseQuant <|> logVar <|> parseLogExprP

-- | parses set expressions
parseSetExpr :: GenParser String () SetExpr
parseSetExpr = try parseInfixSSS <|> setOperand

-- | parses set operands. These are either set variables or sets in parantheses
setOperand :: GenParser String () SetExpr
setOperand = setVar <|> parseSetPref <|> try parseSetComprRange <|> parseSetCompr <|> parseSup <|> parseIf <|> parseSetExprP

-- | this parser parses logical variables
logVar :: GenParser String st LogExpr
logVar = choice [ do s <- word ss; return (Lvar s) | ss <- logVars]

-- | parses set variables
setVar :: GenParser String st SetExpr
setVar = choice [ do s <- word ss; return (Svar s) | ss <- setVars]

-- | parses a logical infix. 
parseLogInfix :: GenParser String () LogExpr
parseLogInfix = do
   e1 <- logOperand
   op <- oneOfWords logLogLogInfixTable
   e2 <- logOperand
   return ( InfixLLL e1 op e2)

-- | parses logical infix with sets as arguments
parseInfixSSL :: GenParser String () LogExpr
parseInfixSSL = do
   s1 <- setOperand
   op <- oneOfWords setSetLogInfixTable
   s2 <- setOperand
   return ( InfixSSL s1 op s2 )

-- | parses set infix
parseInfixSSS :: GenParser String () SetExpr
parseInfixSSS = do
   s1 <- setOperand
   op <- oneOfWords setSetSetInfixTable
   s2 <- setOperand
   return ( InfixSSS s1 op s2 )

-- | parses logical expression in parentheses
parseLogExprP :: GenParser String () LogExpr
parseLogExprP = between (word "(") (word ")") $ parseLogExpr

-- | parses set expressions in parantheses
parseSetExprP :: GenParser String () SetExpr
parseSetExprP = between (word "(") (word ")") $ parseSetExpr


-- | parses set prefix
parseSetPref :: GenParser String () SetExpr
parseSetPref = do
   p <- oneOfWords setPrefixTable
   s <- setOperand
   return (PrefixS p s)

-- | "If" expression
parseIf :: GenParser String () SetExpr
parseIf = do
   word "if"
   word "("
   le <- parseLogExpr
   word ","
   s1 <- parseSetExpr
   word ","
   s2 <- parseSetExpr
   word ")"
   return ( If le s1 s2 )

-- | parses set comprehension, like { x | ph }
parseSetCompr :: GenParser String () SetExpr
parseSetCompr = between ( word "{" ) ( word "}" ) $ do
   svar <- setVar
   word "|"
   le <- parseLogExpr
   return (SetCompr svar le)

-- | parses set comprehension with range, like { x e. CC | ph }
parseSetComprRange :: GenParser String () SetExpr
parseSetComprRange = between ( word "{" ) ( word "}" ) $ do
   s1 <- setVar
   word "e."
   srange <- parseSetExpr
   word "|"
   le <- parseLogExpr
   return (SetComprRange s1 srange le)

-- | parses supremum, like sup ( A , B, R )
parseSup :: GenParser String () SetExpr
parseSup = do
   word "sup"
   word "("
   s1 <- parseSetExpr
   word ","
   s2 <- parseSetExpr
   word ","
   s3 <- parseSetExpr
   word ")"
   return ( Supremum s1 s2 s3 )

-- | parses negation
parseNegation :: GenParser String () LogExpr
parseNegation = do 
   word "-."
   le <- logOperand
   return (Negation le)

-- | parses quantifiers
parseQuant :: GenParser String () LogExpr
parseQuant = do
   q    <- oneOfWords quantTable
   svar <- setVar
   le   <- logOperand
   return (Quant q svar le)

-- | parses quantifiers with range
parseQuantRange :: GenParser String () LogExpr
parseQuantRange = do
   q      <- oneOfWords quantTable
   svar   <- setVar
   word  "e."
   srange <- setOperand
   le     <- logOperand
   return ( QuantRange q svar srange le )

-- | is a set, (actually in a Metamath this means is a class)
parseIsAset :: GenParser String () LogExpr
parseIsAset = do
   s <- setOperand
   word "e."
   word "V"
   return (IsASet s)

-- | substitution expression
parseSubst :: GenParser String () LogExpr
parseSubst = do
   word "["
   s1 <- setVar
   word "/"
   s2 <- setVar
   word "]"
   lop <- logOperand
   return (Subst s1 s2 lop)

-- | parses a triple conjunction "ph /\\ ps /\\ ph". Metamath really understands it
-- right associatively, but here we translate it left-associatively for compatibility
-- with the previous version of the parser
parse3Conj :: GenParser String () LogExpr
parse3Conj = do
   le1 <- logOperand
   word "/\\"
   le2 <- logOperand
   word "/\\"
   le3 <- logOperand
   return (InfixLLL le1 "/\\" ( InfixLLL le2 "/\\" le3 ) )



{------------------------------------------------------------------------}
-- general purpose parsing utilities, mostly to implement the Char part
-- of Parsec to apply for words. The tokens in Metamath are strings, so we parse
-- not lists of Chars like in the Parsec.Char, but lists of strings.

-- | this is 
updatePosWord   :: SourcePos -> String -> SourcePos
updatePosWord sp w = incSourceColumn sp 1

-- | this is like char parser, but for words
word :: String -> GenParser String st String
word s = tokenPrim showWord nextPos testWord
       where
          showWord x = show x
          testWord x = if (x==s) then Just s else Nothing
          nextPos pos x xs = updatePosWord pos x

-- | recognizes words from a provided list
oneOfWords :: [String] -> GenParser String st String
oneOfWords table = choice [word s | s <- table] 

-- | converts Left Right to Maybe
either2Maybe :: Either a b -> Maybe b
either2Maybe (Left _) = Nothing
either2Maybe (Right x) = Just x