(* 
    This file is a part of IsarMathLib - 
    a library of formalized mathematics for Isabelle/Isar.

    Copyright (C) 2007  Slawomir Kolodynski

    This program is free software; Redistribution and use in source and binary forms, 
    with or without modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice, 
   this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright notice, 
   this list of conditions and the following disclaimer in the documentation and/or 
   other materials provided with the distribution.
   3. The name of the author may not be used to endorse or promote products 
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED 
WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR 
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*)

header {*\isaheader{Semigroup\_ZF.thy}*}

theory Semigroup_ZF imports func_ZF FiniteSeq_ZF InductiveSeq_ZF

begin

text{*It seems that the minimal setup needed to talk about a product of a 
  sequence is a set with a binary operation. 
  Such object is called "magma". However, interesting properties
  show up when the binary operation is associative and such alebraic structure
  is called a semigroup. In this theory file we define and study sequences of partial 
  products of sequences of magma and semigroup elements.*}

section{*Sequence of partial products of a sequence*}

text{*It happens often in analysis that we have a (finite) 
  sequence $x_0,x_1,..,x_n$ of elements of some set with a binary 
  operation on it and we want to talk about the sequence the partial products
  (or sums) of this sequence. In romantic mathematics depending of whether 
  we are using multiplicative or additive notation we write 
  $x_0, x_0+x_1,.., x_0+x_1+..+x_n$  or 
  $x_0, x_0\cdot x_1,.., x_0\cdot x_1\cdot..\cdot x_n$ and we know 
  what is the meaning. This section developes this concept so that it can
  be used in a formalized setting.*}

text{*Suppose we have a binary operation $f : X\times X \rightarrow X$. 
  Then every $a\in X$ defines a transformation of $X$ defined by
  $T_a(x) = f\langle x,a\rangle$. In IsarMathLib such transformation is defined
  as @{text "Fix2ndVar(f,a)"} metafunction. Using this notion, 
  given a sequence $x = \{x_k\}_{k\in N}$ of elements of $X$ 
  we can convert (the tail of) it into a sequence of tranformations of $X$. 
  This is defined in @{text "Seq2TransSeq"} below. 
  Then we use that sequence of tranformations to define the sequence
  of products (called @{text "ProdsSeqN"}) by means of 
  @{text "InductiveSeqVarFN"} (defined in @{text "InductiveSeq_ZF"} theory) 
  which implements the inductive sequence determined by a starting point 
  and a sequence of transformations. Finally, we define the product of a 
  sequence as the last element of the sequence of the partial products.
  *}

constdefs
  
  "Seq2TransSeqN(f,a) \<equiv> {\<langle>k,Fix2ndVar(f,a`(k))\<rangle>. k \<in> domain(a)}"

  "ProdsSeqN(f,a) \<equiv> 
  InductiveSeqVarFN(a`(0), domain(Fix2ndVar(f,a`(0))), 
  Seq2TransSeqN(f,Tail(a)),domain(Tail(a)))" 

  "ProdN(f,x) \<equiv> Last(ProdsSeqN(f,x))"

text{*If $X$ is a set with a binary operation $f:X\times X \rightarrow X$ then 
  @{text "Seq2TransSeqN(f,n,a)"} converts a sequence $a$ of elements of $X$  
  to the sequence of corresponding transformations of the set.*}

lemma seq2trans_seq_props: 
  assumes A1: "n \<in> nat" and A2: "f : X\<times>X \<rightarrow> X" and A3: "a:n\<rightarrow>X" and
  A4: "T = Seq2TransSeqN(f,a)"
  shows
  "T : n \<rightarrow> (X\<rightarrow>X)"
  "\<forall>k\<in>n. \<forall>x\<in>X. (T`(k))`(x) = f`\<langle>x,a`(k)\<rangle>"
proof -
  from A3 have D: "domain(a) = n" using func1_1_L1 by simp;
  from A2 A3 have "\<forall>k\<in>n. Fix2ndVar(f,a`(k)) : X\<rightarrow>X"
    using apply_funtype fix_2nd_var_fun by simp;
  then have "{\<langle>k,Fix2ndVar(f,a`(k))\<rangle>. k \<in> n} : n \<rightarrow> (X\<rightarrow>X)"
    by (rule ZF_fun_from_total);
  with A4 D show "T : n \<rightarrow> (X\<rightarrow>X)" 
    using Seq2TransSeqN_def by simp;
  with A4 D have I: "\<forall>k \<in> n. T`(k) = Fix2ndVar(f,a`(k))"
    using Seq2TransSeqN_def ZF_fun_from_tot_val0 by simp;
  { fix k fix x assume A5: "k\<in>n" and "x\<in>X"
    with A1 A3 have 
      "f : X\<times>X \<rightarrow> X"  "x\<in>X"  "a`(k) \<in> X" using apply_funtype
      by auto;
    then have "Fix2ndVar(f,a`(k))`(x) = f`\<langle>x,a`(k)\<rangle>"
      using fix_var_val by simp;
    with A5 I have "(T`(k))`(x) = f`\<langle>x,a`(k)\<rangle>"
      by simp;
  } then show "\<forall>k\<in>n. \<forall>x\<in>X. (T`(k))`(x) = f`\<langle>x,a`(k)\<rangle>"
    by simp;
qed;

text{*Basic properties of the sequence of partial products of a sequence 
  $a = \{a_k\}_{k\in \{0,..,n\} }$.*}

theorem prod_seq_N_props:
  assumes A1: "n \<in> nat" and A2: "f : X\<times>X \<rightarrow> X" and 
  A3: "a : succ(n) \<rightarrow> X" and A4: "P = ProdsSeqN(f,a)"
  shows 
  "P: succ(n) \<rightarrow> X"
  "P`(0) = a`(0)"
  "\<forall>k\<in>n. P`(succ(k)) = f`\<langle>P`(k), a`(succ(k))\<rangle>"
proof -
  let ?b = "Tail(a)"
  let ?T = "Seq2TransSeqN(f,?b)"
  from A1 A3 have D: "domain(Tail(a)) = n"
    using func1_1_L1 tail_props by blast;
  from A1 A3 have  I: "a`(0) \<in> X" 
    using empty_in_every_succ apply_funtype by simp;
  with A2 have II: "domain(Fix2ndVar(f,a`(0))) = X" 
    using fix_2nd_var_fun  func1_1_L1 by blast;
  note A1 I
  moreover 
  from A1 A3 have III: "?b : n \<rightarrow> X" using tail_props by simp;
  with A1 A2 have IV: "?T: n \<rightarrow> (X\<rightarrow>X)" using seq2trans_seq_props
    by simp;
  moreover from A4 D II have V: "P = InductiveSeqVarFN(a`(0), X, ?T, n)"
    using ProdsSeqN_def by simp;
  ultimately show VI: "P: succ(n) \<rightarrow> X" by (rule fin_indseq_var_f_props);
  from A1 I IV V show "P`(0) = a`(0)" by (rule fin_indseq_var_f_props);
  from A1 I IV V have "\<forall>k\<in>n. P`(succ(k)) = (?T`(k))`(P`(k))"
    by (rule fin_indseq_var_f_props);
  moreover
  { fix k assume A5: "k\<in>n" hence "k \<in> succ(n)" by auto;
    with A1 A2 A3 III VI A5 have "(?T`(k))`(P`(k)) = f`\<langle>P`(k), a`(succ(k))\<rangle>"
      using apply_funtype seq2trans_seq_props tail_props by simp }
  ultimately show "\<forall>k\<in>n. P`(succ(k)) = f`\<langle>P`(k), a`(succ(k))\<rangle>"
    by simp;
qed;

text{*A consistency condition: if we make the sequence of 
  magma elements shorter, then we get a shorter partial products
  sequence with the same values as in the original sequence. 
  This can be proven as a special case of @{text "fin_indseq_var_f_restrict"}
  but a proof using @{text "prod_seq_N_props"} and induction turns out to be 
  shorter.*}

lemma prod_seq_N_restrict: assumes 
  A1a: "n \<in> nat" and A1b: "i \<in> nat"  and A2: "f : X\<times>X \<rightarrow> X" and
  A3a: "a : succ(n) \<rightarrow> X"  and A3b: "b : succ(i) \<rightarrow> X"  and
  A4: "n \<subseteq> i" and A5: "\<forall>j \<in> succ(n). b`(j) = a`(j)" and A6: "k \<in> succ(n)"
  shows "ProdsSeqN(f,b)`(k) = ProdsSeqN(f,a)`(k)"
proof -
  let ?P = "ProdsSeqN(f,a)"
  let ?Q = "ProdsSeqN(f,b)"
  note A1a A6
  moreover have "?Q`(0) = ?P`(0)"
  proof -
    have "?P = ProdsSeqN(f,a)" by simp;
    with A1a A2 A3a have "?P`(0) = a`(0)"
      by (rule prod_seq_N_props);
    moreover
    have "?Q = ProdsSeqN(f,b)" by simp
    with A1b A2 A3b have "?Q`(0) = b`(0)"
      by (rule prod_seq_N_props);
    moreover from A1a A5 have "b`(0) = a`(0)"
      using empty_in_every_succ by auto;
    ultimately show "?Q`(0) = ?P`(0)" by simp;
  qed;
  moreover have "\<forall>j \<in> n. 
    ?Q`(j) = ?P`(j) \<longrightarrow> ?Q`(succ(j)) = ?P`(succ(j))"
  proof -
    { fix j 
      assume A7: "j\<in>n" and A8: "?Q`(j) = ?P`(j)"
      have "?P = ProdsSeqN(f,a)" by simp;
      with A1a A2 A3a have 
	"\<forall>k\<in>n. ?P`(succ(k)) = f`\<langle>?P`(k), a`(succ(k))\<rangle>"
	by (rule prod_seq_N_props);
      with A7 have "?P`(succ(j)) = f`\<langle>?P`(j), a`(succ(j))\<rangle>"
	by simp;
      moreover
      have "?Q = ProdsSeqN(f,b)" by simp;
      with A1b A2 A3b have 
	"\<forall>k\<in>i. ?Q`(succ(k)) = f`\<langle>?Q`(k), b`(succ(k))\<rangle>"
	by (rule prod_seq_N_props);
      with A4 A7 have "?Q`(succ(j)) = f`\<langle>?Q`(j), b`(succ(j))\<rangle>"
	by auto;
      moreover from A1a A5 A7 have "b`(succ(j)) = a`(succ(j))"
	using succ_ineq by auto;
      moreover note A8
      ultimately have "?Q`(succ(j)) = ?P`(succ(j))"
	by simp;
    } then show ?thesis by simp;
  qed;
  ultimately show "ProdsSeqN(f,b)`(k) = ProdsSeqN(f,a)`(k)"
    by (rule fin_nat_ind);
qed;

text{*A special case of @{text "prod_seq_N_restrict"} when the longer
  sequence of magma elements is created from the shorter one by appending
  one element.*}

corollary prod_seq_N_append: 
  assumes A1: "n \<in> nat" and  A2: "f : X\<times>X \<rightarrow> X" and
  A3: "a : succ(n) \<rightarrow> X" and A4: "x\<in>X" and A5: "k \<in> succ(n)"
  shows "ProdsSeqN(f,Append(a,x))`(k) = ProdsSeqN(f,a)`(k)"
  proof -
    let ?i = "succ(n)"
    let ?b = "Append(a,x)"
    from A1 A2 A3 A4 A5 have
      "n \<in> nat"  "?i \<in> nat"  "f : X\<times>X \<rightarrow> X"   
      "a : succ(n) \<rightarrow> X"  "?b : succ(?i) \<rightarrow> X"  "n \<subseteq> ?i"   
      "\<forall>j \<in> succ(n). ?b`(j) = a`(j)"   "k \<in> succ(n)"
      using append_props by auto;
    then show "ProdsSeqN(f,?b)`(k) = ProdsSeqN(f,a)`(k)"
      by (rule prod_seq_N_restrict);
qed;
  

(*text{*Elements of the sequence of partial products of a seqence $a$ 
  up to $k$-th element depend only on the sequence $a$ up to $k$-th 
  elelement.*}

lemma prod_seq_N_restrict:
  assumes A1: "n \<in> nat" and A2: "f : X\<times>X \<rightarrow> X" and 
  A3: "a : succ(n) \<rightarrow> X" and A4: "k \<in> succ(n)"
  shows " ProdsSeqN(f,restrict(a,k)) = restrict(ProdsSeqN(f,a),k)"
proof
  
  

text{*When we add an element to the sequence, we add another element
  to the sequence of partial products. The next lemma also tells us what
  is that we are adding to the sequence of partial producst.*}

lemma prod_seq_N_append:  
  assumes A1: "n \<in> nat" and A2: "f : X\<times>X \<rightarrow> X" and 
  A3: "a : succ(n) \<rightarrow> X" and  A4: "x\<in>X" and
  A5: "P = ProdsSeqN(f,Append(a,x))"
  shows "P = Append(ProdsSeqN(f,a), f`\<langle>ProdN(f,a),x\<rangle>)"*)
  
text{*What we really will be using is the notion of the product
  of a sequence, which we define as the last element of
  (inductively defined) sequence of partial products. The next theorem 
  lists some properties of the product of a sequence of magma elements.*}

theorem prodN_props: 
  assumes A1: "n \<in> nat" and A2: "f : X\<times>X \<rightarrow> X" and A3: "a : succ(n) \<rightarrow> X"
  shows
  "ProdN(f,a) =  ProdsSeqN(f,a)`(n)"
  "ProdN(f,a) \<in> X"
proof -
  from A1 A2 A3 have "ProdsSeqN(f,a) : succ(n) \<rightarrow> X"
    using prod_seq_N_props by blast;
  with A1 show "ProdN(f,a) =  ProdsSeqN(f,a)`(n)" and "ProdN(f,a) \<in> X"
    using last_seq_elem apply_funtype ProdN_def by auto;
qed;

text{*A simple lemma about the product of an 1-element sequence.
  Recall that in ZF we have $1=\{ 0\}=$@{text "succ(0)"}. *}

lemma prodN_1elem: 
  assumes A1: "f : X\<times>X \<rightarrow> X" and A2: "a : 1 \<rightarrow> X"
  shows "ProdN(f,a) = a`(0)"
proof -
  let ?P = "ProdsSeqN(f,a)"
  from A1 A2 have
    "0 \<in> nat"  "f : X\<times>X \<rightarrow> X"  
    "a : succ(0) \<rightarrow> X"  "?P = ProdsSeqN(f,a)" 
    by auto;
  then have "?P`(0) = a`(0)" by (rule prod_seq_N_props);
  moreover from A1 A2 have "ProdN(f,a) =  ?P`(0)"
    using prodN_props by auto;
  ultimately show "ProdN(f,a) = a`(0)" by simp;
qed;
  

text{*The next theorem tells us what happens to the product of a sequence
  when we add one more element to it.*}

theorem prodN_append:  
  assumes A1: "n \<in> nat" and  A2: "f : X\<times>X \<rightarrow> X" and
  A3: "a : succ(n) \<rightarrow> X" and A4: "x\<in>X"
  shows 
  "ProdsSeqN(f,Append(a,x))`(n) = ProdN(f,a)"
  "ProdN(f,Append(a,x)) = f`\<langle>ProdN(f,a), x\<rangle>"
proof -
  let ?b = "Append(a,x)"
  let ?P =  "ProdsSeqN(f,?b)"
  from A1 A2 A3 A4 show thesis1: "?P`(n) = ProdN(f,a)"
    using prod_seq_N_append prodN_props by simp;
  from A1 have I: "succ(n) \<in> nat" by auto;
  moreover note A2
  moreover from A3 A4 have II: "?b : succ(succ(n)) \<rightarrow> X"
    using append_props by simp;
  moreover have III: "?P = ProdsSeqN(f,?b)" by simp;
  ultimately have 
    "\<forall>k \<in> succ(n). ?P`(succ(k)) =  f`\<langle>?P`(k), ?b`(succ(k))\<rangle>"
    by (rule prod_seq_N_props);
  with thesis1 have "?P`(succ(n)) =  f`\<langle> ProdN(f,a), ?b`(succ(n))\<rangle>" 
    by auto;
  moreover from A3 A4 have "?b`(succ(n)) = x" using append_props
    by simp;
  moreover
  from I A2 II III have "?P : succ(succ(n)) \<rightarrow> X" 
    by (rule prod_seq_N_props);
  then have "ProdN(f,?b) = ?P`(succ(n))" 
    using last_seq_elem ProdN_def by simp;
  ultimately show  "ProdN(f,Append(a,x)) = f`\<langle>ProdN(f,a), x\<rangle>" 
    by simp;
qed;

section{*Products of sequences of semigroup elements*}

text{*Semigroup is a a magma in which the binary operation is associative.
  In this section we mostly study the products of sequences of elements 
  of semigroup. The goal is to establish the fact that taking the product of 
  a sequence is distributive with respect to concatenation of sequences, 
  i.e for two sequences $a,b$ of the semigroup elements we have 
  $\prod (a\sqcup b) = (\prod a)\cdot (\prod b)$, where "$a \sqcup b$" 
  is concatenation of $a$ and $b$ ($a$@{text "++"}$b$ in Haskell notation).
  Less formally, we want to show that we can discard parantheses in 
  expressions of the form 
  $(a_0\cdot a_1\cdot .. \cdot a_n)\cdot (b_0\cdot .. \cdot b_k)$.
  *}

text{*The definition of the @{text "semigr0"} context below introduces notation
  for writing about finite sequences and semigroup products. 
  In the context we fix the carrier and
  denote it $G$. The binary operation on $G$ is called $f$. 
  All theorems proven in the context @{text "semigr0"} 
  will implicitly assume that $f$ is an associative operation on $G$.
  We will use the multiplicative notation for the semigroup operation.
  The product of a sequence $a$ is denoted $\prod a$. We will write
  $a\hookleftarrow x$ for the result of appending an element $x$ to
  the finite sequence (list) $a$. This is a bit nonstandard, 
  but I don't have a better idea for the "append" notation. Finally,
  $a\sqcup b$ will denote the concatenetion of the lists $a$ and $b$.*}

locale semigr0 =
  
  fixes G and f

  assumes assoc_assum: "f {is associative on} G"

  fixes prod (infixl "\<cdot>" 72)
  defines prod_def [simp]: "x \<cdot> y \<equiv> f`\<langle>x,y\<rangle>"

  fixes seqprod :: "i\<Rightarrow>i" ("\<Prod> _" 71)
  defines seqprod_def [simp]: "\<Prod> a \<equiv> ProdN(f,a)"

  fixes append (infix "\<hookleftarrow>" 72)
  defines append_def [simp]: "a \<hookleftarrow> x \<equiv> Append(a,x)"

  fixes concat (infixl "\<squnion>" 69)
  defines concat_def [simp]: "a \<squnion> b \<equiv> Concat(a,b)";

text{*The next lemma shows our assumption on the associativity
  of the semigroup operation in the notation defined in in the 
  @{text "semigr0"} context.*}

lemma (in semigr0) semigr_assoc: assumes "x \<in> G"  "y \<in> G"  "z \<in> G"
  shows "x\<cdot>y\<cdot>z = x\<cdot>(y\<cdot>z)"
  using prems assoc_assum IsAssociative_def by simp

text{*In the way we define associativity the assumption that
  the $f$ is associative on $G$ also implies that it is a binary
  operation on $X$. *}

lemma (in semigr0) semigr_binop: shows "f : G\<times>G \<rightarrow> G"
  using assoc_assum IsAssociative_def by simp;

text{*Lemma @{text "append_1elem"} written in the notation used in 
  the @{text "semigr0"} context.*}

lemma (in semigr0) append_1elem_nice: 
  assumes "n \<in> nat" and "a: n \<rightarrow> X" and "b : 1 \<rightarrow> X"
  shows "a \<squnion> b = a \<hookleftarrow> b`(0)"
  using prems append_1elem by simp;

text{*Lemma @{text "concat_init_last_elem"} rewritten
  in the notation used in the @{text "semigr0"} context.*}

lemma (in semigr0) concat_init_last: 
  assumes "n \<in> nat"  "k \<in> nat" and 
  "a: n \<rightarrow> X"  and "b : succ(k) \<rightarrow> X"
  shows "(a \<squnion> Init(b)) \<hookleftarrow> b`(k) = a \<squnion> b"
  using prems concat_init_last_elem by simp;

text{*Part of lemma @{text "prodN_append"} written in the
  notation used in teh @{text "semigr0"} context.*}

lemma (in semigr0) prod_append: assumes A1: "n \<in> nat" and
  A2: "a : succ(n) \<rightarrow> G" and A3: "x\<in>G"
  shows "(\<Prod> a\<hookleftarrow>x) = (\<Prod> a) \<cdot> x"
  using prems semigr_binop prodN_append by simp;

text{*The main theorem of the section: taking the product of 
  a sequence is distributive with respect to concatenation of sequences.
  The proof is by induction on the length of the second list.*}

theorem (in semigr0) prod_conc_distr: 
  assumes A1: "n \<in> nat"  "k \<in> nat" and
  A2: "a : succ(n) \<rightarrow> G"   "b: succ(k) \<rightarrow> G"
  shows "(\<Prod> a) \<cdot> (\<Prod> b) = \<Prod> (a \<squnion> b)"
proof -
  from A1 have "k \<in> nat" by simp;
  moreover have "\<forall>b \<in> succ(0) \<rightarrow> G. (\<Prod> a) \<cdot> (\<Prod> b) = \<Prod> (a \<squnion> b)"
  proof -
    { fix b assume A3: "b : succ(0) \<rightarrow> G"
      with A1 A2 have
	"succ(n) \<in> nat"  "a : succ(n) \<rightarrow> G"  "b : 1 \<rightarrow> G" 
	by auto;
      then have "a \<squnion> b = a \<hookleftarrow> b`(0)" by (rule append_1elem_nice);
      with A1 A2 A3 have "(\<Prod> a) \<cdot> (\<Prod> b) = \<Prod> (a \<squnion> b)"
	using apply_funtype prod_append semigr_binop prodN_1elem
	by simp;
    } thus ?thesis by simp;
  qed;
  moreover have "\<forall>j \<in> nat. 
    (\<forall>b \<in> succ(j) \<rightarrow> G. (\<Prod> a) \<cdot> (\<Prod> b) = \<Prod> (a \<squnion> b)) \<longrightarrow>
    (\<forall>b \<in> succ(succ(j)) \<rightarrow> G. (\<Prod> a) \<cdot> (\<Prod> b) = \<Prod> (a \<squnion> b))"
  proof -
    { fix j assume A4: "j \<in> nat" and 
      A5: "(\<forall>b \<in> succ(j) \<rightarrow> G. (\<Prod> a) \<cdot> (\<Prod> b) = \<Prod> (a \<squnion> b))"
      { fix b assume A6: "b : succ(succ(j)) \<rightarrow> G"
	let ?c = "Init(b)"
	from A4 A6 have  T: "b`(succ(j)) \<in> G" and
	  I: "?c : succ(j) \<rightarrow> G" and II: "b = ?c\<hookleftarrow>b`(succ(j))"
	  using apply_funtype init_props by auto;
	from A1 A2 A4 A6 have
	  "succ(n) \<in> nat"  "succ(j) \<in> nat"
	  "a : succ(n) \<rightarrow> G"  "b : succ(succ(j)) \<rightarrow> G"
	  by auto;
	then have III: "(a \<squnion> ?c) \<hookleftarrow> b`(succ(j)) = a \<squnion> b"
	  by (rule concat_init_last);
	from A4 I T have "(\<Prod> ?c\<hookleftarrow>b`(succ(j))) = (\<Prod> ?c) \<cdot> b`(succ(j))"
	  by (rule prod_append);
	with II have 
	  "(\<Prod> a) \<cdot> (\<Prod> b) = (\<Prod> a) \<cdot> ((\<Prod> ?c) \<cdot> b`(succ(j)))"
	  by simp;
	moreover from A1 A2 A4 T I have
	  "(\<Prod> a) \<in> G"   "(\<Prod> ?c) \<in> G"  "b`(succ(j)) \<in> G"
	  using semigr_binop prodN_props by auto;
	ultimately have 
	  "(\<Prod> a) \<cdot> (\<Prod> b) =  ((\<Prod> a) \<cdot> (\<Prod> ?c)) \<cdot> b`(succ(j))"
	  using semigr_assoc by simp;
	with A5 I have "(\<Prod> a) \<cdot> (\<Prod> b) = (\<Prod> (a \<squnion> ?c))\<cdot>b`(succ(j))"
	  by simp;
	moreover
	from A1 A2 A4 I have
	  T1: "succ(n) \<in> nat"  "succ(j) \<in> nat" and
	  "a : succ(n) \<rightarrow> G"   "?c : succ(j) \<rightarrow> G"
	  by auto;
	then have "Concat(a,?c): succ(n) #+ succ(j) \<rightarrow> G"
	  by (rule concat_props);
	with A1 A4 T have
	  "succ(n #+ j) \<in> nat"   
	  "a \<squnion> ?c : succ(succ(n #+j)) \<rightarrow> G"
	  "b`(succ(j)) \<in> G"
	  using succ_plus by auto;
	then have 
	  "(\<Prod> (a \<squnion> ?c)\<hookleftarrow>b`(succ(j))) = (\<Prod> (a \<squnion> ?c))\<cdot>b`(succ(j))"
	  by (rule prod_append);
	with III have "(\<Prod> (a \<squnion> ?c))\<cdot>b`(succ(j)) =  \<Prod> (a \<squnion> b)"
	  by simp;
	ultimately have "(\<Prod> a) \<cdot> (\<Prod> b) = \<Prod> (a \<squnion> b)"
	  by simp;
      } hence "(\<forall>b \<in> succ(succ(j)) \<rightarrow> G. (\<Prod> a) \<cdot> (\<Prod> b) = \<Prod> (a \<squnion> b))"
	by simp;
    } thus ?thesis by blast;
  qed;
  ultimately have "\<forall>b \<in> succ(k) \<rightarrow> G. (\<Prod> a) \<cdot> (\<Prod> b) = \<Prod> (a \<squnion> b)"
    by (rule Nat_ZF_1_L2);
  with A2 show "(\<Prod> a) \<cdot> (\<Prod> b) = \<Prod> (a \<squnion> b)" by simp;
qed;
    
  
end