/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "item_frame.h"

extern void
item_frame_assign(
  struct item_frame*const               o_frame,
  struct item_method*const              o_method)
{

  memset(o_frame, 0, sizeof(*o_frame));

  (*o_frame).m_rect[top_left_x]= 0.0;
  (*o_frame).m_rect[top_left_y]= 0.0;
  (*o_frame).m_rect[top_right_x]= 100.0;
  (*o_frame).m_rect[top_right_y]= 0.0;
  (*o_frame).m_rect[bottom_right_x]= 100.0;
  (*o_frame).m_rect[bottom_right_y]= 24.0;
  (*o_frame).m_rect[bottom_left_x]= 0.0;
  (*o_frame).m_rect[bottom_left_y]= 24.0;

  (*o_frame).m_border.m_style= line_style_solid;
  (*o_frame).m_border.m_thickness= 1.0;
  (*o_frame).m_border.m_color.alpha= 1.0;
  (*o_frame).m_fill.m_color.alpha= 1.0;

  ITEM_METHOD_ASSIGN(o_method, frame);

  return;
}

extern void
item_frame_discharge(
  struct item_frame*const                io_frame)
{
  memset(io_frame, 0, sizeof(*io_frame));
}

extern void
item_frame_draw(
  struct item_frame*const               i_frame,
  struct item_draw_context const*const  i_ctx)
{

  do
  {

    if ((*i_frame).m_fill.m_active)
    {
      cairo_set_source_rgba(
        (*i_ctx).m_cr,
        (*i_frame).m_fill.m_color.red,
        (*i_frame).m_fill.m_color.green,
        (*i_frame).m_fill.m_color.blue,
        (*i_frame).m_fill.m_color.alpha);

      cairo_move_to((*i_ctx).m_cr, (*i_frame).m_rect[top_left_x], (*i_frame).m_rect[top_left_y]);
      cairo_line_to((*i_ctx).m_cr, (*i_frame).m_rect[top_right_x], (*i_frame).m_rect[top_right_y]);
      cairo_line_to((*i_ctx).m_cr, (*i_frame).m_rect[bottom_right_x], (*i_frame).m_rect[bottom_right_y]);
      cairo_line_to((*i_ctx).m_cr, (*i_frame).m_rect[bottom_left_x], (*i_frame).m_rect[bottom_left_y]);
      cairo_close_path((*i_ctx).m_cr);
      cairo_fill((*i_ctx).m_cr);

    }

    item_common_cairo_dash_set((*i_ctx).m_cr, (*i_frame).m_border.m_style);
    cairo_set_line_width((*i_ctx).m_cr, (*i_frame).m_border.m_thickness);
    cairo_set_source_rgba(
      (*i_ctx).m_cr,
      (*i_frame).m_border.m_color.red,
      (*i_frame).m_border.m_color.green,
      (*i_frame).m_border.m_color.blue,
      (*i_frame).m_border.m_color.alpha);

    cairo_move_to((*i_ctx).m_cr, (*i_frame).m_rect[top_left_x], (*i_frame).m_rect[top_left_y]);
    cairo_line_to((*i_ctx).m_cr, (*i_frame).m_rect[top_right_x], (*i_frame).m_rect[top_right_y]);
    cairo_line_to((*i_ctx).m_cr, (*i_frame).m_rect[bottom_right_x], (*i_frame).m_rect[bottom_right_y]);
    cairo_line_to((*i_ctx).m_cr, (*i_frame).m_rect[bottom_left_x], (*i_frame).m_rect[bottom_left_y]);
    cairo_close_path((*i_ctx).m_cr);
    cairo_stroke((*i_ctx).m_cr);

    if (0 == (*i_ctx).m_selected)
    {
      break;
    }

    item_common_rectvect_draw_grips((*i_ctx).m_cr, (double*)i_frame);

  }while(0);

  return;
}

extern void
item_frame_get_bbox(
  struct geom_rect*const                o_bbox,
  void const*const                      i_object)
{
  geom_rectvect_get_bounding_box(o_bbox, (double*)i_object);
  return;
}

extern void
item_frame_in_event(
  enum event_type*const                 o_type,
  struct item_frame const*const         i_frame,
  struct item_in_event const*const      i_ctx)
{

  item_common_rectvect_in_event(o_type, i_ctx, (double*)i_frame);

}

extern void
item_frame_move(
  struct item_frame*const               io_frame,
  struct geom_point const*const         i_event)
{

  item_common_rectvect_move((double*)io_frame, i_event);

  return;
}

extern void
item_frame_normalize(
  struct item_frame*const               io_frame,
  double const                          i_bound_width,
  double const                          i_bound_height)
{
  return;
}

extern int
item_frame_read(
  GError**                              o_error,
  struct item_frame*const               io_frame,
  struct xml *const                     i_xml)
{
  GError*                               l_error;
  int                                   l_exit;
  int                                   l_found;
  GList*                                l_node;
  enum element_tag_type                 l_type;
  char const*                           l_value;
  struct xml*                           l_xml;

  l_error= 0;
  l_exit= 0;

  l_node= (*i_xml).m_children;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_xml= (*l_node).data;

    item_common_lookup(&l_found, &l_type, (*l_xml).m_tag);
    g_strstrip((*l_xml).m_string->str);
    l_value= (*l_xml).m_string->str;

    if (l_found)
    {
      switch(l_type)
      {
        case element_tag_rect_v0:
          (*io_frame).m_rect[0]= xml_convert_double(l_value);
          break;
        case element_tag_rect_v1:
          (*io_frame).m_rect[1]= xml_convert_double(l_value);
          break;
        case element_tag_rect_v2:
          (*io_frame).m_rect[2]= xml_convert_double(l_value);
          break;
        case element_tag_rect_v3:
          (*io_frame).m_rect[3]= xml_convert_double(l_value);
          break;
        case element_tag_rect_v4:
          (*io_frame).m_rect[4]= xml_convert_double(l_value);
          break;
        case element_tag_rect_v5:
          (*io_frame).m_rect[5]= xml_convert_double(l_value);
          break;
        case element_tag_rect_v6:
          (*io_frame).m_rect[6]= xml_convert_double(l_value);
          break;
        case element_tag_rect_v7:
          (*io_frame).m_rect[7]= xml_convert_double(l_value);
          break;
        case element_tag_border:
          l_exit= item_common_read_border(&l_error, &(*io_frame).m_border, l_xml);
          break;
        case element_tag_fill:
          l_exit= item_common_read_fill(&l_error, &(*io_frame).m_fill, l_xml);
          break;
        case element_tag_rotate:
          l_value= xml_lookup_attribute_value(l_xml, "on");
          (*io_frame).m_rotate_shear= xml_convert_boolean(l_value);
          break;
        default:
          break;
      }
    }

    l_node= (*l_node).next;

  }while(1);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern void
item_frame_resize(
  struct item_frame*const               io_frame,
  struct item_resize_event const*const  i_ctx)
{

  item_common_rectvect_resize((double*)io_frame, i_ctx, (*io_frame).m_rotate_shear);

  return;
}

extern int
item_frame_write(
  GError**                              o_error,
  FILE*const                            io_fp,
  struct item_frame const*const         i_frame)
{
  char                                  l_buf[G_ASCII_DTOSTR_BUF_SIZE+1];
  int                                   l_exit;

  l_exit= 0;

  fprintf(io_fp, "\t\t\t<item name=\"frame\">\n");
  fprintf(io_fp, "\t\t\t\t<rect_v0>%s</rect_v0>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_frame).m_rect[0]));
  fprintf(io_fp, "\t\t\t\t<rect_v1>%s</rect_v1>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_frame).m_rect[1]));
  fprintf(io_fp, "\t\t\t\t<rect_v2>%s</rect_v2>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_frame).m_rect[2]));
  fprintf(io_fp, "\t\t\t\t<rect_v3>%s</rect_v3>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_frame).m_rect[3]));
  fprintf(io_fp, "\t\t\t\t<rect_v4>%s</rect_v4>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_frame).m_rect[4]));
  fprintf(io_fp, "\t\t\t\t<rect_v5>%s</rect_v5>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_frame).m_rect[5]));
  fprintf(io_fp, "\t\t\t\t<rect_v6>%s</rect_v6>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_frame).m_rect[6]));
  fprintf(io_fp, "\t\t\t\t<rect_v7>%s</rect_v7>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_frame).m_rect[7]));
  item_common_write_border(o_error, io_fp, &(*i_frame).m_border);
  item_common_write_fill(o_error, io_fp, &(*i_frame).m_fill);
  fprintf(io_fp, "\t\t\t\t<rotate on=\"%s\"/>\n", ((*i_frame).m_rotate_shear ? "true" : "false"));
  fprintf(io_fp, "\t\t\t</item>\n");

  return l_exit;
}
