/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "item_text.h"

extern void
item_text_assign(
  struct item_text*const                o_text,
  struct item_method*const              o_method)
{

  memset(o_text, 0, sizeof(*o_text));
  (*o_text).m_pos_y1= 10.0;
  (*o_text).m_pos_x2= 100.0;
  (*o_text).m_pos_y2= 10.0;
  (*o_text).m_font= pango_font_description_new();
  (*o_text).m_color.alpha= 1.0;

  ITEM_METHOD_ASSIGN(o_method, text);

  return;
}

extern void
item_text_discharge(
  struct item_text*const                io_text)
{
  if ((*io_text).m_font)
  {
    pango_font_description_free((*io_text).m_font);
  }

  if ((*io_text).m_text)
  {
    g_free((*io_text).m_text);
  }

  if ((*io_text).m_value)
  {
    g_free((*io_text).m_value);
  }

  memset(io_text, 0, sizeof(*io_text));

  return;
}

extern void
item_text_draw(
  struct item_text*                     i_text,
  struct item_draw_context const*const  i_ctx)
{
  double                                l_length;
  PangoLayout*                          l_layout;
  double                                l_radians;
  char*                                 l_text;
  
  l_layout= 0;

  do
  {

    cairo_save((*i_ctx).m_cr);

    cairo_set_source_rgba(
      (*i_ctx).m_cr,
      (*i_text).m_color.red,
      (*i_text).m_color.green,
      (*i_text).m_color.blue,
      (*i_text).m_color.alpha);

    l_layout= pango_cairo_create_layout((*i_ctx).m_cr);

    if (item_draw_design == (*i_ctx).m_option.m_type)
    {
      if ((*i_text).m_text && (*i_text).m_text[0])
      {
        l_text= (*i_text).m_text;
      }
      else
      {
        l_text= "default";
      }
    }
    else
    {
      l_text= (*i_text).m_value;
    }

    if (l_text)
    {
      if (0 == (*i_text).m_autowrap)
      {
        pango_layout_set_width(l_layout, -1);
      }
      else
      {
        l_length= geom_line_length((struct geom_line*)i_text);
        pango_layout_set_width(l_layout, (PANGO_SCALE*l_length));
        pango_layout_set_wrap(l_layout, PANGO_WRAP_WORD); 
      }
      pango_layout_set_markup(l_layout, l_text, -1);
    }

    pango_layout_set_font_description(l_layout, (*i_text).m_font);
    cairo_translate((*i_ctx).m_cr, (*i_text).m_pos_x1, (*i_text).m_pos_y1);
    if ((*i_text).m_rotate)
    {
      l_radians= geom_deg2rad((*i_text).m_rotation);
      cairo_rotate((*i_ctx).m_cr, l_radians);
    }
    pango_cairo_update_layout((*i_ctx).m_cr, l_layout);
    pango_cairo_show_layout((*i_ctx).m_cr, l_layout);
    cairo_restore((*i_ctx).m_cr);

    if (0 == (*i_ctx).m_selected)
    {
      if ((*i_ctx).m_option.m_highlight &&
        (item_draw_design == (*i_ctx).m_option.m_type))
      {
        cairo_save((*i_ctx).m_cr);
        cairo_set_source_rgb((*i_ctx).m_cr, 0.50, 0.50, 0.50);
        item_common_cairo_dash_set((*i_ctx).m_cr, line_style_solid);
        cairo_set_line_width((*i_ctx).m_cr, 1.0);
        cairo_move_to((*i_ctx).m_cr, (*i_text).m_pos_x1, (*i_text).m_pos_y1);
        cairo_line_to((*i_ctx).m_cr, (*i_text).m_pos_x2, (*i_text).m_pos_y2);
        cairo_stroke((*i_ctx).m_cr);
        cairo_restore((*i_ctx).m_cr);
      }
      break;
    }

    if (item_draw_print == (*i_ctx).m_option.m_type)
    {
      break;
    }

    cairo_save((*i_ctx).m_cr);
    cairo_set_source_rgb((*i_ctx).m_cr, 0.0, 0.0, 0.0);
    item_common_cairo_dash_set((*i_ctx).m_cr, line_style_solid);
    cairo_set_line_width((*i_ctx).m_cr, 1.0);
    cairo_move_to((*i_ctx).m_cr, (*i_text).m_pos_x1, (*i_text).m_pos_y1);
    cairo_line_to((*i_ctx).m_cr, (*i_text).m_pos_x2, (*i_text).m_pos_y2);
    cairo_stroke((*i_ctx).m_cr);
    cairo_restore((*i_ctx).m_cr);

    cairo_set_source_rgb((*i_ctx).m_cr, 0.0, 1.0, 0.0);

    cairo_rectangle(
      (*i_ctx).m_cr,
      (*i_text).m_pos_x1-4,
      (*i_text).m_pos_y1-4,
      8,
      8);
    cairo_fill((*i_ctx).m_cr);

    cairo_rectangle(
      (*i_ctx).m_cr,
      (*i_text).m_pos_x2-4,
      (*i_text).m_pos_y2-4,
      8,
      8);
    cairo_fill((*i_ctx).m_cr);

  }while(0);

  if (l_layout)
  {
    g_object_unref(l_layout);
  }

  return;
}

extern void
item_text_get_bbox(
  struct geom_rect*const                o_bbox,
  void const*const                      i_object)
{
  struct item_text const*               l_text;

  l_text= (struct item_text const*)i_object;
  (*o_bbox).m_pos_x= (*l_text).m_pos_x1;
  (*o_bbox).m_pos_y= (*l_text).m_pos_y1;
  (*o_bbox).m_len_x= (*l_text).m_pos_x2 - (*l_text).m_pos_x1;
  (*o_bbox).m_len_y= 1.0;

  return;
}

static int
item_text_in_event_grips(
  enum event_type*const                 o_type,
  struct item_text const*const          i_text,
  struct item_in_event const*const      i_ctx)
{
  int                                   l_hit;

  l_hit= 0;

  do
  {

    l_hit= item_common_event_inside(
      i_ctx,
      (*i_text).m_pos_x1-4,
      (*i_text).m_pos_y1-4,
      8,
      8);

    if (l_hit)
    {
      if (0 == (*i_text).m_rotate)
      {
        (*o_type)= event_resize_west;
      }
      else
      {
        (*o_type)= event_resize_north_west;
      }
      break;
    }
    
    l_hit= item_common_event_inside(
      i_ctx,
      (*i_text).m_pos_x2-4,
      (*i_text).m_pos_y2-4,
      8,
      8);

    if (l_hit)
    {
      if (0 == (*i_text).m_rotate)
      {
        (*o_type)= event_resize_east;
      }
      else
      {
        (*o_type)= event_resize_south_east;
      }
      break;
    }

  }while(0);

  return l_hit;
}

extern void
item_text_in_event(
  enum event_type*const                 o_type,
  struct item_text const*const          i_text,
  struct item_in_event const*const      i_ctx)
{
  int                                   l_hit;

  l_hit= 0;

  do
  {

    (*o_type)= event_none;

    l_hit= item_common_line_in_event(i_ctx, (struct geom_line*)i_text);

    if (0 == l_hit)
    {
      break;
    }

    if ((*i_ctx).m_selected)
    {

      l_hit= item_text_in_event_grips(o_type, i_text, i_ctx);

      if (l_hit)
      {
        break;
      }
    }

    (*o_type)= event_move;

  }while(0);

  return;
}

extern void
item_text_move(
  struct item_text*const                io_text,
  struct geom_point const*const         i_event)
{

  (*io_text).m_pos_x1+= (*i_event).m_pos_x;
  (*io_text).m_pos_y1+= (*i_event).m_pos_y;
  (*io_text).m_pos_x2+= (*i_event).m_pos_x;
  (*io_text).m_pos_y2+= (*i_event).m_pos_y;

  return;
}

extern void
item_text_normalize(
  struct item_text*const                io_text,
  double const                          i_bound_width,
  double const                          i_bound_height)
{
  gdouble                               l_swap;

  do
  {

    if ((*io_text).m_rotate)
    {
      break;
    }

    (*io_text).m_rotation= 0.00;

    if ((*io_text).m_pos_x1 > (*io_text).m_pos_x2)
    {
      l_swap= (*io_text).m_pos_x1;
      (*io_text).m_pos_x1= (*io_text).m_pos_x2;
      (*io_text).m_pos_x2= l_swap;
    }

    if ((*io_text).m_pos_y1 > (*io_text).m_pos_y2)
    {
      l_swap= (*io_text).m_pos_y1;
      (*io_text).m_pos_y1= (*io_text).m_pos_y2;
      (*io_text).m_pos_y2= l_swap;
    }

    if (0 > (int)(*io_text).m_pos_x2)
    {
      l_swap= (*io_text).m_pos_x2 - (*io_text).m_pos_x1;
      (*io_text).m_pos_x2= 2.0;
      (*io_text).m_pos_x1= (*io_text).m_pos_x2 - l_swap;
    }

    if (0 > (int)(*io_text).m_pos_y2)
    {
      l_swap= (*io_text).m_pos_y2 - (*io_text).m_pos_y1;
      (*io_text).m_pos_y1= 2.0 - l_swap;
      (*io_text).m_pos_y2= (*io_text).m_pos_y1 + l_swap;
    }

    if (i_bound_width < (int)(*io_text).m_pos_x1)
    {
      l_swap= (*io_text).m_pos_x2 - (*io_text).m_pos_x1;
      (*io_text).m_pos_x1= i_bound_width - 2.0;
      (*io_text).m_pos_x2= (*io_text).m_pos_x1 + l_swap;
    }

    if (i_bound_height < (int)(*io_text).m_pos_y1)
    {
      l_swap= (*io_text).m_pos_y2 - (*io_text).m_pos_y1;
      (*io_text).m_pos_y1= i_bound_height - 2.0;
      (*io_text).m_pos_y2= (*io_text).m_pos_y1 + l_swap;
    }

  }while(0);

  return;
}

extern int
item_text_read(
  GError**                              o_error,
  struct item_text*const                io_text,
  struct xml *const                     i_xml)
{
  GError*                               l_error;
  int                                   l_exit;
  int                                   l_found;
  GList*                                l_node;
  int                                   l_rc;
  GdkRGBA                               l_rgba;
  enum element_tag_type                 l_type;
  char const*                           l_value;
  struct xml*                           l_xml;

  l_error= 0;
  l_exit= 0;

  l_node= (*i_xml).m_children;

  do
  {

    if (0 == l_node)
    {
      break;
    }

    l_xml= (*l_node).data;

    item_common_lookup(&l_found, &l_type, (*l_xml).m_tag);
    g_strstrip((*l_xml).m_string->str);
    l_value= (*l_xml).m_string->str;

    if (l_found)
    {
      switch(l_type)
      {
        case element_tag_pos_x1:
          (*io_text).m_pos_x1= xml_convert_double(l_value);
          break;
        case element_tag_pos_y1:
          (*io_text).m_pos_y1= xml_convert_double(l_value);
          break;
        case element_tag_pos_x2:
          (*io_text).m_pos_x2= xml_convert_double(l_value);
          break;
        case element_tag_pos_y2:
          (*io_text).m_pos_y2= xml_convert_double(l_value);
          break;
        case element_tag_font:
          (*io_text).m_font= pango_font_description_from_string(l_value);
          break;
        case element_tag_color:
          l_rc= gdk_rgba_parse(&l_rgba, l_value);
          if (l_rc)
          {
            (*io_text).m_color= l_rgba;
          }
          break;
        case element_tag_rotate:
          (*io_text).m_rotate= xml_convert_boolean(xml_lookup_attribute_value(l_xml, "on"));
          (*io_text).m_rotation= xml_convert_ulong(l_value);
          break;
        case element_tag_text:
          (*io_text).m_text= g_strdup((*l_xml).m_string->str);
          break;
        default:
          break;
      }
    }

    l_node= (*l_node).next;

  }while(1);

  if (l_error)
  {
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

extern void
item_text_resize(
  struct item_text*const                io_text,
  struct item_resize_event const*const  i_ctx)
{
  gdouble                               l_delta_x;
  gdouble                               l_delta_y;

  switch((*i_ctx).m_type)
  {
    case event_resize_east:
      (*io_text).m_pos_x2+= (*i_ctx).m_event.m_pos_x;
      break;
    case event_resize_north:
      (*io_text).m_pos_y1+= (*i_ctx).m_event.m_pos_y;
      break;
    case event_resize_north_east:
      (*io_text).m_pos_y1+= (*i_ctx).m_event.m_pos_y;
      (*io_text).m_pos_x2+= (*i_ctx).m_event.m_pos_x;
      break;
    case event_resize_north_west:
      (*io_text).m_pos_y1+= (*i_ctx).m_event.m_pos_y;
      (*io_text).m_pos_x1+= (*i_ctx).m_event.m_pos_x;
      break;
    case event_resize_south:
      (*io_text).m_pos_y2+= (*i_ctx).m_event.m_pos_y;
      break;
    case event_resize_south_east:
      (*io_text).m_pos_y2+= (*i_ctx).m_event.m_pos_y;
      (*io_text).m_pos_x2+= (*i_ctx).m_event.m_pos_x;
      break;
    case event_resize_south_west:
      (*io_text).m_pos_y2+= (*i_ctx).m_event.m_pos_y;
      (*io_text).m_pos_x1+= (*i_ctx).m_event.m_pos_x;
      break;
    case event_resize_west:
      (*io_text).m_pos_x1+= (*i_ctx).m_event.m_pos_x;
      break;
    default:
      break;
  }

  l_delta_x= (*io_text).m_pos_x2 - (*io_text).m_pos_x1;
  l_delta_y= (*io_text).m_pos_y2 - (*io_text).m_pos_y1;
  (*io_text).m_rotation= atan2(l_delta_y, l_delta_x) * 180 / G_PI;

  return;
}

extern int
item_text_write(
  GError**                              o_error,
  FILE*const                            io_fp,
  struct item_text const*const          i_text)
{
  char                                  l_buf[G_ASCII_DTOSTR_BUF_SIZE+1];
  int                                   l_exit;
  char*                                 l_text;

  l_exit= 0;

  fprintf(io_fp, "\t\t\t<item name=\"text\">\n");
  fprintf(io_fp, "\t\t\t\t<pos_x1>%s</pos_x1>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_text).m_pos_x1));
  fprintf(io_fp, "\t\t\t\t<pos_y1>%s</pos_y1>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_text).m_pos_y1));
  fprintf(io_fp, "\t\t\t\t<pos_x2>%s</pos_x2>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_text).m_pos_x2));
  fprintf(io_fp, "\t\t\t\t<pos_y2>%s</pos_y2>\n", g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_text).m_pos_y2));
  l_text= pango_font_description_to_string((*i_text).m_font);
  fprintf(io_fp, "\t\t\t\t<font>%s</font>\n", l_text);
  g_free(l_text);
  l_text= gdk_rgba_to_string(&(*i_text).m_color);
  fprintf(io_fp, "\t\t\t\t<color>%s</color>\n", l_text);
  g_free(l_text);
  fprintf(io_fp, "\t\t\t\t<rotate on=\"%s\">%s</rotate>\n", ((*i_text).m_rotate ? "true" : "false"),g_ascii_dtostr(l_buf, sizeof(l_buf), (*i_text).m_rotation));
  fprintf(io_fp, "\t\t\t\t<autowrap on=\"%s\"/>\n", ((*i_text).m_autowrap ? "true" : "false"));

  if ((*i_text).m_text && (*i_text).m_text[0])
  {
    fprintf(io_fp, "\t\t\t\t<text>%s</text>", (*i_text).m_text);
  }

  fprintf(io_fp, "\t\t\t</item>\n");

  return l_exit;
}
