/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler 
*/
/*
Copyright (c) 2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "report_util.h"
#include "export_csv.h"
#include "export_spec.h"
#include "label_pdf.h"
#include "label_spec.h"
#include "layout_pdf.h"
#include "layout_spec.h"

extern enum report_style
report_util_convert_text_to_style(
  char const*                           i_style)
{
  enum report_style                     l_style;
  int                                   l_rc;

  l_style= style_unassigned;

  do
  {

    l_rc= strcmp("layout", i_style);

    if (0 == l_rc)
    {
      l_style= style_layout;
      break;
    }

    l_rc= strcmp("export", i_style);

    if (0 == l_rc)
    {
      l_style= style_export;
      break;
    }
    
    l_rc= strcmp("label", i_style);

    if (0 == l_rc)
    {
      l_style= style_label;
      break;
    }

  }while(0);

  return l_style;
}


extern char*
report_util_get_report_filename(
  char const*                           i_reportfile)
{
  char*                                 l_ptr;
  int                                   l_rc;
  char*                                 l_reportfile;
  unsigned                              l_size;

  l_size= strlen(i_reportfile);
  l_size+= 6;
  l_reportfile= (char*)g_malloc0(l_size);
  g_strlcpy(l_reportfile, i_reportfile, l_size);

  l_ptr= strrchr(l_reportfile, '.');

  if (0 == l_ptr)
  {
    g_strlcat(l_reportfile, ".irpt", l_size);
  }
  else
  {
    l_rc= strcmp(l_ptr, ".irpt");
    if (l_rc)
    {
      g_strlcat(l_reportfile, ".irpt", l_size);
    }
  }

  return l_reportfile;
}

extern int
report_util_open_report(
  GError**                              o_error,
  enum report_style*                    o_style,
  struct xml**                          o_xml,
  char const*                           i_reportfile)
{
  GError*                               l_error;
  int                                   l_exit;
  int                                   l_rc;
  struct xml*                           l_xml;
  char const*                           l_value;

  (*o_style)= style_unassigned;
  (*o_xml)= 0;
  l_error= 0;
  l_exit= 0;
  l_xml= 0;

  do
  {

    l_exit= xml_parse(&l_error, &l_xml, i_reportfile);

    if (l_exit)
    {
      break;
    }

    l_rc= strcmp("iwrite", (*l_xml).m_tag);

    if (l_rc)
    {
      l_error= g_error_new(
        REPORT,
        REPORT_XML_NOT_IWRITE,
        "Invalid XML. Can not find root ('iwrite') element ('%s')",
        i_reportfile);
      _error_log(l_error);
      l_exit= -1;
      break;
    }

    l_value= xml_lookup_attribute_value(l_xml, "style");

    if (l_value)
    {  
      (*o_style)= report_util_convert_text_to_style(l_value);
    }

    if (style_unassigned == (*o_style))
    {
      l_error= g_error_new(
        REPORT,
        REPORT_XML_STYLE_INVALID,
        "Invalid report style ('%s')",
        i_reportfile);
      _error_log(l_error);
      l_exit= -1;
      break;
    } 

    (*o_xml)= l_xml;

  }while(0);

  if (l_error)
  {
    xml_free(l_xml);
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}

static int
report_util_run_export(
  GError**                              o_error,
  struct xml*                           i_xml,
  char const*                           i_password,
  struct goop const*const               i_goop)
{
  int                                   l_exit;
  struct exports_spec                   l_export;

  export_spec_assign(&l_export);

  do
  {

    l_exit= export_spec_read(o_error, &l_export, i_xml);

    if (l_exit)
    {
      break;
    }

    l_export.m_aspect.m_password= g_strdup(i_password);
    l_exit= export_csv_run(&l_export, i_goop);

  }while(0);

  export_spec_discharge(&l_export);

  return l_exit;
}

static int
report_util_run_label(
  GError**                              o_error,
  struct xml*                           i_xml,
  char const*                           i_password,
  struct goop const*const               i_goop)
{
  int                                   l_exit;
  struct label_spec                     l_label;

  label_spec_assign(&l_label);

  do
  {

    l_exit= label_spec_read(o_error, &l_label, i_xml);

    if (l_exit)
    {
      break;
    }

    label_paper_convert_to_points(&l_label.m_paper);
    l_label.m_aspect.m_password= g_strdup(i_password);
    l_exit= label_pdf_run(&l_label, i_goop);

  }while(0);

  label_spec_discharge(&l_label);

  return l_exit;
}

static int
report_util_run_layout(
  GError**                              o_error,
  struct xml*                           i_xml,
  char const*                           i_password,
  struct goop const*const               i_goop)
{
  int                                   l_exit;
  struct layout_spec                    l_layout;

  layout_spec_assign(&l_layout);

  do
  {

    l_exit= layout_spec_read(o_error, &l_layout, i_xml);

    if (l_exit)
    {
      break;
    }

    paper_convert_to_points(&l_layout.m_paper);
    l_layout.m_aspect.m_password= g_strdup(i_password);
    l_exit= layout_pdf_run(&l_layout, i_goop);

  }while(0);

  layout_spec_discharge(&l_layout);

  return l_exit;
}

extern int
report_util_run(
  GError**                              o_error,
  char const*                           i_reportfile,
  char const*                           i_password,
  struct goop const*const               i_goop)
{
  int                                   l_exit;
  char*                                 l_reportfile;
  enum report_style                     l_style;
  struct xml*                           l_xml;

  l_exit= 0;
  l_reportfile= report_util_get_report_filename(i_reportfile);
  l_xml= 0;

  do
  {

    l_exit= report_util_open_report(o_error, &l_style, &l_xml, l_reportfile);

    if (l_exit)
    {
      break;
    }

    switch(l_style)
    {
      case style_export:
        l_exit= report_util_run_export(o_error, l_xml, i_password, i_goop);
        break;
      case style_label:
        l_exit= report_util_run_label(o_error, l_xml, i_password, i_goop);
        break;
      case style_layout:
        l_exit= report_util_run_layout(o_error, l_xml, i_password, i_goop);
        break;
      default:
        /*caught in report_util_open_report */
        break;
    }

  }while(0);

  g_free(l_reportfile);
  xml_free(l_xml);

  return l_exit;
}
