/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler 
*/
/*
Copyright (c) 2016, iwrite authors
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "xml.h"
#include "error.h"

struct parse_data
{
  struct xml*                           m_root;
  GList*                                m_stack;
};

extern gboolean
xml_convert_boolean(
  char const*                           i_value)
{
  gboolean                              l_bool;

  l_bool= 0;

  if (i_value)
  {
    l_bool= (0 == strcmp("true", i_value));
  }

  return l_bool;
}

extern gdouble
xml_convert_double(
  char const*                           i_value)
{
  return g_ascii_strtod(i_value, 0);
}

extern guint64
xml_convert_ulong(
  char const*                           i_value)
{
  return g_ascii_strtoull(i_value, 0, 10);
}


extern char const*
xml_lookup_attribute_value(
  struct xml*const                      io_xml,
  char const*                           i_tag)
{
  int                                   l_rc;
  unsigned                              l_slot;
  char const*                           l_value;

  l_value= 0;

  for (l_slot= 0; (*io_xml).m_attr_slots > l_slot; l_slot++)
  {
    l_rc= strcmp(i_tag, (*io_xml).m_attr[l_slot].m_tag);
    if (0 == l_rc)
    {
      l_value= (*io_xml).m_attr[l_slot].m_value;
      break;
    }
  }

  return l_value;
}

extern void
xml_free(
  struct xml*const                      io_xml)
{
  GList*                                l_next;
  unsigned                              l_slot;
  struct xml*                           l_xml;

  do
  {

    if (0 == io_xml)
    {
      break;
    }

    l_next= (*io_xml).m_children;

    do
    {

      if (0 == l_next)
      {
        break;
      }

      l_xml= (struct xml*)(*l_next).data;
      xml_free(l_xml);
      l_next= (*l_next).next;

    }while(1);

    if ((*io_xml).m_string)
    {
      g_string_free((*io_xml).m_string, 1);
    }

    for (l_slot= 0; (*io_xml).m_attr_slots > l_slot; l_slot++)
    {
      g_free((*io_xml).m_attr[l_slot].m_tag);
      g_free((*io_xml).m_attr[l_slot].m_value);
    }

    g_free((*io_xml).m_attr);

    if ((*io_xml).m_children)
    {
      g_list_free((*io_xml).m_children);
    }

    g_free(io_xml);

  }while(0);

  return;
}

static void
xml_start(
  GMarkupParseContext*                  i_ctx,
  const gchar*                          i_tag,
  const gchar**                         i_names,
  const gchar**                         i_values,
  gpointer                              i_user_data,
  GError**                              io_error)
{
  struct xml*                           l_child;
  GList*                                l_last;
  struct parse_data*                    l_parse;
  unsigned                              l_slot;
  unsigned                              l_slots;
  struct xml*                           l_xml;

  l_parse= (struct parse_data*)i_user_data;
  l_xml= g_new0(struct xml, 1);

  (*l_xml).m_tag= g_strdup(i_tag);
  (*l_xml).m_string= g_string_new(0);

  for (l_slots= 0; i_names[l_slots]; l_slots++);

  if (l_slots)
  {

    (*l_xml).m_attr= g_new0(struct xml_attribute, l_slots);
    (*l_xml).m_attr_slots= l_slots;

    for (l_slot= 0; l_slots > l_slot; l_slot++)
    {
      (*l_xml).m_attr[l_slot].m_tag= g_strdup(i_names[l_slot]);
      g_strstrip((*l_xml).m_attr[l_slot].m_tag);
      (*l_xml).m_attr[l_slot].m_value= g_strdup(i_values[l_slot]);
      g_strstrip((*l_xml).m_attr[l_slot].m_value);
    }

  }

  do
  {

    if ((*l_parse).m_root)
    {
      l_last= g_list_last((*l_parse).m_stack);

      if (0 == l_last)
      {
        (*io_error)= g_error_new(
          GLOBAL,
          GLOBAL_XML_INVALID,
          "XML contains more than one root ('%s') element",
          i_tag); 
        _error_log((*io_error));
        break;        
      }

      l_child= l_last->data;
      (*l_child).m_children= g_list_append((*l_child).m_children, l_xml);
    }
    else
    {
      (*l_parse).m_root= l_xml;
    }

    (*l_parse).m_stack= g_list_append((*l_parse).m_stack, l_xml);

  }while(0);

  return;
}

static void
xml_text(
  GMarkupParseContext*                  i_ctx,
  const gchar*                          text,
  gsize                                 text_len,
  gpointer                              i_user_data,
  GError**                              io_error)
{
  struct parse_data*                    l_parse;
  GList*                                l_last;
  struct xml*                           l_xml;

  l_parse= (struct parse_data*)i_user_data;

  l_last= g_list_last((*l_parse).m_stack);
  l_xml= (struct xml*)l_last->data;

  g_string_append_len((*l_xml).m_string, text, text_len);

  return;
}

static void
xml_end(
  GMarkupParseContext*                  i_ctx,
  const gchar*                          i_tag,
  gpointer                              i_user_data,
  GError**                              io_error)
{
  GList*                                l_last;
  struct parse_data*                    l_parse;
  struct xml*                           l_xml;

  l_parse= (struct parse_data*)i_user_data;
  l_last= g_list_last((*l_parse).m_stack);
  l_xml= l_last->data;
  (*l_parse).m_stack= g_list_remove((*l_parse).m_stack, l_xml);

  return;
}

extern int
xml_parse(
  GError**                              o_error,
  struct xml**                          o_xml,
  char const*                           i_filename)
{
  static GMarkupParser                  l_parser=
  {
    xml_start,
    xml_end,
    xml_text,
    0,
    0
  };
  gchar*                                l_content;
  GMarkupParseContext*                  l_ctx;
  struct parse_data                     l_data;
  GError*                               l_error;
  int                                   l_exit;
  gsize                                 l_length;
  gboolean                              l_rc;

  (*o_xml)= 0;

  memset(&l_data, 0, sizeof(l_data));
  l_error= 0;
  l_exit= 0;

  do
  {

    l_rc= g_file_get_contents(i_filename, &l_content, &l_length, &l_error); 

    if (0 == l_rc)
    {
      l_exit= -1;
      _error_log(l_error);
      break;
    }

    l_ctx= g_markup_parse_context_new(
      &l_parser,
      G_MARKUP_PREFIX_ERROR_POSITION,
      &l_data,
      0);

    l_rc= g_markup_parse_context_parse(
      l_ctx,
      l_content,
      l_length,
      &l_error);

    if (0 == l_rc)
    {
      l_exit= -1;
      _error_log(l_error);
      break;
    }

    l_rc= g_markup_parse_context_end_parse(l_ctx, &l_error);

    if (0 == l_rc)
    {
      l_exit= -1;
      _error_log(l_error);
      break;
    }

    (*o_xml)= l_data.m_root;

  }while(0);

  g_free(l_content);

  if (l_ctx)
  {
    g_markup_parse_context_free(l_ctx);
  }

  if (l_error)
  {
    xml_free(l_data.m_root); 
    g_propagate_error(o_error, l_error);
  }

  return l_exit;
}
