/* 
vim:expandtab:softtabstop=2:tabstop=2:shiftwidth=2:nowrap:ruler 
*/
/*
Copyright (c) 2015-2016, iwrite authors 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright notice,
this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
notice, this list of conditions and the following disclaimer in the
documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include "database_session.h"

enum
{
  RESPONSE_TEST=1,
  RESPONSE_NEW=2,
  RESPONSE_DELETE=3
};

struct database_dialog
{
  GtkEntry*                             m_tag_entry;
  GtkComboBoxText*                      m_tag_comboboxtext;
  GtkComboBoxText*                      m_driver_comboboxtext;
  GtkEntry*                             m_database_entry;
  GtkEntry*                             m_hostname_entry;
  GtkEntry*                             m_ipaddress_entry;
  GtkEntry*                             m_port_entry;
  GtkEntry*                             m_username_entry;
  GtkCheckButton*                       m_prompt_checkbutton;
  GtkButton*                            m_test_button;
  GtkButton*                            m_new_button;
  GtkButton*                            m_save_button;
  GtkButton*                            m_delete_button;
  struct driver_config                  m_driver;
  struct database_config                m_config;
  int                                   m_active;
  guint                                 m_tag_signal_handle;
};

static void
tag_comboboxtext_cb(
  GtkWidget *                           i_widget,
  gpointer                              i_data);

void
user_function (GtkButton *button,
               gpointer   user_data)
{


}

static void
database_property_bind(
  struct database_dialog*const          o_db,
  GtkBuilder*const                      i_builder)
{

  do
  {

    (*o_db).m_tag_entry= GTK_ENTRY(
      gtk_builder_get_object(i_builder, "tag_entry"));

    (*o_db).m_tag_comboboxtext= GTK_COMBO_BOX_TEXT(
      gtk_builder_get_object(i_builder, "tag_comboboxtext"));

    (*o_db).m_tag_signal_handle= g_signal_connect(
      (*o_db).m_tag_comboboxtext,
      "changed",
      G_CALLBACK(tag_comboboxtext_cb),
      o_db);

    (*o_db).m_driver_comboboxtext= GTK_COMBO_BOX_TEXT(
      gtk_builder_get_object(i_builder, "driver_comboboxtext"));

    (*o_db).m_database_entry= GTK_ENTRY(
      gtk_builder_get_object(i_builder, "database_entry"));

    (*o_db).m_hostname_entry= GTK_ENTRY(
      gtk_builder_get_object(i_builder, "hostname_entry"));

    (*o_db).m_ipaddress_entry= GTK_ENTRY(
      gtk_builder_get_object(i_builder, "ip_entry"));

    (*o_db).m_port_entry= GTK_ENTRY(
      gtk_builder_get_object(i_builder, "port_entry"));

    (*o_db).m_username_entry= GTK_ENTRY(
      gtk_builder_get_object(i_builder, "username_entry"));

    (*o_db).m_prompt_checkbutton= GTK_CHECK_BUTTON(
      gtk_builder_get_object(i_builder, "promptpassword_checkbutton"));

    (*o_db).m_test_button= GTK_BUTTON(
      gtk_builder_get_object(i_builder, "test_button"));

    (*o_db).m_new_button= GTK_BUTTON(
      gtk_builder_get_object(i_builder, "new_button"));

    (*o_db).m_save_button= GTK_BUTTON(
      gtk_builder_get_object(i_builder, "save_button"));

    (*o_db).m_delete_button= GTK_BUTTON(
      gtk_builder_get_object(i_builder, "delete_button"));

  }while(0);

  return;
}

static void
database_property_copy_values(
  struct database_aspect *const         o_aspect,
  struct database_dialog const*const    i_db)
{
  gchar const*                          l_text;

  memset(o_aspect, 0, sizeof(*o_aspect));

  l_text= gtk_entry_get_text((*i_db).m_tag_entry);
  if (l_text)
  {
    strncpy((*o_aspect).m_tag, l_text, (sizeof((*o_aspect).m_tag)-1));
  }

  l_text= gtk_combo_box_text_get_active_text((*i_db).m_driver_comboboxtext);
  if (l_text)
  {
    strncpy((*o_aspect).m_driver, l_text, (sizeof((*o_aspect).m_driver)-1));
  }

  l_text= gtk_entry_get_text((*i_db).m_database_entry);
  if (l_text)
  {
    strncpy((*o_aspect).m_dbname, l_text, (sizeof((*o_aspect).m_dbname)-1));
  }

  l_text= gtk_entry_get_text((*i_db).m_hostname_entry);
  if (l_text)
  {
    strncpy((*o_aspect).m_host, l_text, (sizeof((*o_aspect).m_host)-1));
  }

  l_text= gtk_entry_get_text((*i_db).m_ipaddress_entry);
  if (l_text)
  {
    strncpy((*o_aspect).m_hostaddr, l_text, (sizeof((*o_aspect).m_hostaddr)-1));
  }

  l_text= gtk_entry_get_text((*i_db).m_port_entry);
  if (l_text)
  {
    strncpy((*o_aspect).m_port, l_text, (sizeof((*o_aspect).m_port)-1));
  }

  l_text= gtk_entry_get_text((*i_db).m_username_entry);
  if (l_text)
  {
    strncpy((*o_aspect).m_username, l_text, (sizeof((*o_aspect).m_username)-1));
  }

  (*o_aspect).m_prompt_password=
    gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(
      (*i_db).m_prompt_checkbutton));

  return;
}

static void
database_property_clear_values(
  struct database_dialog*const          io_db)
{
  GtkEntryBuffer*                       l_ebuf;

  l_ebuf= gtk_entry_get_buffer((*io_db).m_tag_entry);
  gtk_entry_buffer_delete_text(l_ebuf, 0, -1);
  gtk_combo_box_set_active(GTK_COMBO_BOX((*io_db).m_tag_comboboxtext), -1);
  gtk_combo_box_set_active(GTK_COMBO_BOX((*io_db).m_driver_comboboxtext), -1);
  l_ebuf= gtk_entry_get_buffer((*io_db).m_database_entry);
  gtk_entry_buffer_delete_text(l_ebuf, 0, -1);
  l_ebuf= gtk_entry_get_buffer((*io_db).m_hostname_entry);
  gtk_entry_buffer_delete_text(l_ebuf, 0, -1);
  l_ebuf= gtk_entry_get_buffer((*io_db).m_ipaddress_entry);
  gtk_entry_buffer_delete_text(l_ebuf, 0, -1);
  l_ebuf= gtk_entry_get_buffer((*io_db).m_port_entry);
  gtk_entry_buffer_delete_text(l_ebuf, 0, -1);
  l_ebuf= gtk_entry_get_buffer((*io_db).m_username_entry);
  gtk_entry_buffer_delete_text(l_ebuf, 0, -1);
  gtk_toggle_button_set_active(
    GTK_TOGGLE_BUTTON((*io_db).m_prompt_checkbutton),
    1);

  return;
}

static void
database_property_tag_load(
  struct database_dialog *const         io_db)
{
  int                                   l_slot;

  g_signal_handler_block(
    (*io_db).m_tag_comboboxtext,
    (*io_db).m_tag_signal_handle);

  gtk_combo_box_text_remove_all((*io_db).m_tag_comboboxtext);

  for(l_slot= 0; (*io_db).m_config.m_count > l_slot; l_slot++)
  {
    gtk_combo_box_text_append_text(
      (*io_db).m_tag_comboboxtext,
      (*io_db).m_config.m_aspect[l_slot].m_tag);
  }

  gtk_combo_box_set_active(
    GTK_COMBO_BOX((*io_db).m_tag_comboboxtext),
    (*io_db).m_active);

  g_signal_handler_unblock(
    (*io_db).m_tag_comboboxtext,
    (*io_db).m_tag_signal_handle);

  return;
}

static void
database_property_delete(
  struct database_dialog *const         io_db)
{

  do
  {

    if (-1 == (*io_db).m_active)
    {
      break;
    }

    database_config_delete_group(&(*io_db).m_config, (*io_db).m_active);
    (*io_db).m_active= -1;
    database_property_tag_load(io_db);

  }while(0);

  database_property_clear_values(io_db);

  return;
}

static void
database_property_set_values2(
  struct database_dialog*const          io_db)
{
  struct database_aspect const*         l_aspect;
  int                                   l_rc;
  unsigned                              l_slot;

  do
  {

    if (-1 == (*io_db).m_active)
    {
      break;
    }

    l_aspect= &(*io_db).m_config.m_aspect[(*io_db).m_active];

    gtk_entry_set_text((*io_db).m_tag_entry, (*l_aspect).m_tag);

    gtk_combo_box_set_active(
      GTK_COMBO_BOX((*io_db).m_tag_comboboxtext),
      (*io_db).m_active);

    gtk_combo_box_set_active(
      GTK_COMBO_BOX((*io_db).m_driver_comboboxtext),
      -1);

    for(l_slot= 0; (*io_db).m_driver.m_count > l_slot; l_slot++)
    {
      l_rc= strcmp(
        (*io_db).m_driver.m_module[l_slot].m_tag,
        (*l_aspect).m_driver);

      if (0 == l_rc)
      {
        gtk_combo_box_set_active(
          GTK_COMBO_BOX((*io_db).m_driver_comboboxtext),
          l_slot);
      }
    }

    gtk_entry_set_text(
      (*io_db).m_database_entry, 
      (*l_aspect).m_dbname);

    gtk_entry_set_text(
      (*io_db).m_hostname_entry,
      (*l_aspect).m_host);

    gtk_entry_set_text(
      (*io_db).m_ipaddress_entry,
      (*l_aspect).m_hostaddr);

    gtk_entry_set_text(
      (*io_db).m_port_entry,
      (*l_aspect).m_port);

    gtk_entry_set_text(
      (*io_db).m_username_entry,
      (*l_aspect).m_username);
    
    gtk_toggle_button_set_active(
      GTK_TOGGLE_BUTTON((*io_db).m_prompt_checkbutton),
      (*l_aspect).m_prompt_password);

  }while(0);

  return;
}

static void
database_property_set_values(
  struct database_dialog*const          io_db)
{
  unsigned                              l_slot;

  database_property_tag_load(io_db);

  gtk_combo_box_text_remove_all((*io_db).m_driver_comboboxtext);

  for(l_slot= 0; (*io_db).m_driver.m_count > l_slot; l_slot++)
  {
    gtk_combo_box_text_append_text(
      (*io_db).m_driver_comboboxtext,
      (*io_db).m_driver.m_module[l_slot].m_tag);
  }

  database_property_set_values2(io_db);

  return;
}

static void
database_property_save(
  struct database_aspect *const         io_aspect,
  struct database_dialog *const         io_db)
{
  struct database_aspect                l_aspect;
  int                                   l_rc;
  unsigned                              l_slot;

  do
  {

    memset(&l_aspect, 0, sizeof(l_aspect));
    database_property_copy_values(&l_aspect, io_db);
    (*io_db).m_active= -1;

    if (0 == l_aspect.m_tag[0])
    {
      g_strlcpy(l_aspect.m_tag, DATABASE_CONFIG_DEFAULT_TAG, sizeof(l_aspect.m_tag));
      gtk_entry_set_text((*io_db).m_tag_entry, l_aspect.m_tag);
    }

    for (l_slot= 0; (*io_db).m_config.m_count > l_slot; l_slot++)
    {
      l_rc= strcmp((*io_db).m_config.m_aspect[l_slot].m_tag, l_aspect.m_tag);
      if (0 == l_rc)
      {
        (*io_db).m_active= l_slot;
        break;
      }
    }

    if (-1 == (*io_db).m_active)
    {
      (*io_db).m_config.m_aspect= (struct database_aspect*)
        g_realloc(
          (*io_db).m_config.m_aspect, 
          (1+(*io_db).m_config.m_count)*sizeof(struct database_aspect));
      memset(
        &(*io_db).m_config.m_aspect[(*io_db).m_config.m_count],
        0,
        sizeof(struct database_aspect));
      (*io_db).m_active= (*io_db).m_config.m_count;
      (*io_db).m_config.m_count++;
    }

    memcpy(
      &(*io_db).m_config.m_aspect[(*io_db).m_active],
      &l_aspect,
      sizeof(l_aspect));

    memcpy(io_aspect, &l_aspect, sizeof(l_aspect));

    database_config_save_group(&(*io_db).m_config, (*io_db).m_active);

    database_property_tag_load(io_db);

  }while(0);

  return;
}

static void
database_property_test(
  struct database_dialog const*const    i_db)
{
  struct database_aspect                l_aspect;
  GtkDialog*                            l_dialog;
  GError*                               l_error;
  int                                   l_rc;
  struct database_session               l_session;

  l_error= 0;
  database_aspect_assign(&l_aspect);
  database_session_assign(&l_session);

  do
  {

    database_property_copy_values(&l_aspect, i_db);
    database_aspect_password_fill(&l_rc, &l_aspect);

    if (l_rc)
    {
      break;
    }

    l_rc= database_session_connect(&l_error, &l_session, &l_aspect);

    if (l_rc)
    {
      break;
    }

    l_dialog= GTK_DIALOG(gtk_message_dialog_new(
      get_main_window(),
      GTK_DIALOG_MODAL,
      GTK_MESSAGE_INFO,
      GTK_BUTTONS_OK,
      "Connection successfull"));
    gtk_dialog_run(l_dialog);
    gtk_widget_destroy(GTK_WIDGET(l_dialog));

  }while(0);

  if (l_error)
  {
    l_dialog= GTK_DIALOG(gtk_message_dialog_new(
      get_main_window(),
      GTK_DIALOG_MODAL,
      GTK_MESSAGE_ERROR,
      GTK_BUTTONS_CLOSE,
      (*l_error).message, 
      0));
    gtk_dialog_run(l_dialog);
    gtk_widget_destroy(GTK_WIDGET(l_dialog));
  }

  g_clear_error(&l_error);

  database_aspect_discharge(&l_aspect);
  database_session_discharge(&l_session);

  return;
}

static void
tag_comboboxtext_cb(
  GtkWidget *                           i_widget,
  gpointer                              i_data)
{
  struct database_dialog *              l_db;

  l_db= (struct database_dialog*)i_data;
  (*l_db).m_active= gtk_combo_box_get_active(GTK_COMBO_BOX(i_widget));
  database_property_set_values2(l_db);

  return;
}

extern int
database_aspect_property(
  struct database_aspect *const         io_aspect)
{
  GtkBuilder*                           l_builder;
  GtkDialog*                            l_dialog;
  GError*                               l_error;
  int                                   l_exit;
  struct database_dialog                l_db;
  int                                   l_rc;
  unsigned                              l_slot;
  int                                   l_found;

  l_builder= 0;
  l_dialog= 0;
  l_error= 0;
  l_exit= 0;

  memset(&l_db, 0, sizeof(l_db));
  database_config_assign(&l_db.m_config);
  l_db.m_active= -1;
  driver_config_assign(&l_db.m_driver);

  do
  {

    l_builder= gtk_builder_new();

    l_rc= wrap_gtk_builder_add_from_file(l_builder, "database.glade", &l_error);

    if (0 == l_rc)
    {
      l_exit= -1;
      break;
    }
  
    l_dialog= (GtkDialog*)gtk_builder_get_object(l_builder, "database_dialog");

    if (0 == l_dialog)
    {
      l_error= g_error_new(
        GENERAL,
        GENERIC,
        "Unable to find dialog object: 'database_dialog'");
      l_exit= -1;
      break;
    }

    gtk_window_set_transient_for(GTK_WINDOW(l_dialog), get_main_window());

    if (0 == (*io_aspect).m_tag[0])
    {
      g_strlcpy((*io_aspect).m_tag, DATABASE_CONFIG_DEFAULT_TAG, sizeof((*io_aspect).m_tag));
    }

    l_found= 0;
    database_config_load(&l_db.m_config);

    if (0 == (*io_aspect).m_tag[0])
    {
      g_strlcpy((*io_aspect).m_tag, DATABASE_CONFIG_DEFAULT_TAG, sizeof((*io_aspect).m_tag));
    }

    for (l_slot= 0; l_db.m_config.m_count > l_slot; l_slot++)
    {
      l_rc= strcmp(
        l_db.m_config.m_aspect[l_slot].m_tag,
        (*io_aspect).m_tag);
      if (0 == l_rc)
      {
        memcpy(
          &l_db.m_config.m_aspect[l_slot], 
          io_aspect,
          sizeof(*io_aspect));
        l_found= 1;
        l_db.m_active= l_slot;
        break;
      }
    }

    if (0 == l_found)
    {
      l_db.m_config.m_aspect= (struct database_aspect*)
        g_realloc(
          l_db.m_config.m_aspect, 
          (1+l_db.m_config.m_count)*sizeof(struct database_aspect));
      memcpy(
        &l_db.m_config.m_aspect[l_db.m_config.m_count], 
        io_aspect,
        sizeof(*io_aspect));
      l_db.m_active= l_db.m_config.m_count;
      l_db.m_config.m_count++;
    }

    l_exit= driver_config_load(&l_error, &l_db.m_driver);

    if (l_exit)
    {
      _error_display_prompt(l_error);
      g_clear_error(&l_error);
      l_error= 0;
      l_exit= 0; 
    }

    database_property_bind(&l_db, l_builder);
    database_property_set_values(&l_db);

    gtk_widget_show_all(GTK_WIDGET(l_dialog));
    gtk_window_set_modal(GTK_WINDOW(l_dialog), 1);

    do
    {

      l_exit= gtk_dialog_run(GTK_DIALOG(l_dialog));

      if (GTK_RESPONSE_DELETE_EVENT == l_exit)
      {
        break;
      }

      if (GTK_RESPONSE_CANCEL == l_exit)
      {
        break;
      }

      if (GTK_RESPONSE_OK == l_exit) 
      {
        database_property_save(io_aspect, &l_db);
        break;
      }

      switch(l_exit)
      {
        case RESPONSE_TEST:
          database_property_test(&l_db);
          break;
        case RESPONSE_NEW:
          database_property_save(io_aspect, &l_db);
          database_property_clear_values(&l_db);
          l_db.m_active= -1;
          memset(io_aspect, 0, sizeof(*io_aspect));
          break;
        case RESPONSE_DELETE:
          database_property_delete(&l_db);
          memset(io_aspect, 0, sizeof(*io_aspect));
          break;
        default:
          break;
      }

    }while(1);

  }while(0);

  driver_config_discharge(&l_db.m_driver);
  database_config_discharge(&l_db.m_config);

  if (l_dialog)
  {
    gtk_widget_destroy((GtkWidget*)l_dialog);
  }

  if (l_builder)
  {
    g_object_unref(l_builder);
  }

  if (l_error)
  {
    _error_log(l_error);
    _error_display_prompt(l_error);
  }

  g_clear_error(&l_error);

  return l_exit;
}
